# @dcx/assets

## Introduction to the High Level API (HLA) and Leaf Functions

The assets module contains a High Level API (HLA) in the form of an Object Model along with utility classes that provide a convenient to use interface for accessing the Adobe Cloud Platform. The HLA includes object model classes for [Asset](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html), [Directory](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.directory.html), [File](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html), [Composite](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html) and [Version](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.version.html). 

> Note: There are no plans to support the HLA with the deprecated CC Storage API.

The Object Model classes contain instance methods that mirror the functionality of the underlying storage repository. The core implementation of these instance methods exist in pure functions we call *leaf functions*. These [leaf functions](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#linked) are name exported from the assets module and shared between classes in the Object Model and the [Repository Session](../repo-api-session). Leaf functions are also pure, so there are no observable side effects to the input parameters after execution. 

For methods that make HTTP requests, regardless of whether you use the HLA or a leaf function, you will always get back an AdobePromise which will resolve to either an `AdobeRepoResponse` or (when no useful additional data is concerned) an `AdobeResponse`. The `AdobeRepoResponse` object contains both the [Response object](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_http.adoberesponse.html) from the request (response headers, body, status code etc) as well as a result object which is a generic that in the case of the HLA will be typed depending on the API being called.

```ts
interface AdobeRepoResponse<T> {
    response: AdobeResponse;
    result: T
}
```

There are three difference between using leaf methods and the HLA instance methods

1. Leaf methods will only deserialize HTTP response bodies into simple data objects, like `AdobeAsset`. This is in contrast to the HLA which will deserialize the response body into an HLA class instance that corresponds to the API being called, like `Asset`.
2. The HLA will perform side effects to the asset on which a method is called. For example, calling `getRepoMetadata()` on an instance of a HLA class will hydrate that instance with repository metadata and links. Leaf methods on the other hand are pure and will not hydrate the input asset with the response.
3. Leaf methods assume any AdobeAsset supplied as an argument contains all links required to perform the request, if that link does not exist, leaf methods will throw an `INVALID_PARAMS` error. This differs from the HLA which will perform a head request for you to retrieve the links required for a request if they are missing from the asset.

### HLA Object Model Example

```ts
const { File } = require('@dcx/assets');
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:dc5f83c0-d5e3-4dd5-9b93-265878473100'
};
const sampleJPG = new File(asset, service);

const {
    response,
    result,
} = await sampleJPG.getRepoMetadata();

```

In the example above, the value of `response` would contain both the [Response object](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_http.adoberesponse.html) as well as the result object containing the deserialized repository metadata. I.E the raw response from the repository has been transformed into a class type of Asset, so instead of `repo:name` the name property is just `name`. The `sampleJPG` object is also now hydrated with it's repository metadata and links.

### Leaf Function Example

```ts
const { getRepoMetadata } = require('@dcx/assets');
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:dc5f83c0-d5e3-4dd5-9b93-265878473100'
};

//Leaf functions require the asset to contain the links they need to perform the action
const assetLinks = await getLinksForAsset(service, asset);
asset.links = assetLinks;

//Now that our asset has it's links we can it with the getRepoMetadata leaf method
const {
    response,
    result,
} = await getRepoMetadata(service, asset);
```

In the example above, the `response` object would be an instance of [AdobeResponse](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_http.adoberesponse.html) while the `result` object would contain the raw response body from the http request. Unlike the HLA example above, leaf methods never mutate the data passed in and so the asset object would not change.

We expect most clients will not use the leaf functions and instead will work directly with the HLA Object Model or the [Repository Session](/packages/repo-api-session). 

### When to use leaf functions?

If a client application is looking to add the smallest footprint possible and only requires a very small, focused subset of functionality from the repository API, working with leaf functions is suggested.

## Getting started

### Creating an HTTP Service

Both the HLA and the leaf functions require an instance of the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) class in order to make requests. 

Unlike working with the [@dcx/dcx-js](/packages/dcx-js) module and the [AdobeDCXCompositeXfer](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_dcx_js.adobedcxcompositexfer-1.html) transfer class, you do not need a Repository API Session instance to work with the HLA Module. All that is required is an instance of the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) class that is configured with the endpoint of the Storage Repository. 

```ts
const { createHTTPService } = require('@dcx/http');

const server = 'https://platform-cs-stage.adobe.io';

// Create a service instance providing the endpoint of the server.
const service = createHTTPService(async function (service) {
    // get a new token asynchronously
    const token = await myGetToken();
    service.setAuthToken(token);
}, { server });

// OR, using callbacks
const service = createHTTPService(function (service) {
    myGetToken(function (token) {
        service.setAuthToken(token);
    });
}, { server });

service.setApiKey('MyApiKey');
service.setAdditionalHeaders( {'foo': 'bar'} );
```

Refer to [Authentication](/packages/http/README.md#authentication) for more details on creating an HTTP Service.
For more details on configuration options, see [@dcx/http](/packages/http/README.md).

### Return type

The returned objects for APIs defined in the `@dcx/assets` module follow a few rules:

1. If the API is asynchronous and may be cancelable, `AdobePromise` is returned.
2. If the API is asynchronous and is definitely not cancelable, `Promise` may be returned.
3. APIs that make requests to the Platform are always asynchronous and cancelable. These APIs will typically resolve to an object containing a `response` key ([`AdobeResponse`](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_http.adoberesponse.html) type)), and other keys depending on the API.
4. If an API returns some data object that conforms to `AdobeAsset`, the data will be tranformed to the `AdobeAsset` type. The original response object containing, for example, `repo:` prefixed properties, will be available in the `response` object.
5. Same as above, but for HLA classes; `AdobeAsset` will often be replaced with `Asset` instances. This may not occur in cases where there could be significant overhead in creating those classes.

### Working with Links

In accordance with the the [Hypertext Application Language (HAL)](https://tools.ietf.org/id/draft-kelly-json-hal-01.html) convention, the APIs aligned with the Repository API standard rely on link relations to help users identify the dynamic URLs found in the HTTP responses. Using the HLA abstracts all the intricacies of working with links in the repository API. If you plan to work with the leaf methods then some utility classes are provided to aid in working with links.

### Fetching Links of an Asset

Fetching asset links can be done with the [fetchLinksIfMissing](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#fetchlinksifmissing) leaf method.

```ts
const { fetchLinksIfMissing, LinkRelation } = require('@dcx/assets');
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:dc5f83c0-d5e3-4dd5-9b93-265878473100'
};

const links = await fetchLinksIfMissing(service, asset, [LinkRelation.REPO_METADATA]);
```

The [fetchLinksIfMissing](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#fetchlinksifmissing) method will first check the asset object itself for the links, if not found it will check the optionally configured links cache. If the links are not found in either location a head operation is performed on the asset and the links are returned.

The links will be returned as a LinkSet which is a Dictionary of [LinkRelation keys](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_common_types.html#linkrelationkey-1) and Link objects. The @dcx/util module provides utility methods for working with templated and untemplated HAL Links.

For non templated links use [getLinkHref(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_util.html#getlinkhref)

```ts
import { getLinkHref, getLinkHrefTemplated } from '@dcx/util';
const repoMetadataHref = getLinkHref(links, LinkRelation.REPO_METADATA);
```

For templated links use [getLinkHrefTemplated(...)](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_util.html#getlinkhreftemplated)

```ts
import { getLinkHref, getLinkHrefTemplated } from '@dcx/util';
const renditionOpts: RenditionOptions = {
    size: 512
};
const renditionHref = getLinkHrefTemplated(asset.links, LinkRelation.RENDITION, renditionOpts);
```

### Resolving an Asset

To unconditionally resolve an asset, you can use the [`resolveAsset()`](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#resolveAsset) leaf method. This can be done using any resolvable asset object:

1. Object with an `assetId` property

>[!NOTE]
>
> **Note:** Although not required, it's recommended to provide the `repositoryId` along with assetId for caching and performance reasons.

2. Object with a `path` and `repositoryId`

3. Object with a `links` property containing at least one of ID, Path, RepoMetadata, or Primary links.

>[!NOTE]
>
>**Note:** Primary link only works for assets that have a primary resource; this may not be the case for composite assets without a manifest.

Once an asset is resolved or contains the links required, clients are free to use leaf methods or instantiate HLA class instances. Note that HLA class instances may be created without resolving an asset, but care should be taken to avoid creating the wrong class. To use the AssetFactory it's recommended to resolve the asset first to properly set a format, or manually set the format.

### Caching Links

The assets module provides a [RepositoryLinksCache](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_assets.repositorylinkscache.html) class that can be used to cache links returned in the header of any response. This caching comes out of the box if you use the [Repository Session](/packages/repo-api-session) but must be configured if you are only using the HLA.

The constructor for all HLA classes require either an instance of the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) class or a ServiceConfig object. The ServiceConfig interface simply just defines the service and the an instance of a RepositoryLinksCache. If you would like to take advantage of caching in the HLA you will need to provide a ServiceConfig object to HLA class constructors instead of an [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) instance.

```ts

import { RepositoryLinksCache, ServiceConfig } from '@dcx/assets';

const server = 'https://platform-cs-stage-va6.adobe.io';
const service = createHTTPService(authHandler, { server });

const cache = new RepositoryLinksCache();
const serviceConfig: ServiceConfig = {
    service: service,
    cache: cache
};

const asset = new Asset(TEST_DEFAULT_ASSET, serviceConfig);
```

### Working with PageResources

Page Resources provide page-at-a-time access to other Resources, specifically Resources that enumerate items (i.e., Directories and Versions Resources, as well as the Discoverable Assets Document and Discoverable Repositories Document).

Pagination is supported out of the box in the HLA for any Resource that includes a page link relation. 

For example, the `getPagedVersions()` method on the File class returns a PageResource object which can be used to iterate through the pages.

```ts
const asset = {
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg',
    assetId: 'urn:aaid:sc:US:ab5f22c0-d5d3-4dd5-9b93-2652378473583'
};
const asset = new File(asset, service);

const {
    response,
    paged,
} = await asset.getPagedVersions();
```

Use `hasNextPage()` to check if another page exists.

```ts
pageResource.hasNextPage();
```

To load the next page call `getNextPage()`

```ts
const secondPage = await pageResource.getNextPage();
```

[Page options](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_assets.pageoptions.html) can also be used to filter and sort the paged results.

```ts
const opts = { limit: 2 };
pageResource.getPage(opts: PageOptions)
```

Some operations support the ability to embed additional resources in the response. Currently, only the discoverable assets & directory listing API's support this functionality. The discoverable assets API supports embedding both Effective Privileges and Repository resources while the directory listing API just supports Effective Privileges.

```ts
const discoverableAssets = await getDiscoverableAssets(service, {
    embed: [LinkRelation.EFFECTIVE_PRIVILAGES, LinkRelation.REPOSITORY]
});
```

### The Index Repository

The entry point for all clients of the Repository API is the Index Repository. The index repository provides clients a resource to discover all of the other Repositories that they have access to. This Repository also provides clients with a list of Discoverable Assets and other information that's needed to interact with the Storage Repository API.

Resources that are part of the Index Repository are exposed via Leaf Methods only.

#### Discoverable Assets

The [Discoverable Assets Document](https://developers.corp.adobe.com/index-repository-api/docs/discoverable-assets-document.md) lists the set of Discoverable Assets for the requesting user.

[getDiscoverableAssets](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getdiscoverableassets) can be used to get a paged resource of Discoverable Assets.

```ts
const { getDiscoverableAssets } = require('@dcx/assets');
const pageOpts = { limit: 3 }
const { paged } = await getDiscoverableAssets(service, pageOpts);

//Get Next page
await paged.getNextPage();
```

#### Discoverable Repositories

The [Discoverable Repositories Document](https://developers.corp.adobe.com/index-repository-api/docs/discoverable-repositories-document.md) is a Repository that contains, for the requesting user, at least one Discoverable Asset. The Discoverable Repositories Document enumerates the list of Discoverable Repositories and embeds the Repository Resource for each Repository.

[getDiscoverableRepos](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getdiscoverablerepos) can be used to get a paged resource of Discoverable Repositories.

```ts
const { getDiscoverableRepos } = require('@dcx/assets');
const pageOpts = { limit: 1 }
const { paged } = await getDiscoverableRepos(service, pageOpts);

//Get Next page
await paged.getNextPage();
```


### Asset vs. AdobeAsset vs. AdobeAssetEmbedded

`Asset` is a class, the basis of the HLA (see below) which contains properties and convenience APIs. 

`AdobeAsset` is simple data object conforming to the AdobeAsset interface; it is a subset of the `Asset` class, containing only properties. The properties of AdobeAsset are all optional, and each maps to a field in the underlying cloud resource's Repository Metadata. These fields differ from the response object in that they don't have namespace prefixes like `repo:` and `storage:`

To convert from a data object from an API call to an AdobeAsset, the `deserializeAsset(...)` utility method can be used. This method does not mutate the incoming object, instead it returns a new object.

`AdobeAssetEmbedded` is another subset of AdobeAsset; one with strong typing for embedded properties. This type is returned from some APIs, but it's also possible for the object to conform to AdobeAssetEmbedded while not being typed as such. Much of these situations depend on generic types, and much of it is handled implicitly, but some things can only be determined at runtime. For example, `deserializeAsset(...)` will return an AdobeAssetEmbedded conforming object, if the input data contains the `_embedded` key.

Anywhere `AdobeAsset` is an argument will also accept `AdobeAssetEmbedded`. The same is true for `Asset` instances; however, some benefits of the HLA classes are lost if using them as arguments in this way. For example, an Asset could be provided to the leaf method `getRepoMetadata(...)`, but the asset's properties (etag, md5, etc.) will not be updated.

### Creating instances of HLA Assets

There are 2 ways to create instances of Asset (or it's subclasses).

1. Using the constructor
   
```ts
import { Asset, Directory, File } from '@dcx/assets';
const asset = new Asset({ repositoryId, assetId }, service);
const dir = new Directory({ repositoryId, assetId }, service);
const file = new File({ repositoryId, assetId }, service);
```

This method provides no safety or guarantees about the asset type. For example, it's possible to create an instance of `File` using asset data that should be a `Directory` type. Clients are recommended to check the `format` property of an asset before instantiating specific types, or only use with assets the type is certain. 

2. Using the AssetFactory
   
```ts
import { AssetFactory, hydrateAsset, DirectoryMediaType } from '@dcx/assets';
const asset = hydrateAsset({ repositoryId, assetId }, service);
// asset is type Asset
// OR
const factory = new AssetFactory(service);
const dir = factory.hydrate({ repositoryId, assetId, format: DirectoryMediaType });
// dir is type Directory
```

The AssetFactory switches on the input asset's `format` property. If no format is defined, it will return the base `Asset` type. Additionally, since the AssetFactory can create any subclass of Asset at runtime, the impact to your bundle size may be unnecessarily large, depending on the APIs required.

An Asset instance may also be "upgraded" after the fact, by using the AssetFactory. For example:

```ts
import { AssetFactory } from '@dcx/assets';
const factory = new AssetFactory(service);
const asset = factory.hydrate({ repositoryId, assetId });
await asset.getRepoMetadata(); // assumes assetId points to a directory class
// asset is type Asset
const dir = factory.hydrate(asset);
```

If it's current subclass already matches the most specific subclass possible, it will return **the same instance**, it will not duplicate or mutate the object. This is useful for cases where you would like to get the correct type after making some API calls that would determine the asset's format.

## Working with the Asset Class

At the core of the Assets module is the [Asset class](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html) and the [Asset Interface](file:///Users/dylan/Documents/Code/git.corp.adobe.com/dcx-js/docgen/interfaces/_dcx_common_types.adobeasset-1.html), they are the base types that other model classes like [Directory](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.directory.html), [File](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html), [Composite](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html) and [Version](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.version.html) either directly or indirectly inherit from.

Below are some examples of the functionality currently supported on the Asset class. 

|Method|High Level API|Leaf method|
|-|-|-|
|[headPrimaryResource](#docHeadPrimaryResource)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#headprimaryresource)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#headprimaryresource)|
|[getPrimaryResource](#docGetPrimaryResource)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#getprimaryresource)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getprimaryresource)|
|[getRepoMetadata](#docGetRepoMetadata)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#getrepometadata)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getrepometadata)|
|[getAppMetadata](#docGetAppMetadata)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#getappmetadata) |[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getappmetadata)|
|[putAppMetadata](#docPutAppMetadata)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#putappmetadata)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#putappmetadata)|
|[patchAppMetadata](#docPatchAppMetadata)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#patchappmetadata)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#patchappmetadata)|
|[getEffectivePrivileges](#docGetEffectivePrivileges)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#geteffectiveprivileges)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#geteffectiveprivileges)|
|[getACLPolicy](#docGetACLPolicy)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#getaclpolicy)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getaclpolicy)|
|[checkACLPrivilege](#docCheckACLPrivilege)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#checkaclprivilege)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#checkaclprivilege)|
|[patchACLPolicy](#docPatchACLPolicy)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#patchaclpolicy)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#patchaclpolicy)|
|[deleteACLPolicy](#docDeleteACLPolicy)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#deleteaclpolicy)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#deleteaclpolicy)|


### Primary Resource

The role and function of the [Primary Resource](https://developers.corp.adobe.com/storage-api/docs/reference/primary-landing.md) varies between different Asset classes. For Files, the Primary Resource contains the data associated with the File. For Directories, the Primary Resource contains a JSON representation of the Directory contents. For Composites, the Primary Resource contains a representation of the Composite as a DCX Snapshot.

Likewise, the availability of read and write APIs for the Primary Resource differs. `File#updatePrimaryResource()` is, expectedly, only available on the `File` class.

#### Head Primary Resource

Performs a HEAD operation on the Asset

```ts
const { result } = await asset.headPrimaryResource();
```

#### Get Primary Resource

Get Primary resource with expected response type of 'json'.

```ts
const { result } = await directoryAsset.getPrimaryResource('json');
```

Get Primary resource with expected response type of 'buffer'.

```ts
// defaults to "defaultbuffer"
// buff is type Buffer on Node, ArrayBuffer on browser
const { result } = await fileAsset.getPrimaryResource(); 
```

### Repository Metadata Resource

The [Repository Metadata Resource](https://developers.corp.adobe.com/storage-api/docs/reference/repository-metadata.md) contains metadata about an Asset as maintained by a Repository.

#### Get Repository Metadata

Retrieves the repository metadata resource for the asset

```ts
const { result } = await asset.getRepoMetadata();
```

### Application Metadata

The [Application Metadata Resource](https://developers.corp.adobe.com/storage-api/docs/reference/application-metadata.md#) contains data about the Asset that is generated and maintained by clients of the Repository. The `@dcx/assets` module provides APIs for get, put, and patching application metadata.

#### Get Application Metadata

Returns the application metadata associated with an asset.

```ts
const { result, etag } = await asset.getAppMetadata();
// result is type any
// etag is the fetched app metadata's etag
```

```ts
// OR using strong typed returned object
const { result } = await asset.getAppMetadata<MyCustomInterface>();
// result is type MyCustomInterface
```

#### Put Application Metadata

Creates application metadata for an asset.

```ts
// Conditional update with etag
const newAppMetadata = JSON.stringify({foo: 'bar'});
const { result } = await asset.putAppMetadata(newAppMetadata, etag);
```

```ts
// Unconditional update with object
const newAppMetadata: Record<string, string> = {foo: 'bar'};
const { result } = await asset.putAppMetadata(newAppMetadata);
// OR
const { result } = await asset.putAppMetadata(newAppMetadata, "*");
```

```ts
// Like with getAppMetadata, the input object can be strongly typed with generics
const obj: MyCustomInterface = {...};
const { result } = await asset.putAppMetadata<MyCustomInterface>(obj, "*");
```

#### Patch Application Metadata

Update application metadata associated with the asset.

For methods of creating `JSONPatchDocument`s, [see below](#creating-json-patches).

```ts
const patchDoc: JSONPatchDocument = ...;
const { result } = await asset.patchAppMetadata(patchDoc, etag);
// result.etag is the new app metadata resource's etag
```

## Access Control

### Effective Privileges

The [ACL Effective Resource](https://developers.corp.adobe.com/storage-api/docs/reference/acl-effective.md) returns the effective Privileges the current user has for an Asset. More specifically, it lists which Privileges the user has for each resource associated with the Asset.

#### Get Effective Privileges

Retrieves the effective privileges for the resources of an asset

```ts
const file = new File({ repositoryId, assetId }, service);

const { result } = await file.getEffectivePrivileges();
```

#### Get ACL Policy

Retrieves the ACL resource for the asset.

```ts
const file = new File({ repositoryId, assetId }, service);

const { result } = await file.getACLPolicy();
```

#### Check ACL Privilege

Checks whether the current user has the requested Privilege on the specified Resource of an Asset.

The representation of the ACL Check Resource is not defined by the Repository API Specification. Therefore, users should ignore any representation returned in the response body and just inspect the status code.

Below are the possible status codes and what they signify in relation to the checked Privilege.

* 404 Not Found: Neither ACK nor READ is granted on the target Resource, and the Access Control on the target Resource cannot be checked.
* 204 No Content: The checked Privilege is granted.
* 403 Forbidden: The checked Privilege is not granted.

```ts
const file = new File({ repositoryId, assetId }, service);

const { result } = await file.checkACLPrivilege(
    "write",
    LinkRelation.REPOSITORY
);
```

#### Patch ACL Policy

Issues a Patch operation against an assets ACL Policy resource to merge an `Access Control Entry (ACE)` to the assets ACL.

```ts
const file = new File({ repositoryId, assetId }, service);

const patchPolicy = [{
    op: 'add',
    path: '/repo:acl/-':
    value: {
        'repo:principal': {
            'xdm:provider': {
                '@id': 'https://ims-na1.adobelogin.com//'
            },
            '@id': '67c566904aa96e029920faae@AdobeID',
            '@type': 'https://ns.adobe.com/adobecloudplatform/ims/user'
        },
        'repo:modifier': 'grant',
        'repo:privileges': ['ack', 'read', 'write', 'delete'],
        'repo:relations': ['http://ns.adobe.com/adobecloud/rel/primary'],
        'repo:inheritance': 'deep'
    }
}];

const { result } = await file.patchACLPolicy(patchPolicy);
```

#### Delete ACL Policy

Issues a DELETE operation against an assets ACL Policy removing all ACEs.

```ts
const file = new File({ repositoryId, assetId }, service);

const { response } = await file.deleteACLPolicy();
```

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.assets.html) for a list of all methods on the Asset class.

## Working with Files

The [File](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html) class extends [Asset](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html) and includes extra functionality specific to working with files.

|Method|High Level API|Leaf method|
|-|-|-|
|[updatePrimaryResource](#docUpdatePrimaryResource)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html#updateprimaryresource)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#updateprimaryresource)|
|[getRendition](#docGetRendition)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html#getrendition)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getrendition)|
|[getPagedVersions](#docGetPagedVersions)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html#getpagedversions)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getpagedversions)|
|[getVersionResource](#docGetVersionResource)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html#getversionresource) |[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getversionresource)|
|[patchVersions](#docPatchVersions)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html#patchversions)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#patchversions)|

### Primary Resource

#### Update Primary Resource

Updates the asset's primary resource. If the size is less then `maxSingleTransferSize` the the data can be passed directly to function as in the first two examples below. If the data is larger than `maxSingleTransferSize` then the data must be transferred using block transfer and a callback function to slice the data is provided in place of the data itself.

See [Put Composite Component](docPutCompositeComponent) for more information on Block Transfers.

```ts
//Update a JSON asset
const { response } = await file.updatePrimaryResource(
    JSON.stringify({ some: { object: { with: 'stuff', foo: 123, bar: 'Lorem Ipsum' } } }),
    'application/json'
);

//Update a binary asset w/buffer less than `maxSingleTransferSize` 
const someBuffer = ...;
const { response } = await file.updatePrimaryResource(someBuffer, 'image/jpeg');

//Block upload a large buffer
const someLargeBuffer = ...;
const onSliceBuffer = (startBuf, endBuf): Promise<Buffer> => {
    const slice = someLargeBuffer.slice(startBuf, endBuf);
    return new Promise((resolve) => {
        resolve(slice);
    });
};

const { response } = await file.updatePrimaryResource(
    onSliceBuffer,
    'image/jpeg',
    12300000
);
```

### Renditions

A rendition is a derived representation of an asset or of a part thereof. Renditions are typically lossy—that is, they contain less data than the asset from which they are derived—and represented in a more broadly understood media type. 

The example below will retrieve a rendition of a composite file, but would work the same for any other [media types](https://git.corp.adobe.com/pages/caf/api-spec/chapters/common_resources/renditions_resources.html#media-types) supported by the platform. The [getRendition](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html#getrendition) method accepts [RenditionOptions](https://git.corp.adobe.com/DMA/dcx-js/blob/dev/packages/common-types/src/rendition.ts#L20) to customize the rendition you want returned. You can also provide the [response type](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_http.html#responsetype) you want returned.

#### Get Rendition

```ts
const { File, RenditionType } = require('@dcx/assets');

// Create a simple object representing the asset
const sampleCompositeAsset = new File({ repositoryId, assetId }, service);

// Get a rendition of composite
const { result } = await sampleCompositeAsset.getRendition({ size: 400, type: RenditionType.IMAGE_PNG }, 'defaultbuffer');
// result is type Buffer on Node, ArrayBuffer on browser
```

### Versions

Repositories retain a list of versions of assets. A new version is created when a Content Resource is created or updated in a File or a Composite (Components are an exception to this rule). Directories on the other hand are not versioned and will not contain a version LinkRelation.

The assets module provides a HLA class for working with a version instance ([Version](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.version.html)) as well as a list of versions ([VersionSet](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.versionset.html)).

#### Get Paged Versions

Retrieves a paged list of versions for the asset.

```ts
const pageOpts = { limit: 1 };
const versionsResponse = await asset.getPagedVersions(pageOpts);
const nextPage = versionsResponse.paged.getNextPage();
```

#### Get Version Resource

Retrieves a specific version of the asset.

```ts 
// Get a version of an asset, indexing starts at 0, so below we are asking for the second version of the asset
const { result } = await asset.getVersionResource('1');
```

#### Patch Versions

A version can be patched for purposes of marking it as a milestone version.

```ts
const file = new File({ repositoryId, assetId }, service);

const addPatchDoc = [
    {
        op: 'add',
        path: '/children/version=0/milestone',
        value: {
            label: 'My First Milestone',
            description: 'Label description'
        }
    }
];

const patchResponse = await file.patchVersions(JSON.stringify(addPatchDoc), ':1611932908062');
```

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.file.html) for a list of all methods on the File class.

### Working with Composites

The assets module contains a [Composite class](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html) for working with DCX Composites in the storage repository.

Composite formats organize content as a set of Components bound together with a Manifest. Composites are built on top of the Digital Composite Technology DCX framework, which describes both the organization of the Components as well as the transfer and synchronization algorithms used to operate on them.


|Method|High Level API|Leaf method|
|-|-|-|
|[getComponent](#docGetComponent)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html#getcomponent)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getcompositecomponent)|
|[putComponent](#docPutCompositeComponent)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html#putcomponent)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#putcompositecomponent)|

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html) for a list of all methods on the Composite class.

#### Get Component

Returns a component from a composite

```ts
const composite = new Composite({ repositoryId, assetId }, service);
const componentId = '43a912b8-79e0-4976-9c0e-689dd026e465';
const componentVersion = '3';
const responseType = 'json';

const result = await composite.getComponent(componentId, componentVersion, responseType);
```

#### Put Component 

Components can be directly uploaded by performing a PUT operation on the component resource on the composite asset. This method should only be used if the component size is less then `maxSingleTransferSize` which can be retrieved from the '/rel/block/init' link.

```ts
import { getLinkProperty } from '@dcx/util';
const maxSingleTransferSize = getLinkProperty(
    asset.links,
    LinkRelation.BLOCK_UPLOAD_INIT,
    BlockTransferProperties.MAX_SINGLE_TRANSFER_SIZE
);
```    

To initiate a direct upload to the component resource a string or buffer containing the component data should be passed to the putCompositeComponent method. While the size parameter is optional it is recommended for clients to provide this value if it is known.

If clients pass a buffer or string that is larger then `maxSingleTransferSize` then dcx-js will automatically handle slicing the buffer and performing a block upload of the component. This is not recommended and clients should try and execute the correct mode of transfer upfront.

#### Put Component w/Direct Upload 

```ts
import { generateUuid } from '@dcx/util';
import { Composite } from '@dcx/assets';

const buffer = ...;
const composite = new Composite(asset, service);

const componentId = generateUuid();
const contentType = 'application/json';
const isNew = true;
const size = 63340016;
const componentMD5 = 'Dq6vQLzdarl88fvv5vgKcA==';

const putNewResponse = await composite.putComponent(
    componentId,
    buffer,
    contentType,
    isNew,
    size,
    componentMD5
);
```

#### Put Component w/BlockUpload (Browser)

```ts
import { generateUuid } from '@dcx/util';
import { Composite } from '@dcx/assets';

const composite = new Composite({ repositoryId, assetId }, service);

const links = await fetchLinksIfMissing(service, composite, [
    LinkRelation.COMPONENT,
    LinkRelation.BLOCK_UPLOAD_INIT
]);
composite.setLinks(links);

// On change callback from a file input tag
document.querySelector('#myfile').addEventListener('change', async function() {
    var file = this.files[0];
    var reader = new FileReader();

    const uuid = generateUuid();
    
    const onSliceBuffer = (startBuf, endBuf) => {
        return new Promise((resolve) => {
            var slice = file.slice(startBuf, endBuf);
            reader.onload = async function(buffer) {
                var byteArray = new Int8Array(buffer.target.result);
                resolve(byteArray);
            } 
            reader.readAsArrayBuffer(slice);
        });
    };

    await composite.putComponent(
        uuid,
        onSliceBuffer,
        'video/quicktime',
        true,
        63340016
    )
})
```

#### Put Component (Leaf Method) w/BlockUpload

If a client just needs to push components and wants to keep bundle sizes small they can use the leaf methods powering the HLA to perform a direct or block upload of a component.

```ts
import { generateUuid } from '@dcx/util';
import { putCompositeComponent } from '@dcx/assets';

const composite: AdobeAsset = { repositoryId, assetId };

const links = await fetchLinksIfMissing(service, composite, [
    LinkRelation.COMPONENT,
    LinkRelation.BLOCK_UPLOAD_INIT
]);
composite.links = links;

const buffer = ...;
const exampleGetSliceCallback = (startBuf, endBuf): Promise<any> => {
    const slice = buffer.slice(startBuf, endBuf);
    return new Promise((resolve) => {
        resolve(slice);
    });
};

const componentId = generateUuid();
const contentType = 'video/quicktime';
const isNew = true;
const size = 3340016;

const putNewResponse = await putCompositeComponent(
    service,
    composite,
    componentId,
    onSliceBuffer,
    contentType,
    isNew,
    size
);
```

#### Block Upload Progress Monitoring

Clients can monitor the progress of block uploads by assigning a callback function to the onProgress property. The callback will be called after every block completes. If an incorrect file size is provided and the transferDocument needs to be extended then totalBytes will report the new extended size. The totalBytes value in the final callback to onProgress will represent the total bytes transferred which will likely be less the extended totalBytes value that would have been reported after extending. For this reason we provide an indeterminate flag that clients can use to update the UI accordingly.

```ts
blockUploadPromise.blockUpload.onProgress = (bytesCompleted, totalBytes, indeterminate) => {
   console.log(`${bytesCompleted} / ${totalBytes}`);
};
```

#### Block Upload Pausing & Resuming

Block uploads can be paused and resumed. When a blockUpload is paused, any pending block uploads are allowed to complete but no new uploads will be started while in the paused state. Currently pausing will pause all block uploads requests and not just the BlockUpload instance that is paused. Since our HLA encapsulate both direct upload and block upload workflow. In order to guarantee the blockUpload object presenst in API returned promise object. Here we provide a util function called pauseBlockTransfer to help client wait for blockUpload to be present and then pause it. This Util function return an instance of BlockTransfer that can be used later to resume the block transfer. This Util function also accept a call back function that allow client to monitor the block transfer state change.

```ts
const putComponentPromise = testComposite.putComponent(
    componentId,
    componentBuffer,
    RenditionType.IMAGE_JPG,
    undefined,
    componentBuffer.byteLength,
);

// pause the transfer if blockTransfer exists
const maybePausedBlockTransfer = await pauseBlockTransfer(putCompositePromise);
if (maybePausedBlockTransfer) {
  // resume or store the block transfer for later resumption
  maybePausedBlockTransfer.resume();
}
const result = await putComponentPromise;
```

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.composite.html) for a list of all methods on the Composite class.

## Working with Directories

[Directories](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.directory.html) are assets that provide the ability to organize elements into a hierarchy. Each directory has 0 or more children. Each child is either a file, entity, composite or directory. 

|Method|High Level API|Leaf method|
|-|-|-|
|[getPagedChildren](#docGetPagedChildren)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.directory.html#getpagedchildren)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#getpagedchildren)|

|[createAsset](#docCreateAsset)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.directory.html#createasset)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#createasset)|

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.directory.html) for a list of all methods on the Directory class.

### Get Paged Children

Returns a paged set of results for the directory listing

```ts
// Create an instance of the Directory class
const directory = new Directory({ repositoryId, assetId }, service);

// Create some (Optional) page options
const pageOpts = { limit: 2 };

// Get a paged directory listing
const pageResource = await directory.getPagedChildren(pageOpts);

// Get the next page
const nextPage = await pageResource.paged.getNextPage();
```

### Create Asset

Create an asset relative to the current directory.

```ts
// Must set the contentType to the type of object you want to create
const parent = new Directory({ repositoryId, assetId }, service);
const contentType = 'some/type';
const path = 'demo.test';

const { result } = await directory.createAsset(
    path,
    false, /* createIntermediates */
    contentType,
    undefined /* resourceDesignator */
);
// result is type Asset, to create subclass use AssetFactory or a constructor

// OR, to create an asset inside of a nested directory that may or may not exist
// and return repository metadata along with the newly created asset
const { result } = await directory.createAsset(
    'nested/dir/file.test',
    true, /* createIntermediates */
    contentType,
    LinkRelation.REPO_METADATA /* resourceDesignator */
);
```

## Operations
The following operations are supported as individual leaf methods and as convenience methods on the Asset HLA:

|Operation|High Level API|Leaf method|
|-|-|-|
|[Move](#docMove)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#move)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#moveasset)|
|[Copy](#docCopy)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#copy)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#copyasset)|
|[Package](#docPackage)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#package)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#packageassets)|
|[Discard](#docDiscard)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#discard)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#discardasset)|
|[Restore](#docRestore)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#restore)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#restoreasset)|
|[Delete](#docDelete)|[HLA docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#delete)|[Leaf docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#deleteasset)|

> Note: HLA instances have added syntax sugar on top of leaf methods. For example, destination assets may or may not contain a `repositoryId` (in which case it defaults to the instance's repositoryId). Additionally, a destination may be provided as a string which is treated as either an `assetId` or a `path`, depending on whether the string conforms to the Adobe URN format.

Clients may opt to create their own operation document using either the [`OperationDocumentBuilder` class](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.operationdocumentbuilder.html) or manually with their own logic. This operation document can be supplied directly to the [`doOperation` leaf method](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_assets.html#dooperation) for more complex combinations of operations.

> Note: `OperationDocumentBuilder` provides no additional client-side validation. It's still possible for clients to create invalid combinations of operations (ie. delete and move the same asset).


### Move

```ts
const assetData: AdobeAsset = { repositoryId, assetId };
const asset = new Asset(assetData, service);

// using a path destination, same repositoryId as asset
await asset.move('some/new/path', createIntermediates, overwriteExisting);

// using an asset destination
await asset.move({ path: 'some/new/path', newRepositoryId });
```

### Copy

```ts
const asset = new Asset(assetData, service);

// using a path destination, same repositoryId as asset
await asset.copy('some/new/path', createIntermediates, overwriteExisting);

// using an asset destination
await asset.copy({ path: 'some/new/path', newRepositoryId });
```

### Package

```ts
const asset = new Asset(assetData, service);

// using HLA
await asset.package('some/new/path', createIntermediates, overwriteExisting);

// using leaf method
const { result } = await packageAssets(
    service, 
    [
        { path: 'some/path1', repositoryId }, 
        { path: 'some/path2', repositoryId }
    ], 
    { path: 'some/package.zip', newRepositoryId }, 
    createIntermediates, 
    overwriteExisting
);
// on success, result.success === true
```

### Discard

```ts
const asset = new Asset(assetData, service);

await asset.discard(recursive);
```

### Restore

```ts
const asset = new Asset(assetData, service);

await asset.discard(); // only discarded assets can be restored
await asset.restore();
```

### Delete

```ts
const asset = new Asset(assetData, service);

await asset.delete(recursive);
```

## Bulk Operations

Bulk requests allow users to combine multiple read or write requests into a single HTTP request. Bulk requests return an array of [response objects](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_http.adoberesponse.html) for the sub-requests. There are [restrictions](https://developers.corp.adobe.com/storage-api/docs/reference/bulk-requests.md#restrictions) around what can and cannot be included in a bulk request as well as what order the requests should be placed in. dcx-js will guard against too many sub-requests being included in the bulk request or mixing operations types (READ and WRITE). The rest of the restrictions is up to clients to follow.

### Example Bulk READ

```ts
const { BulkRequestDescriptor, File } = require('@dcx/assets');
const file = new File({ repositoryId, assetId }, service);

const req1: BulkRequestDescriptor = {
    method: 'GET',
    href: getLinkHref(file.links, LinkRelation.EFFECTIVE_PRIVILAGES),
    headers: { accept: 'application/json' }
};

const req2: BulkRequestDescriptor = {
    method: 'GET',
    href: getLinkHref(file.links, LinkRelation.ACL_POLICY)
};

const req3: BulkRequestDescriptor = {
    method: 'GET',
    href: getLinkHref(file.links, LinkRelation.EMBEDDED_METADATA, 'id'),
    headers: { accept: 'application/json' }
};

const {
    result: subRequestResponseArray, 
    response: httpResponse 
} = await file.performBulkRequest([req1, req2, req3]);
```

### Example Bulk WRITE

```ts
const composite = new Composite({ repositoryId, assetId }, service);

const aclPatch = JSON.stringify({
    'repo:acl': [
        {
            'repo:principal': {
                'xdm:provider': {
                    '@id': 'https://ims-na1.adobelogin.com/'
                },
                '@id': '67c566904aa96e029920faae@AdobeID',
                '@type': 'https://ns.adobe.com/adobecloudplatform/ims/user'
            },
            'repo:modifier': 'grant',
            'repo:privileges': ['ack', 'read', 'write', 'delete'],
            'repo:relations': ['http://ns.adobe.com/adobecloud/rel/primary'],
            'repo:inheritance': 'deep'
        }
    ]
});

const req1: BulkRequestDescriptor = {
    method: 'PATCH',
    href: getLinkHref(composite.links, LinkRelation.ACL_POLICY),
    body: aclPatch,
    headers: {
        'content-type': 'application/vnd.adobecloud.accesscontrolpolicy+json',
        'content-length': getDataLength(aclPatch)
    }
};

const xmpPatch = JSON.stringify([
    {
        op: 'add',
        path: '/xmp:CreatorTool',
        value: '13.76'
    }
]);

const req2: BulkRequestDescriptor = {
    method: 'PATCH',
    href: getLinkHref(composite.links, LinkRelation.EMBEDDED_METADATA),
    body: xmpPatch,
    headers: {
        'content-type': 'application/json-patch+json',
        'content-length': getDataLength(xmpPatch),
        'if-match': '*'
    }
};

const res = await composite.performBulkRequest([req1, req2]);
```

See the [docs](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_assets.asset.html#checkaclprivilege) for a list of all the methods on the Asset class.

## Creating JSON Patches

There are 2 ways to create JSON Patch documents in dcx-js.

1. `PatchDocumentBuilder` class from @dcx/assets
Allows adding specific operations to a document, then retrieving that document as a string or object. Useful if you are making unconditional changes to the resource.

```ts
import { createPatchDocumentBuilder } from '@dcx/assets';
import { JSONPatchDocument } from '@dcx/common-types';

const docBuilder = createPatchDocumentBuilder();
docBuilder.addToList('/some/path', 'value', 'end');
docBuilder.copy('/source', '/destination');

const doc: JSONPatchDocument = docBuilder.operations;
// OR
const docString: string = docBuilder.getDocument();
```

2. `createJSONPatch(...)` utility method from @dcx/util
Creates a JSONPatchDocument from two objects, an `input` and an `output`. Applying the returned JSON Patch to `input` will result in `output`. This is useful if you are making changes to the existing resource and have already fetched it. The operations performed to reach the output object is optimized using Levenshtein distance. 

> Note: The operation "move" on Array/Object elements is not supported, meaning arrays or objects that can be optimally reached with a single move will be replaced instead. This situation is relatively rare; if you know this is the outcome at compilation it would be better to use a single operation with the document builder above.

```ts
import { createJSONPatch } from '@dcx/util';
import { JSONPatchDocument } from '@dcx/common-types';

const objBefore = {...};
const objAfter = {...};

const patchDoc: JSONPatchDocument = createJSONPatch(objBefore, objAfter);
```

## Streaming

Download APIs that may return large resources can be used with streaming. Currently dcx-js returns system native readable streams when a `'stream'` `responseType` is requested through the download APIs.  For node.js the stream response type is expected to be an [IncomingMessage](https://nodejs.org/api/http.html#class-httpincomingmessage) and for the browser, it is expected to be a [ReadableStream](https://developer.mozilla.org/en-US/docs/Web/API/ReadableStream).  In the future,  we hope to shift all streams over to WHATWG ReadableStream, to align on a single stream implementation everywhere.

In order to assist with managing the differing stream types currently exported by dcx-js a few narrowing functions are provided:

```ts
import {isNodeReadableStream} from '@dcx-js/util'
const {response} = await session.getCompositeComponent(composite, component.id, component.version, 'stream');
if (isNodeReadableStream(response)) {
    // handle node.js ReadableStream
    response.on('end', handleEnd);
    response.on('error', handleError);
    response.on('readable', handleReadable);
    // or pipe it
    response.pipe(writeStream);
} else {
    // handle WHATWG Stream
    const reader = response.getReader();
    reader.pipeTo(writeStream);
}
```
