/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPAccessControlList,
    ACPRepoMetadataResource,
    ACPRepository,
    AdobeAsset,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeMinimalAsset,
    AdobeRepoMetadata,
    AdobeResponse,
    AdobeResponseType,
    AssetWithRepoAndPathOrId,
    CopyResourceDesignator,
    EffectivePrivileges,
    JSONPatchDocument,
    Link,
    LinkedResource,
    LinkMode,
    LinkSet,
    Privilege,
    RepoDownloadStreamableReturn,
    RequestDescriptor,
    RequireSome,
    ResolvableResourceRelations,
    ResourceDesignator,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    expandURITemplate,
    getLinkHref,
    getLinkHrefTemplated,
    getLinkProperty,
    isObject,
    merge,
    pruneUndefined,
    validateParams,
} from '@dcx/util';
import { AdobeRepositoryLinksCache } from './cache/RepositoryLinksCache';
import { RepoResponseResult, RepoResponseSuccessResult } from './common';
import { AssetType, AssetTypes } from './enum/asset_types';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation, LinkRelationKey } from './enum/link';
import { JSONMediaType, JSONPatchMediaType } from './enum/media_types';
import { Properties } from './enum/properties';
import { STREAMABLE_RESPONSE_TYPES } from './enum/response_types';
import { AdobeStreamableContext, OptionalContext } from './LeafContext';
import {
    _copyResources,
    copyAsset,
    CopyResourcesOperationResult,
    deleteAsset,
    discardAsset,
    IDBasedOperationSource,
    moveAsset,
    packageAssets,
    PathOrIdAssetDesignator,
    restoreAsset,
} from './operations';
import { _getUrlFallbackDirect } from './private';
import { constructServiceEndpoint, getReposityLinksCache, getService, ServiceConfig } from './Service';
import { BulkRequestDescriptor, performBulkRequest } from './util/bulk';
import { isRepoResponseResultLike, isResolvableAsset } from './util/duck_type';
import { getHTTPResource, headHTTPResource } from './util/http';
import { parseLinksFromResponseHeader } from './util/link';
import { deserializeAsset } from './util/serialization';
import {
    assertAssetIsResolvable,
    assertLinksContain,
    assertLinksContainAny,
    doLinksContain,
    makeStatusValidator,
} from './util/validation';
export { ACPRepoMetadataResource, AdobeAsset } from '@dcx/common-types'; // re-export for convenience, should use common-types

const dbg = newDebug('dcx:assets:asset');
const dbgl = newDebug('dcx:assets:asset:leaf');

export interface Asset<T extends AdobeAsset> extends AdobeAsset {
    readonly type: AssetType;

    /**
     * Performs a HEAD operation on the Asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    headPrimaryResource(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;

    /**
     * Downloads an asset's primary resource
     * @param responseType - Type to tranform response into
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getPrimaryResource<R extends AdobeResponseType>(
        responseType?: R,
        additionalHeaders?: Record<string, string>,
    ): RepoDownloadStreamableReturn<R>;

    /**
     * Retrieves the repository metadata resource for the asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getRepoMetadata(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<AdobeRepoMetadata>, AdobeDCXError>;

    /**
     * @hidden
     * Retrieves the repository metadata for the base directory of the asset
     */
    getBaseDirectoryMetadata(): AdobePromise<AdobeRepoMetadata>;

    /**
     * Fetches, Hydrates and returns the assets links
     *
     * @returns {AdobePromise<LinkSet>}
     */
    getLinks(additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;

    /**
     * Get the repository resource for an asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getRepositoryResource(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError>;

    /**
     * Performs a HEAD request on the application metadata resource.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    headAppMetadata(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError>;

    /**
     * Returns the application metadata associated with the asset.
     *
     * @example
     * ```ts
     * const { result, etag, response } = await asset.getAppMetadata(etag);
     * // result is type object, the app metadata (undefined if etag was provided and not modified)
     * // etag is type string, the app metadata's etag
     * // response is type AdobeResponse
     * ```
     *
     * @note
     * Browser clients may prefer *not* using the etag parameter, instead relying on
     * the browser's HTTP cache to add the header, unless you are managing a cache yourself.
     *
     * @note
     * If etag provided and resource has not been modified, result will be `null`.
     *
     * @param etag    If specified, will be passed as the If-None-Match header value.
     *                  Note: this etag refers to the application metadata resource.
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getAppMetadata<R = any>(
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<R> & { etag: string }, AdobeDCXError>;

    /**
     * Creates application metadata for an asset.
     * @example
     * ```ts
     * const { result, response } = await asset.putAppMetadata(obj, etag);
     * // result is type object, shape: { etag: string }, with etag being the
     * //                        value of the new app metadata resource's etag
     * // response is type AdobeResponse
     * ```
     *
     * @note Replaces the entire resource.
     *
     * @param metadata  New application metadata JSON object
     * @param etag      If specified, will be passed as the If-Match header value.
     *                      Note: this etag refers to the application metadata resource.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    putAppMetadata<R = string | Record<string, unknown>>(
        metadata: R,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError>;

    /**
     * Update application metadata associated with the asset.
     *
     * @param patchDoc              JSON Patch Document to apply.
     *                              {@link https://git.corp.adobe.com/pages/caf/api-spec/#patch|RFC 6902}
     * @param etag                  ETag of the application metadata resource.
     *                              Used as the If-Match header value.
     *                              For unconditional update, use wildcard "*".
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     * @returns {AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError>}
     */
    patchAppMetadata<R = string | JSONPatchDocument>(
        patchDoc: R,
        etag: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError>;

    /**
     * Retrieves the effective privileges for the various resources of the asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getEffectivePrivileges(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError>;

    /**
     * Executes a bulk request that allows clients to combine multiple read or write operations into a single HTTP request.
     *
     * Bulk Requests have the limitations below:
     * 1) Bulk requests must pertain to the Resources of a single Asset.
     * 2) User agents can either read multiple Resources (with HEAD and GET requests) or write multiple Resources
     *    (with POST, PUT, PATCH, and DELETE requests), but cannot mix these.
     * 3) Bulk requests may be issued using both ID-based and path-based Links, but these may not be mixed in
     *    a single request. The addressing mode used in each sub-request must match that of the bulk request.
     *
     * @note Currently only supports bulk READ Operations
     *
     * @param requests      A list of BulkRequestDescriptor's to be included as part of the bulk request
     * @param linkMode      The link mode used by the requests
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    performBulkRequest(
        requests: BulkRequestDescriptor[],
        linkMode?: LinkMode,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<AdobeResponse[]>, AdobeDCXError>;

    /**
     * Fetch's the links for the asset if the optionally provded link relation is missing from the instance
     * @param linksToPopulate       The link relation that is needed
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    fetchLinksIfMissing(
        linksToPopulate: LinkRelationKey[],
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<Asset, AdobeDCXError, RequestDescriptor>;

    /**
     * Retrieves the ACL resource for an asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getACLPolicy(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;

    /**
     * Checks whether the current user has the requested Privilege on the specified Resource of an Asset.
     * @param privilege             The Privilege to be checked. Legal values are read, write, delete, and ack.
     * @param relation              The LinkRelation type of the Resource whose Privilege will be checked.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    checkACLPrivilege(
        privilege: Omit<Privilege, 'none'>,
        relation: LinkRelationKey,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError>;

    /**
     * Patch the ACL policy for this asset using a JSON Patch Document or stringified representation.
     * @param policy A JSON Patch Document in string on JSON format representing the patch operations to perform on the ACL
     * @param etag Optional etag for the ACL policy. If supplied, the patch will only be performed if the remote policy's etag matches the one provided.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     * @see {@link https://acpcs.corp.adobe.com/apis/?choose=aclfrontservice#operation/jsonPatchPolicy patch documentation}
     */
    patchACLPolicy(
        policy: JSONPatchDocument | string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;

    /**
     * Issues a DELETE request against an asset's ACL Policy removing all ACEs.
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     * @returns {AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>}
     */
    deleteACLPolicy(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>;

    /**
     * ******************************************************************************
     * Operations
     * ******************************************************************************
     */

    /**
     * Copy the asset to some destination.
     *
     * @note
     * To copy to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param destAsset             Asset containing either path or assetId
     * @param createIntermediates   Whether to create intermediate directories if missing.
     * @param overwriteExisting     Whether to overwrite an existing asset.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @param manifestPatch         A JSON Patch Document to be applied to the target asset's manifest.
     */
    copy(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
        manifestPatch?: JSONPatchDocument,
    ): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>;

    /**
     * Move the asset, uses the operation endpoint.
     *
     * @note
     * To move to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destAsset - Asset containing either path or assetId
     * @param {boolean} createIntermediates - Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting - Whether to overwrite the existing asset.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    move(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>;

    /**
     * Deletes the asset, irreversible.
     * If the asset is a directory, recursive is a required parameter.
     *
     * @param {string}  [etag]              Etag of asset to delete, if undefined will delete unconditionally.
     * @param {boolean} [recursive]         Whether the delete should delete subdirectories recursively, required for directory.
     * @param {Record<string, string>} additionalHeaders    Additional headers to apply to HTTP Requests.
     *
     * @returns AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>
     */
    delete(
        etag?: string,
        recursive?: boolean,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;

    /**
     * Discards the asset, can be reversed with restore.
     *
     * @param {boolean} [etag]              Etag of the asset to discard, if undefined will discard unconditionally.
     * @param {boolean} [recursive]         Whether the delete should delete subdirectories recursively, required for directory.
     * @param {Record<string, string>} [additionalHeaders]    Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>}
     */
    discard(
        etag?: string,
        recursive?: boolean,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;

    /**
     * Restores the asset, can only be performed on discarded assets.
     *
     * @param {Record<string, string>} [additionalHeaders]    Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    restore(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>;

    /**
     * Package the asset into a zip file.
     *
     * @note
     * To package to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be packaged to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destAsset - Asset containing either path or assetId.
     * @param {boolean} createIntermediates - Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting - Whether to overwrite the existing asset.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>}
     */
    package(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;

    /**
     * Copies resources from source to target asset using the COPY_RESOURCES operation.
     * See {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html}
     *
     *
     * @param {PathOrIdAssetDesignator} sourceAsset                     The source asset
     * @param {PathOrIdAssetDesignator} targetAsset                     The destination asset
     * @param {CopyResourceDesignator[]} resources                      An array of resource designators to be copied from the source to the target asset.
     * @param {boolean} [intermediates]                                 Whether to create intermediate directories if missing.
     * @param {JSONPatchDocument} [manifestPatch]                       A JSON Patch Document to be applied to the target asset's manifest.
     * @param {Record<string, string>} [additionalHeaders]              Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
     */
    copyResources(
        targetAsset: PathOrIdAssetDesignator,
        resources: CopyResourceDesignator[],
        intermediates?: boolean,
        manifestPatch?: JSONPatchDocument,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>;
}

export class Asset<T extends AdobeAsset = AdobeAsset> implements Asset<T>, LinkedResource {
    readonly type: AssetType = AssetTypes.Asset;

    /**
     * Underlying Data Source
     */
    protected _data: T = {} as T;

    /**
     * Service Instance
     */
    protected _svc: AdobeHTTPService;

    /**
     * Links Cache
     */
    protected _cache: AdobeRepositoryLinksCache<LinkSet> | undefined;

    /**
     * The assets links
     */
    private _links: LinkSet;

    constructor(
        data: T | ACPRepoMetadataResource,
        svcOrSvcConfig: AdobeHTTPService | ServiceConfig,
        links: LinkSet = {},
    ) {
        this._data = deserializeAsset(data as T) as T;
        this._svc = getService(svcOrSvcConfig);
        this._cache = getReposityLinksCache(svcOrSvcConfig);

        this._links = merge(
            {},
            (data as AdobeAsset).links || {},
            (data as ACPRepoMetadataResource)['_links'] || {},
            links,
        );
    }

    /**
     * ******************************************************************************
     * LinkedResource Interface Implementation
     * ******************************************************************************
     */

    /**
     * Set links on asset
     * @param links
     */
    setLinks(links: LinkSet): void {
        this._links = links;
        this._updateCachedLinks();
    }

    /**
     * Returns the links for this asset if they have been hydrated
     *
     * @returns {LinkSet}
     */
    get links(): LinkSet {
        return this._links;
    }

    set links(links: LinkSet) {
        this.setLinks(links);
    }

    /**
     * Sets the links
     * @param {string} relationship     The link relation key
     * @param {Link} link               The link
     */
    setLink(relationship: string, link: Link): void {
        this._links[relationship] = link;
        this._updateCachedLinks();
    }

    /**
     * Returns a Link by relation
     * @param {stirng} relationship     The link relation key
     *
     * @returns {Link}
     */
    getLink(relationship: string): Link {
        return this._links[relationship];
    }

    /**
     * Removes a link
     * @param relationship      The relation key of the link to remove
     *
     * @returns {void}
     */
    removeLink(relationship: string) {
        delete this._links[relationship];
        this._updateCachedLinks();
    }

    /**
     * @private
     *
     * Updates the cache with the assets links
     *
     * @returns {void}
     */
    private _updateCachedLinks() {
        if (this._cache) {
            this._cache.setValueWithAsset(this.links, this.asset);
        }
    }

    /**
     * Gets a property for a link relationship
     * @param relationship      The link relation key
     * @param property          The property to fetch
     * @param linkMode          The link mode
     *
     * @returns {string | undefined}
     */
    getLinkProperty(relationship: string, property: string, linkMode: LinkMode = 'id'): string | undefined {
        dbg('getLinkProperty()');

        return getLinkProperty({ _links: this._links }, relationship, property, linkMode);
    }

    /**
     * Returns a Link Href for a templated link
     * @param {string} relationship                                                                                   The link relation key
     * @param {Record<string, (string | number) | (string | number)[] | Record<string, string | number>>} values      The values to populate the template with
     * @param {LinkMode} linkMode                                                                                     The link mode
     *
     * @returns {string | undefined}
     */
    getLinkHrefTemplated(
        relationship: string,
        values: Record<string, (string | number) | (string | number)[] | Record<string, string | number>>,
        linkMode: LinkMode = 'id',
    ): string | undefined {
        dbg('getLinkHrefTemplated()');

        return getLinkHrefTemplated({ _links: this._links }, relationship, values, linkMode);
    }

    /**
     * Returns the href of a link
     * @param relationship      The link relation key
     * @param linkMode          The link mode
     *
     * @returns {string | undefined}
     */
    getLinkHref(relationship: string, linkMode: LinkMode = 'id'): string | undefined {
        dbg('getLinkHref()');

        return getLinkHref({ _links: this._links }, relationship, linkMode);
    }

    /**
     * ******************************************************************************
     * Getters/Setters
     * ******************************************************************************
     */

    /**
     * Returns an AdobeAsset representation of this Asset instance
     *
     * @returns {AdobeAsset}
     */
    public get asset(): AdobeAsset {
        return {
            ...this._data,
            ...this.links,
        };
    }

    /**
     * Creates a ServiceConfig object
     *
     * @returns {ServiceConfig}
     */
    get serviceConfig(): ServiceConfig {
        return {
            service: this._svc,
            cache: this._cache,
        };
    }

    /**
     * The Repository ID of the Repository storing the Asset.
     *
     * @returns {string | undefined}
     */
    get repositoryId(): string | undefined {
        return this._data.repositoryId;
    }
    /**
     * Sets the repositoryId of the asset
     *
     * @returns {void}
     */
    set repositoryId(val: string | undefined) {
        this._data.repositoryId = val;
    }

    /**
     * A unique identifier given to every addressable Asset in a given Repository.
     */
    get assetId(): string | undefined {
        return this._data.assetId;
    }
    set assetId(val: string | undefined) {
        this._data.assetId = val;
    }

    /**
     * An Asset's location in the Directory hierarchy.
     */
    get path(): string | undefined {
        return this._data.path;
    }
    set path(val: string | undefined) {
        this._data.path = val;
    }

    /**
     * The name of the Asset.
     */
    get name(): string | undefined {
        return this._data.name;
    }

    /**
     * An ETag is an HTTP response header returned by an HTTP/1.1-compliant web server, used to determine change in content of a Resource at a given URL. This property is required if the Asset has a Primary Resource.
     */
    get etag(): string | undefined {
        return this._data.etag;
    }
    set etag(val: string | undefined) {
        this._data.etag = val;
    }

    /**
     * Identifier of the head version of the Asset. Not present for an Asset which does not yet have a version.
     */
    get version(): string | undefined {
        return this._data.version;
    }
    set version(val: string | undefined) {
        this._data.version = val;
    }

    /**
     * The media type of the Resource.
     */
    get format(): string | undefined {
        return this._data.format;
    }
    set format(val: string | undefined) {
        this._data.format = val;
    }

    /**
     * The class of an Asset is inferred from its Media Type at the time the Asset is created.
     */
    get assetClass(): string | undefined {
        return this._data.assetClass;
    }

    /**
     * The server date and time when the Resource was created in the Repository, such as when a File is first uploaded or a Directory is created by the server as the parent of a new Asset.
     */
    get createDate(): string | undefined {
        return this._data.createDate;
    }
    /**
     * The server date and time when the Resource was last modified in the Repository, such as when a new version of an Asset is uploaded or a Directory's child Resource is added or removed.
     */
    get modifyDate(): string | undefined {
        return this._data.modifyDate;
    }
    /**
     * Time the asset was discarded directly or by inheritance. Does not exist for active assets.
     */
    get discardDate(): string | undefined {
        return this._data.discardDate;
    }
    /**
     * The ID of the user who initiated the action that caused the Resource to be created in the Repository
     */
    get createdBy(): string | undefined {
        return this._data.createdBy;
    }
    /**
     * The ID of the user who initiated the action that most recently caused the Resource to be modified in the Repository.
     */
    get modifiedBy(): string | undefined {
        return this._data.modifiedBy;
    }
    /**
     * Identifier of the user that discarded the asset directly or by inheritance. Does not exist for active assets.
     */
    get discardedBy(): string | undefined {
        return this._data.discardedBy;
    }
    /**
     * A timestamp capturing the time at which this Asset was created on the client, and which may therefore precede the time at which it was created in the Repository. (It can also be later, due to clock skew.) Can be set only at the time of Asset creation and, if not set, defaults to repo:createDate.
     */
    get deviceCreateDate(): string | undefined {
        return this._data.deviceCreateDate;
    }
    /**
     * A timestamp capturing the time at which this Asset was last modified on the client, and which may therefore precede the time at which the corresponding modification was uploaded to the Repository. (It can also be later, due to clock skew.) If not specified with any request that creates a new version of the Asset, it will be updated to repo:modifyDate.
     */
    get deviceModifyDate(): string | undefined {
        return this._data.deviceModifyDate;
    }
    /**
     * ID of the root asset
     */
    get baseAssetId(): string | undefined {
        return this._data.baseAssetId;
    }
    set baseAssetId(val: string | undefined) {
        this._data.baseAssetId = val;
    }

    /**
     * The Asset's state, indicating whether the Asset is active, discarded, or deleted.
     */
    get state(): string | undefined {
        return this._data.state;
    }

    /**
     * The size of the Asset in bytes.
     */
    get size(): number | undefined {
        return this._data.size;
    }
    set size(val: number | undefined) {
        this._data.size = val;
    }

    get md5(): string | undefined {
        return this._data.md5;
    }

    set defaultScheduledDeletionDuration(val: number | undefined) {
        this._data.defaultScheduledDeletionDuration = val;
    }

    get defaultScheduledDeletionDuration(): number | undefined {
        return this._data.defaultScheduledDeletionDuration;
    }

    set scheduledDeletionDate(val: string | undefined) {
        this._data.scheduledDeletionDate = val;
    }

    get scheduledDeletionDate(): string | undefined {
        return this._data.scheduledDeletionDate;
    }

    set width(val: number | undefined) {
        this._data.width = val;
    }

    get width(): number | undefined {
        return this._data.width;
    }

    set length(val: number | undefined) {
        this._data.length = val;
    }

    get length(): number | undefined {
        return this._data.length;
    }

    /**
     * ******************************************************************************
     * Link APIs
     * ******************************************************************************
     */

    fetchLinksIfMissing(
        linksToPopulate: LinkRelationKey[] = [],
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<Asset, AdobeDCXError> {
        dbg('fetchLinksIfMissing()');

        return _fetchLinksIfMissing(this.serviceConfig, this, linksToPopulate, undefined, additionalHeaders).then(
            ({ result, response }) => {
                this._updateDataWithResponse(response as AdobeResponse);

                return this;
            },
        );
    }

    useLinkOrResolveResource<R extends AdobeResponseType = AdobeResponseType>(
        resource: ResolvableResourceRelations,
        responseType?: R,
    ): AdobePromise<RepoResponseResult<AdobeAsset, R>, AdobeDCXError> {
        dbg('useLinkOrResolveResource()');

        return useLinkOrResolveResource<R>(this.serviceConfig, this, resource, responseType).then((resp) => {
            this._updateDataWithResponse(resp.response);
            this.setLinks(merge(this.links, resp.result.links));
            return resp;
        });
    }

    headPrimaryResource(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>, AdobeDCXError> {
        dbg('headPrimaryResource()');

        return this.fetchLinksIfMissing([LinkRelation.PRIMARY], additionalHeaders)
            .then(() => {
                return headPrimaryResource(this.serviceConfig, this, additionalHeaders);
            })
            .then((response) => {
                this._updateDataWithResponse(response);
                return response;
            });
    }

    getRepoMetadata(): AdobePromise<RepoResponseResult<AdobeRepoMetadata>, AdobeDCXError> {
        dbg('getRepoMetadata()');

        return this.useLinkOrResolveResource(LinkRelation.REPO_METADATA, 'json').then((res) => {
            this._data = merge(this._data, res.result, deserializeAsset(res.response.response));
            const resp = res.response.response;
            if (resp && resp._links) {
                this.setLinks(merge(this.links, resp._links));
            }

            return {
                result: this._data as unknown as AdobeRepoMetadata,
                response: res.response,
            };
        });
    }

    headAppMetadata(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError> {
        dbg('headAppMetadata()');

        return this.fetchLinksIfMissing([LinkRelation.APP_METADATA], additionalHeaders).then(() => {
            return headAppMetadata(this.serviceConfig, this);
        });
    }

    getAppMetadata<R = any>(
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<R> & { etag: string }, AdobeDCXError> {
        dbg('getAppMetadata()');

        validateParams(['etag', etag, 'string', true]);

        return this.fetchLinksIfMissing([LinkRelation.APP_METADATA], additionalHeaders).then(() => {
            return getAppMetadata<R>(this._svc, this, etag, additionalHeaders);
        });
    }

    putAppMetadata<R = string | Record<string, unknown>>(
        metadata: R,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError> {
        dbg('putAppMetadata()');

        validateParams(['etag', etag, 'string', true], ['metadata', metadata, ['object', 'string']]);

        return this.fetchLinksIfMissing([LinkRelation.APP_METADATA], additionalHeaders).then(() => {
            return putAppMetadata<R>(this._svc, this, metadata, etag, additionalHeaders);
        });
    }

    patchAppMetadata<R = string | JSONPatchDocument>(
        patchDoc: R,
        etag: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError> {
        dbg('patchAppMetadata()');
        validateParams(['patchDoc', patchDoc, ['string', 'object[]']], ['etag', etag, 'string']);

        return this.fetchLinksIfMissing([LinkRelation.APP_METADATA], additionalHeaders).then(() => {
            return patchAppMetadata<R>(this._svc, this, patchDoc, etag, additionalHeaders);
        });
    }

    /* istanbul ignore next */
    getBaseDirectoryMetadata(): AdobePromise<AdobeRepoMetadata, AdobeDCXError> {
        dbg('getBaseDirectoryMetadata()');

        return getBaseDirectoryMetadata(this._svc, this);
    }

    getLinks(additionalHeaders?: Record<string, string>): AdobePromise<LinkSet, AdobeDCXError> {
        dbg('getLinks()');

        if (isObject(this.links) && Object.keys(this.links).length > 0) {
            return AdobePromise.resolve<LinkSet>(this.links);
        }

        return getLinksForAsset(this._svc, this, additionalHeaders).then((links) => {
            this.setLinks(links);
            return links;
        });
    }

    getRepositoryResource(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError> {
        dbg('getRepositoryResource()');

        return this.fetchLinksIfMissing([LinkRelation.REPOSITORY], additionalHeaders)
            .then(() => {
                return getRepositoryResource(this._svc, this, additionalHeaders);
            })
            .then((res) => {
                this.repositoryId = res.result['repo:repositoryId'];
                return res;
            });
    }

    getEffectivePrivileges(additionalHeaders): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError> {
        dbg('getEffectivePrivileges()');

        return this.fetchLinksIfMissing([LinkRelation.EFFECTIVE_PRIVILAGES], additionalHeaders).then(() => {
            return getEffectivePrivileges(this._svc, this);
        });
    }

    performBulkRequest(
        requests: BulkRequestDescriptor[],
        linkMode?: LinkMode,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<AdobeResponse[]>, AdobeDCXError> {
        dbg('performBulkRequest()');

        validateParams(['requests', requests, 'array'], ['linkMode', linkMode, 'string', true, ['id', 'path']]);

        return this.fetchLinksIfMissing([LinkRelation.BULK_REQUEST], additionalHeaders).then(() => {
            return performBulkRequest(this._svc, this, requests, linkMode, additionalHeaders);
        });
    }

    getACLPolicy(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError> {
        dbg('getACLPolicy()');

        return this.fetchLinksIfMissing([LinkRelation.ACL_POLICY], additionalHeaders).then(() => {
            return getACLPolicy(this._svc, this);
        });
    }

    checkACLPrivilege(
        privilege: Omit<Privilege, 'none'>,
        relation: LinkRelationKey,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError> {
        dbg('checkACLPrivilege()');

        // TODO: validate relation as enum
        validateParams(['privilege', privilege, 'string'], ['relation', relation, 'string']);

        return this.fetchLinksIfMissing([LinkRelation.ACCESS_CHECK], additionalHeaders).then(() => {
            return checkACLPrivilege(this._svc, this, privilege, relation);
        });
    }

    patchACLPolicy(
        policy: JSONPatchDocument | string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError> {
        dbg('patchACLPolicy()');

        validateParams(['policy', policy, ['string', 'object']]);

        return this.fetchLinksIfMissing([LinkRelation.ACL_POLICY], additionalHeaders).then(() =>
            patchACLPolicy(this._svc, this, policy, etag),
        );
    }

    deleteACLPolicy(
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<'json'>, AdobeDCXError> {
        dbg('deleteACLPolicy()');
        return this.fetchLinksIfMissing([LinkRelation.ACL_POLICY], additionalHeaders).then(() => {
            return deleteACLPolicy(this._svc, this);
        });
    }

    getPrimaryResource<R extends AdobeResponseType = 'defaultbuffer'>(
        responseType?: R,
        additionalHeaders?: Record<string, string>,
    ): RepoDownloadStreamableReturn<R> {
        dbg('getPrimaryResource()');

        validateParams(['responseType', responseType, 'string']);

        const ctx = {};
        return this._withSourcePromise(ctx)
            .then(() => this.fetchLinksIfMissing([LinkRelation.PRIMARY], additionalHeaders))
            .then(() => {
                return getPrimaryResource.call(
                    ctx,
                    this._svc,
                    this,
                    responseType,
                    additionalHeaders,
                ) as RepoDownloadStreamableReturn<R>;
            });
    }

    /**
     * ******************************************************************************
     * Operations APIs
     * ******************************************************************************
     */

    /**
     * Copies an asset for the src to the destination
     * @param destPathOrURN         Asset containing either path or assetId
     * @param createIntermediates   Whether to create intermediate directories if missing.
     * @param overwriteExisting     Whether to overwrite an existing asset.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @param manifestPatch         A JSON Patch Document to be applied to the target asset's manifest.
     */
    copy(
        destination: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
        manifestPatch?: JSONPatchDocument,
    ): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError> {
        dbg('copy()');

        validateParams(
            ['destination', destination, ['object', 'string']],
            ['createIntermediates', createIntermediates, 'boolean'],
            ['overwriteExisting', overwriteExisting, 'boolean'],
            ['manifestPatch', manifestPatch, ['object', 'string'], true],
        );

        return copyAsset(
            this.serviceConfig,
            { repositoryId: this.repositoryId, assetId: this.assetId, path: this.path } as PathOrIdAssetDesignator,
            destination,
            createIntermediates,
            overwriteExisting,
            additionalHeaders,
            manifestPatch,
        ).then(({ response, result }) => {
            return {
                response,
                result: new Asset(result, this.serviceConfig),
            };
        });
    }

    copyResources(
        targetAsset: PathOrIdAssetDesignator,
        resources: CopyResourceDesignator[],
        intermediates?: boolean | undefined,
        manifestPatch?: JSONPatchDocument | undefined,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError> {
        dbg('copyResources()');

        validateParams(
            ['targetAsset', targetAsset, 'object'],
            ['resources', resources, 'array'],
            ['manifestPatch', manifestPatch, ['object', 'string'], true],
            ['intermediates', intermediates, 'boolean', true],
        );

        return _copyResources(
            this.serviceConfig,
            {
                repositoryId: this.repositoryId,
                assetId: this.assetId,
                path: this.path,
                version: this.version,
            } as PathOrIdAssetDesignator,
            targetAsset,
            resources,
            intermediates,
            manifestPatch,
            additionalHeaders,
        );
    }

    /**
     * Moves an asset to the destination
     * @param destPathOrURN
     * @param createIntermediates
     * @param overwriteExisting
     * @param additionalHeaders
     */
    move(
        destination: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Asset>, AdobeDCXError> {
        dbg('move()');

        validateParams(
            ['destination', destination, ['object', 'string']],
            ['createIntermediates', createIntermediates, 'boolean'],
            ['overwriteExisting', overwriteExisting, 'boolean'],
        );

        return moveAsset(
            this.serviceConfig,
            { repositoryId: this.repositoryId, assetId: this.assetId, path: this.path } as PathOrIdAssetDesignator,
            destination,
            createIntermediates,
            overwriteExisting,
            additionalHeaders,
        ).then(({ response, result }) => {
            this._data = { ...this._data, ...pruneUndefined(result) };
            return {
                response,
                result: this,
            };
        });
    }

    delete(
        etag?: string,
        recursive = false,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError> {
        dbg('delete()');

        validateParams(['etag', etag, 'string', true], ['recursive', recursive, 'boolean']);

        return deleteAsset(
            this.serviceConfig,
            { repositoryId: this.repositoryId, assetId: this.assetId, path: this.path } as PathOrIdAssetDesignator,
            etag,
            recursive,
            additionalHeaders,
        ).then((res) => {
            this._data.state = 'DELETED';
            return res;
        });
    }

    discard(
        etag?: string,
        recursive = false,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError> {
        dbg('discard()');

        validateParams(['etag', etag, 'string', true], ['recursive', recursive, 'boolean']);

        return discardAsset(
            this.serviceConfig,
            { repositoryId: this.repositoryId, assetId: this.assetId, path: this.path } as PathOrIdAssetDesignator,
            etag,
            recursive,
            additionalHeaders,
        ).then((res) => {
            this._data.state = 'DISCARDED';
            return res;
        });
    }

    /**
     * Package the asset.
     * @param destPathOrURN
     * @param createIntermediates
     * @param overwriteExisting
     * @param additionalHeaders
     */
    package(
        destination: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError> {
        dbg('package()');

        validateParams(
            ['destination', destination, ['object', 'string']],
            ['createIntermediates', createIntermediates, 'boolean'],
            ['overwriteExisting', overwriteExisting, 'boolean'],
        );

        return packageAssets(
            this._svc,
            { repositoryId: this.repositoryId, assetId: this.assetId, path: this.path } as PathOrIdAssetDesignator,
            destination,
            createIntermediates,
            overwriteExisting,
            additionalHeaders,
        );
    }

    restore(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<Asset>, AdobeDCXError> {
        dbg('restore()');

        return restoreAsset(
            this._svc,
            {
                repositoryId: this.repositoryId,
                assetId: this.assetId,
            } as IDBasedOperationSource,
            additionalHeaders,
        ).then(({ response, result }) => {
            this._data = { ...this._data, ...pruneUndefined(result), state: 'ACTIVE' };
            return {
                response,
                result: this,
            };
        });
    }

    /**
     * Update internal data object with response from mutating API call.
     *
     * @param response
     * @returns {AdobeResponse}
     */
    protected _updateDataWithResponse(response: AdobeResponse): AdobeResponse {
        if (!response) {
            return response;
        }
        this._data.etag = response.headers.etag || this._data.etag;
        this._data.version = response.headers.version || this._data.version;
        this._data.assetId = response.headers['asset-id'] || this._data.assetId;
        this._data.md5 = response.headers['content-md5'] || this._data.md5;
        this._data.repositoryId = response.headers['repository-id'] || this._data.repositoryId;

        return response;
    }

    protected _withSourcePromise<T extends Record<string | number | symbol, unknown>>(
        source: T,
    ): AdobePromise<void, AdobeDCXError, T> {
        return AdobePromise.resolve(undefined, source);
    }
}

/**
 * Downloads an asset's primary resource
 * @this [OptionalContext<AdobeStreamableContext>] Optional this binding for AdobeStreamableContext
 * @param svc               HTTPService
 * @param asset             The asset
 * @param responseType      Type to tranform response into
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export function getPrimaryResource<T extends AdobeResponseType = 'defaultbuffer'>(
    this: OptionalContext<AdobeStreamableContext>,
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    responseType?: T,
    additionalHeaders?: Record<string, string>,
): RepoDownloadStreamableReturn<T> {
    dbg('getPrimaryResource()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['responseType', responseType, 'enum', true, STREAMABLE_RESPONSE_TYPES],
    );
    assertLinksContain(asset.links, [LinkRelation.PRIMARY]);

    const primaryHref = getLinkHref(asset.links, LinkRelation.PRIMARY);
    const ctx = {};
    return _getUrlFallbackDirect.call(
        ctx,
        svc,
        asset,
        primaryHref,
        LinkRelation.PRIMARY,
        responseType,
        undefined,
        undefined,
        additionalHeaders,
    ) as RepoDownloadStreamableReturn<T>;
}

/**
 * Performs a HEAD operation on the Asset
 *
 * @param svc                Service or service config
 * @param asset              The asset
 * @param additionalHeaders  Additional headers to apply to HTTP Requests
 */
export function headPrimaryResource(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeResponse<'void'>, AdobeDCXError> {
    dbg('headPrimaryResource()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.PRIMARY]);

    const primaryHref = getLinkHref(asset.links, LinkRelation.PRIMARY);

    const service = getService(svc);
    return service
        .invoke<'void'>(HTTPMethods.HEAD, primaryHref, additionalHeaders, undefined, {
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            const linkSet = parseLinksFromResponseHeader(response);
            asset.links = merge(asset.links || {}, linkSet);
            const cache = getReposityLinksCache(svc);
            if (cache) {
                cache.setValueWithAsset(asset.links || {}, asset);
            }

            return response;
        });
}

const WELL_KNOWN_RESOLVE_BY_ID_TEMPLATE = '/content/directory/resolve{?repositoryId,id,resource,mode}';
const WELL_KNOWN_RESOLVE_BY_PATH_TEMPLATE = '/content/directory/resolve{?repositoryId,path,resource,mode}';

/**
 * Returns the resolve link for an Asset
 *
 * @note Resolve link is classified as well known by CA-771 https://jira.corp.adobe.com/browse/CA-771?src=confmacro
 *
 * @param svc               Service or service config
 * @param asset             The asset
 * @param mode              The resolve by mode to return
 * @param resource          Optional resource
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export function getResolveLinkForAsset(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AssetWithRepoAndPathOrId,
    mode: 'id' | 'path' = 'id',
    resource?: ResourceDesignator,
    additionalHeaders?: Record<string, string>,
): AdobePromise<string> {
    dbgl('getResolveLinkForAsset()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['mode', mode, 'enum', false, ['id', 'path']],
        ['resource', resource, ['string', 'object'], true],
    );
    assertAssetIsResolvable(asset);

    const params: Record<string, string | undefined> = {
        repositoryId: asset.repositoryId,
        id: asset.assetId,
        path: asset.path,
        mode,
        resource: isObject(resource) ? JSON.stringify(resource) : (resource as string),
    };

    const service = getService(svc);
    const endpoint = constructServiceEndpoint(
        asset.assetId ? WELL_KNOWN_RESOLVE_BY_ID_TEMPLATE : WELL_KNOWN_RESOLVE_BY_PATH_TEMPLATE,
        service,
    );
    return AdobePromise.resolve<string>(expandURITemplate(endpoint, pruneUndefined<Record<string, string>>(params)));
}

/**
 * Call resolveByID or resolveByPath on an asset.
 * If resource is specified (and valid) the additional resource will be included in the response object.
 *
 * @note
 * This method currently only takes a LinkRelationKey as the resource argument.
 * It will be updated with support for Resource Designator objects with https://jira.corp.adobe.com/browse/DCX-4232.
 *
 * @param svc               Service or service config
 * @param asset             The asset to resolve, must be resolvable
 * @param mode              Whether to return path- or id-based links
 * @param resource          Additional resource to include in the response
 * @param responseType      Optional parsing of additional resource, defaults to 'json'
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export function resolveAsset<T extends AdobeResponseType = 'json'>(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AssetWithRepoAndPathOrId,
    mode: 'path' | 'id' = 'id',
    resource?: ResourceDesignator,
    responseType?: T,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeAsset, T | 'void'>, AdobeDCXError> {
    dbgl('resolveAsset()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['mode', mode, 'enum', true, ['id', 'path']],
        ['resource', resource, ['string', 'object'], true],
    );
    assertAssetIsResolvable(asset);

    const service = getService(svc);
    const cache = getReposityLinksCache(svc);

    // Set pending in cache only if using ID-based links and have assetId and repositoryId.
    const setPending = cache && asset.assetId && asset.repositoryId && mode === 'id';
    if (setPending) {
        dbgl('rA() set pending');
        (cache as AdobeRepositoryLinksCache<LinkSet>).setPending(asset.assetId as string, asset.repositoryId);
    }

    return getResolveLinkForAsset(svc, asset, mode, resource, additionalHeaders)
        .then<AdobeResponse<T | 'void'>>((href) =>
            resource == null
                ? headHTTPResource(service, href, additionalHeaders)
                : getHTTPResource<T>(service, href, additionalHeaders, responseType),
        )
        .then<RepoResponseResult<AdobeAsset, T | 'void'>>((response) => {
            const parsed = _parseResolvedAsset(asset, response, mode === 'id', cache);

            return {
                response,
                result: parsed,
            };
        })
        .catch((e) => {
            // remove from pending
            /* istanbul ignore next */
            if (setPending) {
                (cache as AdobeRepositoryLinksCache<LinkSet>).deleteWithAsset(asset);
            }
            throw e; // pass error along
        });
}

/**
 * Same as `fetchLinksForAssetWithResponse()`, but only returns LinkSet object.
 *
 * @see fetchLinksForAssetWithResponse
 *
 * @param svc        Service or service config
 * @param asset                     The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function fetchLinksForAsset(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeMinimalAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<LinkSet> {
    dbgl('fetchLinksForAsset()');
    return fetchLinksForAssetWithResponse(svc, asset, additionalHeaders).then((res) => {
        return res.result.links;
    });
}

/**
 * Fetch asset's Repository Metadata using one of (in order of preference):
 * 1. HEAD ResolveBy* API
 * 2. HEAD api:id
 * 3. HEAD api:repo_metadata
 * 4. HEAD api:primary
 * 5. HEAD api:path
 *
 * @note
 * Always makes an API call. Does not fetch links from cache.
 * To fetch links from cache, use `fetchLinksIfMissing()` and provide
 * a list of LinkRelations required.
 *
 * @param svc        Service or service config
 * @param asset              The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function fetchLinksForAssetWithResponse(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeMinimalAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeAsset & Required<Pick<AdobeAsset, 'links'>>, 'void'>> {
    dbgl('fetchAsset()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);

    // Preferable method to get links: resolve API
    // resolveAsset() handles caching of links.
    if (isResolvableAsset(asset)) {
        return resolveAsset(svc, asset, 'id', undefined, undefined, additionalHeaders).then((res) => {
            return res as RepoResponseResult<AdobeAsset & Required<Pick<AdobeAsset, 'links'>>, 'void'>;
        });
    }

    // Otherwise, try to head any link that will return links.
    let linkToHead: LinkRelationKey;

    try {
        linkToHead = assertLinksContainAny(asset.links, [
            LinkRelation.ID,
            LinkRelation.REPO_METADATA,
            LinkRelation.PRIMARY,
            LinkRelation.PATH,
        ]);
    } catch (e) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            'Asset is not resolvable. Must contain repositoryId & path, assetId, or links.',
            e as Error,
        );
    }

    const hrefToHead = getLinkHref(asset.links, linkToHead);
    const service = getService(svc);
    const cache = getReposityLinksCache(svc);

    // Can't set pending in cache since we don't have both the repository ID and asset ID,
    // but we can still fetch the links and return them.

    return service
        .invoke<'void'>(HTTPMethods.HEAD, hrefToHead, additionalHeaders, undefined, {
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            // Can only be guaranteed the links are ID-based if using the ID link.
            return {
                result: _parseResolvedAsset(asset, response, linkToHead === LinkRelation.ID, cache),
                response,
            };
        });
}

/**
 * Returns cached links or fetches if none cached.
 * If partial links are cached, will return them.
 *
 * @param svc        Service or service config
 * @param asset                            The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getLinksForAsset(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<LinkSet> {
    dbgl('getLinksForAsset()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);

    const linksCandidate = asset.links || (asset as Record<string, LinkSet>)[Properties.LINKS];
    if (isObject(linksCandidate) && Object.keys(linksCandidate).length !== 0) {
        return AdobePromise.resolve<LinkSet>(linksCandidate);
    }

    const cache = getReposityLinksCache(svc);
    if (cache) {
        const cachedLinks = cache.getValueWithAsset(asset);
        /* istanbul ignore if */
        if (cachedLinks) {
            return AdobePromise.resolve<LinkSet>(cachedLinks);
        }
    }

    return fetchLinksForAsset(svc, asset as AdobeMinimalAsset, additionalHeaders);
}

/**
 * Retrieves the repository metadata resource for the asset
 *
 * @param svc        Service or service config
 * @param asset            The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getRepoMetadata(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeRepoMetadata>, AdobeDCXError> {
    dbgl('getRepoMetadata()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.REPO_METADATA]);

    const href = getLinkHref(asset.links, LinkRelation.REPO_METADATA) as string;
    return svc
        .invoke<'json'>(HTTPMethods.GET, href, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            const linkSet = parseLinksFromResponseHeader(response);
            const metadata = response.response;

            //Header links could contain links not found in the response body
            metadata[Properties.LINKS] = merge({}, metadata[Properties.LINKS], linkSet);
            return {
                result: metadata,
                response: response,
            };
        });
}

/* istanbul ignore next */
export function getBaseDirectoryMetadata(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
): AdobePromise<AdobeRepoMetadata, AdobeDCXError> {
    dbgl('getBaseDirectoryMetadata()');

    throw new DCXError(DCXError.NOT_IMPLEMENTED, 'Method not implemented.');
}

/**
 * Get the repository resource for an asset
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        Asset whose links to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getRepositoryResource(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError> {
    dbgl('getRepositoryResource()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.REPOSITORY]);

    const repositoryResourceHref = getLinkHref(asset.links, LinkRelation.REPOSITORY);
    return svc
        .invoke(HTTPMethods.GET, repositoryResourceHref, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            return {
                result: response.response as ACPRepository,
                response: response,
            };
        });
}

/**
 * Performs a HEAD request on the application metadata resource.
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        The asset to HEAD the app metadata for
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function headAppMetadata(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeResponse<'void'>, AdobeDCXError> {
    dbg('headAppMetadata()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.APP_METADATA]);

    const appMetaHref = getLinkHref(asset.links, LinkRelation.APP_METADATA);
    const service = getService(svc);
    return service
        .invoke<'void'>(HTTPMethods.HEAD, appMetaHref, additionalHeaders, undefined, {
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            const links = parseLinksFromResponseHeader(response);
            asset.links = merge(asset.links || {}, links);
            const cache = getReposityLinksCache(svc);
            if (cache) {
                cache.setValueWithAsset(asset.links!, asset);
            }

            return response;
        });
}

/**
 * Returns the application metadata associated with the asset.
 *
 * @note
 * Browser clients may prefer *not* using the etag parameter, instead relying on
 * the browser's HTTP cache to add the header, unless you are managing a cache yourself.
 *
 * @note
 * If etag provided and resource has not been modified, result will be `null`.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             The asset to fetch the app metadata for
 * @param etag              If specified, will be passed as the If-None-Match header value.
 *                             Note: this etag refers to the application metadata resource.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getAppMetadata<T = any>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    etag?: string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<T, 'json'> & { etag: string }, AdobeDCXError> {
    dbgl('getAppMetadata()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object'], ['etag', etag, 'string', true]);

    const appMetadataHref = getLinkHref(asset.links, LinkRelation.APP_METADATA);
    const headers = { ...additionalHeaders };
    if (etag) {
        headers[HeaderKeys.IF_NONE_MATCH] = etag;
    }
    return svc
        .invoke<'json'>(HTTPMethods.GET, appMetadataHref, headers, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator([304]),
        })
        .then((response) => {
            let result = response.response;
            let resEtag = response.headers.etag;
            if (response.statusCode === 304) {
                result = null;
                resEtag = etag as string;
            }
            return {
                result,
                response,
                etag: resEtag,
            };
        });
}

/**
 * Update application metadata associated with the asset.
 *
 * @example
 * ```ts
 * const { result, response } = await asset.putAppMetadata(obj, etag);
 * // result is type object, shape: { etag: string }, with etag being the
 * //                        value of the new app metadata resource's etag
 * // response is type AdobeResponse
 * ```
 *
 * @note Replaces the entire resource.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             Asset whose links to retrieve
 * @param metadata          New application metadata JSON object
 * @param etag              If specified, will be passed as the If-None-Match header value.
 *                              Note: this etag refers to the application metadata resource.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function putAppMetadata<T = any>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    metadata: T,
    etag?: string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError> {
    dbgl('putAppMetadata()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['metadata', metadata, ['object', 'string']],
        ['etag', etag, 'string', true],
    );
    assertLinksContain(asset.links, [LinkRelation.APP_METADATA]);

    const href = getLinkHref(asset.links, LinkRelation.APP_METADATA);
    return svc
        .invoke(
            HTTPMethods.PUT,
            href,
            pruneUndefined(
                Object.assign(additionalHeaders, {
                    [HeaderKeys.IF_MATCH]: etag,
                    [HeaderKeys.CONTENT_TYPE]: JSONMediaType,
                }),
            ),
            typeof metadata === 'string' ? metadata : JSON.stringify(metadata),
            { isStatusValid: makeStatusValidator() },
        )
        .then((response) => {
            return {
                response,
                result: {
                    etag: response.headers.etag,
                },
            };
        });
}

/**
 * Update application metadata associated with the asset.
 *
 * @param {AdobeHTTPService} svc    HTTP Service or ServiceConfig
 * @param {AdobeAsset} asset        Asset whose links to retrieve
 * @param patchDoc                  JSON Patch Document to apply.
 *                                  {@link https://git.corp.adobe.com/pages/caf/api-spec/#patch|RFC 6902}
 * @param etag                      ETag of the application metadata resource.
 *                                    Used as the If-Match header value.
 *                                    For unconditional update, use wildcard "*".
 * @param additionalHeaders         Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError>}
 */
export function patchAppMetadata<T = string | JSONPatchDocument>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    patchDoc: T,
    etag: string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError> {
    dbgl('patchAppMetadata()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['metadata', patchDoc, ['object[]', 'string']],
        ['etag', etag, 'string'],
    );
    assertLinksContain(asset.links, [LinkRelation.APP_METADATA]);

    const href = getLinkHref(asset.links, LinkRelation.APP_METADATA);
    return svc
        .invoke(
            HTTPMethods.PATCH,
            href,
            pruneUndefined(
                Object.assign(additionalHeaders, {
                    [HeaderKeys.IF_MATCH]: etag,
                    [HeaderKeys.CONTENT_TYPE]: JSONPatchMediaType,
                }),
            ),
            typeof patchDoc === 'string' ? patchDoc : JSON.stringify(patchDoc),
            { isStatusValid: makeStatusValidator() },
        )
        .then((response) => {
            return {
                response,
                result: {
                    etag: response.headers.etag,
                },
            };
        });
}

/**
 * Retrieves the effective privileges for the various resources of the asset
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        Asset whose links to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getEffectivePrivileges(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError> {
    dbgl('getEffectivePrivileges()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.EFFECTIVE_PRIVILAGES]);

    const effectivePrivilagesHref = getLinkHref(asset.links, LinkRelation.EFFECTIVE_PRIVILAGES);
    return svc
        .invoke(HTTPMethods.GET, effectivePrivilagesHref, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            return {
                result: response.response as EffectivePrivileges,
                response: response,
            };
        });
}

/**
 * Retrieves the ACL resource for an asset
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        Asset whose links to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getACLPolicy(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError> {
    dbgl('getACLPolicy()');
    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.ACL_POLICY]);
    const policyHref = getLinkHref(asset.links, LinkRelation.ACL_POLICY);

    return svc
        .invoke(HTTPMethods.GET, policyHref, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            return {
                result: response.response as ACPAccessControlList,
                response: response,
            };
        });
}

/**
 * Checks whether the current user has the requested Privilege on the specified Resource of an Asset.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             Asset to check the ACL privilege against
 * @param privilege         The Privilege to be checked. Legal values are read, write, delete, and ack.
 * @param relation          The LinkRelation type of the Resource whose Privilege will be checked.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function checkACLPrivilege(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    privilege: Omit<Privilege, 'none'>,
    relation: LinkRelationKey,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError> {
    dbgl('checkACLPrivilege()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['privilege', privilege, 'string', false, ['ack', 'read', 'write', 'delete']],
    );
    assertLinksContain(asset.links, [LinkRelation.ACCESS_CHECK]);

    const href = getLinkHrefTemplated(asset.links, LinkRelation.ACCESS_CHECK, {
        privilege: privilege.toString(),
        relation: relation,
    });

    return svc
        .invoke<'json'>(
            HTTPMethods.GET,
            href,
            Object.assign({ directive: 'acl-check-no-body' }, additionalHeaders),
            undefined,
            {
                responseType: 'json',
                isStatusValid: makeStatusValidator([403]),
            },
        )
        .then((result) => {
            return {
                result: result.statusCode === 403 ? false : true,
                response: result,
            };
        });
}

/**
 * Patch the ACL policy for this asset using a JSON Patch Document or stringified representation.
 * @param svc HTTP Service or ServiceConfig
 * @param asset Asset to issue the ACL PATCH on
 * @param policy A JSON Patch Document representing the patch operations to perform on the ACL
 * @param etag Optional etag for the ACL policy. If supplied, the patch will only be performed if the remote policy's etag matches the one provided.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 * @see {@link https://acpcs.corp.adobe.com/apis/?choose=aclfrontservice#operation/jsonPatchPolicy patch documentation}
 */
export function patchACLPolicy(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    policy: JSONPatchDocument | string,
    etag?: string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError> {
    dbgl('patchACLPolicy()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['policy', policy, ['string', 'object']],
        ['etag', etag, 'string', true],
    );
    assertLinksContain(asset.links, [LinkRelation.ACL_POLICY]);

    const headers: Record<string, string> = pruneUndefined(
        Object.assign(additionalHeaders, {
            [HeaderKeys.CONTENT_TYPE]: JSONPatchMediaType,
            [HeaderKeys.IF_MATCH]: etag,
        }),
    );
    const href = getLinkHref(asset.links, LinkRelation.ACL_POLICY);

    const service = getService(svc);

    return service
        .invoke<'json'>(
            HTTPMethods.PATCH,
            href,
            headers,
            typeof policy === 'string' ? policy : JSON.stringify(policy),
            {
                responseType: 'json',
                isStatusValid: makeStatusValidator(),
            },
        )
        .then((result) => {
            return {
                result: result.response,
                response: result,
            };
        });
}

/**
 * Issues a DELETE request against an asset's ACL Policy removing all ACEs.
 *
 * @param {AdobeHTTPService} svc                        HTTP Service or ServiceConfig
 * @param {AdobeAsset} asset                            Asset object
 * @param {Record<string, string>} additionalHeaders    Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>}
 */
export function deleteACLPolicy(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<'json'>, AdobeDCXError> {
    dbgl('deleteACLPolicy()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.ACL_POLICY]);

    const href = getLinkHref(asset.links, LinkRelation.ACL_POLICY);

    const service = getService(svc);

    return service
        .invoke<'json'>(HTTPMethods.DELETE, href, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((result) => {
            return {
                result: result.response,
                response: result,
            };
        });
}

/**
 * Issues GET on a resource link if it exists in either the asset object
 * or in the links cache. If neither exists, use the resolve API to fetch
 * the resource in a single request.
 *
 * @note The list of relations that can be fetched using the resolve API will likely
 * change over time. `ResolvableResourceRelations` will have to be updated with
 * each new valid resource.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             Asset object
 * @param resource
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function useLinkOrResolveResource<T extends AdobeResponseType = AdobeResponseType>(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    resource: ResolvableResourceRelations,
    responseType?: T,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeAsset, T>, AdobeDCXError> {
    dbgl('useLinkOrResolveResource()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['resource', resource, 'string'], // don't validate as an enum so that clients can try to use new resolve API params before we implement them
    );

    const service = getService(svc);

    let href: string | undefined = undefined;
    try {
        href = getLinkHref(asset.links, resource);
    } catch (_) {
        // noop
    }

    return AdobePromise.resolve(href)
        .then<string | LinkSet | undefined>((hrefOrUndef) => {
            if (typeof hrefOrUndef === 'string') {
                dbgl('uLORR() asset has link');
                return hrefOrUndef;
            }

            const cache = getReposityLinksCache(svc);
            return getLinksFromCache(asset, cache, [resource]);
        })
        .then((linksOrHref) => {
            if (typeof linksOrHref === 'string') {
                return linksOrHref;
            }

            try {
                return getLinkHref(linksOrHref, resource);
            } catch (_) {
                return;
            }
        })
        .then<AdobeResponse | RepoResponseResult<AdobeAsset>>((hrefOrUndef) => {
            if (typeof hrefOrUndef === 'string') {
                dbgl('uLORR() cache or asset had link');
                return service.invoke<T>(HTTPMethods.GET, hrefOrUndef, additionalHeaders, undefined, {
                    isStatusValid: makeStatusValidator(),
                    responseType,
                });
            }

            /* istanbul ignore if */
            if (!isResolvableAsset(asset)) {
                throw new DCXError(
                    DCXError.INVALID_PARAMS,
                    'Asset is not resolvable. Must contain repository ID + path/id or links.',
                );
            }

            return resolveAsset<T>(svc, asset, 'id', resource, responseType, additionalHeaders);
        })
        .then((resp) => {
            if (isRepoResponseResultLike(resp)) {
                return resp;
            }

            // This was a GET to a link that already existed somewhere.
            // Don't need to mess with caching just return the asset passed in,
            // along with the response from the GET.
            return {
                result: asset,
                response: resp,
            };
        });
}

/**
 * Fetch set of links if required link array does not exist in links property or cache.
 *
 * @param svc                   Service or service config
 * @param asset                 Asset with or without links
 * @param linksToPopulate       Required links
 * @param suppressMissingErrors Suppress error when links fetched but not found
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
export function fetchLinksIfMissing(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    /* istanbul ignore next */
    linksToPopulate: LinkRelationKey[] = [],
    suppressMissingErrors = false,
    additionalHeaders?: Record<string, string>,
): AdobePromise<LinkSet, AdobeDCXError, RequestDescriptor> {
    return _fetchLinksIfMissing(svc, asset, linksToPopulate, suppressMissingErrors, additionalHeaders).then(
        ({ result }) => {
            return result;
        },
    );
}

/**
 *
 * @param asset The asset for which links should be resolved
 * @param linksToPopulate the reltypes being requested to be resolved
 */
function _canUseLinksAPI(
    asset: AdobeAsset,
    linksToPopulate: LinkRelationKey[],
): asset is RequireSome<AdobeAsset, 'assetId'> {
    return (
        typeof asset.assetId === 'string' &&
        asset.assetId.length > 0 &&
        linksToPopulate.every((reltype) => !_linksThatRequireFullResolve.has(reltype))
    );
}

/**
 * The Links API utilizes a well-known path and cannot return some reltypes.
 * This is the list of reltypes that are not returned by the Links API.
 */
export const _linksThatRequireFullResolve = new Set<LinkRelationKey>([
    // Hierarchical Links
    LinkRelation.BASE_DIRECTORY,
    LinkRelation.RESOLVE_BY_ID,
    LinkRelation.RESOLVE_BY_PATH,
    LinkRelation.REPO_OPS,
    LinkRelation.REPOSITORY,
    LinkRelation.DIRECTORY,
    LinkRelation.DISCARD,
    LinkRelation.RESTORE,
    LinkRelation.PATH,
    // Deprecated links
    // Links API does not currently support the ANNOTATIONS reltype,
    //  but will return a new ANNOTATIONS reltype in the future
    LinkRelation.ANNOTATIONS,
]);

const WELL_KNOWN_LINKS_API_TEMPLATE = '/links{?assetId,repositoryId,clientRegion}';
/**
 * Returns the well-known Links API (ResolveLite) URL for a given asset
 * @param svc
 * @param asset
 */
function _getLinksAPIUrlForAsset(
    svc: AdobeHTTPService | ServiceConfig,
    asset: RequireSome<AdobeAsset, 'assetId'>,
): string {
    dbgl('getLinksAPIUrlForAsset()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);

    const service = getService(svc);
    const endpoint = constructServiceEndpoint(WELL_KNOWN_LINKS_API_TEMPLATE, service);
    const { assetId, repositoryId, contentRegion } = asset;
    const values = pruneUndefined<Record<string, string>>({ assetId, repositoryId, contentRegion });
    return expandURITemplate(endpoint, values);
}
/**
 * fetch links via the Links API (ResolveLite)
 * @see {@link https://wiki.corp.adobe.com/pages/viewpage.action?pageId=2605664337 Proposal: Links API (ResolveLite)}
 * @param svc HTTPService
 * @param asset The asset to have the links fetched for
 * @param additionalHeaders Additional headers to be provided with the request
 */
function _fetchLinksViaLinksAPI(
    svc: AdobeHTTPService | ServiceConfig,
    asset: RequireSome<AdobeAsset, 'assetId'>,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeAsset & Required<Pick<AdobeAsset, 'links'>>, 'json'>> {
    const href = _getLinksAPIUrlForAsset(svc, asset);
    const service = getService(svc);
    return service
        .invoke(HTTPMethods.GET, href, additionalHeaders, undefined, { responseType: 'json' })
        .then((response) => {
            const cache = getReposityLinksCache(svc);
            asset.links = Object.assign({}, asset.links, response.response._links);
            cache?.setValueWithAsset(asset.links!, asset);
            return {
                response,
                result: asset as AdobeAsset & Required<Pick<AdobeAsset, 'links'>>,
            };
        });
}

/**
 * Fetch set of links if required link array does not exist in links property or cache.
 *
 * @internal
 * @private
 *
 * @param svc                   Service or service config
 * @param asset                 Asset with or without links
 * @param linksToPopulate       Required links
 * @param suppressMissingErrors Suppress error when links fetched but not found
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
function _fetchLinksIfMissing(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    linksToPopulate: LinkRelationKey[] = [],
    suppressMissingErrors = false,
    additionalHeaders?: Record<string, string>,
): AdobePromise<{ result: LinkSet; response?: AdobeResponse }, AdobeDCXError, RequestDescriptor> {
    dbgl('fetchLinksIfMissing()', linksToPopulate);

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['linksToPopulate', linksToPopulate, 'string[]'],
        ['suppressMissingErrors', suppressMissingErrors, 'boolean', true],
    );
    if (doLinksContain(asset.links, linksToPopulate)) {
        dbgl('fLIM() links exist');
        return AdobePromise.resolve<{ result: LinkSet }, AdobeDCXError, RequestDescriptor>({
            result: asset.links as LinkSet,
        });
    }

    const cache = getReposityLinksCache(svc);

    return getLinksFromCache(asset, cache, linksToPopulate, true)
        .then<LinkSet | RepoResponseResult<AdobeAsset>>((cachedLinks) => {
            if (cachedLinks) {
                return cachedLinks;
            }
            if (_canUseLinksAPI(asset, linksToPopulate)) {
                return _fetchLinksViaLinksAPI(svc, asset, additionalHeaders);
            }
            dbgl('fLIM() fetching links');
            return fetchLinksForAssetWithResponse(svc, asset as AssetWithRepoAndPathOrId, additionalHeaders);
        })
        .then((fetchedResponseOrCachedLinks) => {
            let links: LinkSet = fetchedResponseOrCachedLinks as LinkSet;

            let response = undefined as unknown as AdobeResponse;
            let fetchedAsset = undefined as unknown as AdobeAsset;
            if (isRepoResponseResultLike(fetchedResponseOrCachedLinks)) {
                response = fetchedResponseOrCachedLinks.response;
                links = fetchedResponseOrCachedLinks.result.links as LinkSet;
                fetchedAsset = fetchedResponseOrCachedLinks.result;
            }
            if (asset.links !== links) {
                asset.links = merge(asset.links || {}, links);

                const cache = getReposityLinksCache(svc);
                if (cache) {
                    cache.setValueWithAsset(asset.links!, asset);
                }
            }

            if (!suppressMissingErrors && !doLinksContain(links, linksToPopulate)) {
                throw new DCXError(
                    DCXError.INVALID_PARAMS,
                    'Required links could not be fetched for asset.',
                    undefined,
                    response,
                    pruneUndefined({ required: linksToPopulate, asset: fetchedAsset }),
                );
            }

            dbgl('fLIM() fetchedOCached exists');
            return { result: links || asset.links, response };
        });
}

/**
 * Fetches an assets links from the provided cache
 *
 * @param {AdobeAsset} asset                                            Asset object that identifies the composite asset.
 * @param {AdobeRepositoryLinksCache<LinkSet> | undefined} cache        The cache
 * @param {LinkRelationKey[]} requiredLinks                             The expected links
 * @param {boolean} setPending                                          Put the asset key in the cache in a pending state to prevent it being loaded multiple times.
 *
 * @returns {AdobePromise<LinkSet | undefined, AdobeDCXError, RequestDescriptor>}
 */
export function getLinksFromCache(
    asset: AdobeAsset,
    cache: AdobeRepositoryLinksCache<LinkSet> | undefined,
    requiredLinks?: LinkRelationKey[],
    setPending?: boolean,
): AdobePromise<LinkSet | undefined, AdobeDCXError, RequestDescriptor> {
    dbgl('getLinksFromCache()');

    if (!cache) {
        return AdobePromise.resolve(undefined);
    }

    const promiseOrUndef = cache.getValueWithAsset(asset);
    if (promiseOrUndef == null) {
        // not cached, resolve immediately
        if (setPending) {
            cache.setPending(asset.assetId as string, asset.repositoryId);
        }
        return AdobePromise.resolve(undefined);
    }

    return AdobePromise.resolve(promiseOrUndef).then((cachedLinks) => {
        if (!requiredLinks) {
            return cachedLinks;
        } else if (doLinksContain(cachedLinks, requiredLinks)) {
            // If requiredLinks is specified, even if cachedLinks exist,
            // ensure that it has the exact required link(s).
            return cachedLinks;
        }

        if (setPending) {
            cache.setPending(asset.assetId as string, asset.repositoryId);
        }
    });
}

export function assetTransformer(data: ACPRepoMetadataResource): [string, AdobeAsset] {
    dbgl('assetTransformer()');

    const asset = deserializeAsset(data);
    asset.links = merge({}, (data as AdobeAsset).links, data['_links']);
    return [asset.assetId as string, asset];
}

export function copyResources(
    svc: AdobeHTTPService | ServiceConfig,
    sourceAsset: PathOrIdAssetDesignator,
    targetAsset: PathOrIdAssetDesignator,
    resources: CopyResourceDesignator[],
    intermediates?: boolean,
    manifestPatch?: JSONPatchDocument,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError> {
    return _copyResources(svc, sourceAsset, targetAsset, resources, intermediates, manifestPatch, additionalHeaders);
}

/**
 * Private
 */

/**
 * Parse response headers from an API call that returns
 * headers containing links and asset data, ie. resolveBy*.
 *
 * Does not mutate incoming asset.
 *
 * @param {AdobeAsset} asset                            Incoming asset.
 * @param {AdobeResponse} response                      Response with no body required
 * @param {boolean} idBasedResponse                     Is this resolve by id?
 * @param {AdobeRepositoryLinksCache<LinkSet>} cache    Links cache
 *
 * @returns {AdobeAsset} With some properties set as defined.
 */
function _parseResolvedAsset(
    asset: AdobeAsset,
    response: AdobeResponse,
    idBasedResponse: boolean,
    cache?: AdobeRepositoryLinksCache<LinkSet>,
): AdobeAsset & Required<Pick<AdobeAsset, 'assetId' | 'format' | 'repositoryId' | 'etag' | 'links'>> {
    const links = parseLinksFromResponseHeader(response);
    const parsed = {
        ..._getAssetData(asset),
        ...pruneUndefined({
            assetId: response.headers['asset-id'] || response.headers['x-resource-id'],
            format: response.headers[HeaderKeys.CONTENT_TYPE],
            md5: response.headers['content-md5'],
            etag: response.headers['etag'],
            version: response.headers['version'],
            repositoryId: response.headers['repository-id'],
        }),
        links,
    } as AdobeAsset & Required<Pick<AdobeAsset, 'assetId' | 'format' | 'repositoryId' | 'etag' | 'links'>>;

    // Cache if provided with a cache.
    if (idBasedResponse && links && Object.keys(links).length > 0 && cache) {
        cache.setValueWithAsset(links, parsed);
    }

    return parsed;
}

/**
 * Get asset data.
 * If using HLA return the asset property object, otherwise return the object.
 *
 * @param {AdobeAsset | Record<string, unknown>} asset     The Asset
 *
 * @returns {AdobeAsset}
 */
function _getAssetData(asset: AdobeAsset | Record<string, unknown>): AdobeAsset {
    if (!isObject((asset as Record<string, unknown>).asset)) {
        return asset;
    }

    return (asset as Record<string, unknown>).asset as AdobeAsset;
}
