/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeBlockDownload,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeResponse,
    AdobeResponseType,
    BlockTransferState,
    RepoDownloadStreamableReturn,
} from '@dcx/common-types';
import DCXError, { isAdobeDCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    EventEmitter,
    arrayBufferToString,
    assert,
    concatUint8Arrays,
    getFirstRegexpCapture,
    pruneUndefined,
    validateParams,
} from '@dcx/util';
import { AdobeStreamableContext, OptionalContext } from '../LeafContext';
import { HeaderKeys } from '../enum/header_keys';
import { HTTPMethods } from '../enum/http_methods';
import { maybeGetBlockTransfer } from '../util/block_transfer';
import { makeStatusValidator } from '../util/validation';
import { blockTransferManager } from './BlockTransferManager';
import { BlockTransferStates } from './common';
const dbg = newDebug('dcx:assets:blockdownload');
const dbgl = newDebug('dcx:assets:blockdownload:leaf');

const _START_BYTE = 0;
const _END_BYTE = 1;

/** @internal */
interface BlockResolution {
    response: AdobeResponse<'stream' | 'defaultbuffer'>;
    index: number;
    lane: number;
}

export interface BlockDownloadOptions {
    startByte?: number;
    endByte?: number;
    blockSize?: number;
    url?: string;
    totalSize?: number;
    maxConcurrentRequests?: number;
}

export interface BlockDownloadEvents {
    stateChanged: (state: BlockTransferState, blockDownload: AdobeBlockDownload) => void;
    [key: string]: (...args: any[]) => unknown;
}
function* getId() {
    let id = 0;
    while (true) {
        yield id++;
    }
}
const idGenerator = getId();

export class BlockDownload
    extends EventEmitter<BlockDownloadEvents>
    // eslint-disable-next-line prettier/prettier
    implements AdobeBlockDownload
{
    private _contentType?: string;
    private _dbgId: number;
    /** Current state of the transfer. */
    private _state: BlockTransferState = BlockTransferStates.NOT_INITIALIZED;

    /** Error object when in ERROR state. */
    private _error: AdobeDCXError | undefined;

    /** The size for each individual block request. */
    private _blockSize: number;

    /** Collected bytes, only used for buffer responseType */
    private _bytes!: Uint8Array;

    /** Blocks not yet handled due to arriving out of order. */
    private _cachedBlocks: Map<number, BlockResolution> = new Map();

    /** Index of block requests made */
    private _blockRequestIndex = 0;
    /** Index of block data pushed to stream or added to buffer. */
    private _blockHandledIndex = 0;

    /** First byte to download. */
    private _startByte?: number;

    /** Last byte to download. If not provided, becomes totalSize. */
    private _endByte?: number;

    /** The byte range to use for the next request. */
    private _currentByteRange: [start: number, end: number] = [
        undefined as unknown as number,
        undefined as unknown as number,
    ];

    /** Number of concurrent requests to issue. */
    private _maxConcurrentRequests: number;

    /** Total size of the asset. */
    private _totalSize?: number;

    /** Block download presigned URL. */
    private _url?: string;

    /** Pending requests. */
    private _pending: AdobePromise<BlockResolution | void>[] = [];
    private _resolve!: () => void;
    private _reject!: (reason: AdobeDCXError) => void;
    private _promise: AdobePromise<BlockDownload>;
    private _service: AdobeHTTPService;

    constructor(svc: AdobeHTTPService, responseType: 'buffer', options: BlockDownloadOptions = {}) {
        super(['stateChanged']);

        dbg('constructor');

        const { startByte, blockSize, endByte, url, totalSize, maxConcurrentRequests } = Object.assign(
            {
                blockSize: blockTransferManager.downloadChunkSize,
                maxConcurrentRequests: 4,
            },
            pruneUndefined(options),
        ) as BlockDownloadOptions & Required<Pick<BlockDownloadOptions, 'blockSize'>>;

        validateParams(
            ['svc', svc, 'object'],
            ['responseType', responseType, 'enum', false, ['buffer']],
            ['blockSize', blockSize, '+number'],
            ['url', url, 'string', true],
            ['startByte', startByte, 'number', true],
            ['endByte', endByte, 'number', true],
            ['totalSize', totalSize, 'number', true],
            ['maxConcurrentRequests', maxConcurrentRequests, '+number'],
        );
        this._dbgId = idGenerator.next().value as number;
        this._maxConcurrentRequests = maxConcurrentRequests as number;
        this._blockSize = Math.round(blockSize);
        this._service = svc;
        this._url = url;
        this._startByte = startByte;
        this._endByte = endByte;
        this._totalSize = totalSize;
        this._bytes = new Uint8Array();

        this._promise = new AdobePromise<BlockDownload>((resolve, reject) => {
            this._resolve = () => {
                dbg(this._dbgId, 'resolving');
                this.removeAllHandlers();
                resolve(this);
            };

            this._reject = (err: AdobeDCXError) => {
                dbg(this._dbgId, 'rejecting: ', err);

                this.removeAllHandlers();
                reject(err);
            };
        });
    }

    /**
     * The content type of the retrieved asset.
     */
    public get contentType(): string {
        return this._contentType ?? '';
    }

    /**
     * The size of the content or undefined if not known.
     */
    public get totalSize(): number | undefined {
        return this._totalSize;
    }

    /**
     * The download byte array.
     *
     * @type {Uint8Array}
     */
    public get buffer(): Uint8Array {
        return this._bytes;
    }

    /**
     * The current state of the transfer.
     */
    public get state(): BlockTransferState {
        return this._state;
    }

    /**
     * Get promise that resolves when the BlockDownload completes.
     */
    public get promise(): AdobePromise<AdobeBlockDownload> {
        return this._promise;
    }

    /**
     * Make a request for a single block.
     *
     * @param {number | undefined}  startByte    - First byte to download.
     *                                             May be undefined if downloading last N bytes, or entire asset.
     * @param {number}              endByte      - Last byte to download.
     *                                             If negative, downloads the last <lastByte> bytes.
     * @param {number}              blockIndex   - The order index of the current block.
     * @param {number}              laneIndex    - The download lane to use.
     */
    private _requestBlock(
        startByte: number | undefined,
        endByte: number | undefined,
        blockIndex: number,
        laneIndex: number,
    ): AdobePromise<BlockResolution, AdobeDCXError> {
        dbg(this._dbgId, '_requestBlock(): ', startByte, endByte, blockIndex, laneIndex);

        const headers = buildRangeHeader(startByte, endByte);
        return this._service
            .invoke<'defaultbuffer'>(HTTPMethods.GET, this._url as string, headers, undefined, {
                responseType: 'defaultbuffer',
                isStatusValid: makeStatusValidator(),
                isExternalRequest: true,
            })
            .then((response) => {
                return {
                    response,
                    index: blockIndex,
                    lane: laneIndex,
                };
            })
            .catch(this._handleErrorAndThrow.bind(this));
    }

    /**
     * Initialize block download.
     * If provided with all required properties during construction,
     * this method only moves to the next state.
     *
     * It's possible a BlockDownload is constructed without a totalSize,
     * in which case this method will download the first block (if possible)
     * to determine the total size.
     *
     * @param {string} url          - Presigned direct download URL
     * @param {number} totalSize    - Total size of the asset to download
     */
    public init(
        url: string = this._url as string,
        totalSize: number = this._totalSize as number,
    ): AdobePromise<AdobeBlockDownload> {
        dbg(this._dbgId, 'init(): ', url, totalSize);

        if (this._state === BlockTransferStates.INITIALIZED && url === this._url && totalSize === this._totalSize) {
            return AdobePromise.resolve(this);
        }

        this._assertStateIsValid('init');
        this._shiftState(BlockTransferStates.INITIALIZING);

        this._url = url;

        this._initByteRange(totalSize);

        /**
         * If totalSize is defined, initialization is done.
         *
         * If totalSize is undefined & the request is for the
         * last N bytes, where N is less than or equal to a
         * single block, initialization is done (using bytes=-N range).
         * when->this._currentByteRange[0] == undefined
         *
         * If totalSize is undefined & N > a single block,
         * _initByteRange() will have thrown an error.
         */
        if (this._totalSize !== Infinity || this._currentByteRange[_START_BYTE] == null) {
            this._shiftState(BlockTransferStates.INITIALIZED);
            return AdobePromise.resolve(this);
        }

        /**
         * If totalSize is undefined, and we're starting from byte 0.
         *  when-> this._currentByteRange[0] == 0
         *  then-> make a single block request to determine total size
         */
        const { startByte, endByte, blockIndex } = this._nextBlockData();
        let newState = BlockTransferStates.INITIALIZED;
        return this._requestBlock(startByte, endByte, blockIndex, 0)
            .then((res) => {
                this._updateTotalSize(res);
                if (!endByte || endByte > (this._endByte as number)) {
                    newState = BlockTransferStates.FINALIZING;
                }
                return res;
            })
            .then(this._handleBlock.bind(this))
            .then(() => this._shiftState(newState))
            .catch(this._handleErrorAndThrow.bind(this));
    }

    /**
     * Start download loop.
     */
    public start(): AdobePromise<BlockDownload, AdobeDCXError> {
        dbg(this._dbgId, 'start()');

        this._assertStateIsValid('start');
        this._shiftState(BlockTransferStates.STARTED);

        if (this._currentByteRange[_START_BYTE] != null) {
            this._start();
            return this._promise;
        }

        // First byte is undefined, downloading last bytes in a single request
        const [start, end] = this._currentByteRange;
        const blockIndex = this._blockRequestIndex;

        this._blockRequestIndex += 1;
        this._currentByteRange = [end + 1, end];

        this._requestBlock(start, end, blockIndex, 0)
            .then(this._handleBlock.bind(this))
            .catch(this._handleError.bind(this));
        return this._promise;
    }

    public pause(): AdobePromise<BlockDownload, AdobeDCXError> {
        dbg(this._dbgId, 'pause()');

        this._assertStateIsValid('pause');
        this._shiftState(BlockTransferStates.PAUSING);

        return AdobePromise.allSettled(this._pending).then(() => {
            this._shiftState(BlockTransferStates.PAUSED);
            blockTransferManager.startNextWaiting('download');
            return this;
        });
    }

    public resume(): BlockDownload {
        dbg(this._dbgId, 'resume()');
        if (this.state === BlockTransferStates.PAUSED) {
            this._shiftState(BlockTransferStates.STARTED);
            this._start();
        }
        return this;
    }

    public cancel(): AdobePromise<BlockDownload, AdobeDCXError> {
        dbg(this._dbgId, 'cancel()');

        this._assertStateIsValid('cancel');
        this._shiftState(BlockTransferStates.CANCELED);

        this._reject(new DCXError(DCXError.ABORTED, 'BlockDownload aborted.'));
        blockTransferManager.startNextWaiting('download');
        return this._promise;
    }

    /**
     * Set current state to waiting.
     * Done by the BlockTransferManager.
     *
     * @internal
     */
    _setWaiting() {
        this._shiftState(BlockTransferStates.WAITING);
    }

    /**
     * Start main loop.
     */
    private _start(): void {
        dbg(this._dbgId, '_start()');

        for (let i = 0; i < this._maxConcurrentRequests; i++) {
            this._loop(i).catch(this._handleError.bind(this));
        }
    }

    /**
     * Check if transfer is complete.
     */
    private get _loopShouldContinue(): boolean {
        const shouldContinue =
            this._state === BlockTransferStates.STARTED &&
            this._currentByteRange[_START_BYTE] <= (this._endByte as number);

        dbg(this._dbgId, '_loopShouldContinue() ', shouldContinue, this._currentByteRange, this._endByte);
        return shouldContinue;
    }

    /**
     * Main control loop.
     *
     * Loop through requests until no more data to read.
     */
    private async _loop(lane: number) {
        dbg(this._dbgId, '_loop(): ', lane);

        let finalize = false;
        while (this._loopShouldContinue && !finalize) {
            const { startByte, endByte, blockIndex, done } = this._nextBlockData();
            finalize = done;
            const prom = this._requestBlock(startByte, endByte, blockIndex, lane)
                .then(this._handleBlock.bind(this))
                .catch(this._handleError.bind(this));

            this._pending[lane] = prom;
            await prom;
        }

        dbg(this._dbgId, `_loop(${lane}) done, ${finalize}`);

        if (finalize) {
            dbg(this._dbgId, `_loop(${lane}) finalize`);
            this._shiftState(BlockTransferStates.FINALIZING);
        }
    }

    private _nextBlockData(): { startByte: number; endByte?: number; blockIndex: number; done: boolean } {
        dbg(this._dbgId, '_nextBlockData()');
        // Get the block index, then increment
        const blockIndex = this._blockRequestIndex;
        this._blockRequestIndex += 1;

        // Get the start and end byte, then increment
        const [startByte, endByte] = this._currentByteRange;
        this._currentByteRange[_START_BYTE] += this._blockSize;
        this._currentByteRange[_END_BYTE] = Math.min(
            this._currentByteRange[_END_BYTE] + this._blockSize,
            this._endByte as number,
        );

        const done = endByte >= (this._endByte as number);

        return {
            startByte,
            endByte,
            blockIndex,
            done,
        };
    }

    /**
     * Initialize the current byte range, total size, and start/end byte.
     *
     * startByte == undefined && endByte == undefined
     *  => download entire asset
     *
     * startByte == N && endByte == undefined
     *  => download N-end
     *
     * startByte == undefined && endByte == N
     *  => download first N bytes
     *
     * startByte == undefined && endByte == -N
     *  => download last N bytes
     *
     * @param totalSize
     */
    private _initByteRange(totalSize = this._totalSize): void {
        dbg(this._dbgId, '_initByteRange(): ', totalSize);

        this._totalSize = totalSize;

        if (this._totalSize != null && this._totalSize < 0) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Total size must be positive.');
        }

        if (!this._totalSize) {
            this._totalSize = Infinity;
        }

        // No end to range, use the total size
        if (!this._endByte) {
            this._endByte = this._totalSize;
        }

        // startByte not defined or 0, endByte undefined or totalSize:
        // Downloading entire asset.
        if (!this._startByte && this._endByte === this._totalSize) {
            this._startByte = 0;
            this._currentByteRange = [0, this._blockSize - 1];
            return;
        } else if (!this._startByte && this._endByte < 0 && this._totalSize !== Infinity) {
            // * startByte == undefined && endByte == -N
            // *  => download last N bytes
            this._startByte = Math.max(0, this._totalSize + this._endByte);
            this._endByte = this._totalSize;
        } else if (!this._startByte && this._endByte > 0) {
            this._startByte = 0;
        }

        // endByte may still be Infinity, but will be updated in a later if clause
        if (this._startByte != null) {
            this._currentByteRange[_START_BYTE] = Math.max(this._startByte, 0);
        }

        // endByte undefined or positive, set to the smaller of endByte or startByte+blockSize
        if (this._endByte == null || this._endByte > 0) {
            this._currentByteRange[_END_BYTE] = Math.min(this._endByte, (this._startByte || 0) + this._blockSize - 1);
        }

        if (this._startByte != null || this._endByte === Infinity) {
            return;
        }

        // startByte is undefined and endByte is negative, downloading last N bytes.
        if (this._totalSize === Infinity) {
            // Can't determine startByte without a total size.
            if (-this._endByte > this._blockSize) {
                throw new DCXError(DCXError.INVALID_PARAMS, 'Cannot download last N bytes without a total size.');
            }

            // For segments that are lte a single block,
            // just use the `bytes=-N` notation.
            this._currentByteRange = [undefined as unknown as number, this._endByte];
            return;
        }

        // If totalSize is set, use that to determine start/end.
        this._startByte = Math.max(0, this._totalSize + this._endByte);
        this._endByte = this._totalSize;
        const end = Math.min(this._startByte + this._blockSize - 1, this._endByte);

        this._currentByteRange = [this._startByte, end];
    }

    /**
     * Move to complete state after all pending blocks are complete.
     */
    private _finalize(): AdobePromise<void> {
        dbg(this._dbgId, '_finalize() start');
        blockTransferManager.startNextWaiting('download');
        return AdobePromise.allSettled(this._pending).then(() => {
            this._checkCachedBlocks();
            this._shiftState(BlockTransferStates.COMPLETE);
        });
    }

    /**
     * Attempt to parse and set totalSize from the Content-Range header value.
     *
     * @param blockRes - The block response
     * @returns {BlockResolution}
     */
    private _updateTotalSize(blockRes: BlockResolution): BlockResolution {
        dbg(this._dbgId, '_updateTotalSize()');

        /* istanbul ignore else */
        if (this._totalSize == null || this._totalSize === Infinity) {
            try {
                const contentRange = blockRes.response.headers['content-range'];
                const size = parseInt(contentRange.split('/')[1]);
                assert(() => !isNaN(size), 'Invalid number.');
                this._totalSize = size;
                /* istanbul ignore else */
                if (this._endByte === Infinity) {
                    this._endByte = this._totalSize;
                }
            } catch (e) {
                /* istanbul ignore next */
                throw new DCXError(
                    DCXError.INVALID_DATA,
                    'Could not determine total size.',
                    e as Error,
                    blockRes.response,
                );
            }
        }
        dbg(this._dbgId, '_uTS(): ', this._totalSize, this._endByte);

        return blockRes;
    }

    /**
     * Handle async error.
     *
     * @param err
     */
    private _handleError(err: Error | AdobeDCXError): void {
        dbg(this._dbgId, '_handleError(): ', err);
        this._shiftState(BlockTransferStates.ERROR);

        this._error = err as AdobeDCXError;
        if (!isAdobeDCXError(err)) {
            this._error = new DCXError(DCXError.UNEXPECTED, 'An unexpected error occurred.', err);
        }

        this._reject(this._error);
    }

    /**
     * Handle error by rejecting outer promise and pass along to next catch.
     *
     * @param err
     */
    private _handleErrorAndThrow(err: Error | AdobeDCXError): void {
        this._handleError(err);
        throw this._error;
    }

    /**
     * Handle a block response.
     * Add the data to the collector, or push to stream if it is the current block
     * Otherwise cache the block to be handled in-order at a later time.
     *
     * @param blockResponse
     * @returns
     */
    private _handleBlock(blockResponse: BlockResolution): void {
        if (this._endByte === Infinity) {
            // Update total size if not yet set
            this._updateTotalSize(blockResponse);
        }

        const { index, response } = blockResponse;

        dbg(this._dbgId, `_handleBlock(${index})`);
        // Blocks must be handled in order. If this is not the current block to be handled, cache it and return.
        if (index !== this._blockHandledIndex) {
            dbg(this._dbgId, `_handleBlock(${index}) cached`);
            this._cachedBlocks.set(index, blockResponse);
            return;
        }

        // This is the next block to be handled, remove it from the cache if it is cached
        dbg(this._dbgId, `_handleBlock(${index}) handled`);
        this._pushBlockData(response as AdobeResponse<'defaultbuffer'>);
        this._markCurrentBlockHandled();
    }

    /**
     * Increment _blockHandledIndex and handle that block if it is cached.
     */
    private _markCurrentBlockHandled() {
        this._cachedBlocks.delete(this._blockHandledIndex);
        this._blockHandledIndex += 1;
        this._checkCachedBlocks();
    }

    /**
     * If the current block is cached, handle it.
     */
    private _checkCachedBlocks(): void {
        const block = this._cachedBlocks.get(this._blockHandledIndex);
        if (block) {
            this._handleBlock(block);
        }
    }

    /**
     * Push downloaded block to stream or buffer.
     *
     * @param {AdobeResponse} response
     */
    private _pushBlockData(response: AdobeResponse<'defaultbuffer'>): void {
        // Don't push to closed stream
        /* istanbul ignore next */
        if (this._state === BlockTransferStates.ERROR) {
            return;
        }
        this._contentType = this._contentType || response.headers[HeaderKeys.CONTENT_TYPE];
        /* istanbul ignore next */
        const byteArr =
            typeof Buffer !== 'undefined' && response.response instanceof Buffer
                ? response.response
                : new Uint8Array((response as unknown as AdobeResponse<'arraybuffer'>).response);

        this._bytes = concatUint8Arrays(this._bytes, byteArr);
    }

    /**
     * Move to a new state, finalize if needed.
     * Skips state change if already in complete or error states.
     *
     * @param newState
     */
    private _shiftState(newState: BlockTransferState): BlockDownload {
        dbg(this._dbgId, '_shiftState(): ', newState);

        if (
            this._state === BlockTransferStates.COMPLETE ||
            this._state === BlockTransferStates.ERROR ||
            this._state === BlockTransferStates.CANCELED
        ) {
            return this;
        }

        this._state = newState;
        this.emit('stateChanged', [this._state, this]);
        if (newState === BlockTransferStates.FINALIZING) {
            this._finalize();
        }
        if (newState === BlockTransferStates.COMPLETE) {
            Promise.all(this._pending).then(this._resolve.bind(this));
        }

        return this;
    }

    /**
     * Check current state is valid for method call.
     *
     * @throws {AdobeDCXError}
     *
     * @param method
     * @param currentState
     */
    private _assertStateIsValid(
        method: 'init' | 'start' | 'pause' | 'cancel',
        currentState: BlockTransferState = this._state,
    ): void {
        dbg(this._dbgId, '_assertStateIsValid() ', method, currentState);

        let valid = false;
        const msg = 'Invalid state transition.';

        switch (method) {
            case 'init':
                if (currentState === BlockTransferStates.NOT_INITIALIZED) {
                    valid = true;
                }
                break;
            case 'start':
                if (
                    currentState === BlockTransferStates.INITIALIZED ||
                    currentState === BlockTransferStates.WAITING ||
                    currentState === BlockTransferStates.STARTED
                ) {
                    valid = true;
                }
                break;
            case 'pause':
                if (currentState === BlockTransferStates.INITIALIZING || currentState === BlockTransferStates.STARTED) {
                    valid = true;
                }
                break;
            case 'cancel':
                valid = true;
                break;
        }
        if (!valid) {
            dbg(this._dbgId, '_aSIV() throw ', msg);
            throw new DCXError(DCXError.INVALID_STATE, msg, undefined, undefined, { method, currentState });
        }
    }
}

export function newBlockDownload(
    svc: AdobeHTTPService,
    responseType: 'buffer',
    options?: BlockDownloadOptions,
): BlockDownload {
    return new BlockDownload(svc, responseType, options);
}

/**
 * Wait for a transfer that may use BlockDownload.
 * Resolves when either the download is complete, or the blockDownload property becomes available.
 * @deprecated - waitForBlockDownloadToStart has deprecated using maybeGetBlockTransfer instead.
 * @param res
 */
export async function waitForBlockDownloadToStart<T extends RepoDownloadStreamableReturn<any>>(res: T): Promise<void> {
    await maybeGetBlockTransfer(res);
}

/**
 * Block download an asset.
 *
 * @this {OptionalContext<AdobeStreamableContext>}
 *
 * @param svc
 * @param url
 * @param startByte
 * @param endByte
 * @param responseType
 * @param isPresignedUrl
 * @param totalSize
 * @returns
 */
export function _doBlockDownload<T extends AdobeResponseType = 'defaultbuffer'>(
    this: OptionalContext<AdobeStreamableContext>,
    svc: AdobeHTTPService,
    url: string,
    startByte?: number,
    endByte?: number,
    responseType: T = 'defaultbuffer' as T,
    isPresignedUrl?: boolean,
    totalSize?: number,
    additionalHeaders?: Record<string, string>,
): RepoDownloadStreamableReturn<T> {
    dbgl('_doBlockDownload()');

    // Skip blockDownloadClass creation all together if streaming.
    const blockDownload =
        responseType === 'stream'
            ? undefined
            : newBlockDownload(svc, 'buffer', {
                  startByte,
                  endByte,
              });
    const ctx = (this ?? {}) as AdobeStreamableContext;
    ctx.blockDownload = blockDownload as AdobeBlockDownload | undefined;
    let p = AdobePromise.resolve(url, typeof this !== 'undefined' ? this : {});
    if (!isPresignedUrl) {
        // Need to fetch presigned URL using the (presumably) download link passed in.
        p = p
            .then(() =>
                // DCX-11103 requests that the default for all block transfers is to request transfer acceleration
                // Because the `repo:accelerated` parameter is suggested to be deprecated as part of CA-967,
                // we are opting for setting the priority header on the block upload init request to trigger
                // RAPI to utilize EALinks, which also achieve the same transfer acceleration per CSA-374.
                // Consumers are still able to override the priority value by providing one of their own.
                svc.invoke<'text'>(HTTPMethods.GET, url, { priority: 'u=1', ...additionalHeaders }, undefined, {
                    responseType: 'text',
                    isStatusValid: makeStatusValidator(),
                    retryOptions: {
                        pollCodes: [202],
                        pollHeader: 'location',
                        pollMethod: 'GET',
                    },
                }),
            )
            .then((resp) => {
                const href =
                    resp.response.indexOf('href') > 0
                        ? getFirstRegexpCapture(resp.response, `"href":\\s*"([^;"]*)"`)
                        : '';
                if (typeof href !== 'string' || href === '') {
                    throw new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'No block download href found in response.',
                        undefined,
                        resp,
                    );
                }

                totalSize = totalSize || parseInt(getFirstRegexpCapture(resp.response, `"size":\\s*(\\d+)`));
                return href;
            });
    }

    return p.then(async (presignedUrl: string) => {
        if (blockDownload) {
            return Promise.race([
                blockDownload
                    .init(presignedUrl, totalSize)
                    .then(() => blockTransferManager.addAndStartDownload(blockDownload)),
                blockDownload.promise,
            ]).then(() => {
                return {
                    statusCode: 200,
                    headers: pruneUndefined({
                        [HeaderKeys.CONTENT_TYPE]: blockDownload.contentType,
                        [HeaderKeys.CONTENT_LENGTH]: blockDownload.totalSize,
                    }),
                    responseType,
                    response: _convertBufferResponse(blockDownload!.buffer, responseType, blockDownload.contentType),
                    message: 'OK',
                } as AdobeResponse;
            });
        }
        // when using stream response type, invoke the presignedUrl directly with the entire requested byte range.
        return svc.invoke<'stream'>(HTTPMethods.GET, presignedUrl, buildRangeHeader(startByte, endByte), undefined, {
            responseType: 'stream',
            isExternalRequest: true,
        });
    });
}

function _convertBufferResponse<T extends Omit<AdobeResponseType, 'blob'>>(
    response: Uint8Array,
    responseType: T,
    contentType?: string,
);
function _convertBufferResponse(response: Uint8Array, responseType: 'blob', contentType: string);
function _convertBufferResponse(typedArr: Uint8Array, responseType: AdobeResponseType, contentType?: string) {
    if (responseType === 'defaultbuffer' || responseType === 'buffer' || responseType === 'arraybuffer') {
        return typedArr.buffer;
    }

    if (responseType === 'blob') {
        return new Blob([typedArr], { type: contentType });
    }

    const text = arrayBufferToString(typedArr);
    if (responseType === 'text') {
        return text;
    }

    try {
        return JSON.parse(text as string);
    } catch (_) {
        /* istanbul ignore next */
        return text;
    }
}

/**
 * Build a range request header object
 *
 * @example
 * const headers = buildRangeHeader(10, 100);
 * // headers == { range: bytes=10-100 }
 *
 * const headers = buildRangeHeader(undefined, 100);
 * // headers == { range: bytes=0-100 }
 *
 * const headers = buildRangeHeader(10, undefined);
 * // headers == { range: bytes=10- }
 *
 * const headers = buildRangeHeader(undefined, -10);
 * // headers == { range: bytes=-10 }
 *
 * @param startByte
 * @param endByte
 * @returns
 */
export function buildRangeHeader(startByte?: number, endByte?: number): Record<string, string> {
    if (startByte == null && endByte == null) {
        return {};
    }

    const range = `bytes=${startByte != null ? startByte : endByte != null && endByte > 0 ? '0' : ''}-${
        endByte != null ? Math.abs(endByte) : ''
    }`;

    return {
        range,
    };
}
