/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import AdobePromise from '@dcx/promise';
import { validateParams } from '@dcx/util';
import { BlockDownload } from './BlockDownload';
import { BlockUpload } from './BlockUpload';
import { BlockTransferStates } from './common';

/**
 * BlockTransferManager manages the queuing ans sequencing of mulitple active BlockUploads/Downloads
 */
export interface BlockTransferManager {
    /**
     * Ordered array of promises for all block uploads
     */
    readonly uploads: BlockUpload[];

    /**
     * Ordered array of promises for all block uploads
     */
    readonly downloads: BlockDownload[];

    /**
     * Block size to use.
     * Defaults to 1024*1024*10 bytes.
     */
    downloadChunkSize: number;

    /**
     * Array of the currently pending block upload HTTP requests
     */
    pendingUploadRequests: AdobePromise[];

    /**
     * Array of currently pending block download HTTP requests
     */
    pendingDownloadRequests: AdobePromise[];
}

/**
 * Private/internal BlockTransferManager API
 */
class BlockTransferManagerImpl implements BlockTransferManager {
    private _uploads: BlockUpload[] = [];
    private _downloads: BlockDownload[] = [];
    private _pendingUploadRequests = [];
    private _pendingDownloadRequests = [];
    private _downloadChunkSize: number = 1024 * 1024 * 10;

    public get downloads(): BlockDownload[] {
        return this._downloads;
    }

    public get uploads(): BlockUpload[] {
        return this._uploads;
    }

    public set downloadChunkSize(val: number) {
        validateParams(['downloadChunkSize', val, '+number']);
        this._downloadChunkSize = val;
    }
    public get downloadChunkSize(): number {
        return this._downloadChunkSize;
    }

    get pendingUploadRequests(): AdobePromise[] {
        return this._pendingUploadRequests;
    }

    get pendingDownloadRequests(): AdobePromise[] {
        return this._pendingDownloadRequests;
    }

    resetUploads() {
        this._uploads = [];
        this._pendingUploadRequests = [];
    }

    /**
     * Add BlockUpload to pending list.
     * If none others are active, start it.
     * If there are queued & running requests, set the state to waiting.
     *
     * @param upload
     */
    addAndStartUpload(upload: BlockUpload): AdobePromise<BlockUpload> {
        return this._addAndStart('upload', upload);
    }

    /**
     * Add BlockDownload to pending list.
     * If none others are active, start it.
     * If there are queued & running requests, set the state to waiting.
     *
     * @param download
     */
    addAndStartDownload(download: BlockDownload): AdobePromise<BlockDownload> {
        if (download.state !== BlockTransferStates.INITIALIZED) {
            return Promise.resolve(download) as unknown as AdobePromise;
        }
        return this._addAndStart('download', download);
    }

    startNextWaiting(type: 'upload' | 'download') {
        const transfers = type === 'upload' ? this._uploads : this._downloads;
        const doneTransfers: BlockUpload[] | BlockDownload[] = [];
        let done = false;
        for (const transfer of transfers) {
            if (!done && transfer && transfer.state === BlockTransferStates.WAITING) {
                transfer.start();
                done = true;
            } else if (
                transfer.state === BlockTransferStates.CANCELED ||
                transfer.state === BlockTransferStates.ERROR ||
                transfer.state === BlockTransferStates.FINALIZING ||
                transfer.state === BlockTransferStates.COMPLETE
            ) {
                doneTransfers.push(transfer as any);
            }
        }
        const filtered = (transfers as any).filter((transfer) => !doneTransfers.includes(transfer));
        if (type === 'download') {
            this._downloads = filtered;
        } else {
            this._uploads = filtered;
        }
    }

    private _addAndStart<
        T extends 'upload' | 'download',
        S extends BlockUpload | BlockDownload = T extends 'upload'
            ? BlockUpload
            : T extends 'download'
            ? BlockDownload
            : never,
    >(type: 'upload' | 'download', transfer: S): AdobePromise<S> {
        const transfers = (type === 'upload' ? this._uploads : this._downloads) as unknown as S[];
        const willStart = transfers.filter(
            (u) =>
                u &&
                (u.state === BlockTransferStates.NOT_INITIALIZED ||
                    u.state === BlockTransferStates.INITIALIZED ||
                    u.state === BlockTransferStates.INITIALIZING ||
                    u.state === BlockTransferStates.STARTED),
        );
        if (willStart.length === 0) {
            transfer.start();
        } else {
            transfer._setWaiting();
        }
        transfers.push(transfer);
        return transfer.promise as AdobePromise<S>;
    }
}

export const blockTransferManager = new BlockTransferManagerImpl();
