/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPTransferDocument,
    AdobeAsset,
    AdobeBlockUpload,
    AdobeComponentUploadRecord,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeRepoUploadResult,
    AdobeResponse,
    BasicLink,
    BlockTransferDocument,
    BlockTransferState,
    GetSliceCallback,
    LinkRelationKey,
    LinkSet,
    RepoMetaPatch,
    ResourceDesignator,
    SliceableData,
    UploadProgressCallback,
} from '@dcx/common-types';
import { DCXError, ProblemTypes, isAdobeDCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    EventEmitter,
    arrayBufferToString,
    generateUuid,
    getLinkHref,
    getLinkHrefTemplated,
    isAnyFunction,
    isObject,
    merge,
    mergeDeep,
    normalizeHeaders,
    pruneUndefined,
    validateParams,
} from '@dcx/util';
import { ServiceConfig, getReposityLinksCache, getService } from '../Service';
import { RepoResponseResult } from '../common';
import { HeaderKeys } from '../enum/header_keys';
import { HTTPMethods } from '../enum/http_methods';
import { LinkRelation } from '../enum/link';
import { BlockTransferMediaType } from '../enum/media_types';
import { BlockTransferProperties, Properties } from '../enum/properties';
import { _directUpload, getDefaultSliceCallback } from '../private';
import {
    DEFAULT_MAX_CONCURRENT_REQUESTS,
    _parseUploadableData,
    deserializeUploadComponentRecord,
    getDataLength,
    shouldUseBlockTransferForUpload,
} from '../util/block_transfer';
import { parseHttpResponseContent } from '../util/bulk';
import { isBlob, isTransferDocument } from '../util/duck_type';
import { parseLinkString, parseLinksFromResponseHeader } from '../util/link';
import { deserializeAsset } from '../util/serialization';
import { assertLinksContain, makeStatusValidator } from '../util/validation';
import { blockTransferManager } from './BlockTransferManager';
import { BlockTransferStates } from './common';

const dbg = newDebug('dcx:assets:blockupload');
const dbgl = newDebug('dcx:assets:blockupload:leaf');

/**
 * The BlockUpload class manages block transfers of large documents
 */
export class BlockUpload extends EventEmitter<BlockUploadEvents> implements AdobeBlockUpload {
    /**
     * Contains the result of the BlockUpload
     */
    public uploadRecord!: AdobeComponentUploadRecord;
    public createdAsset?: AdobeAsset;
    /**
     * Contains the result of the BlockUpload
     */
    public finalizeResponse: AdobeResponse | undefined;
    /**
     * An internal id assigned to every instance of this class.
     */
    private _internalBlockUploadId: string = generateUuid();
    /**
     * The HTTPService to make requests
     */
    private _service: AdobeHTTPService;
    /**
     * The promise resolve method that indicates completion for all blocks
     */
    private _resolve!: (state: AdobeBlockUpload) => void;
    /**
     * The promise rejection method that indicates an error during upload
     */
    private _reject!: (reason: AdobeDCXError) => void;
    /**
     * The overall promise which resolves when all blocks are uploaded
     */
    private _promise: AdobePromise<AdobeBlockUpload>;
    /**
     * Current state of block upload
     */
    private _state: BlockTransferState = BlockTransferStates.NOT_INITIALIZED;
    /**
     * Holds the index of the next block to be requested
     */
    private _currentBlockIndex = 0;
    /**
     * The BlockTransferDocument which specifies things like the blockSize and transfer urls
     * Links are expected to be on the transfer document once initialized. During an assetmoved
     * operation, the links are removed from an existing transfer document so that the
     * transfer document can be re-used while only the links are re-generated.
     */
    private _blockTransferDocument!: Required<BlockTransferDocument> & { links: BlockTransferDocument['links'] };
    /**
     * A accessor convenience for storing the block transfer urls
     */
    private _transferBlockLinks!: BasicLink[];
    /**
     * An array of the currently active block upload requests
     */
    private _pendingBlockRequests: AdobePromise[] = [];
    /**
     * A user provided callback responsible for slicing the buffer in chunks for block upload
     */
    private _getSliceCallback: GetSliceCallback;
    /**
     * The number of bytes that have been currently uploaded
     */
    private _bytesUploaded = 0;
    /**
     * The total number of blocks that were uploaded. Used to trim unused transfer url's from finalize body
     */
    private _totalBlocksUploaded = 0;
    /**
     * Transfer is indeterminate if the TransferDocument has been extended
     */
    private _indeterminateTransfer = false;
    /**
     * Maximum number of Concurrent upload block requests
     */
    private _maxConcurrentRequests = DEFAULT_MAX_CONCURRENT_REQUESTS;
    /**
     * On progress callback handler, if transfer is extended going into an indeterminate state
     */
    public onProgress?: UploadProgressCallback;
    /**
     * Optional Params required when clients don't call blockInit themselves
     */
    private _asset!: AdobeAsset;
    private _dataSize!: number;
    private _relationType!: LinkRelationKey;
    private _contentType!: string;
    private _componentId!: string;
    private _md5?: string;
    private _ifMatch?: string;
    private _relPath?: string;
    private _createIntermediates?: boolean;
    private _respondWith?: ResourceDesignator;
    private _repoMetaPatch?: RepoMetaPatch;

    /**
     * This class can be initialized with an existing blockTransferDocument acquired via the HLA or leaf methods
     * @param {AdobeHTTPService} service                            The HTTPService
     * @param {GetSliceCallback} getSliceCallback               The callback to slice the data buffer
     * @param {BlockTransferDocument} blockTransferDocument     The BlockTransferDocument which defined the blocks and block size
     */
    constructor(
        service: AdobeHTTPService,
        getSliceCallback: GetSliceCallback,
        blockTransferDocument: BlockTransferDocument,
    );

    /**
     * This class can be initialized with an existing blockTransferDocument acquired via the HLA or leaf methods with additional parameters
     * @param {AdobeHTTPService} service                The HTTPService
     * @param {GetSliceCallback} getSliceCallback   The callback to slice the data buffer, data sh
     * @param {Required<BlockTransferDocument>} blockTransferDocument     The BlockTransferDocument which defined the blocks and block size
     * @param {LinkRelation} relationType           (optional) The block upload relation type
     * @param {number} dataSize                     (optional) The total size of the component
     * @param {string} contentType                  (optional) The content type of the component being uploaded
     * @param {string} componentId                  (optional) The componentId to upload the component against
     * @param {string} md5                          (optional) MD5 of the data being uploaded
     * @param relPath                               (optional) the relative path must be specified.
     * @param createIntermediates                   (optional) should intermediate directories be automatically created.
     * @param respondWith                           (optional) Resource of the newly-created Asset to return in the response, if any.
     * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/advanced/block.html#completing-a-block-transfer Completing a Block Transfer}
     */
    constructor(
        service: AdobeHTTPService,
        getSliceCallback: GetSliceCallback,
        blockTransferDocument: Required<BlockTransferDocument>,
        relationType?: LinkRelationKey,
        dataSize?: number,
        contentType?: string,
        componentId?: string,
        md5?: string,
        etag?: string,
        relPath?: string,
        createIntermediates?: boolean,
        respondWith?: ResourceDesignator,
        repoMetaPatch?: RepoMetaPatch,
        maxConcurrentRequests?: number,
    );
    /**
     * This class can also be initialized with the parameters required to initialize the block transfer. The init method
     * will handle calling blockInit for clients.
     * @param {AdobeHTTPService} service                The HTTPService
     * @param {GetSliceCallback} getSliceCallback   The callback to slice the data buffer, data sh
     * @param {AdobeAsset} asset                    The asset associated with the block upload
     * @param {LinkRelation} relationType           The block upload relation type
     * @param {number} dataSize                     The total size of the component
     * @param {string} contentType                  The content type of the component being uploaded
     * @param {string} componentId                  (optional) The componentId to upload the component against
     * @param {string} md5                          (optional) MD5 of the data being uploaded
     * @param relPath                               (optional) When creating an asset, the relative path must be specified.
     * @param createIntermediates                   (optional) When creating an asset, should intermediate directories be automatically created.
     * @param respondWith                           (optional) When creating an asset, Resource of the newly-created Asset to return in the response, if any.
     * @param repoMetaPatch                         (optional) When creating an asset, it is possible to patch select fields from the Repository Metadata Resource.
     * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/advanced/block.html#completing-a-block-transfer Completing a Block Transfer}
     */
    constructor(
        service: AdobeHTTPService,
        getSliceCallback: GetSliceCallback,
        asset: AdobeAsset,
        relationType: LinkRelationKey,
        dataSize: number,
        contentType: string,
        componentId?: string,
        md5?: string,
        etag?: string,
        relPath?: string,
        createIntermediates?: boolean,
        respondWith?: ResourceDesignator,
        repoMetaPatch?: RepoMetaPatch,
        maxConcurrentRequests?: number,
    );
    /**
     * Overloaded constructor handling both cases
     * @param {AdobeHTTPService} service                                                    The HTTPService
     * @param {GetSliceCallback} getSliceCallback                                           The callback to slice the data buffer
     * @param {Required<BlockTransferDocument> | AdobeAsset} blockTransferDocumentOrAdobeAsset        (optional) The asset associated with the block upload or transfer document with all attributes required
     * @param {LinkRelation} relationType                                                   (optional) The block upload relation type
     * @param {number} dataSize                                                             (optional) The total size of the component
     * @param {string} contentType                                                          (optional) The content type of the component being uploaded
     * @param {string} componentId                                                          (optional) The componentId to upload the component against
     * @param {string} md5                                                                  (optional) MD5 of the data being uploaded
     * @param {string} relPath                                                              (optional) When creating an asset, the relative path must be specified.
     * @param repoMetaPatch                                                                 (optional) When creating an asset, it is possible to patch select fields from the Repository Metadata Resource.
     * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/advanced/block.html#completing-a-block-transfer Completing a Block Transfer}
     */
    constructor(
        service: AdobeHTTPService,
        getSliceCallback: GetSliceCallback,
        blockTransferDocumentOrAdobeAsset: Required<BlockTransferDocument> | AdobeAsset,
        relationType?: LinkRelationKey,
        dataSize?: number,
        contentType?: string,
        componentId?: string,
        md5?: string,
        etag?: string,
        relPath?: string,
        createIntermediates?: boolean,
        respondWith?: ResourceDesignator,
        repoMetaPatch?: RepoMetaPatch,
        maxConcurrentRequests?: number,
    ) {
        super(['stateChanged']);
        this._service = service;
        this._getSliceCallback = getSliceCallback;

        // Do we have a BlockTransferDocument or do we have to call blockInit?
        if (isTransferDocument(blockTransferDocumentOrAdobeAsset)) {
            validateParams([
                BlockTransferProperties.REPO_SIZE,
                blockTransferDocumentOrAdobeAsset[BlockTransferProperties.REPO_SIZE],
                'number',
            ]);

            this._blockTransferDocument = blockTransferDocumentOrAdobeAsset;
            this._transferBlockLinks = this._blockTransferDocument[Properties.LINKS][LinkRelation.BLOCK_TRANSFER];
            this._dataSize = this._blockTransferDocument[BlockTransferProperties.REPO_SIZE];
            this._relationType = this._blockTransferDocument[BlockTransferProperties.REPO_REL_TYPE];
            this._shiftState(BlockTransferStates.INITIALIZED);
            dbg(
                `BlockUpload Initialized: Transfer document found with ${this._transferBlockLinks.length} links. ` +
                    `BlockUploadId: ${this._internalBlockUploadId}`,
            );
        } else {
            // We are handling init, make sure we have what we need.
            validateParams(
                ['relationType', relationType, 'string'],
                ['dataSize', dataSize, 'number'],
                ['contentType', contentType, 'string'],
                ['componentId', componentId, 'string', true],
                ['etag', etag, 'string', true],
            );
            this._asset = blockTransferDocumentOrAdobeAsset;

            // Assert if the asset does not contain the block upload link
            assertLinksContain(
                this._asset.links,
                [LinkRelation.BLOCK_UPLOAD_INIT],
                DCXError.UNEXPECTED,
                '/rel/block/init missing from BlockTransferDocument.',
            );
            this._relationType = relationType as LinkRelationKey;
            this._dataSize = dataSize as number;
            this._contentType = contentType as string;
            this._componentId = componentId as string;
            this._md5 = md5;
            this._ifMatch = etag;
        }

        this._relPath = relPath;
        this._createIntermediates = createIntermediates;
        this._respondWith = respondWith;
        this._repoMetaPatch = repoMetaPatch;
        this._maxConcurrentRequests = maxConcurrentRequests || DEFAULT_MAX_CONCURRENT_REQUESTS;

        // Create the promise to control the entire block upload lifecycle
        this._promise = new AdobePromise((resolve, reject) => {
            this._reject = reject;
            this._resolve = resolve;
        });

        blockTransferManager.uploads.push(this);
    }

    /**
     * Initializes the block transfer, only needs to be called if a BlockTransferDocument is not provided to the constructor
     */
    public init(additionalHeaders?: Record<string, string>): AdobePromise<AdobeBlockUpload, AdobeDCXError> {
        this._assertStateIsValid('init');
        // Do we have a blockTransferDocument? If not the client is expecting us to handle the blockInit
        if (!this._blockTransferDocument || !this._blockTransferDocument[Properties.LINKS]) {
            this._shiftState(BlockTransferStates.INITIALIZING);
            const transferDocument: ACPTransferDocument = pruneUndefined(
                Object.assign(
                    {
                        [BlockTransferProperties.REPO_REL_TYPE]: this._relationType,
                        [BlockTransferProperties.REPO_IF_MATCH]: this._ifMatch,
                        [BlockTransferProperties.REPO_SIZE]: this._dataSize,
                        [BlockTransferProperties.DC_FORMAT]: this._contentType,
                        [BlockTransferProperties.COMPONENT_ID]: this._componentId,
                        [BlockTransferProperties.REPO_MD5]: this._md5,
                    },
                    this._blockTransferDocument,
                ),
            );
            return initBlockUpload(this._service, this._asset, transferDocument, additionalHeaders).then((response) => {
                this._blockTransferDocument = response.result;
                this._transferBlockLinks = this._blockTransferDocument[Properties.LINKS][LinkRelation.BLOCK_TRANSFER];
                this._shiftState(BlockTransferStates.INITIALIZED);
                dbg(
                    `BlockUpload Initialized: Transfer document found with ${this._transferBlockLinks.length} links. ` +
                        `BlockUploadId: ${this._internalBlockUploadId}`,
                );
                return this;
            });
        }

        return AdobePromise.resolve(this);
    }

    /**
     * Getter for the current state of the block_transfer
     */
    public get state(): BlockTransferState {
        return this._state;
    }

    public get promise(): AdobePromise<AdobeBlockUpload> {
        return this._promise;
    }

    /**
     * Called by clients when they wish to start the block transfer
     */
    public start(): AdobePromise<AdobeBlockUpload> {
        this._assertStateIsValid('start');
        // If we have initialized and we are the next block upload in the queue
        if (
            blockTransferManager.uploads[0] === this &&
            (this._state === BlockTransferStates.INITIALIZED || this._state === BlockTransferStates.PAUSED)
        ) {
            dbg(`Starting the transfer of BlockUpload: ${this._internalBlockUploadId}`);
            this._shiftState(BlockTransferStates.STARTED);
            this._uploadLoop();
        }
        return this._promise;
    }

    /**
     * Pauses the block transfers
     */
    public pause(): AdobePromise<AdobeBlockUpload> {
        this._assertStateIsValid('pause');
        this._shiftState(BlockTransferStates.PAUSING);
        return AdobePromise.allSettled(this._pendingBlockRequests).then(() => {
            this._shiftState(BlockTransferStates.PAUSED);
            dbg(`BlockUploading has been paused.  BlockUploadId: ${this._internalBlockUploadId}`);
            return this;
        });
    }

    /**
     * Resumes the block transfers
     */
    public resume(): AdobeBlockUpload {
        this._assertStateIsValid('resume');
        dbg(`BlockUploading has been resumed.  BlockUploadId: ${this._internalBlockUploadId}`);
        this.start();
        return this;
    }

    /**
     * Cancels the block requests
     */
    cancel() {
        this._assertStateIsValid('cancel');
        this._shiftState(BlockTransferStates.CANCELED);
        dbg(`A BlockUpload has been canceled... BlockUploadId: ${this._internalBlockUploadId}`);
        this._promise.cancel();
        this._cancel();
    }

    /**
     * Set current state to waiting.
     * Done by the BlockTransferManager.
     *
     * @internal
     */
    _setWaiting() {
        this._shiftState(BlockTransferStates.WAITING);
    }

    /**
     * Uploads the next block and adds to pendingBlockRequests
     */
    public uploadNextBlock(data: SliceableData): AdobePromise<AdobeResponse> {
        // Only add the next block if we are not paused or cancelled
        this._assertStateIsValid('uploadNextBlock');
        // Very unlikely to be true unless users are manually handling block uploads.
        /* istanbul ignore if  */
        if (this._isEmptyBlock(data)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Trying to upload empty data block.');
        }

        // Get the block upload link for this block
        const currentBlockHref =
            this._blockTransferDocument[Properties.LINKS][LinkRelation.BLOCK_TRANSFER][this._currentBlockIndex].href;

        // Cleanup callback
        let removeReq: () => void; //eslint-disable-line prefer-const

        // Upload the block
        dbg(`Uploading a block... BlockUploadId: ${this._internalBlockUploadId}`);
        const dataLength = getDataLength(data);
        const blockRequestPromise = this._uploadBlock(data, currentBlockHref)
            .then((res) => {
                this._totalBlocksUploaded++;
                this._updateProgress(dataLength);

                // Cleanup
                removeReq();
                dbg(
                    `A block has completed... ${this._pendingBlocksCount} requests still active. ` +
                        `BlockUploadId: ${this._internalBlockUploadId}`,
                );
                return res;
            })
            .catch((error) => {
                dbg(`A block upload has failed. BlockUploadId: ${this._internalBlockUploadId}`);
                removeReq();
                this._shiftState(BlockTransferStates.ERROR);
                // If fails, cancel & reject
                this._reject(
                    new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'A block has failed during upload',
                        error,
                        error.response,
                    ),
                );
                this.cancel();
            });

        // Push request promise onto pending block requests stack and set cleanup callback
        removeReq = this._pushPendingBlockRequest(blockRequestPromise);
        return blockRequestPromise;
    }

    /**
     * Returns the number of currently active transfer requests across any BlockUpload instance
     */
    private get _pendingBlocksCount() {
        return blockTransferManager.pendingUploadRequests.filter((p) => !!p).length;
    }

    /**
     * Returns the blockLockPromise which indicates if we are blocked from any new requests
     */
    private _nextBlockLock(): Promise<void> {
        if (this._pendingBlocksCount < this._maxConcurrentRequests) {
            return Promise.resolve();
        }

        // If a promise is resolved from any active BlockUpload instance, unblock.
        // If a BlockUpload class has completed or already sent all the required requests,
        // this unlocking is should be disregarded.
        return Promise.race(blockTransferManager.pendingUploadRequests.filter((p) => !!p));
    }
    /**
     * Handles the assetmoved problem type returned by ACP when an asset has migrated from one
     * region to another before the block upload process has finished. When the assetmoved
     * problem type is encountered, it is necessary to repeat the entire block upload process,
     * including initialization. This is being done within the block upload class to maintain
     * priority order with the block transfer manager and re-use any existing objects created
     * on the heap for the purposes of this upload.
     */
    private _handleAssetMoved(error) {
        dbg('_handleAssetMoved');
        // abort any pendingBlocks
        this._pendingBlockRequests.forEach((request) => {
            request.abort();
        });
        // reset existing arrays
        this._pendingBlockRequests.length = 0;
        this._transferBlockLinks.length = 0;
        // extract links and update if necessary
        this._asset.links = { ...this._asset.links, ...parseLinksFromResponseHeader(error.response) };
        // reset block index
        this._currentBlockIndex = 0;
        this._bytesUploaded = 0;
        this._state = BlockTransferStates.NOT_INITIALIZED;
        // reset blockTransfer links on the transfer document
        this._blockTransferDocument[Properties.LINKS] = undefined;
        // re-init the block transfer
        this.init().then(() => {
            return this.start();
        });
        // because assetmoved is invoked within the context of the upload loop,
        // we throw to interrupt remaining behavior in the upload loop and allow the above
        // start method to trigger a new uploadLoop from the beginning again.
        throw BlockTransferStates.INITIALIZING;
    }
    /**
     * Recursive function which attempts to add the next block for upload.
     */
    private _uploadLoop() {
        // Uploading the next block is blocked by a lock which only opens if the current
        // active requests is < MaxConcurrentRequests (default = 4). If we are at max requests
        // then we await the lock to open before adding the next. The loop continues until there are
        // no more blocks to upload.
        this._nextBlockLock()
            //Break loop if in finalized/finalizing state
            .then(() => {
                /* istanbul ignore if  */
                if (this._state === BlockTransferStates.FINALIZING || this._state === BlockTransferStates.COMPLETE) {
                    throw BlockTransferStates.COMPLETE;
                } else if (this._state === BlockTransferStates.PAUSING || this._state === BlockTransferStates.PAUSED) {
                    // We are paused, let's break the upload loop.
                    throw BlockTransferStates.PAUSED;
                } else if (this._state === BlockTransferStates.CANCELED) {
                    throw BlockTransferStates.CANCELED;
                } else if (this._state === BlockTransferStates.ERROR) {
                    throw BlockTransferStates.ERROR;
                }
            })
            // Get the next block based on _currentBlockIndex
            .then(() => this._getBlockAtIndex(this._currentBlockIndex))
            .then<SliceableData | void>((blockData) => {
                if (this._isEmptyBlock(blockData)) {
                    // Check if the next block is valid
                    dbg(`No more blocks.  BlockUploadId: ${this._internalBlockUploadId}`);

                    // Block returned is not valid, wait for all pending requests in this
                    // BlockUpload instance to complete and then finialize.
                    // ACP may respond with a 409 assetmoved on finalize
                    // Finalize will throw BlockTransferState.INITIALIZING when that happens
                    // to signal to the upload loop that it should be restarting
                    return Promise.all(this._pendingBlockRequests).then(this._finalize.bind(this));
                } else if (
                    blockData &&
                    getDataLength(blockData) > 0 &&
                    this._currentBlockIndex >= this._transferBlockLinks.length
                ) {
                    // When performing an extend, a 409 assetmoved error may be returned
                    // in those instances the block transfer must be re-initialized and retried from the beginning
                    // This is handled separately by the
                    // otherwise the transfer document is extended by 150% and the existing blockData is used
                    return this._extend().then(() => {
                        return blockData;
                    });
                }

                return blockData;
            })
            .then((blockData) => {
                // Did the last step return undefined for the next block? If so we are done.
                if (!blockData) {
                    throw BlockTransferStates.COMPLETE;
                }

                // Kick off the next upload, don't await the request to finish, handled by blockLock.
                this.uploadNextBlock(blockData);

                // Increment current block index
                this._currentBlockIndex++;
            })
            .then(() => {
                // Start the upload loop again, if we are at max pending requests it will await one to finish
                this._uploadLoop();
            })
            .catch((e) => {
                // If an error was throw somewhere else in the loop
                if (typeof e !== 'string') {
                    this._continueBlockUploads();
                    this._reject(e);
                }
                if (e === BlockTransferStates.INITIALIZING) {
                    dbg(
                        `BlockUpload loop must be re-started due to assetmoved BlockUploadId: ${this._internalBlockUploadId}`,
                    );
                    return;
                }
                // Did we throw COMPLETE in order to break the loop?
                if (e === BlockTransferStates.COMPLETE) {
                    dbg(`BlockUpload loop is complete. BlockUploadId: ${this._internalBlockUploadId}`);
                    return;
                } else if (e === BlockTransferStates.PAUSED) {
                    dbg(
                        `BlockUpload loop is terminated due to paused state. ` +
                            `BlockUploadId: ${this._internalBlockUploadId}`,
                    );
                    return;
                } else if (e === BlockTransferStates.CANCELED) {
                    dbg(
                        `BlockUpload loop is terminated due to the upload being canceled. ` +
                            `BlockUploadId: ${this._internalBlockUploadId}`,
                    );
                    this._continueBlockUploads();
                    return;
                } else if (e === BlockTransferStates.ERROR) {
                    dbg(
                        `BlockUpload loop is terminated due to error state. ` +
                            `BlockUploadId: ${this._internalBlockUploadId}`,
                    );
                    this._continueBlockUploads();
                    return;
                }
                // Some unknown error
            });
    }

    /**
     * Returns the buffer slice for the block at an index
     * @param blockIndex    The block index to get the slice for
     */
    private _getBlockAtIndex(blockIndex: number): Promise<SliceableData> | undefined {
        dbg(`_getBlockAtIndex(${blockIndex})`);
        // Assert if we are not in the STARTED state
        // this._assertStateIsValid('getBlockAtIndex');
        const size = Math.min(this._dataSize, this._blockTransferDocument[BlockTransferProperties.REPO_BLOCK_SIZE]);
        if (this._state === BlockTransferStates.STARTED) {
            const startByte = blockIndex * size;
            dbg('calling _getSliceCallback', startByte, startByte + size);
            return this._getSliceCallback(startByte, startByte + size).catch((error) => {
                throw new DCXError(
                    DCXError.UNEXPECTED_RESPONSE,
                    'The getSliceCallback threw an unexpected error.',
                    error,
                );
            });
        }
    }

    /**
     * Uploads the block
     * @param body    Data frame to upload
     * @param href      The upload href
     */
    private _uploadBlock(
        body: Blob | Buffer | ArrayBuffer | ArrayBufferView | string,
        href: string,
    ): AdobePromise<AdobeResponse> {
        const promise = this._service.invoke(HTTPMethods.PUT, href, undefined, body, {
            isStatusValid: makeStatusValidator(),
            isExternalRequest: true,
        });
        return promise;
    }

    /**
     * Returns true if the buffer passed is empty or invalid
     * @param block     The data to check for validity
     */
    private _isEmptyBlock(block: SliceableData | undefined): boolean {
        if (typeof block === 'string') {
            return block.length === 0;
        }
        if (isBlob(block)) {
            return block.size === 0;
        }
        return !block || block.byteLength === 0;
    }

    /**
     * Extends the block transfer by 150%
     */
    private _extend(): AdobePromise<AdobeResponse, AdobeDCXError> {
        assertLinksContain(
            this._blockTransferDocument[Properties.LINKS],
            [LinkRelation.BLOCK_EXTEND],
            DCXError.UNEXPECTED,
            'The transfer document does not contain an extend href',
        );

        // Round up, decimal size values are not supported in extend.
        const extendSize = Math.ceil(this._blockTransferDocument[BlockTransferProperties.REPO_SIZE] * 1.5);
        const extendHref = getLinkHrefTemplated(
            this._blockTransferDocument[Properties.LINKS],
            LinkRelation.BLOCK_EXTEND,
            {
                size: extendSize,
            },
        );

        return this._service
            .invoke(HTTPMethods.POST, extendHref, {}, undefined, {
                isStatusValid: makeStatusValidator(),
                responseType: 'json',
            })
            .then((result) => {
                // Save response as new BlockTransferDocument
                this._indeterminateTransfer = true;
                this._blockTransferDocument = result.response;
                this._transferBlockLinks = this._blockTransferDocument[Properties.LINKS][LinkRelation.BLOCK_TRANSFER];
                dbg(
                    `Transfer document was extended to ${this._transferBlockLinks.length} transfer links. ` +
                        `BlockUploadId: ${this._internalBlockUploadId}`,
                );
                return result;
            })
            .catch((error) => {
                if (isAdobeDCXError(error) && error.problemType === ProblemTypes.ASSET_MOVED) {
                    this._handleAssetMoved(error);
                }
                throw new DCXError(
                    DCXError.UNEXPECTED_RESPONSE,
                    'An unexpected error occurred while extending the block transfer document.',
                    error,
                    error.response,
                );
            });
    }

    /**
     * Adds the blockRequest to the pending array and returns a cleanup method
     * @param blockRequest     The block request to add
     */
    private _pushPendingBlockRequest(blockRequest: AdobePromise): () => void {
        const i = this._pendingBlockRequests.push(blockRequest);
        const j = blockTransferManager.pendingUploadRequests.push(blockRequest);
        return () => {
            // Subtract 1 since array index starts at 0 & push return array length
            // Remove from internal pending block requests
            delete this._pendingBlockRequests[i - 1];

            // Remove from upload manager pending block requests
            delete blockTransferManager.pendingUploadRequests[j - 1];
        };
    }

    /**
     * Send progress event.
     *
     * If incrementOrDone is true, send the final progress event with 100% completion.
     * If it's a number, add it to bytes sent.
     *
     * @param incrementOrDone
     *
     */
    private _updateProgress(incrementOrDone: number | true) {
        dbg('_updateProgress()', incrementOrDone);

        if (typeof incrementOrDone === 'number') {
            this._bytesUploaded += incrementOrDone;
        }

        if (this.onProgress && isAnyFunction(this.onProgress)) {
            try {
                this.onProgress(
                    this._bytesUploaded,
                    Math.max(this._blockTransferDocument[BlockTransferProperties.REPO_SIZE], this._bytesUploaded),
                    this._indeterminateTransfer,
                );
            } catch (e) {
                console.error('Error in onProgress callback', e);
            }
        }
    }

    /**
     * Private cancel method
     */
    private _cancel() {
        // TODO: return promise for things getting cancelled

        this._pendingBlockRequests.map((p) => {
            p && p.cancel();
        });

        // Don't resolve if we hit an error, let's the catch method reject
        if (this._state !== BlockTransferStates.ERROR) {
            this._resolve(this);
        }
    }

    /**
     * Asserts if the current state is valid
     * @param method    The method asserting
     */
    private _assertStateIsValid(
        method: 'init' | 'start' | 'getBlockAtIndex' | 'uploadNextBlock' | 'pause' | 'resume' | 'cancel',
        stateTest?: string,
    ) {
        // If state test is sent in use that, use for unit tests.
        const state = stateTest || this._state;
        switch (method) {
            case 'init':
                if (state !== BlockTransferStates.NOT_INITIALIZED && state !== BlockTransferStates.INITIALIZED) {
                    throw new DCXError(DCXError.INVALID_STATE, 'BlockUpload has already been initialized');
                }
                break;
            case 'start':
                if (state === BlockTransferStates.NOT_INITIALIZED) {
                    throw new DCXError(DCXError.INVALID_STATE, 'Please call init before starting the block upload');
                }
                break;
            case 'uploadNextBlock':
                if (state === BlockTransferStates.PAUSED || state === BlockTransferStates.CANCELED) {
                    throw new DCXError(DCXError.INVALID_STATE, 'Cannot add block when Paused or Cancelled');
                }
                break;
            case 'getBlockAtIndex':
                if (state !== BlockTransferStates.STARTED) {
                    throw new DCXError(DCXError.INVALID_STATE, `Cannot fetch block while in the ${state} state`);
                }
                break;
            case 'cancel':
                if (
                    state !== BlockTransferStates.STARTED &&
                    state !== BlockTransferStates.FINALIZING &&
                    state !== BlockTransferStates.PAUSING &&
                    state !== BlockTransferStates.PAUSED &&
                    state !== BlockTransferStates.ERROR
                ) {
                    throw new DCXError(DCXError.INVALID_STATE, `Trying to cancel while in an invalid state ${state}`);
                }
        }
    }

    private _continueBlockUploads() {
        dbg('continueBlockUploads()');
        // Only allow this instance to shift itself off the stack
        if (blockTransferManager.uploads[0] === this) {
            // Shift pendingBlockUploads array, start next blockUpload in array if any
            blockTransferManager.uploads.shift();
            if (blockTransferManager.uploads.length > 0) {
                const blockUpload = blockTransferManager.uploads[0];
                dbg(`Another block upload found in the queue, starting...`);
                blockUpload.start();
            } else if (this._pendingBlocksCount === 0) {
                dbg(`There are no more pending block transfers.. Clean up blockUploadManager..`);
                blockTransferManager.resetUploads();
            }
        }
    }

    /**
     * Finalize the blockupload, calls finalize and polls for result
     */
    private _finalize(): AdobePromise<void, AdobeDCXError> {
        dbg(`Finalizing block transfer.  BlockUploadId: ${this._internalBlockUploadId}`);
        this._shiftState(BlockTransferStates.FINALIZING);
        const finalizeHref = getLinkHrefTemplated(
            this._blockTransferDocument[Properties.LINKS],
            LinkRelation.BLOCK_FINALIZE,
            pruneUndefined({
                path: this._relPath,
                intermediates: this._createIntermediates,
                respondWith: isObject(this._respondWith) ? JSON.stringify(this._respondWith) : this._respondWith,
                repoMetaPatch: this._repoMetaPatch,
            }),
        );

        // Only send include the transfer URL's used in the finalize body
        this._blockTransferDocument[Properties.LINKS][LinkRelation.BLOCK_TRANSFER] = this._transferBlockLinks.slice(
            0,
            this._totalBlocksUploaded,
        );
        // Kick off finalize, polling for result
        return this._service
            .invoke(
                HTTPMethods.POST,
                finalizeHref,
                { [HeaderKeys.CONTENT_TYPE]: BlockTransferMediaType },
                JSON.stringify(pruneUndefined(this._blockTransferDocument)),
                {
                    isStatusValid: makeStatusValidator(),
                    retryOptions: { pollHeader: 'location', pollCodes: [202], timeoutAfter: 120000 },
                    responseType: 'arraybuffer',
                },
            )
            .then((result) => {
                // Did we get back OK?
                /* istanbul ignore else  */
                if (result.statusCode === 200 || result.statusCode === 201) {
                    // The Resources was successfully uploaded.
                    dbg(`Finalize complete.  BlockUploadId: ${this._internalBlockUploadId}`);
                    const response = parseHttpResponseContent(new Uint8Array(result.response));
                    if (this._relationType === LinkRelation.PRIMARY) {
                        this.finalizeResponse = response;
                        this.createdAsset = pruneUndefined({
                            assetId: response.headers['asset-id'],
                            repositoryId: response.headers['repository-id'],
                            links: parseLinkString(response.headers.link),
                            etag: response.headers['etag'],
                            md5: response.headers['content-md5'],
                        });
                        if (response.response && this._respondWith) {
                            try {
                                const responseJSON = JSON.parse(arrayBufferToString(response.response));
                                response.response = responseJSON;
                                this.createdAsset = mergeDeep(this.createdAsset, deserializeAsset(responseJSON));
                            } catch (error) {
                                throw new DCXError(
                                    DCXError.UNEXPECTED,
                                    'Unexpected error parsing respondWith parameter',
                                    error as Error,
                                );
                            }
                        }
                    } else {
                        this.finalizeResponse = result;
                        try {
                            this.uploadRecord = deserializeUploadComponentRecord(
                                this._blockTransferDocument[BlockTransferProperties.DC_FORMAT],
                                this._blockTransferDocument[BlockTransferProperties.COMPONENT_ID],
                                this._bytesUploaded,
                                arrayBufferToString(result.response),
                            );
                        } catch (error) {
                            throw new DCXError(
                                DCXError.UNEXPECTED,
                                'An error occurred while deserializing upload component record.',
                                error as Error,
                                response,
                            );
                        }
                    }
                    this._shiftState(BlockTransferStates.COMPLETE);
                    this._updateProgress(true);
                    this._resolve(this);
                }
                this._continueBlockUploads();
            })
            .catch((error) => {
                if (isAdobeDCXError(error) && error.problemType === ProblemTypes.ASSET_MOVED) {
                    this._handleAssetMoved(error);
                }
                // We hit an error, reject
                dbg(`Error occurred finalizing the block transfer.. Rejecting`);
                this._reject(
                    new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'An error occurred while finalizing the block transfer.',
                        error,
                        error.response,
                    ),
                );
                this._continueBlockUploads();
            });
    }

    /**
     * Move to a new state, finalize if needed.
     * Skips state change if already in complete or error states.
     *
     * @param newState
     */
    private _shiftState(newState: BlockTransferState): BlockUpload {
        dbg(`_shiftState(): ${newState}`);

        if (
            this._state === BlockTransferStates.COMPLETE ||
            this._state === BlockTransferStates.ERROR ||
            this._state === BlockTransferStates.CANCELED
        ) {
            return this;
        }

        this._state = newState;
        this.emit('stateChanged', [this._state]);

        return this;
    }
}

/**
 * Initializes a control message (via the Block Initiate URL), in which the user agent
 * tells the service that it wishes to transfer content, and the service tells the user
 * agent how to go about doing that.
 * @param svc                       HTTPService instance
 * @param asset                     The asset the component is related to
 * @param transferDocument          The transfer document which is used to initialize, process, and complete block uploads.
 * @param additionalHeaders         (Optional) Any additional header to include in the request
 */
export function initBlockUpload(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    transferDocument: ACPTransferDocument,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<Required<BlockTransferDocument>>, AdobeDCXError> {
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['transferDocument', transferDocument, 'object'],
        ['additionalHeaders', additionalHeaders, 'object', true],
    );

    assertLinksContain(asset.links, [LinkRelation.BLOCK_UPLOAD_INIT]);

    // TODO: Should be removed once ACP supports Resource Designator in the transfer document
    if (transferDocument['repo:resource']) {
        if (!transferDocument['repo:resource'].reltype) {
            throw new DCXError(DCXError.INVALID_DATA, 'reltype param is required in the Resource Designator');
        }

        transferDocument[BlockTransferProperties.REPO_REL_TYPE] = transferDocument['repo:resource'].reltype;

        if (transferDocument['repo:resource'].component_id) {
            transferDocument[BlockTransferProperties.COMPONENT_ID] = transferDocument['repo:resource'].component_id;
        }
        if (transferDocument['repo:resource'].etag) {
            transferDocument[BlockTransferProperties.REPO_IF_MATCH] = transferDocument['repo:resource'].etag;
        }
        delete transferDocument['repo:resource'];
    }

    //If we are block uploading a component, include the componentId in the transfer document.
    if (transferDocument[BlockTransferProperties.REPO_REL_TYPE] === LinkRelation.COMPONENT) {
        if (!transferDocument[BlockTransferProperties.COMPONENT_ID]) {
            throw new DCXError(DCXError.INVALID_DATA, 'Component Id required to block upload to a component');
        }
    }

    // Initialize block transfer
    const blockUploadInitLink = getLinkHref(asset.links, LinkRelation.BLOCK_UPLOAD_INIT);
    additionalHeaders[HeaderKeys.CONTENT_TYPE] = BlockTransferMediaType;

    const headers = normalizeHeaders(additionalHeaders) as Record<string, string>;
    // DCX-11103 requests that the default for all block transfers is to request transfer acceleration
    // Because the `repo:accelerated` parameter is suggested to be deprecated as part of CA-967,
    // we are opting for setting the priority header on the block upload init request to trigger
    // RAPI to utilize EALinks, which also achieve the same transfer acceleration per CSA-374.
    // Consumers are still able to override the priority value by providing one of their own.
    headers.priority = headers.priority || 'u=1';
    return svc
        .invoke(HTTPMethods.POST, blockUploadInitLink, headers, JSON.stringify(transferDocument), {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            return {
                response,
                result: response.response as Required<BlockTransferDocument>,
            };
        });
}

/**
 * @internal
 * @private
 */
type _UploadParams = {
    additionalHeaders?: Record<string, string>;
    asset: AdobeAsset;
    componentId?: string;
    contentType: string;
    dataOrSliceCallback: SliceableData | GetSliceCallback;
    etag?: string;
    maybeIsNew?: boolean;
    md5?: string;
    progressCb?: UploadProgressCallback;
    relation: LinkRelationKey;
    size?: number;
    svc: AdobeHTTPService | ServiceConfig;
    blockSize?: number;
    maxConcurrentRequests?: number;
};

/**
 * Perform an upload, Will attempt to use block transfer if the asset size is greater than the block transfer threshold.
 * @internal
 * @private
 */
export function _upload({
    additionalHeaders,
    asset,
    componentId,
    contentType,
    dataOrSliceCallback,
    etag,
    maybeIsNew,
    md5,
    progressCb,
    relation,
    size,
    svc,
    blockSize,
    maxConcurrentRequests,
}: _UploadParams): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }> {
    dbgl('_upload()');
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['size', size, 'number', true],
        ['md5', md5, 'string', true],
        ['etag', etag, 'string', true],
    );
    const uploadableData = _parseUploadableData(dataOrSliceCallback, size);
    const dataSize = getDataLength(uploadableData);
    const service = getService(svc);
    if (shouldUseBlockTransferForUpload(asset, dataSize)) {
        return _doBlockUpload({
            asset,
            additionalHeaders,
            componentId,
            contentType,
            dataOrSliceCallback,
            etag,
            md5,
            progressCb,
            relation,
            size,
            service,
            blockSize,
            maxConcurrentRequests,
        });
    }
    // We are going to perform a direct upload, let's make sure we have the link we need.
    assertLinksContain(asset.links, [relation]);

    // Block upload not required, direct upload instead
    const href = getLinkHrefTemplated(asset.links, relation, { component_id: componentId });
    return AdobePromise.resolve(undefined, { blockUpload: undefined, progress: progressCb })
        .then(() => {
            // If we were provided a callback function, call it to get the data, otherwise return the data
            return Promise.resolve(
                isAnyFunction(dataOrSliceCallback) ? dataOrSliceCallback(0, dataSize) : dataOrSliceCallback,
            );
        })
        .then((data) =>
            _directUpload({
                asset,
                additionalHeaders,
                contentType,
                data,
                etag,
                headHref: href,
                href,
                maybeIsNew,
                relation,
                service,
            }),
        )
        .then((response) => {
            let links: LinkSet = {};
            try {
                links = parseLinksFromResponseHeader(response);
                asset.links = merge(asset.links || {}, links);
                const cache = getReposityLinksCache(svc);
                if (cache) {
                    cache.setValueWithAsset(asset.links!, asset);
                }
            } catch (_) {
                // noop
            }
            /* istanbul ignore next */
            return {
                response: response,
                result: {
                    revision: response.headers['revision'] || response.headers['version'],
                    location: response.headers['location'],
                    links,
                    etag: response.headers['etag'],
                    version: response.headers['version'] || response.headers['revision'],
                    md5: response.headers['md5'] || response.headers['content-md5'],
                    length: dataSize,
                    type: contentType,
                },
                isBlockUpload: false,
                asset: {
                    assetId: asset.assetId || response.headers['asset-id'],
                    repositoryId: asset.repositoryId || response.headers['repository-id'],
                    links: asset.links || links,
                },
            };
        })
        .catch((error) => {
            //Some other unexpected error
            /* istanbul ignore next  */
            throw error;
        });
}

/**
 * @internal
 * @private
 */
type _DoBlockUploadParams = {
    service: AdobeHTTPService;
    asset: AdobeAsset;
    additionalHeaders?: Record<string, string>;
    dataOrSliceCallback: SliceableData | GetSliceCallback;
    contentType: string;
    progressCb?: UploadProgressCallback;
    relation: LinkRelationKey;
    size?: number;
    componentId?: string;
    md5?: string;
    etag?: string;
    relPath?: string;
    createIntermediates?: boolean;
    respondWith?: ResourceDesignator;
    blockSize?: number;
    repoMetaPatch?: RepoMetaPatch;
    maxConcurrentRequests?: number;
};

/**
 * Perform a block upload.
 * @internal
 * @private
 */
export function _doBlockUpload({
    service,
    asset,
    additionalHeaders,
    dataOrSliceCallback,
    contentType,
    progressCb,
    relation,
    size,
    componentId,
    md5,
    etag,
    relPath,
    createIntermediates,
    respondWith,
    blockSize,
    repoMetaPatch,
    maxConcurrentRequests,
}: _DoBlockUploadParams): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload: AdobeBlockUpload }> {
    if (!size) {
        throw new DCXError(
            DCXError.INVALID_DATA,
            `A size estimate is required when a GetSliceCallback is provided. ` +
                `The size is used to determine the number of requests required to block transfer the asset.`,
        );
    }

    const getSliceCallback = isAnyFunction(dataOrSliceCallback)
        ? (dataOrSliceCallback as GetSliceCallback)
        : getDefaultSliceCallback(dataOrSliceCallback);

    assertLinksContain(asset.links, [LinkRelation.BLOCK_UPLOAD_INIT]);

    const promise = initBlockUpload(
        service,
        asset,
        pruneUndefined({
            [BlockTransferProperties.REPO_REL_TYPE]: relation,
            [BlockTransferProperties.REPO_IF_MATCH]: etag,
            [BlockTransferProperties.REPO_SIZE]: size,
            [BlockTransferProperties.DC_FORMAT]: contentType,
            [BlockTransferProperties.COMPONENT_ID]: componentId,
            [BlockTransferProperties.REPO_MD5]: md5,
            [BlockTransferProperties.REPO_BLOCK_SIZE]: blockSize,
        }),
        additionalHeaders,
    );

    return promise
        .then((response) => {
            const blockUpload = new BlockUpload(
                service,
                getSliceCallback,
                response.result,
                relation,
                size,
                contentType,
                componentId,
                md5,
                etag,
                relPath,
                createIntermediates,
                respondWith,
                repoMetaPatch,
                maxConcurrentRequests,
            );
            blockUpload.onProgress = progressCb;
            Object.assign(promise, { blockUpload });

            return blockUpload.init(additionalHeaders);
        })
        .then((blockUpload) => {
            return blockUpload.start();
        })
        .then((blockUpload) => {
            const finalizeResponse = blockUpload.finalizeResponse || { headers: {} };

            return {
                response: finalizeResponse as AdobeResponse<'json'>,
                result: blockUpload.uploadRecord! || blockUpload.createdAsset,
                blockUpload: blockUpload,
                isBlockUpload: true,
                asset: {
                    assetId: finalizeResponse.headers['asset-id'] || asset.assetId,
                    repositoryId: finalizeResponse.headers['repository-id'] || asset.repositoryId,
                    etag: finalizeResponse.headers['etag'],
                    links: asset.links,
                },
            };
        });
}

export type BlockUploadEvents = {
    stateChanged: (state: BlockTransferState) => void;
};
