/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPTransferDocument,
    AdobeAsset,
    AdobeBlockUpload,
    AdobeDCXComposite,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeRepoUploadResult,
    AdobeResponse,
    AdobeResponseType,
    BlockTransferDocument,
    ComponentResourceDesignator,
    ComponentUploadInfo,
    GetSliceCallback,
    JSONPatchDocument,
    LinkRelationKey,
    LinkSet,
    ManifestData,
    RepoDownloadStreamableReturn,
    ResponseTypeMap,
    SliceableData,
    UploadProgressCallback,
    UploadRecipe,
} from '@dcx/common-types';
import { DCXError, ProblemTypes, _responseToError } from '@dcx/error';
import { AdobeDCXLogger, newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    arrayBufferToString,
    chunkArray,
    expandURITemplate,
    getLinkHref,
    getLinkHrefTemplated,
    isObject,
    normalizeHeaders,
    pruneUndefined,
    validateParams,
    verifyUuid,
} from '@dcx/util';
import { fetchLinksIfMissing } from './Asset';
import { _doBlockDownload } from './BlockTransfer/BlockDownload';
import { _upload, initBlockUpload } from './BlockTransfer/BlockUpload';
import { File, getVersionResource } from './File';
import { ServiceConfig, getReposityLinksCache, getService } from './Service';
import { RepoResponse, RepoResponseResult } from './common';
import { EmbeddedMetadataRepresentation, patchEmbeddedMetadata, putEmbeddedMetadata } from './embedded_metadata';
import { AssetType, AssetTypes } from './enum/asset_types';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { JSONPatchMediaType, JSONProblemMediaType, ManifestMediaType } from './enum/media_types';
import { BlockTransferProperties, Properties } from './enum/properties';
import { PathOrIdAssetDesignator } from './operations';
import { _getUrlFallbackDirect } from './private';
import { getBlockDownloadThreshold, shouldUseBlockTransferForUpload } from './util/block_transfer';
import { BulkRequestDescriptor, performBulkRequest } from './util/bulk';
import { isAdobeDCXCompositeLike, isBufferLike, isServiceConfig } from './util/duck_type';
import { parseLinksFromResponseHeader } from './util/link';
import { assertLinksContain, makeStatusValidator } from './util/validation';
const dbg = newDebug('dcx:assets:composite');
const dbgl = newDebug('dcx:assets:composite:leaf');
const logger = AdobeDCXLogger.getInstance();

type AdobeCompositeData = AdobeAsset;

export interface AdobeManifestResult<T> {
    manifestData: T;
    manifestEtag: string;
}

export type ComponentRequestByPath = {
    component_path: string;
    responseType?: AdobeResponseType;
    skipBlockDownload?: boolean;
    subrequestHeaders?: Record<string, string>;
};

// Merge the response and data to the request to use as an individual ComponentResponse
type ComponentResponse<C extends ComponentRequestByPath> = C &
    RepoResponse<{
        data?: C['responseType'] extends AdobeResponseType
            ? ResponseTypeMap[C['responseType']]
            : ResponseTypeMap['defaultbuffer'];
        error?: DCXError;
    }>;

// Internal Utility type to convert a union of types to an intersection of types
type UnionToIntersection<U> = (U extends any ? (k: U) => any : never) extends (k: infer I) => any ? I : never;

// Internal Utility type to map the response
type ToResponseObject<C> = C extends ComponentRequestByPath
    ? { [P in C['component_path']]: ComponentResponse<C> }
    : never;

// Object with `response` (AdobeResponse) and `results` members
export type ManifestAndComponentsByPathResponse<C extends ComponentRequestByPath[]> = {
    manifest: RepoResponse<{
        data?: ManifestData;
        error?: DCXError<unknown>;
    }>;
    components: UnionToIntersection<{ [I in keyof C]: ToResponseObject<C[I]> }[number]>;
    responses: AdobeResponse[];
};

export interface AdobeComposite extends File {
    headManifest(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError>;

    getManifest<T = Record<string, unknown>>(
        version: string,
        etag: string,
        additionalHeaders: Record<string, string>,
    ): AdobePromise<RepoResponse<AdobeManifestResult<T>, 'json'>, AdobeDCXError>;

    getManifestUrl(versionId?: string): AdobePromise<string, AdobeDCXError>;

    getManifestAndComponentsByPath<C extends ComponentRequestByPath[]>(
        components: C,
        version?: string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<ManifestAndComponentsByPathResponse<C>, AdobeDCXError>;

    getComponentUrl(componentId: string, version?: string): AdobePromise<string, AdobeDCXError>;

    getComponent<T extends AdobeResponseType = 'defaultbuffer'>(
        componentId: string,
        version?: string,
        responseType?: T,
        additionalHeaders?: Record<string, string>,
        componentSize?: number,
    ): RepoDownloadStreamableReturn<T>;

    updateManifest<T = string | Record<string, unknown>>(
        manifest: T,
        overwrite: boolean,
        validationLevel?: number,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse, AdobeDCXError>;

    putComponent(
        componentId: string,
        data: Buffer | ArrayBuffer | Blob | string,
        contentType: string,
        maybeIsNew?: boolean,
        size?: number,
        md5?: string,
        progressCb?: UploadProgressCallback,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }>;

    putComponent(
        componentId: string,
        getSliceCallback: GetSliceCallback,
        contentType: string,
        maybeIsNew?: boolean,
        size?: number,
        md5?: string,
        progressCb?: UploadProgressCallback,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }>;

    putEmbeddedMetadata<T = Record<string, unknown>>(
        data: T,
        etag: string | undefined,
        format: EmbeddedMetadataRepresentation,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>>;

    patchEmbeddedMetadata(
        data: JSONPatchDocument | string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>>;

    copy(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
        manifestPatch?: JSONPatchDocument,
    ): AdobePromise<RepoResponseResult<Composite, 'json'>, AdobeDCXError>;
}

export class Composite extends File implements AdobeComposite {
    readonly type: AssetType = AssetTypes.Composite;

    constructor(data: AdobeCompositeData, svc: AdobeHTTPService | ServiceConfig, links?: LinkSet) {
        super(data, svc, links);
    }
    /**
     * Performs a HEAD operation on the composites manifest
     */
    headManifest(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError> {
        dbg('headManifest()');

        return this.fetchLinksIfMissing([LinkRelation.MANIFEST], additionalHeaders)
            .then(() => headCompositeManifest(this._svc, this, additionalHeaders))
            .then((res) => this._updateDataWithResponse(res));
    }
    /**
     * Returns the composites manifest
     * @param {string} version                              The version of the manifest to fetch
     * @param {string} etag                                 ETag of manifest version
     * @param {Object} additionalHeaders                    Additional headers to be applied to RAPI requests
     */
    getManifest<T = Record<string, unknown>>(
        version?: string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponse<AdobeManifestResult<T>, 'json'>, AdobeDCXError> {
        dbg('getManifest()');

        validateParams(['version', version, 'string', true], ['etag', etag, 'string', true]);

        return this.fetchLinksIfMissing([LinkRelation.MANIFEST], additionalHeaders).then(() => {
            return getCompositeManifest<T>(this._svc, this, version, etag, additionalHeaders);
        });
    }
    /**
     * Returns the URL of the composites manfest
     * @param {string} versionId                            The version of the manifest to return a link for
     */
    getManifestUrl(versionId: string): AdobePromise<string, AdobeDCXError> {
        dbg('getManifestUrl()');

        // No version id, will use the manifest link.
        // With version id, need to fetch the version resource, so page is required.
        const requiredLink = versionId == null ? LinkRelation.MANIFEST : LinkRelation.PAGE;

        return this.fetchLinksIfMissing([requiredLink]).then(() => getCompositeManifestUrl(this._svc, this, versionId));
    }
    /**
     * @param components An array of ComponentRequestsByPath. NOTE: Paths are restricted to well-known paths only as determined by specific composite media type.
     * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Download+Components+by+DCX+Path Proposal: Download DCX Components by Path}
     * If using components that do not exist on the given well-known paths, please use @see performBulkRequest
     * @example
     *  ```
     *  const components = [
     *    {
     *      component_path: "/model/database.json",
     *      responseType: "json" // default value "defaultbuffer",
     *      skipBlockDownload: true  // To Skip block download in case of large asset and get presigned url instead
     *    }
     *  ]
     *  ```
     * @param version           Version of the composite manifest to fetch (components all come from this version of the manifest)
     * @param etag              Etag to use within the if-match header of the request
     * @param additionalHeaders Additional headers to attach to HTTP requests
     * @example
     * ```
     * {
     *   manifest: {
     *     data?: ManifestData // JSON parsed manifest data
     *     error?: DCXError // present if there was an error fetching the manifest
     *     response: AdobeResponse // response for the manifest sub-request (or secondary follow-up manifest request)
     *   },
     *   components: {
     *     "/model/database.json": { // component paths are the keys
     *       component_path: "/model/database.json",
     *       data?: ResponseTypeMap[ResponseType], // parsed into the requested responseType
     *       error?: DCXError, // present if there was an error fetching the manifest
     *       response: AdobeResponse // response for component sub-request (or follow-up request)
     *     }
     *   },
     *   responses: AdobeResponse[] // array of Bulk request responses (1 response per bulk request issued)
     * }
     * ```
     */
    getManifestAndComponentsByPath<C extends ComponentRequestByPath[]>(
        components: C,
        // Composite version (default, is current branch, if no local current branch, HEAD is pulled
        version?: string,
        // optional if-match header
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<ManifestAndComponentsByPathResponse<C>, AdobeDCXError> {
        dbgl('getManifestAndComponentsByPath()');
        return getManifestAndComponentsByPath<C>(this._svc, this, components, version, etag, additionalHeaders);
    }

    /**
     * Returns a component from a composite
     * @param componentId       Component ID to fetch
     * @param componentRevision Component revision to fetch
     * @param responseType      Type to tranform response into
     *                              (defaults to Buffer on Node, ArrayBuffer on browser)
     * @param additionalHeaders Additional headers to add to request
     * @param componentSize component size
     */
    getComponent<T extends AdobeResponseType = 'defaultbuffer'>(
        componentId: string,
        componentRevision: string,
        responseType?: T,
        additionalHeaders?: Record<string, string>,
        componentSize?: number,
    ): RepoDownloadStreamableReturn<T> {
        dbg('getComponent()');

        const ctx = {};

        return this._withSourcePromise(ctx).then(() => {
            return getCompositeComponent.call(
                ctx,
                this._svc,
                this,
                componentId,
                componentRevision,
                responseType,
                additionalHeaders,
                componentSize,
            ) as RepoDownloadStreamableReturn<T>;
        });
    }

    /**
     * Returns a component from a composite by path
     *
     * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Download+Components+by+DCX+Path Proposal: Download DCX Components by Path}
     *
     * @param componentPath     Component path to fetch
     * @param responseType      Type to tranform response into
     *                             (defaults to Buffer on Node, ArrayBuffer on browser)
     * @param additionalHeaders Additional headers to add to request
     *
     * @example
     * ```
     * const component = await composite.getComponentByPath('/model/database.json', 'json');
     * ```
     **/
    getComponentByPath<T extends AdobeResponseType = 'defaultbuffer'>(
        componentPath: string,
        responseType: T = 'defaultbuffer' as T,
        additionalHeaders?: Record<string, string>,
    ): RepoDownloadStreamableReturn<T> {
        dbg('getComponentByPath()');

        return getCompositeComponentByPath(this.serviceConfig, this, componentPath, responseType, additionalHeaders);
    }
    /**
     * Return the URL of a composite component. The revision is required if the URL will be used for a GET operation.
     * When using the URL to upload a component (PUT), revision is not required.
     * @param componentId       Component ID to return URL for
     * @param componentRevision           The component revision.
     * @param additionalHeaders Additional headers to attach to HTTP requests
     */
    getComponentUrl(
        componentId: string,
        componentRevision?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<string, AdobeDCXError> {
        dbg('getComponentUrl()');

        return this.fetchLinksIfMissing([LinkRelation.COMPONENT], additionalHeaders).then(() => {
            return getCompositeComponentUrl(this._svc, this, componentId, componentRevision);
        });
    }

    updateManifest<T = string | Record<string, unknown>>(
        manifest: T,
        overwrite: boolean,
        validationLevel?: number,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse, AdobeDCXError> {
        dbg('updateManifest()');

        return this.fetchLinksIfMissing([LinkRelation.MANIFEST], additionalHeaders)
            .then(() =>
                updateCompositeManifest<T>(
                    this._svc,
                    this,
                    manifest,
                    overwrite,
                    validationLevel,
                    etag,
                    additionalHeaders,
                ),
            )
            .then(this._updateDataWithResponse.bind(this));
    }

    /**
     * Uploads a composite component. If the size of the data is less then repo:maxSingleTransferSize data can be directly uploaded.
     * @param componentId       The componentId of the component to be uploaded, componentId supposes to be an uuid
     * @param data              A buffer containing the data to be uploaded
     * @param contentType       The contentType of the component being uploaded
     * @param maybeIsNew        Do we think this is a new component?
     * @param size              The expected size of this component
     * @param md5               The md5 of the component
     * @param progressCb        A Callback to be invoked with upload progress updates
     * @param additionalHeaders Additional headers to attach to HTTP Requests
     */
    putComponent(
        componentId: string,
        data: Buffer | ArrayBuffer | Blob | string,
        contentType: string,
        maybeIsNew?: boolean,
        size?: number,
        md5?: string,
        progressCb?: UploadProgressCallback,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }>;

    /**
     * Uploads a composite component. If the size of the data is greater then repo:maxSingleTransferSize a GetSliceCallback should be provided instead of the data.
     * @param componentId           The componentId of the component to be uploaded. componentId supposes to be an uuid.
     * @param getSliceCallback      A callback function to return a slice of data to upload
     *                                  The GetSliceCallback function accepts a start byte where the buffer slice should start and an end byte where
     *                                  the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer
     *                                  of length 0 indicating the end of the buffer.
     * @param contentType           The contentType of the component being uploaded
     * @param maybeIsNew            Do we think this is a new component?
     * @param size                  The expected size of this component
     * @param md5                   The md5 of the component
     * @param progressCb            A Callback to be invoked with upload progress updates
     * @param additionalHeaders     Additional headers to attach to HTTP Requests
     */
    putComponent(
        componentId: string,
        getSliceCallback: GetSliceCallback,
        contentType: string,
        maybeIsNew?: boolean,
        size?: number,
        md5?: string,
        progressCb?: UploadProgressCallback,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }>;

    /**
     * Uploads a composite component
     * @param componentId           The componentId of the component to be uploaded
     * @param dataOrSliceCallback   Either the component data or a GetSliceCallback.
     *                                  If the size of the data is greater then repo:maxSingleTransferSize a GetSliceCallback should be provided instead of the data.
     *                                  The GetSliceCallback function accepts a start byte where the buffer slice should start and an end byte where
     *                                  the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer
     *                                  of length 0 indicating the end of the buffer.
     * @param contentType           The contentType of the component being uploaded
     * @param maybeIsNew            Do we think this is a new component?
     * @param size                  The expected size of this component
     * @param md5                   The md5 of the component
     * @param progressCb            A Callback to be invoked with upload progress updates
     * @param additionalHeaders     Additional headers to attach to HTTP Requests
     */
    putComponent(
        componentId: string,
        dataOrSliceCallback: SliceableData | GetSliceCallback,
        contentType: string,
        maybeIsNew?: boolean,
        size?: number,
        md5?: string,
        progressCb?: UploadProgressCallback,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }> {
        dbg('putComponent()');

        if (maybeIsNew && !verifyUuid(componentId)) {
            throw new DCXError(DCXError.INVALID_PARAMS, `Component id is not a uuid`);
        } else if (!verifyUuid(componentId)) {
            logger.warn(`Existing component id is not a uuid`);
        }

        return this.fetchLinksIfMissing([LinkRelation.COMPONENT], additionalHeaders).then(() => {
            return putCompositeComponent(
                this.serviceConfig,
                this,
                componentId,
                dataOrSliceCallback,
                contentType,
                maybeIsNew,
                size,
                md5,
                progressCb,
                additionalHeaders,
            );
        });
    }

    /**
     * Update entire embedded/XMP metadata resource.
     *
     * @note
     * Replaces existing XMP resource.
     *
     * @note
     * Currently only supported on composites.
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * The argument `etag` will be updated to optional when the service supports it.
     * See {@link https://jira.corp.adobe.com/browse/SYSE-7940|ticket}
     * and {@link https://jira.corp.adobe.com/browse/SYSE-5943|another ticket}
     *
     * @param {T = Record<string, unknown>} data            New embedded metadata
     * @param {string}                      etag            ETag of metadata resource to update.
     * @param {'json'|'xml'}                [format='json'] Defines the representation of the body, either XML or JSON.
     *                                                      If using XML, clients must pass the data as a string.
     *                                                      If using XML with TypeScript, clients must specify the generic as `string`.
     *                                                      Defaults to json.
     * @param {Record<string, string>}      additionalHeaders     Additional headers to attach to HTTP Requests
     */
    putEmbeddedMetadata<T = Record<string, unknown>>(
        data: T,
        etag?: string,
        format: EmbeddedMetadataRepresentation = 'json',
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>> {
        validateParams(
            ['data', data, ['object', 'object[]', 'string']],
            ['etag', etag, 'string', true],
            ['format', format, 'enum', false, ['json', 'xml']],
        );
        return this.fetchLinksIfMissing([LinkRelation.EMBEDDED_METADATA], additionalHeaders)
            .then(() => putEmbeddedMetadata(this._svc, this, data, etag, format, additionalHeaders))
            .then(async (res) => {
                await this.headManifest(additionalHeaders);
                return res;
            });
    }

    /**
     * Update embedded/XMP metadata using JSON Patch document.
     *
     * @see
     * https://tools.ietf.org/html/rfc6902#page-6
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * The argument `etag` will be updated to optional when the service supports it.
     * see: https://jira.corp.adobe.com/browse/SYSE-7940
     * and: https://jira.corp.adobe.com/browse/SYSE-5943
     *
     * @param {JSONPatchDocument | string}      data      Data to use as PATCH body
     * @param {string}                          etag      ETag of the embedded metadata resource to be updated
     * @param {Record<string, string>}          additionalHeaders     Additional headers to attach to HTTP Requests
     */
    patchEmbeddedMetadata(
        data: JSONPatchDocument | string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>> {
        validateParams(['data', data, ['object', 'object[]', 'string']], ['etag', etag, 'string', true]);

        return this.fetchLinksIfMissing([LinkRelation.EMBEDDED_METADATA], additionalHeaders)
            .then(() => patchEmbeddedMetadata(this._svc, this, data, etag, additionalHeaders))
            .then(async (res) => {
                await this.headManifest();
                return res;
            });
    }

    copy(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
        manifestPatch?: JSONPatchDocument,
    ): AdobePromise<RepoResponseResult<Composite, 'json'>, AdobeDCXError> {
        return super
            .copy(destAsset, createIntermediates, overwriteExisting, additionalHeaders, manifestPatch)
            .then(({ response, result }) => {
                const composite = new Composite(result, this.serviceConfig);
                return {
                    response,
                    result: composite,
                };
            });
    }
}

export function headCompositeManifest(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeResponse> {
    dbgl('headCompositeManifest()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);
    assertLinksContain(asset.links, [LinkRelation.MANIFEST]);

    const manifestHref = getLinkHref(asset.links, LinkRelation.MANIFEST);
    return svc.invoke(HTTPMethods.HEAD, manifestHref, additionalHeaders, undefined, {
        responseType: 'json',
        isStatusValid: makeStatusValidator(),
    });
}

export function getCompositeManifest<T = unknown>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    version?: string,
    etag?: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponse<AdobeManifestResult<T>, 'json'>, AdobeDCXError> {
    dbgl('getCompositeManifest()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['version', version, 'string', true],
        ['etag', etag, 'string', true],
    );

    assertLinksContain(asset.links, [LinkRelation.MANIFEST]);

    return getCompositeManifestUrl(svc, asset, version, additionalHeaders)
        .then((manifestLink) => {
            return svc.invoke(
                HTTPMethods.GET,
                manifestLink,
                Object.assign(additionalHeaders ?? {}, etag ? { [HeaderKeys.IF_NONE_MATCH]: etag } : {}),
                undefined,
                {
                    responseType: 'json',
                    isStatusValid: makeStatusValidator([304]),
                },
            );
        })
        .then((response) => {
            return {
                manifestData: (response.response || null) as T,
                manifestEtag: response.headers['etag'] as string,
                response: response,
            };
        });
}

export function getCompositeManifestUrl(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    versionId?: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<string, AdobeDCXError> {
    dbgl('getCompositeManifestUrl()');

    return _getUrl(svc, asset, LinkRelation.MANIFEST, versionId, additionalHeaders);
}

/**
 * Get the component download url by path, which has to be composite version specific.
 *
 * @internal
 *
 * @param svc                   Service or service config
 * @param asset                 Asset with a version
 * @param component_path        Component path
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
export function _getComponentPathUrl(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    component_path: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<string, AdobeDCXError> {
    dbgl('_getComponentPathUrl()');

    return _getUrl(svc, asset, LinkRelation.COMPONENT, asset.version, additionalHeaders).then((href) =>
        expandURITemplate(href, pruneUndefined({ component_path })),
    );
}

/**
 * Get the resource url based on link relation, which could be version specific or not
 *
 * @internal
 *
 * @param svc                   Service or service config
 * @param asset                 Asset
 * @param relation              Link relation
 * @param versionid             Optional asset version, which is to get version specific url
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
export function _getUrl(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    relation: LinkRelationKey,
    versionId?: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<string, AdobeDCXError> {
    dbgl('_getUrl()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object'], ['versionId', versionId, 'string', true]);
    return AdobePromise.resolve()
        .then(() => {
            if (versionId) {
                return getVersionResource(svc, asset, versionId, additionalHeaders);
            }
        })
        .then((versionResource) => {
            if (versionId != null) {
                if (!versionResource || !isObject(versionResource) || typeof versionResource.result !== 'object') {
                    /* istanbul ignore next */
                    throw new DCXError(
                        DCXError.UNEXPECTED_RESPONSE,
                        'Invalid version resource.',
                        undefined,
                        versionResource ? versionResource.response : undefined,
                    );
                }

                assertLinksContain(versionResource.result[Properties.LINKS], [relation]);
                return getLinkHref(versionResource.result[Properties.LINKS], relation);
            }
            return getLinkHref(asset.links, relation);
        });
}

/**
 * @param components An array of ComponentRequestsByPath. NOTE: Paths are restricted to well-known paths only as determined by specific composite media type.
 * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Download+Components+by+DCX+Path Proposal: Download DCX Components by Path}
 * If using components that do not exist on the given well-known paths, please use @see  performBulkRequest
 * @example
 *  ```
 *  const components = [
 *    {
 *      component_path: "/model/database.json",
 *      responseType: "json" // Optional, default value "defaultbuffer",
 *      skipBlockDownload: true  // To skip block download of large asset and get presigned url instead
 *    }
 *  ]
 *  ```
 * @param version Version of the composite manifest to fetch (components all come from this version of the manifest)
 * @param etag Etag to use within the if-match header of the request
 * @param additionalHeaders Additional headers to attach to HTTP requests
 * @example
 * ```
 * {
 *   manifest: {
 *     data?: ManifestData // JSON parsed manifest data
 *     error?: DCXError // present if there was an error fetching the manifest
 *     response: AdobeResponse // response for the manifest sub-request (or secondary follow-up manifest request)
 *   },
 *   components: {
 *     "/model/database.json": { // component paths are the keys
 *       component_path: "/model/database.json",
 *       data?: ResponseTypeMap[ResponseType], // parsed into the requested responseType
 *       error?: DCXError, // present if there was an error fetching the manifest
 *       response: AdobeResponse // response for component sub-request (or follow-up request)
 *     }
 *   },
 *   responses: AdobeResponse[] // array of Bulk request responses (1 response per bulk request issued)
 * }
 * ```
 */
export function getManifestAndComponentsByPath<C extends ComponentRequestByPath[] = []>(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset | AdobeDCXComposite,
    components: C,
    version?: string,
    etag?: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<ManifestAndComponentsByPathResponse<C>, AdobeDCXError> {
    const BULK_SUBREQUEST_LIMIT = 10;
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['components', components, 'array'],
        ['version', version, 'string', true],
        ['etag', etag, 'string', true],
    );

    const requiredLinks: LinkRelationKey[] = [
        LinkRelation.MANIFEST,
        LinkRelation.COMPONENT,
        LinkRelation.BULK_REQUEST,
        LinkRelation.BLOCK_DOWNLOAD,
    ];
    if (version) {
        requiredLinks.push(LinkRelation.PAGE);
    }
    return fetchLinksIfMissing(svc, asset, requiredLinks, undefined, additionalHeaders).then(async (links) => {
        asset.links = Object.assign({}, asset.links, links);
        const service = getService(svc);
        const requests = components.map<BulkRequestDescriptor |(ComponentRequestByPath & WithHref)>(
            ({ component_path, responseType, subrequestHeaders }) => ({
                method: HTTPMethods.GET,
                href: getLinkHrefTemplated(asset.links, LinkRelation.COMPONENT, { component_path }),
                headers: Object.assign(subrequestHeaders || {}, additionalHeaders),
                component_path,
                responseType,
            }),
        );
        requests.unshift({
            method: HTTPMethods.GET,
            href: await getCompositeManifestUrl(service, asset, version, additionalHeaders),
            headers: Object.assign(etag ? { [HeaderKeys.IF_NONE_MATCH]: etag } : {}, additionalHeaders),
        });
        // chunk to maximum of 10 sub-requests per bulk request. See storage-api docs:
        // https://developers.corp.adobe.com/storage-api/docs/reference/bulk-requests.md#restrictions
        const bulkRequests = await Promise.all<RepoResponseResult<AdobeResponse[]>>(
            chunkArray(requests, BULK_SUBREQUEST_LIMIT).map((subRequests) =>
                performBulkRequest(service, asset, subRequests, undefined, additionalHeaders, true),
            ),
        );
        return AdobePromise.resolve<ManifestAndComponentsByPathResponse<C>>(
            // convert an array of bulk request responses into a single result
            await bulkRequests.reduce(
                async (prevPromise, bulkRequest) => {
                    const result = await prevPromise;
                    // push the bulk response onto the result
                    result.responses.push(bulkRequest.response);

                    // Bulk request did not succeed do not attempt to process sub-responses
                    /* istanbul ignore if */
                    if (bulkRequest.response.statusCode !== 200) {
                        return result;
                    }
                    const { componentResponses, manifestResponse } = _categorizeBulkResults(
                        bulkRequest.result,
                        requests,
                    );
                    // parse sub-responses for all components and merge onto the result
                    // The result is an object whose keys are the component_path values of each component requested
                    const skipBlockDownloadMap = {};
                    components.forEach((component: ComponentRequestByPath) => {
                        if (component.hasOwnProperty('skipBlockDownload')) {
                            skipBlockDownloadMap[component.component_path] = component.skipBlockDownload;
                        }
                    });
                    Object.assign(
                        result.components as any,
                        await _mapComponentResponses(
                            requests.slice(1) as (ComponentRequestByPath & WithHref)[],
                            componentResponses,
                            svc,
                            asset.links!,
                            additionalHeaders,
                            skipBlockDownloadMap,
                        ),
                    );

                    // If the manifest wasn't requested with this bulkRequest, move onto the next bulk request
                    /* istanbul ignore if */
                    if (!manifestResponse) {
                        return result;
                    }

                    // manifest request was successful
                    if (manifestResponse.statusCode === 200) {
                        result.manifest.data = _convertToRequestedResponseType(manifestResponse.response, 'json');
                        result.manifest.response = manifestResponse;
                        if (isAdobeDCXCompositeLike(asset) && typeof asset.current?.parse === 'function') {
                            asset.current.parse(arrayBufferToString(manifestResponse.response));
                            asset.current.versionId = manifestResponse.headers['version'];
                        }
                        asset.links = parseLinksFromResponseHeader(manifestResponse);
                        const cache = getReposityLinksCache(svc);
                        if (cache) {
                            cache.setValueWithAsset(asset.links, asset);
                        }

                        return result;
                    }

                    // When an etag is supplied and the manifest is not modified
                    if (manifestResponse.statusCode === 304) {
                        result.manifest.response = manifestResponse;
                        return result;
                    }

                    if (manifestResponse.statusCode === 404) {
                        result.manifest.error =
                            _responseToError(manifestResponse) ||
                            new DCXError(
                                DCXError.NO_COMPOSITE,
                                'Composite missing or deleted',
                                undefined,
                                manifestResponse,
                            );
                        return result;
                    }

                    if (
                        // Server requesting use of Block Transfer
                        manifestResponse.headers[HeaderKeys.CONTENT_TYPE] === JSONProblemMediaType &&
                        manifestResponse.response.type === ProblemTypes.RESPONSE_TOO_LARGE
                    ) {
                        try {
                            const bdlHref = manifestResponse.headers.location
                                ? manifestResponse.headers.location
                                : getLinkHrefTemplated(asset.links, LinkRelation.BLOCK_DOWNLOAD, {
                                      resource: JSON.stringify({ reltype: LinkRelation.MANIFEST }),
                                  });
                            const bdlResult = await _doBlockDownload(
                                service,
                                bdlHref,
                                undefined,
                                undefined,
                                'json',
                                true,
                                undefined,
                                additionalHeaders,
                            );
                            result.manifest.data = bdlResult.response;
                            result.manifest.response = bdlResult;
                            if (isAdobeDCXCompositeLike(asset) && typeof asset.current?.parse === 'function') {
                                asset.current.parse(JSON.stringify(bdlResult.response));
                            }
                            asset.links = parseLinksFromResponseHeader(bdlResult);
                        } catch (err) {
                            result.manifest.error =
                                err instanceof DCXError
                                    ? err
                                    : new DCXError(
                                          DCXError.UNEXPECTED,
                                          'Error fetching manifest via block download',
                                          err,
                                      );
                        }
                        // Return result after block download
                        return result;
                    }

                    // manifest request error
                    result.manifest.error =
                        _responseToError(manifestResponse) ||
                        new DCXError(
                            DCXError.UNEXPECTED_RESPONSE,
                            manifestResponse.response.title || 'Failed to fetch manifest. Operation failed.',
                            undefined,
                            manifestResponse,
                        );

                    return result;
                },
                Promise.resolve({
                    manifest: {},
                    components: {},
                    responses: [],
                } as unknown as ManifestAndComponentsByPathResponse<C>),
            ),
        );
    });
}

type CategorizedUpdateManifestBulkResponses = {
    manifestResponse: AdobeResponse;
    repoMetadataResponse: AdobeResponse;
};

function _categorizeUpdateManifestBulkResult(
    results: AdobeResponse[],
    requests: BulkRequestDescriptor[],
): CategorizedUpdateManifestBulkResponses {
    const responses = {
        manifestResponse: {},
        repoMetadataResponse: {},
    } as CategorizedUpdateManifestBulkResponses;

    requests.forEach((request) => {
        const response = results.find((res) => res.headers[HeaderKeys.CONTENT_ID] === request.href);
        if (!response) {
            throw new DCXError(
                DCXError.UNEXPECTED_RESPONSE,
                'Bulk sub-response content-id did not match any bulk request',
                undefined,
                response,
            );
        }

        if (request.href.includes(':repometadata')) {
            responses.repoMetadataResponse = response;
        } else {
            responses.manifestResponse = response;
        }
    });

    return responses;
}

type CategorizedBulkResponses = {
    manifestResponse: AdobeResponse;
    componentResponses: AdobeResponse[];
};

type WithHref = { href: string };

function _categorizeBulkResults<C extends (BulkRequestDescriptor | (ComponentRequestByPath & WithHref))[]>(
    results: AdobeResponse[],
    requests: C,
): CategorizedBulkResponses {
    return results.reduce(
        (responses, response) => {
            const request = requests.find((request) => request.href === response.headers[HeaderKeys.CONTENT_ID]);
            if (!request) {
                throw new DCXError(
                    DCXError.UNEXPECTED_RESPONSE,
                    'Bulk sub-response content-id did not match any bulk request',
                    undefined,
                    response,
                );
            }

            if ('component_path' in request) {
                responses.componentResponses.push(response);
            } else {
                responses.manifestResponse = response;
            }
            return responses;
        },
        {
            componentResponses: [],
        } as unknown as CategorizedBulkResponses,
    ) as CategorizedBulkResponses;
}

async function _mapComponentResponses<C extends (ComponentRequestByPath & WithHref)[]>(
    componentRequests: C,
    componentResponses: AdobeResponse[],
    svc: AdobeHTTPService | ServiceConfig,
    assetLinks: LinkSet,
    additionalHeaders?: Record<string, string>,
    skipBlockDownloadMap?: Record<string, boolean>,
): Promise<RepoResponse<ToResponseObject<C[number]>[]>> {
    return (
        await Promise.all(
            componentResponses.map(async (response) => {
                /* istanbul ignore if */
                if (
                    response.headers[HeaderKeys.CONTENT_TYPE] === JSONProblemMediaType &&
                    response.response.type === ProblemTypes.RESPONSE_TOO_LARGE
                ) {
                    const componentRequest = _locateRequestFromResponse(response, componentRequests);
                    const bdlHref = response.headers.location
                        ? response.headers.location
                        : getLinkHrefTemplated(assetLinks, LinkRelation.BLOCK_DOWNLOAD, {
                              resource: JSON.stringify({ component_path: componentRequest.component_path }),
                          });

                    if (skipBlockDownloadMap && skipBlockDownloadMap[componentRequest.component_path]) {
                        return {
                            statusCode: 200,
                            headers: pruneUndefined({
                                [HeaderKeys.CONTENT_TYPE]: response.headers['content-type'],
                                [HeaderKeys.CONTENT_LENGTH]: response.headers['content-length'],
                                [HeaderKeys.CONTENT_ID]: response.headers[HeaderKeys.CONTENT_ID],
                            }),
                            responseType: 'application/json',
                            response: { href: bdlHref },
                            message: 'OK',
                        } as AdobeResponse;
                    }
                    const blockDownload = await _doBlockDownload(
                        getService(svc),
                        bdlHref,
                        undefined,
                        undefined,
                        'defaultbuffer',
                        true,
                        undefined,
                        additionalHeaders,
                    );
                    // copy the content-id over to the block download response
                    Object.assign(blockDownload.headers, {
                        [HeaderKeys.CONTENT_ID]: response.headers[HeaderKeys.CONTENT_ID],
                    });
                    return blockDownload as AdobeResponse<'defaultbuffer'>;
                }
                return response;
            }),
        )
    ).reduce((result, response) => {
        const request = _locateRequestFromResponse(response, componentRequests);
        try {
            const responseError = _responseToError(response);
            result[request.component_path] = Object.assign({}, request, {
                response,
                [responseError ? 'error' : 'data']: responseError
                    ? responseError
                    : _convertToRequestedResponseType(
                          response.response,
                          (request.responseType as AdobeResponseType) || 'defaultbuffer',
                          response.headers['content-type'],
                      ),
            });
        } catch (error) {
            /* istanbul ignore next */
            result[request.component_path] = Object.assign({}, request, {
                response,
                error: new DCXError(
                    DCXError.UNEXPECTED,
                    'Error parsing sub-response into requested responseType',
                    error as Error,
                ),
            });
        }
        return result;
    }, {} as unknown as RepoResponse<ToResponseObject<C[number]>[]>);
}

function _locateRequestFromResponse<C extends (ComponentRequestByPath & WithHref)[]>(
    response: AdobeResponse,
    requests: C,
): C[number] {
    /* istanbul ignore if */
    if (!response.headers[HeaderKeys.CONTENT_ID]) {
        throw new DCXError(
            DCXError.UNEXPECTED_RESPONSE,
            'Sub-response is missing content-id header',
            undefined,
            response,
        );
    }
    const request = requests.find(({ href }) => href === response.headers[HeaderKeys.CONTENT_ID]);
    if (!request) {
        throw new DCXError(
            DCXError.UNEXPECTED_RESPONSE,
            'Bulk sub-response content-id did not match any bulk request',
            undefined,
            response,
        );
    }
    return request;
}
function _convertToRequestedResponseType<T extends Omit<AdobeResponseType, 'blob'>>(
    response: Uint8Array,
    responseType: T,
    contentType?: string,
);
function _convertToRequestedResponseType(response: Uint8Array, responseType: 'blob', contentType: string);
function _convertToRequestedResponseType<T extends AdobeResponseType>(
    response: Uint8Array | Record<string, unknown>,
    responseType: T,
    contentType?: string,
): ResponseTypeMap[T] {
    if (!isBufferLike(response)) {
        return response as ResponseTypeMap[T];
    }
    switch (responseType) {
        case 'text':
            return arrayBufferToString(response) as ResponseTypeMap[T];

        case 'json':
            return JSON.parse(arrayBufferToString(response));

        case 'blob':
            return new Blob([response], { type: contentType }) as ResponseTypeMap[T];

        case 'buffer':
        case 'defaultbuffer':
            return response as ResponseTypeMap[T];

        case 'arraybuffer':
            return response.buffer as ResponseTypeMap[T];
    }

    /* istanbul ignore next */
    throw new DCXError(DCXError.INVALID_PARAMS, 'requested response type is not supported');
}

export function getCompositeComponentUrl(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    componentId: string,
    componentRevision?: string,
): string {
    dbgl('getCompositeComponentUrl()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['componentId', componentId, 'string'],
        ['componentRevision', componentRevision, 'string', true],
    );
    assertLinksContain(asset.links, [LinkRelation.COMPONENT]);

    return getLinkHrefTemplated(asset.links, LinkRelation.COMPONENT, {
        component_id: componentId,
        revision: componentRevision,
    });
}

/**
 * Fetch pre-signed URL for a regular file asset
 * @param svc                   Service or service config
 * @param asset                 Asset object that identifies the asset
 * @param componentResource     If provided, generate presignedUrl for this component resource instead of asset primary resource
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/advanced/block.html#block-download Block Download Specification}
 */
export function getPresignedUrl(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    componentResource?: ComponentResourceDesignator,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<string, 'json'>, AdobeDCXError> {
    dbgl('getPresignedUrl()');
    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);

    return fetchLinksIfMissing(svc, asset, [LinkRelation.BLOCK_DOWNLOAD], undefined, additionalHeaders)
        .then((linkSet) => {
            const resource = componentResource ? JSON.stringify(componentResource) : undefined;
            const url = getLinkHrefTemplated(linkSet, LinkRelation.BLOCK_DOWNLOAD, { resource });
            return (isServiceConfig(svc) ? svc.service : svc).invoke(
                HTTPMethods.GET,
                url,
                // DCX-11103 requests that the default for all block transfers is to request transfer acceleration
                // Because the `repo:accelerated` parameter is suggested to be deprecated as part of CA-967,
                // we are opting for setting the priority header on the block upload init request to trigger
                // RAPI to utilize EALinks, which also achieve the same transfer acceleration per CSA-374.
                // Consumers are still able to override the priority value by providing one of their own.
                { priority: 'u=1', ...additionalHeaders },
                undefined,
                {
                    isStatusValid: makeStatusValidator(),
                    responseType: 'json',
                    retryOptions: {
                        pollCodes: [202],
                        pollHeader: 'location',
                        pollMethod: HTTPMethods.GET,
                    },
                },
            );
        })
        .then((response) => {
            /**
             * response = {
             *   href: string;
             *   size: number;
             *   type: string;
             * }
             */
            if (typeof response.response.href !== 'string') {
                throw new DCXError(DCXError.INVALID_DATA, 'Direct download URL not found.', undefined, response);
            }
            return {
                response,
                result: response.response.href,
            };
        });
}

/**
 * Fetch pre-signed URL for a composite component
 * @param svc                   Service or service config
 * @param asset                 Asset object that identifies the asset
 * @param componentId           Component id
 * @param componentRevision     Component revision
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
export function getCompositeComponentPresignedUrl(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    componentId: string,
    componentRevision: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<string, 'json'>, AdobeDCXError> {
    dbgl('getCompositeComponentPresignedUrl()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['componentId', componentId, 'string'],
        ['componentRevision', componentRevision, 'string'],
    );

    return getPresignedUrl(
        svc,
        asset,
        { reltype: LinkRelation.COMPONENT, revision: componentRevision, component_id: componentId },
        additionalHeaders,
    );
}

/**
 * Get a composite component by path. Must be a well-known path for a given composite type.
 *
 * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Download+Components+by+DCX+Path Proposal: Download DCX Components by Path}
 *
 * @param svc - AdobeHTTPService instance or ServiceConfig object
 * @param asset - AdobeAsset instance
 * @param componentPath - path to the component
 * @param responseType - response type
 * @param additionalHeaders - additional headers
 * @param skipBlockDownload - to skip block download & get presignedURL instead
 * @returns - AdobePromise<RepoResponseResult<T, AdobeResponseType>, AdobeDCXError>
 *
 * @example
 * ```
 * const response = await getCompositeComponentByPath(
 *   serviceConfig,
 *   asset,
 *   '/path/to/component',
 *   'blob',
 *   { 'x-foo': 'bar' },
 * );
 * ```
 */
export function getCompositeComponentByPath<T extends AdobeResponseType = 'defaultbuffer'>(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    componentPath: string,
    responseType: T = 'defaultbuffer' as T,
    additionalHeaders?: Record<string, string>,
    skipBlockDownload?: boolean,
): RepoDownloadStreamableReturn<T> {
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['componentPath', componentPath, 'string'],
        ['responseType', responseType, 'string', true],
        ['additionalHeaders', additionalHeaders, 'object', true],
    );
    return fetchLinksIfMissing(svc, asset, [LinkRelation.COMPONENT, LinkRelation.PAGE], undefined, additionalHeaders)
        .then((links) => {
            asset.links = links;
            if (asset.version) {
                return _getComponentPathUrl(getService(svc), asset, componentPath);
            }
        })
        .then((componentVersionUrl) =>
            _getUrlFallbackDirect(
                getService(svc),
                asset,
                componentVersionUrl ??
                    getLinkHrefTemplated(asset.links, LinkRelation.COMPONENT, { component_path: componentPath }),
                LinkRelation.COMPONENT,
                responseType,
                undefined,
                undefined,
                additionalHeaders,
                skipBlockDownload,
            ),
        );
}

/**
 * Get a composite component by id and revision.
 * @param svc - AdobeHTTPService instance or ServiceConfig object
 * @param asset - AdobeAsset instance
 * @param componentId - component id
 * @param componentRevision - component revision
 * @param responseType - response type
 * @param additionalHeaders - additional headers
 * @param componentSize - component size
 * @returns - AdobePromise<RepoResponseResult<T, AdobeResponseType>, AdobeDCXError>
 */
export function getCompositeComponent<T extends AdobeResponseType = 'defaultbuffer'>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    componentId: string,
    componentRevision: string,
    responseType: T = 'defaultbuffer' as T,
    additionalHeaders?: Record<string, string>,
    componentSize?: number,
): RepoDownloadStreamableReturn<T> {
    dbgl('getCompositeComponent()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['componentId', componentId, 'string'],
        ['componentRevision', componentRevision, 'string'],
        ['responseType', responseType, 'string', true],
        ['additionalHeaders', additionalHeaders, 'object', true],
        ['componentSize', componentSize, 'number', true],
    );
    const ctx = {};
    if (!componentSize || componentSize < getBlockDownloadThreshold()) {
        const href = getCompositeComponentUrl(svc, asset, componentId, componentRevision);
        return _getUrlFallbackDirect.call(
            ctx,
            svc,
            asset,
            href,
            LinkRelation.COMPONENT,
            responseType,
            componentId,
            componentRevision,
            additionalHeaders,
        ) as RepoDownloadStreamableReturn<T>;
    }

    return getCompositeComponentPresignedUrl(svc, asset, componentId, componentRevision, additionalHeaders).then(
        ({ response, result: bdlHref }) => {
            return _doBlockDownload.call(
                ctx,
                svc,
                bdlHref,
                undefined,
                undefined,
                responseType,
                true,
                response.response.size,
                additionalHeaders,
            );
        },
    ) as RepoDownloadStreamableReturn<T>;
}

export function updateCompositeManifest<T = string | Record<string, unknown>>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    manifest: T,
    overwrite: boolean,
    validationLevel = 1,
    etag?: string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<AdobeResponse<'json'>, AdobeDCXError> {
    dbgl('updateCompositeManifest() ', overwrite, etag);

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['manifest', manifest, ['object', 'string']],
        ['overwrite', overwrite, 'boolean'],
        ['validationLevel', validationLevel, '+number'],
        ['etag', etag, 'string', true],
    );
    if (validationLevel < 1) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'ValidationLevel must be >=1');
    }
    assertLinksContain(asset.links, [LinkRelation.BULK_REQUEST, LinkRelation.REPO_METADATA]);

    return getCompositeManifestUrl(svc, asset, undefined, additionalHeaders).then(async (manifestLink) => {
        const headers = Object.assign({}, additionalHeaders);
        if (overwrite) {
            headers[HeaderKeys.IF_MATCH] = '*';
        } else if (etag) {
            headers[HeaderKeys.IF_MATCH] = etag;
        }

        const manifestContentType = `${ManifestMediaType}; validation-level=${validationLevel}`;
        headers[HeaderKeys.CONTENT_TYPE] = manifestContentType;
        const manifestBody = typeof manifest === 'string' ? manifest : JSON.stringify(manifest);

        const requests: BulkRequestDescriptor[] = [
            { method: HTTPMethods.PUT, href: manifestLink, headers, body: manifestBody },
        ];
        if (asset.deviceModifyDate) {
            const repoMetadataBody = JSON.stringify([
                { op: 'add', path: `/${[Properties.REPO_DEVICE_MODIFY_DATE]}`, value: asset.deviceModifyDate },
            ]);
            const repoMetadataHeader: Record<string, string> = {
                [HeaderKeys.CONTENT_TYPE]: JSONPatchMediaType,
            };
            requests.push({
                method: HTTPMethods.PATCH,
                href: getLinkHref(asset.links, LinkRelation.REPO_METADATA) as string,
                headers: repoMetadataHeader,
                body: repoMetadataBody,
            });
        }

        const { result, response } = await performBulkRequest(svc, asset, requests, undefined, additionalHeaders, true);
        const { manifestResponse, repoMetadataResponse } = _categorizeUpdateManifestBulkResult(result, requests);

        dbgl('uCM() status code for manifest response: ', manifestResponse.statusCode);
        if (manifestResponse.statusCode === 412 && overwrite) {
            dbgl('uCM() retry 412 without overwrite');

            // update failed, try without overwrite since it may be new
            return updateCompositeManifest(
                svc,
                asset,
                manifest,
                false /* overwrite */,
                validationLevel,
                undefined /* etag */,
                additionalHeaders,
            );
        } else if (manifestResponse.statusCode === 409 && overwrite) {
            dbgl('uCM() retry 409 without overwrite');

            return updateCompositeManifest(svc, asset, manifest, false, validationLevel, etag, additionalHeaders);
        } else if (manifestResponse.statusCode === 409) {
            throw new DCXError(DCXError.UPDATE_CONFLICT, 'Manifest has been changed', undefined, manifestResponse);
        } else if (manifestResponse.statusCode === 412) {
            throw new DCXError(DCXError.PRECONDITION_FAILED, 'Precondition failed', undefined, manifestResponse);
        } else if (manifestResponse.statusCode === 400) {
            const manifestResError = _responseToError(manifestResponse);
            throw new DCXError(
                manifestResError?.code as string,
                manifestResError?.message,
                undefined,
                manifestResponse,
            );
        } else {
            const validator = makeStatusValidator();
            const validOrError = validator(manifestResponse.statusCode, manifestResponse);
            if (validOrError !== true) {
                throw new DCXError(
                    (validOrError as DCXError).code || DCXError.UNEXPECTED_RESPONSE,
                    (validOrError as unknown as { _message: string })._message ||
                        (validOrError as AdobeDCXError).message,
                    (validOrError as AdobeDCXError).underlyingError,
                    manifestResponse,
                );
            }
        }

        dbgl('uCM() status code for metadata response: ', repoMetadataResponse.statusCode);
        if (
            asset.deviceModifyDate &&
            !(
                repoMetadataResponse.statusCode === 200 ||
                repoMetadataResponse.statusCode === 201 ||
                repoMetadataResponse.statusCode === 204
            )
        ) {
            throw new DCXError(
                DCXError.UNEXPECTED_RESPONSE,
                'Unexpected HTTP Response',
                undefined,
                repoMetadataResponse,
            );
        }
        manifestResponse['xhr'] = response.xhr; // For backward compatibility of unit tests
        return manifestResponse;
    });
}

/**
 *
 *
 * Uploads a composite component
 * @remarks Service can override the block size client specified @see {@link https://git.corp.adobe.com/caf/api-spec/blob/gh-pages/schemas/repository/transfer-document.schema.json#L45}
 * @param service                                       An HTTPService instance
 * @param asset                                         The asset associated with the PUT request
 * @param componentId                                   The componentId of the component to be uploaded. ComponentId supposes to be uuid.
 * @param dataOrSliceCallback                           Either the component data or a GetSliceCallback.
 *                                                       If the size of the data is greater then repo:maxSingleTransferSize a GetSliceCallback should be provided instead of the data.
 *                                                       The GetSliceCallback function accepts a start byte where the buffer slice should start and an end byte where
 *                                                       the buffer slice should end (not inclusive). The callback should return a Promise containing the buffer slice or a buffer
 *                                                       of length 0 indicating the end of the buffer.
 * @param contentType                                   The contentType of the component being uploaded
 * @param maybeIsNew                                    Do we think this is a new component?
 * @param size                                          The expected size of this component
 * @param md5                                           The md5 of the component
 * @param progressCb                                    A progress callback that will be invoked with updates to upload progress
 * @param additionalHeaders                             Additional headers to attach to HTTP Requests
 * @param blockSize                                     Desired block size for the upload in bytes
 */
export function putCompositeComponent(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    componentId: string,
    dataOrSliceCallback: SliceableData | GetSliceCallback,
    contentType: string,
    maybeIsNew?: boolean,
    size?: number,
    md5?: string,
    progressCb?: UploadProgressCallback,
    additionalHeaders?: Record<string, string>,
    blockSize?: number,
): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }> {
    validateParams(
        ['service', svc, 'object'],
        ['asset', asset, 'object'],
        ['componentId', componentId, 'string'],
        ['contentType', contentType, 'string'],
        ['maybeIsNew', maybeIsNew, 'boolean', true],
        ['size', size, 'number', true],
        ['blockSize', blockSize, 'number', true],
        ['md5', md5, 'string', true],
    );
    if (maybeIsNew && !verifyUuid(componentId)) {
        throw new DCXError(DCXError.INVALID_PARAMS, `Component id is not a uuid`);
    } else if (!verifyUuid(componentId)) {
        logger.warn(`Existing component id is not a uuid`);
    }

    const uploadPromise = _upload({
        svc,
        asset,
        dataOrSliceCallback,
        contentType,
        relation: LinkRelation.COMPONENT,
        size,
        componentId,
        md5,
        maybeIsNew,
        additionalHeaders,
        progressCb,
        blockSize,
    });

    return uploadPromise.then(({ response, result, isBlockUpload, asset: compositeAsset }) => {
        const res = {
            response: response,
            result: { ...result, id: componentId, type: contentType },
            isBlockUpload,
            asset: compositeAsset,
        };

        Object.defineProperty(res, 'compositeAsset', {
            get: () => {
                // FUTURE: Log deprecation warning for this accessor,
                // should use `asset` property
                return compositeAsset;
            },
        });
        return res;
    });
}

/**
 * For returning the information necessary to perform the uploads either direct to ACP or via block upload.
 * @param svc  An HTTPService instance
 * @param asset The asset associated with the PUT request
 * @param uploads An array of ComponentUploadInfo
 * @param additionalHeaders Additional headers to attach to HTTP Requests
 * @returns An array of UploadRecipe
 */
export function getCompositeComponentsUrlsForUpload(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    uploads: ComponentUploadInfo[],
    additionalHeaders?: Record<string, string>,
): AdobePromise<UploadRecipe[], AdobeDCXError> {
    return fetchLinksIfMissing(
        svc,
        asset,
        [LinkRelation.COMPONENT, LinkRelation.BLOCK_UPLOAD_INIT],
        undefined,
        additionalHeaders,
    ).then((links) => {
        const service = getService(svc);
        getReposityLinksCache(svc)?.setValueWithAsset(links, asset);
        asset.links = { ...asset.links, ...links };
        const additionalHeadersWithAuth = normalizeHeaders(
            pruneUndefined({
                ...additionalHeaders,
                [HeaderKeys.AUTHORIZATION]: service.authProvider.authToken,
                [HeaderKeys.X_API_KEY]: service.authProvider.apiKey,
            }),
        );
        return Promise.all(
            uploads.map((upload) =>
                shouldUseBlockTransferForUpload(asset, upload.size)
                    ? _getBlockUploadRecipe(service, asset, upload, additionalHeadersWithAuth)
                    : _getDirectUploadRecipe(service, asset, upload, additionalHeadersWithAuth),
            ),
        );
    });
}

function _getDirectUploadRecipe(
    service: AdobeHTTPService,
    asset: AdobeAsset,
    upload: ComponentUploadInfo,
    directUploadHeaders: Record<string, string>,
): AdobePromise<UploadRecipe, AdobeDCXError> {
    return AdobePromise.resolve({
        blockSize: upload.size,
        uploadRequestParameters: [
            {
                href: getCompositeComponentUrl(service, asset, upload.componentId),
                method: HTTPMethods.PUT,
                headers: directUploadHeaders,
            },
        ],
    });
}

function _getBlockUploadRecipe(
    service: AdobeHTTPService,
    asset: AdobeAsset,
    upload: ComponentUploadInfo,
    blockUploadHeaders: Record<string, string>,
): AdobePromise<UploadRecipe, AdobeDCXError> {
    //if it's block upload, then generate block upload init request and add an empty record into return record, later will update the record based on component Id.
    //This is to make sure the order of the return record is the same as the order of the upload array
    //add a placeholder for the upload recipe
    const transferDocument: ACPTransferDocument = pruneUndefined({
        'repo:reltype': LinkRelation.COMPONENT,
        'repo:size': upload.size,
        'dc:format': upload.contentType,
        component_id: upload.componentId,
    });
    return initBlockUpload(service, asset, transferDocument, blockUploadHeaders)
        .then((blockUploadResponse) => {
            if (blockUploadResponse.response.statusCode !== 200) {
                throw new DCXError(
                    DCXError.UNEXPECTED_RESPONSE,
                    'Unexpected response from block upload init',
                    blockUploadResponse.response,
                );
            }
            const blockTransferDocument = blockUploadResponse.result as BlockTransferDocument;
            return {
                blockSize: blockTransferDocument[BlockTransferProperties.REPO_BLOCK_SIZE]!,
                uploadRequestParameters: blockTransferDocument[Properties.LINKS][LinkRelation.BLOCK_TRANSFER].map(
                    ({ href }) => ({
                        href,
                        method: HTTPMethods.PUT,
                    }),
                ),
                finalizeRequestParameters: {
                    href: getLinkHrefTemplated(
                        blockTransferDocument[Properties.LINKS],
                        LinkRelation.BLOCK_FINALIZE,
                        {},
                    ),
                    method: HTTPMethods.POST,
                    headers: blockUploadHeaders,
                    body: `${JSON.stringify(blockTransferDocument)}`,
                },
            };
        })
        .catch((err) => {
            throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'Unexpected response from block upload init', err);
        });
}
