/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPRepoMetadataResource,
    AdobeAsset,
    AdobeAssetEmbedded,
    AdobeHTTPService,
    BodyType,
    EmbeddableResource,
    GetSliceCallback,
    LinkSet,
    ProgressCallback,
    RepoMetaPatch,
    RequestDescriptor,
    ResourceDesignator,
    SliceableData,
    UploadProgressCallback,
} from '@dcx/common-types';
import AdobeDCXError, { DCXError, ProblemTypes, unexpectedResponse } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    appendPathElements,
    getLinkHref,
    getLinkHrefTemplated,
    isAnyFunction,
    isObject,
    merge,
    mergeDeep,
    pruneUndefined,
    validateParams,
} from '@dcx/util';
import { Asset, fetchLinksIfMissing } from './Asset';
import { _doBlockUpload } from './BlockTransfer/BlockUpload';
import { AdobeGetPageResult, Page, PageOptions, PageResource } from './Pagination';
import { ServiceConfig, getReposityLinksCache, getService } from './Service';
import { RepoResponse, RepoResponseResult } from './common';
import { AssetType, AssetTypes } from './enum/asset_types';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { Properties } from './enum/properties';
import { PathOrIdAssetDesignator } from './operations';
import { _parseUploadableData, getDataLength, shouldUseBlockTransferForUpload } from './util/block_transfer';
import { parseMultipartResponseParts } from './util/bulk';
import { parseLinksFromResponseHeader } from './util/link';
import { deserializeAsset } from './util/serialization';
import { assertLinksContain, makeStatusValidator } from './util/validation';

const dbg = newDebug('dcx:assets:directory');
const dbgl = newDebug('dcx:assets:directory:leaf');

export type AdobeDirectoryData = AdobeAsset & Page;

export interface AdobeGetDirectoryDataResult {
    result: AdobeDirectoryData;
}

/**
 * Instance type is returned (From instance method or session)
 */
export interface AdobeGetDirectoryResult {
    result: Directory;
}

export interface AdobeDirectory extends Asset<AdobeDirectoryData> {
    children: AdobeAsset[];

    /**
     * Returns a paged set of results for the directory listing
     * @param {PageOptions} pageOpts              Page options
     */
    getPagedChildren<T extends EmbeddableResource = never>(
        pageOpts?: PageOptions<T>,
    ): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAsset>>>;

    /**
     * Create an asset relative to the current directory.2
     *
     * @param {string} relPath                                      Path2 relative to current directory.
     * @param {boolean} createIntermediates                         Whether to create intermediate directories if they don't exist.
     * @param {string} contentType                                  Content type of the new asset.
     * @param {ResourceDesignator} [resourceDesignator]             Resource to be included in response.
     * @param {Record<string, string[]>} [additionalHeaders = {}]   Additional headers for the request.
     */
    createAsset(
        relPath: string,
        createIntermediates: boolean,
        contentType: string,
        resourceDesignator?: ResourceDesignator,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Asset>, AdobeDCXError>;

    /**
     * Copy the asset to some destination.
     *
     * @note
     * To copy to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destAsset            Asset containing either path or assetId
     * @param {boolean} createIntermediates     Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting       Whether to overwrite an existing asset.
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    copy(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
    ): AdobePromise<RepoResponseResult<Directory, 'json'>, AdobeDCXError>;
}

export class Directory extends Asset<AdobeDirectoryData> implements AdobeDirectory {
    readonly type: AssetType = AssetTypes.Directory;

    public children: AdobeAsset[] = [];

    constructor(
        data: Partial<AdobeDirectoryData | ACPRepoMetadataResource>,
        svc: AdobeHTTPService | ServiceConfig,
        links?: LinkSet,
    ) {
        super(data, svc, links);
        this.children = data[Properties.CHILDREN];
    }

    /**
     * Returns a paged set of results for the directory listing
     * @param pageOpts          Page options
     * @param additionalHeaders Additional headers to be provided with HTTP requests
     */
    getPagedChildren<T extends EmbeddableResource = never>(
        pageOpts?: PageOptions<T>,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>> {
        dbg('getPagedChildren()');

        return this.fetchLinksIfMissing([LinkRelation.PAGE], additionalHeaders).then(() => {
            return getPagedChildren<T>(this._svc, this, pageOpts, additionalHeaders);
        });
    }

    /**
     *
     * Create an asset relative to the current directory.
     *
     * @param {string} relPath                                      Path relative to current directory.
     * @param {boolean} createIntermediates                         Whether to create intermediate directories if they don't exist.
     * @param {string} contentType                                  Content type of the new asset.
     * @param {ResourceDesignator} [resourceDesignator]             Resource to be included in response.
     * @param {Record<string, string[]>} [additionalHeaders = {}]   Additional headers for the request.
     */
    createAsset(
        relPath: string,
        createIntermediates: boolean,
        contentType: string,
        resourceDesignator?: ResourceDesignator,
        additionalHeaders?: Record<string, string>,
        primaryResource?: SliceableData | GetSliceCallback,
        primaryResourceSize?: number,
        repoMetaPatch?: RepoMetaPatch,
        progressCb?: UploadProgressCallback,
    ): AdobePromise<RepoResponseResult<Asset>, AdobeDCXError> {
        dbg('createAsset()');
        return createAsset(
            this._svc,
            this,
            relPath,
            createIntermediates,
            contentType,
            resourceDesignator as ResourceDesignator,
            additionalHeaders,
            primaryResource,
            primaryResourceSize,
            repoMetaPatch,
            progressCb,
        ).then((res) => ({
            result: new Asset(res.result, this.serviceConfig),
            response: res.response,
        }));
    }

    /**
     * Move the asset, uses the operation endpoint.
     *
     * @note
     * To move to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destination          Asset containing either path or assetId
     * @param {boolean} createIntermediates     Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting       Whether to overwrite the existing asset.
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    copy(
        destination: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
    ): AdobePromise<RepoResponseResult<Directory, 'json'>, AdobeDCXError> {
        return super.copy(destination, createIntermediates, overwriteExisting).then(({ response, result }) => {
            const dir = new Directory(result, this.serviceConfig);
            return {
                response,
                result: dir,
            };
        });
    }
}

/**
 * Transforms a raw Directory JSON representation into an AdobeAsset
 *
 * @param {ACPRepoMetadataResource} data        JSON representation of a directory
 *
 * @returns {[string, AdobeAsset]}
 */
export function directoryTransformer(data: ACPRepoMetadataResource): [string, AdobeAsset] {
    dbgl('directoryTransformer()');

    const asset = deserializeAsset(data, data[Properties.PAGE]?.embed) as AdobeDirectoryData;
    asset.links = merge({}, (data as AdobeAsset).links, data['_links']);
    const children = data.children || data[Properties.CHILDREN];
    if (children && children.length > 0) {
        asset.children = children.map((child: AdobeAsset) => {
            return deserializeAsset(child);
        });
    } else {
        asset.children = [];
    }
    return [asset.assetId as string, asset];
}

/**
 * Returns a Directory object
 *
 * @param {AdobeHTTPService} svc        HTTP Service
 * @param {AdobeAsset} dirAsset         The asset
 *
 * @returns {AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>}
 */
export function getDirectory(
    svc: AdobeHTTPService | ServiceConfig,
    dirAsset: AdobeAsset,
): AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError> {
    dbgl('directoryTransformer()');

    // this uses links leaf methods
    // if a client wants to get a directory without link handling, they could just use getPrimaryResource
    return fetchLinksIfMissing(svc, dirAsset, [LinkRelation.PRIMARY]).then((links) => {
        const service = getService(svc);
        const primaryHref = getLinkHref(links, LinkRelation.PRIMARY);
        return getDirectoryByURL(service, primaryHref);
    });
}

/**
 * Returns a Directory object by URL
 *
 * @param {AdobeHTTPService} svc        HTTP Service
 * @param {AdobeAsset} dirAsset         The asset
 *
 * @returns {AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>}
 */
export function getDirectoryByURL(
    svc: AdobeHTTPService,
    url: string,
): AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError> {
    dbgl('getDirectoryByURL()');

    return svc
        .invoke(HTTPMethods.GET, url, undefined, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((result) => {
            return {
                result: result.response as AdobeDirectory,
                response: result,
            };
        });
}

/**
 * Returns a PageResource object that allows for traversing a paged set of results
 *
 * @param {AdobeHTTPService} svc        HTTP Service
 * @param {AdobeAsset} dirAsset         The asset
 * @param {PageOptions} pageOpts        Paging options
 *
 * @returns {AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>>}
 */
export function getPagedChildren<T extends EmbeddableResource = never>(
    svc: AdobeHTTPService,
    dirAsset: AdobeAsset,
    pageOpts: Omit<PageOptions<T>, 'itemTransformer'> = {},
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>> {
    dbgl('getPagedChildren()');
    assertLinksContain(dirAsset.links, [LinkRelation.PAGE]);
    if (pageOpts && pageOpts.embed && pageOpts.embed.includes(<T>LinkRelation.REPOSITORY)) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            'Repository Resource embeds on directory listings are not supported',
        );
    }
    try {
        const pageResource = new PageResource<AdobeAsset>(
            dirAsset.links as LinkSet,
            svc,
            directoryTransformer,
            'api:primary',
        );
        return pageResource.getPage<T>(pageOpts, additionalHeaders);
    } catch (error) {
        return AdobePromise.reject(error);
    }
}

/**
 * Create an asset relative to the current directory.
 *
 * @param service               The HTTPService
 * @param parentDir             The parent directory to create the asset in
 * @param relPath               Path relative to current directory.
 * @param createIntermediates   Whether to create intermediate directories if they don't exist.
 * @param contentType           Content type of the new asset.
 * @param respondWith           Resource to be included in response.
 * @param additionalHeaders     Additional headers for the request.
 * @param primaryResource       The primary resource to be uploaded.
 * @param primaryResourceSize   The size of the primary resource.
 * @param repoMetaPatch         Patch to apply to repository metadata immediately after create.
 * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/single.html#repository-metadata-patch-support Repository Metadata Patch Support}
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError, RequestDescriptor>}
 */
export function createAsset(
    svc: AdobeHTTPService | ServiceConfig,
    parentDir: AdobeAsset,
    relPath: string,
    createIntermediates: boolean,
    contentType: string,
    respondWith?: ResourceDesignator,
    additionalHeaders: Record<string, string> = {},
    primaryResource?: SliceableData | GetSliceCallback,
    primaryResourceSize?: number,
    repoMetaPatch?: RepoMetaPatch,
    progressCb?: ProgressCallback,
): AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError, RequestDescriptor> {
    dbgl('createAsset()');

    validateParams(
        ['service', svc, 'object'],
        ['parentDir', parentDir, 'object'],
        ['relPath', relPath, 'string'],
        ['createIntermediates', createIntermediates, 'boolean'],
        ['contentType', contentType, 'string'],
        ['respondWith', respondWith, ['string', 'object'], true],
        ['additionalHeaders', additionalHeaders, 'object', true],
        ['repoMetaPatch', repoMetaPatch, 'object', true],
    );
    const resourceDesignatorStr = isObject(respondWith) ? JSON.stringify(respondWith) : (respondWith as string);

    return fetchLinksIfMissing(svc, parentDir, [LinkRelation.CREATE]).then((links) => {
        const href = getLinkHrefTemplated(links, LinkRelation.CREATE, {
            path: relPath,
            intermediates: createIntermediates.toString(),
            respondWith: resourceDesignatorStr,
            mode: 'id', // not used yet, but will make RAPI return ID-based links
            repoMetaPatch,
        });
        const headers = Object.assign({}, { [HeaderKeys.CONTENT_TYPE]: contentType }, additionalHeaders);
        const service = getService(svc);
        const parsedUploadable = primaryResource
            ? _parseUploadableData(primaryResource, primaryResourceSize)
            : undefined;
        const parsedUploadableSize = parsedUploadable ? getDataLength(parsedUploadable) : 0;

        if (primaryResource && shouldUseBlockTransferForUpload(parentDir, parsedUploadableSize)) {
            return _doBlockUpload({
                service,
                contentType,
                relation: LinkRelation.PRIMARY,
                asset: parentDir,
                dataOrSliceCallback: primaryResource!,
                size: parsedUploadableSize,
                relPath,
                createIntermediates,
                respondWith,
                repoMetaPatch,
                additionalHeaders,
                progressCb,
            }).then(({ result, response }) => {
                const name = relPath.split('/').slice(-1);
                return {
                    result: pruneUndefined(mergeDeep({ name }, result)),
                    response,
                };
            });
        }

        return AdobePromise.resolve().then(async () => {
            const body = isAnyFunction(primaryResource)
                ? await primaryResource(0, parsedUploadableSize)
                : parsedUploadable;
            return service
                .invoke<'json'>(HTTPMethods.POST, href, headers, body as BodyType | undefined, {
                    responseType: 'json',
                    isStatusValid: makeStatusValidator([413]),
                    reuseRequestDesc: {
                        id: 'createAsset',
                        method: HTTPMethods.POST,
                        href,
                        headers,
                        progress: progressCb,
                    },
                })
                .then((resp) => {
                    if (resp.statusCode === 413) {
                        return _doBlockUpload({
                            service,
                            contentType,
                            relation: LinkRelation.PRIMARY,
                            asset: parentDir,
                            dataOrSliceCallback: primaryResource!,
                            size: getDataLength(_parseUploadableData(primaryResource!, primaryResourceSize)),
                            relPath,
                            createIntermediates,
                            respondWith,
                            repoMetaPatch,
                            additionalHeaders,
                            progressCb,
                        })
                            .catch((err) => {
                                if (err.problemType === ProblemTypes.ASSET_NAME_CONFLICT) {
                                    // This branch is expected to be able to be removed once DCX-10172 is complete
                                    // The initial 413 response does not include the asset id but may wind up creating the asset (prior to DCX-10172 being complete)
                                    const existingAsset = {
                                        // if problemType exists, we can assert that the err.response property also exists.
                                        assetId: err.response!.response?.['repo:assetId'],
                                        links: parseLinksFromResponseHeader(err.response!),
                                    };
                                    return _doBlockUpload({
                                        service,
                                        contentType,
                                        relation: LinkRelation.PRIMARY,
                                        asset: existingAsset,
                                        dataOrSliceCallback: primaryResource!,
                                        size: getDataLength(
                                            _parseUploadableData(primaryResource!, primaryResourceSize),
                                        ),
                                        relPath,
                                        createIntermediates,
                                        respondWith,
                                        repoMetaPatch,
                                        additionalHeaders,
                                        progressCb,
                                    });
                                }
                                throw err;
                            })
                            .then(({ result, response }) => {
                                const name = relPath.split('/')[relPath.split('/').length - 1];
                                return {
                                    result: pruneUndefined(mergeDeep({ name }, result)) as AdobeAsset,
                                    response,
                                };
                            });
                    }
                    const name = relPath.split('/')[relPath.split('/').length - 1];
                    let path: string | undefined = undefined;
                    if (parentDir.path) {
                        path = appendPathElements(parentDir.path, relPath);
                    }
                    const links = parseLinksFromResponseHeader(resp);
                    const respHeaders = resp.headers;
                    // rejecting on resource fetch error.
                    // istanbul ignore next
                    if (respHeaders[HeaderKeys.CONTENT_TYPE]?.includes('multipart/mixed')) {
                        const parts = parseMultipartResponseParts(resp);
                        const resourceErrorResponse = parts[1];
                        if (resourceErrorResponse.statusCode === 404) {
                            throw new AdobeDCXError(
                                DCXError.ASSET_NOT_FOUND,
                                'Asset was created successfully but repository metadata could not be found.',
                                undefined,
                                resp,
                            );
                        } else if (resourceErrorResponse.statusCode === 403) {
                            throw new AdobeDCXError(
                                DCXError.FORBIDDEN,
                                'Asset was created successfully but Permission denied for fetching repository metadata.',
                                undefined,
                                resp,
                            );
                        } else {
                            throw unexpectedResponse('Unexpected Server Response', undefined, resp);
                        }
                    }

                    const respBody = isObject(resp.response) ? resp.response : { etag: '', md5: '' };
                    const assetId = respHeaders['asset-id'] || respHeaders['x-resource-id'];
                    const repositoryId = parentDir.repositoryId; // this may be undefined, if links were passed in

                    let etag: string = respBody.etag as string;
                    let md5: string = respBody.md5 as string;
                    // if resource designator is set, etag and md5 in headers may be that resource's values
                    // if it isn't, get those from the headers
                    if (respondWith == null) {
                        etag = respHeaders['etag'];
                        md5 = respHeaders['content-md5'];
                    }

                    // or if repository metadata is returned, the body contains them
                    // use that response as the base asset
                    const asset: AdobeAsset =
                        isObject(resp.response) &&
                        respondWith &&
                        (respondWith === LinkRelation.REPO_METADATA ||
                            (isObject(respondWith) && respondWith.reltype === LinkRelation.REPO_METADATA))
                            ? deserializeAsset(resp.response)
                            : {};

                    const cache = getReposityLinksCache(svc);
                    if (cache) {
                        cache.setValueWithAsset(links, asset);
                    }
                    return {
                        result: pruneUndefined(
                            mergeDeep(
                                { name },
                                asset,
                                pruneUndefined({ links, assetId, etag, md5, repositoryId, format: contentType, path }),
                            ),
                        ),
                        response: resp,
                    };
                });
        });
    });
}
