/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPRepoMetadataResource,
    ACPRepository,
    AdobeAsset,
    AdobeAssetEmbedded,
    AdobeHTTPService,
    AdobeRepository,
    EmbeddableResource,
} from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { merge, validateParams } from '@dcx/util';
import { AdobeGetPageResult, PageOptions, PageResource } from './Pagination';
import { ServiceConfig, constructServiceEndpoint, getReposityLinksCache, getService } from './Service';
import { RepoResponse } from './common';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { Properties } from './enum/properties';
import { IndexRepository, getIndexRepository, parseLinksFromResponseHeader } from './util/link';
import { deserializeAsset, deserializeRepository } from './util/serialization';
import { makeStatusValidator } from './util/validation';
const dbg = newDebug('dcx:assets:discoverable');

/**
 * Returns a paged resource of Discoverable Assets
 *
 * @param svc               The http service
 * @param pageOpts          Paging options
 * @param additionalHeaders Additional headers to attach to HTTP requests
 */
export function getDiscoverableAssets<T extends EmbeddableResource = never>(
    svc: AdobeHTTPService | ServiceConfig,
    pageOpts: Omit<PageOptions<T>, 'itemTransformer'> = {},
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>> {
    dbg('getDiscoverableAssets()');

    validateParams(['svc', svc, 'object'], ['pageOpts', pageOpts, 'object']);

    const service = getService(svc);
    return getIndexRepository(svc, additionalHeaders)
        .then((indexRepository: IndexRepository) => {
            const pageResource = new PageResource<AdobeAsset>(
                indexRepository.assetLinks,
                service,
                discoverableAssetTransformer,
                'api:primary',
            );
            return pageResource.getPage(pageOpts, additionalHeaders);
        })
        .then((pageResponse) => {
            return {
                result: pageResponse.response.response,
                paged: pageResponse.paged,
                response: pageResponse.response,
            };
        });
}

/**
 * Transforms a ACPRepoMetadataResource to an AdobeAsset for use in paging
 *
 * @param {ACPRepoMetadataResource} data    Raw json of an ACPRepoMetadataResource
 *
 * @returns {[string, AdobeAsset]}
 */
export function discoverableAssetTransformer(data: ACPRepoMetadataResource): [string, AdobeAsset] {
    dbg('discoverableAssetTransformer()');

    const dataItem = data[Properties.EMBEDDED][LinkRelation.REPO_METADATA];
    const asset = deserializeAsset(dataItem, data[Properties.EMBEDDED]);
    asset.links = merge({}, (data as AdobeAsset).links, dataItem['_links']);
    return [asset.assetId as string, asset];
}

/**
 * Transforms a ACPRepository to an AdobeRepository for use in paging
 *
 * @param {ACPRepository} data              Raw json of an ACPRepository
 *
 * @returns {[string, AdobeRepository]}
 */
export function discoverableReposTransformer(data: ACPRepository): [string, AdobeRepository] {
    dbg('discoverableReposTransformer()');

    const dataItem = data[Properties.EMBEDDED][LinkRelation.PRIMARY];
    const repository = deserializeRepository(dataItem);
    repository.links = dataItem['_links'];
    // TODO: add checks instead of casting here
    return [repository.repositoryId as string, repository as AdobeRepository];
}

/**
 * Returns a paged resource of Discoverable Repos
 *
 * @param svc               The http service
 * @param pageOpts          Paging options
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export function getDiscoverableRepos(
    svc: AdobeHTTPService | ServiceConfig,
    pageOpts: Omit<PageOptions<never>, 'itemTransformer' | 'embed'> = {},
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeRepository>>> {
    dbg('getDiscoverableRepos()');

    validateParams(['svc', svc, 'object'], ['pageOpts', pageOpts, 'object']);

    const service = getService(svc);
    const repositoriesLink = constructServiceEndpoint('/repositories', service);

    return AdobePromise.resolve(undefined)
        .then(async () => {
            const cache = getReposityLinksCache(svc);
            let repositoriesLinks;
            if (cache) {
                repositoriesLinks = await cache.getRepositoryLinks();
            }

            if (!repositoriesLinks) {
                const response = await service.invoke(
                    HTTPMethods.HEAD,
                    repositoriesLink,
                    additionalHeaders,
                    undefined,
                    {
                        isStatusValid: makeStatusValidator(),
                    },
                );

                repositoriesLinks = parseLinksFromResponseHeader(response);
            }

            if (cache) {
                cache.setRepositoryLinks(repositoriesLinks);
            }

            const pageResource = new PageResource<AdobeRepository>(
                repositoriesLinks,
                service,
                discoverableReposTransformer as any,
            );
            return pageResource.getPage(pageOpts, additionalHeaders);
        })
        .then((pageResponse) => {
            return {
                result: pageResponse.response.response,
                paged: pageResponse.paged,
                response: pageResponse.response,
            };
        });
}
