/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeAsset, AdobeHTTPService } from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import { validateParams } from '@dcx/util';
import { Asset } from './Asset';
import { Composite } from './Composite';
import { Directory } from './Directory';
import { File } from './File';
import { ServiceConfig } from './Service';
import { AssetTypes } from './enum/asset_types';
import { DirectoryMediaType } from './enum/media_types';

const dbg = newDebug('dcx:assets:factory');

/**
 * Create an HLA instance type from AdobeAsset or Asset
 *
 * @param {AdobeAsset | Asset} asset        Asset object
 * @param {AdobeHTTPService} svc            The http service
 *
 * @returns {T}
 */
export function hydrateAsset<T extends Asset = Asset>(
    asset: AdobeAsset | Asset,
    svc: AdobeHTTPService | ServiceConfig,
): T {
    dbg('hydrateAsset()');

    validateParams(['asset', asset, 'object'], ['svc', svc, 'object']);

    const format = asset.format || '';
    const type = format.endsWith('+dcx')
        ? AssetTypes.Composite
        : format === DirectoryMediaType
        ? AssetTypes.Directory
        : format.split('/').length > 1
        ? AssetTypes.File
        : AssetTypes.Asset;

    switch (type) {
        case AssetTypes.Directory:
            if ((asset as Asset).type === type && asset instanceof Directory) {
                return asset as unknown as T;
            }
            return new Directory(asset, svc) as unknown as T;

        case AssetTypes.Composite:
            if ((asset as Composite).type === type && asset instanceof Composite) {
                return asset as unknown as T;
            }
            return new Composite(asset, svc) as unknown as T;

        case AssetTypes.File:
            if ((asset as File).type === type && asset instanceof File) {
                return asset as unknown as T;
            }
            return new File(asset, svc) as unknown as T;

        case AssetTypes.Asset:
            if ((asset as Asset).type === type && asset instanceof Asset) {
                return asset as unknown as T;
            }

            return new Asset(asset, svc) as T;
        /* istanbul ignore next */
        default:
            return new Asset(asset, svc) as T;
    }
}

export class AssetFactory {
    private _svc: AdobeHTTPService | ServiceConfig;
    constructor(svc: AdobeHTTPService | ServiceConfig) {
        this._svc = svc;
    }

    hydrate<T extends Asset>(asset: AdobeAsset): T {
        return hydrateAsset(asset, this._svc);
    }
}
