/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPTransferDocument,
    ACPVersion,
    AdobeAsset,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeResponse,
    BlockTransferDocument,
    JSONPatchDocument,
    LinkSet,
} from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { getLinkHref, getLinkHrefTemplated, validateParams } from '@dcx/util';
import { initBlockUpload } from './BlockTransfer/BlockUpload';
import { AdobeFileBase, AdobeFileData, VersionContentType } from './FileBase';
import { AdobeGetPageResult, PageOptions, PageResource } from './Pagination';
import { ServiceConfig } from './Service';
import {
    AdobeVersion,
    AdobeVersionPatchOperation,
    Version,
    adobeVersionTransformer,
    versionTransformer,
} from './Version';
import { RepoResponse, RepoResponseResult } from './common';
import { AssetType, AssetTypes } from './enum/asset_types';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { JSONPatchMediaType } from './enum/media_types';
import { Properties, VersionProperties } from './enum/properties';
import { PathOrIdAssetDesignator } from './operations';
import { assertLinkProperty, assertLinksContain, makeStatusValidator } from './util/validation';

const dbg = newDebug('dcx:assets:file');
const dbgl = newDebug('dcx:assets:file:leaf');

export interface File extends AdobeFileBase {
    /**
     * Retrieves a paged list of versions for the asset
     * @param pageOpts          Optional paging and embedding parameters
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     *
     * @returns {AdobePromise<RepoResponse<AdobeGetPageResult<Version>>>}
     */
    getPagedVersions(
        pageOpts: Omit<PageOptions<never>, 'embed'>,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponse<AdobeGetPageResult<Version>>>;

    /**
     * Retrieves a specific version of the asset
     * @param version              Resource version
     * @param additionalHeaders    Additional headers to be applied to HTTP requests
     *
     * @returns {AdobePromise<RepoResponseResult<Version>>}
     */
    getVersionResource(
        version: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Version>>;

    /**
     * Update a version
     *
     * @param {string | AdobeVersionPatchOperation[]} patchDoc      The patch document to update the version with
     * @param {string} etag                                         The etag3 of the version to update
     * @param {Record<string, string>} additionalHeaders            Additional headers to be applied to HTTP requests
     *
     * @returns {AdobePromise<AdobeResponse, AdobeDCXError>}
     */
    patchVersions(
        etag: string,
        patchDoc: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse>;

    /**
     * Initializes a control message (via the Block Initiate URL), in which the user agent
     * tells the service that it wishes to transfer content, and the service tells the user
     * agent how to go about doing that.
     * @param transferDocument          The transfer document which is used to initialize, process, and complete block uploads.
     * @param additionalHeaders         Any additional header to include in the request
     *
     * @returns {AdobePromise<RepoResponseResult<Required<BlockTransferDocument>>, AdobeDCXError>}
     */
    initBlockUpload(
        transferDocument: ACPTransferDocument,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Required<BlockTransferDocument>>, AdobeDCXError>;

    /**
     * Copy the file to some destination.
     *
     * @note
     * To copy to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param destAsset             Asset containing either path or assetId
     * @param createIntermediates   Whether to create intermediate directories if missing.
     * @deprecated File overriding using copy operation is no longer supported by the API
     * @param overwriteExisting     Whether to overwrite an existing asset.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     */
    copy(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
        manifestPatch?: JSONPatchDocument,
    ): AdobePromise<RepoResponseResult<File, 'json'>, AdobeDCXError>;
}

export class File extends AdobeFileBase implements File {
    readonly type: AssetType = AssetTypes.File;

    constructor(data: AdobeFileData, svc: AdobeHTTPService | ServiceConfig, links?: LinkSet) {
        super(data, svc, links);
    }

    getPagedVersions(
        pageOpts: Omit<PageOptions<never>, 'embed'> = { itemTransformer: versionTransformer },
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeVersion>>> {
        dbg('getPagedVersions()');

        validateParams(['pageOpts', pageOpts, 'object', true]);

        return this.fetchLinksIfMissing([LinkRelation.PAGE], additionalHeaders).then(() => {
            return getPagedVersions(this._svc, this, pageOpts, additionalHeaders);
        });
    }

    getVersionResource(
        version: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Version | undefined>> {
        dbg('getVersionResource()');

        validateParams(['version', version, 'string']);

        return this.fetchLinksIfMissing([LinkRelation.PAGE], additionalHeaders).then(() =>
            getVersionResource(this._svc, this, version, additionalHeaders).then((response) => ({
                result: response.result ? new Version(response.result, this._svc) : undefined,
                response: response.response,
            })),
        );
    }

    patchVersions(
        patchDoc: string | AdobeVersionPatchOperation[],
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse, AdobeDCXError> {
        dbg('patchVersions()');

        validateParams(['patchDoc', patchDoc, ['string', 'array']], ['etag', etag, 'string', true]);
        return this.fetchLinksIfMissing([LinkRelation.VERSION_HISTORY], additionalHeaders).then(() => {
            return patchVersions(this._svc, this, patchDoc, etag, additionalHeaders);
        });
    }

    initBlockUpload(
        transferDocument: ACPTransferDocument,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<Required<BlockTransferDocument>>, AdobeDCXError> {
        validateParams(
            ['transferDocument', transferDocument, 'object'],
            ['additionalHeaders', additionalHeaders, 'object', true],
        );
        return this.fetchLinksIfMissing([LinkRelation.BLOCK_UPLOAD_INIT], additionalHeaders).then(() =>
            initBlockUpload(this._svc, this, transferDocument, additionalHeaders),
        );
    }

    copy(
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting: boolean,
        additionalHeaders?: Record<string, string>,
        manifestPatch?: JSONPatchDocument,
    ): AdobePromise<RepoResponseResult<File, 'json'>, AdobeDCXError> {
        return super
            .copy(destAsset, createIntermediates, overwriteExisting, additionalHeaders, manifestPatch)
            .then(({ response, result }) => {
                const file = new File(result, this.serviceConfig);
                return {
                    response,
                    result: file,
                };
            });
    }
}

/**
 * Issues a PATCH request against the specified version in the patchDoc.
 *
 * @param svc               - The http service
 * @param asset             - Asset object that identifies the asset
 * @param patchDoc          - A JSON Patch Document (RFC 6902 and https://git.corp.adobe.com/pages/caf/api-spec/#patch)
 * @param etag              - If specified then its value will be passed with the If-Match header.
 *                              Version resource will only be updated if eTag matches what is on the server.
 *                              Note that this etag refers to the Versions resource. The etag property on the asset argument is ignored.
 */
export function patchVersions(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    patchDoc: string | AdobeVersionPatchOperation[],
    etag?: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeResponse, AdobeDCXError> {
    dbgl('patchVersions()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['patchDoc', patchDoc, ['string', 'array']],
        ['etag', etag, 'string', true],
    );
    assertLinksContain(asset.links, [LinkRelation.VERSION_HISTORY]);

    const headers = { ...additionalHeaders, [HeaderKeys.CONTENT_TYPE]: JSONPatchMediaType };
    if (etag) {
        headers[HeaderKeys.IF_MATCH] = etag;
    }

    const versionsHref = getLinkHref(asset.links, LinkRelation.VERSION_HISTORY);
    return svc.invoke(
        HTTPMethods.PATCH,
        versionsHref,
        headers,
        typeof patchDoc === 'string' ? patchDoc : JSON.stringify(patchDoc),
        {
            isStatusValid: makeStatusValidator(),
        },
    );
}

/**
 * Returns a version of an Asset
 *
 * @param svc               - The http service
 * @param asset             - Asset object that identifies the asset
 * @param version           - The version to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export function getVersionResource(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    version: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<ACPVersion>> {
    dbgl('getVersionResource()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object'], ['version', version, 'string']);
    assertLinksContain(asset.links, [LinkRelation.PAGE]);
    assertLinkProperty(asset.links as LinkSet, LinkRelation.PAGE, 'type', VersionContentType);
    const versionHref = getLinkHrefTemplated(asset.links, LinkRelation.PAGE, { version: version });
    return svc
        .invoke<'json'>(HTTPMethods.GET, versionHref, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((resp) => {
            return {
                result:
                    resp.response[VersionProperties.TOTAL_CHILDREN] > 0
                        ? resp.response[Properties.CHILDREN][0]
                        : undefined,
                response: resp,
            };
        });
}

/**
 * Returns a paged list of an assets versions
 *
 * @param svc                 - The http service
 * @param asset               - Asset object that identifies the asset
 * @param pageOpts            - The paging object.
 *                              @see {@link https://developers.corp.adobe.com/storage-api/docs/reference/page.md Page Object Referece }
 * @param additionalHeaders   Additional headers to be applied to HTTP requests
 */
export function getPagedVersions(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    pageOpts: Omit<PageOptions<never>, 'embed'> = {},
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeVersion>>> {
    dbgl('getPagedVersions()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object'], ['pageOpts', pageOpts, 'object']);
    assertLinksContain(asset.links, [LinkRelation.PAGE]);
    assertLinkProperty(asset.links as LinkSet, LinkRelation.PAGE, 'type', VersionContentType);

    const pageResource = new PageResource<AdobeVersion>(
        asset.links as LinkSet,
        svc,
        pageOpts.itemTransformer || adobeVersionTransformer,
        LinkRelation.VERSION_HISTORY,
    );
    return pageResource.getPage(pageOpts, additionalHeaders);
}
