/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeAsset,
    AdobeBlockUpload,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeRepoUploadResult,
    AdobeResponse,
    AdobeResponseType,
    GetSliceCallback,
    JSONPatchDocument,
    LinkRelationKey,
    LinkSet,
    RenditionOptions,
    RepoDownloadStreamableReturn,
    ResponseTypeMap,
    SliceableData,
} from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    LinkProvider,
    assert,
    getLinkHrefTemplated,
    isAnyFunction,
    provideLink,
    pruneUndefined,
    validateParams,
} from '@dcx/util';
import { Asset } from './Asset';
import { _doBlockDownload } from './BlockTransfer/BlockDownload';
import { _doBlockUpload, _upload } from './BlockTransfer/BlockUpload';
import { AdobeStreamableContext, OptionalContext } from './LeafContext';
import { ServiceConfig, getService } from './Service';
import { RepoResponseResult, defaultBufferResponseType } from './common';
import {
    EmbeddedMetadataRepresentation,
    getEmbeddedMetadata,
    patchEmbeddedMetadata,
    putEmbeddedMetadata,
} from './embedded_metadata';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { Properties } from './enum/properties';
import { STREAMABLE_RESPONSE_TYPES } from './enum/response_types';
import { assertLinksContain, makeStatusValidator } from './util/validation';

const dbg = newDebug('dcx:assets:filebase');
const dbgl = newDebug('dcx:assets:filebase:leaf');

export const VersionContentType = 'application/vnd.adobe.versions+json';

export interface AdobeFileData extends AdobeAsset {
    format?: string;
    version?: string;
    width?: number;
    height?: number;
    renderable?: boolean;
}

export interface AdobeFileBase extends Asset<AdobeFileData> {
    getRendition<T extends AdobeResponseType>(
        opts?: RenditionOptions,
        responseType?: T,
        linkProvider?: LinkProvider<RenditionOptions>,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<T>>;

    blockDownload<T extends AdobeResponseType>(
        startByte?: number,
        endByte?: number,
        resource?: LinkRelationKey,
        componentId?: string,
        version?: string,
        responseType?: T,
    ): RepoDownloadStreamableReturn<T>;

    /**
     * ******************************************************************************
     * Embedded metadata APIs
     * ******************************************************************************
     */

    /**
     * Get embedded metadata for the composite.
     *
     * @param {'json' | 'xml'}          format              Whether to return as JSON or XML.
     * @param {Record<string, string>}  additionalHeaders   Additional headers to be applied to HTTP requests
     */
    getEmbeddedMetadata<T = any>(
        format?: EmbeddedMetadataRepresentation,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<T>>;

    /**
     * Update entire embedded/XMP metadata resource.
     *
     * @note
     * Replaces existing XMP resource.
     *
     * @note
     * Currently only supported on composites.
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * The argument `etag` will be updated to optional when the service supports it.
     * See {@link https://jira.corp.adobe.com/browse/SYSE-7940|ticket}
     * and {@link https://jira.corp.adobe.com/browse/SYSE-5943|another ticket}
     *
     * @param {T = Record<string, unknown>} data            New embedded metadata
     * @param {string}                      etag            ETag of metadata resource to update.
     * @param {'json'|'xml'}                [format='json'] Defines the representation of the body, either XML or JSON.
     *                                                      If using XML, clients must pass the data as a string.
     *                                                      If using XML with TypeScript, clients must specify the generic as `string`.
     *                                                      Defaults to json.
     * @param {Record<string, string>}      additionalHeaders     Additional headers to attach to HTTP Requests
     */
    putEmbeddedMetadata<T = Record<string, unknown>>(
        data: T,
        etag: string /** TODO: Revisit optional etag, see: https://jira.corp.adobe.com/browse/SYSE-5943 */,
        format?: EmbeddedMetadataRepresentation,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>>;

    /**
     * Update embedded/XMP metadata using JSON Patch document.
     *
     * @see
     * https://tools.ietf.org/html/rfc6902#page-6
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * The argument `etag` will be updated to optional when the service supports it.
     * see: https://jira.corp.adobe.com/browse/SYSE-7940
     * and: https://jira.corp.adobe.com/browse/SYSE-5943
     *
     * @param {JSONPatchDocument | string}      data      Data to use as PATCH body
     * @param {string}                          etag      ETag of the embedded metadata resource to be updated
     * @param {Record<string, string>}          additionalHeaders     Additional headers to attach to HTTP Requests
     */
    patchEmbeddedMetadata(
        data: JSONPatchDocument | string,
        etag: string /** TODO: Revisit optional etag, see: https://jira.corp.adobe.com/browse/SYSE-5943 */,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>>;

    /**
     * Update the asset's primary resource.
     *
     * @note Implicitly creates a new version of the asset.
     *
     * @param dataOrSliceCallback   Data to use for update, or callback to provide blocks for block upload.
     *                              If a callback is used, block upload will be used, regardless of size of the asset.
     * @param contentType           The content type of the resource to upload.
     * @param [size]                The size of the resource in bytes.
     * @param [etag]                The etag of the existing resource.
     * @param [md5]                 The MD5 hash of the resource, used for block upload finalize.
     * @param [additionalHeaders]   Additional headers to attach to HTTP Requests
     */
    updatePrimaryResource(
        dataOrSliceCallback: SliceableData | GetSliceCallback,
        contentType: string,
        size?: number,
        etag?: string,
        md5?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }>;
}

export abstract class AdobeFileBase extends Asset<AdobeFileData> implements AdobeFileBase {
    constructor(data: AdobeFileData, svc: AdobeHTTPService | ServiceConfig, links?: LinkSet) {
        super(data, svc, links);

        this._data.width = data.width != null ? data.width : (data as Record<string, number>)[Properties.IMAGE_WIDTH];
        this._data.length =
            data.length != null ? data.length : (data as Record<string, number>)[Properties.IMAGE_LENGTH];

        // assume rendition link existence means it can be rendered
        this._data.renderable =
            data.renderable != null
                ? data.renderable
                : typeof this._data.links === 'object'
                ? LinkRelation.RENDITION in this._data.links
                : undefined;
    }

    public get width(): number | undefined {
        return this._data.width;
    }

    public get length(): number | undefined {
        return this._data.length;
    }

    public get renderable(): boolean | undefined {
        return this._data.renderable;
    }
    /**
     * Get's a rendition of the asset according to the (optional) rendition options
     * @param renditionOpts     Rendition preferences. Either longest side should be
     *                              given Or both width and height in options.
     * @param responseType      Type to tranform the response body into.
     * @param additionalHeaders Additional headers to be provided with HTTP requests
     */
    getRendition<T extends AdobeResponseType = 'json'>(
        opts?: RenditionOptions,
        responseType?: T,
        linkProvider?: LinkProvider<RenditionOptions>,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<ResponseTypeMap[T], T>> {
        dbg('getRendition()');

        validateParams(
            ['opts', opts, 'object', true],
            ['responseType', responseType, 'string', true],
            ['linkProvider', linkProvider, 'object', true],
        );

        return this.fetchLinksIfMissing([LinkRelation.RENDITION], additionalHeaders).then(() => {
            return getRendition<T>(this._svc, this, opts, responseType, linkProvider, additionalHeaders);
        });
    }

    /**
     * Embedded Metadata APIs
     */

    /**
     * Returns the embedded metadata for the File
     * @param {XMPResponseType}         format              Response format
     * @param {Record<string, string>}  additionalHeaders   Additional headers to be applied to HTTP requests
     */
    getEmbeddedMetadata<T = any>(
        format: EmbeddedMetadataRepresentation = 'json',
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponseResult<T>> {
        dbg('getEmbeddedMetadata()');

        validateParams(['format', format, 'enum', false, ['json', 'xml']]);

        return this.fetchLinksIfMissing([LinkRelation.EMBEDDED_METADATA], additionalHeaders).then(() => {
            return getEmbeddedMetadata<T>(this._svc, this, format, additionalHeaders);
        });
    }

    putEmbeddedMetadata<T = Record<string, unknown>>(
        data: T,
        etag?: string,
        format: EmbeddedMetadataRepresentation = 'json',
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>> {
        validateParams(
            ['data', data, ['object', 'object[]', 'string']],
            ['etag', etag, 'string', true],
            ['format', format, 'enum', false, ['json', 'xml']],
        );

        return this.fetchLinksIfMissing([LinkRelation.EMBEDDED_METADATA], additionalHeaders).then(() =>
            putEmbeddedMetadata(this._svc, this, data, etag, format, additionalHeaders),
        );
    }

    patchEmbeddedMetadata(
        data: JSONPatchDocument | string,
        etag?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeResponse<'void'>> {
        validateParams(['data', data, ['object', 'object[]', 'string']], ['etag', etag, 'string', true]);

        return this.fetchLinksIfMissing([LinkRelation.EMBEDDED_METADATA], additionalHeaders).then(() =>
            patchEmbeddedMetadata(this._svc, this, data, etag, additionalHeaders),
        );
    }

    /**
     * Perform block download of specified byte range.
     * To get the last N bytes of the file, use `endByte` set to a negative number.
     *
     * @example
     * ```js
     * // fetch last 10 bytes
     * const res = await file.blockDownload(undefined, -10);
     *
     * // fetch first 10 bytes
     * const res = await file.blockDownload(0, 10);
     * ```
     *
     * @param {number} [startByte] - first byte to fetch
     * @param {number} [endByte] - last byte to fetch
     * @param {LinkRelationKey} [resource] - resource to fetch
     * @param {string} [componentId] - specific component ID to fetch
     * @param {string} [version] - version of resource to fetch
     * @param {AdobeResponseType} [responseType = 'buffer'] - type to transform response body into, defaults to Buffer (Node) or ArrayBuffer (browser)
     */
    blockDownload<T extends AdobeResponseType = 'defaultbuffer'>(
        startByte?: number,
        endByte?: number,
        resource?: LinkRelationKey,
        componentId?: string,
        version?: string,
        responseType?: T,
        additionalHeaders?: Record<string, string>,
    ): RepoDownloadStreamableReturn<T> {
        dbg('blockDownload()');

        validateParams(
            ['startByte', startByte, 'number', true],
            ['endByte', endByte, 'number', true],
            ['resource', resource, 'string', true],
            ['componentId', componentId, 'string', true],
            ['version', version, 'string', true],
            ['responseType', responseType, 'enum', true, STREAMABLE_RESPONSE_TYPES],
        );

        assert(
            () => startByte == null || endByte == null || startByte < endByte,
            'endByte must be greater than startByte',
        );

        const ctx = {};
        return this._withSourcePromise(ctx)
            .then(() => this.fetchLinksIfMissing([LinkRelation.BLOCK_DOWNLOAD]))
            .then(() => {
                return doBlockDownload.call(
                    ctx,
                    this._svc,
                    this,
                    startByte,
                    endByte,
                    resource,
                    componentId,
                    version,
                    responseType,
                    additionalHeaders,
                ) as RepoDownloadStreamableReturn<T>;
            });
    }

    updatePrimaryResource(
        dataOrSliceCallback: SliceableData | GetSliceCallback,
        contentType: string,
        size?: number,
        etag?: string,
        md5?: string,
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }> {
        dbg('updatePrimaryResource()');
        const requiredLink = isAnyFunction(dataOrSliceCallback) ? LinkRelation.BLOCK_UPLOAD_INIT : LinkRelation.PRIMARY;

        return this.fetchLinksIfMissing([requiredLink], additionalHeaders)
            .then(() =>
                updatePrimaryResource(
                    this.serviceConfig,
                    this,
                    dataOrSliceCallback,
                    contentType,
                    size,
                    etag,
                    md5,
                    additionalHeaders,
                ),
            )
            .then((res) => {
                this._data.etag = res.result.etag;
                this._data.md5 = res.result.md5;
                this._data.length = res.result.length;
                this._data.version = res.result.version;

                return res;
            });
    }
}

export function getRendition<T extends AdobeResponseType = 'defaultbuffer'>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    renditionOptions?: RenditionOptions,
    responseType: T = 'defaultbuffer' as T,
    linkProvider?: LinkProvider<RenditionOptions>,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<ResponseTypeMap[T], T>> {
    dbgl('getRendition()');
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['renditionOptions', renditionOptions, 'object', true],
        ['linkProvider', linkProvider, 'object', true],
    );
    assertLinksContain(asset.links, [LinkRelation.RENDITION]);
    const renditionHref = linkProvider
        ? provideLink(asset.links![LinkRelation.RENDITION]!, linkProvider, renditionOptions)
        : getLinkHrefTemplated(asset.links, LinkRelation.RENDITION, { ...renditionOptions });
    return svc
        .invoke<T>(HTTPMethods.GET, renditionHref, additionalHeaders, undefined, {
            responseType: defaultBufferResponseType(responseType),
            isStatusValid: makeStatusValidator(),
        })
        .then((resp) => {
            return {
                result: resp.response,
                response: resp,
            };
        });
}

/**
 * Perform block download of specified byte range.
 * @this
 * @param svc
 * @param assetOrPresignedUrl
 * @param startByte
 * @param endByte
 * @param resource
 * @param componentId
 * @param version
 * @param responseType
 * @returns
 */
export function doBlockDownload<T extends AdobeResponseType = 'defaultbuffer'>(
    this: OptionalContext<AdobeStreamableContext>,
    svc: AdobeHTTPService,
    assetOrPresignedUrl: AdobeAsset | string,
    startByte?: number,
    endByte?: number,
    resource?: LinkRelationKey,
    componentId?: string,
    version?: string,
    responseType?: T,
    additionalHeaders?: Record<string, string>,
): RepoDownloadStreamableReturn<T> {
    dbgl('doBlockDownload()');

    validateParams(
        ['svc', svc, 'object'],
        ['assetOrPresignedUrl', assetOrPresignedUrl, ['object', 'string']],
        ['startByte', startByte, 'number', true],
        ['endByte', endByte, 'number', true],
        ['resource', resource, 'string', true],
        ['componentId', componentId, 'string', true],
        ['version', version, 'string', true],
        ['responseType', responseType, 'enum', true, STREAMABLE_RESPONSE_TYPES],
    );

    assert(() => startByte == null || endByte == null || startByte < endByte, 'endByte must be greater than startByte');
    const ctx = (this ?? {}) as AdobeStreamableContext;
    // using presigned url passed in
    if (typeof assetOrPresignedUrl === 'string') {
        return _doBlockDownload.call(
            ctx,
            svc,
            assetOrPresignedUrl,
            startByte,
            endByte,
            responseType,
            true,
            undefined,
            additionalHeaders,
        ) as RepoDownloadStreamableReturn<T>;
    }

    // otherwise, fetching the block href
    const asset = assetOrPresignedUrl;
    assertLinksContain(asset.links, [LinkRelation.BLOCK_DOWNLOAD]);
    const resourceDesignator = pruneUndefined({
        reltype: resource,
        component_id: componentId,
        revision: version,
    });
    const blockDLHref = getLinkHrefTemplated(asset.links, LinkRelation.BLOCK_DOWNLOAD, {
        resource: typeof resource !== 'undefined' ? JSON.stringify(resourceDesignator) : undefined,
    });

    return _doBlockDownload.call(
        ctx,
        svc,
        blockDLHref,
        startByte,
        endByte,
        responseType,
        false,
        undefined,
        additionalHeaders,
    ) as RepoDownloadStreamableReturn<T>;
}

/**
 * Update an asset's primary resource.
 *
 * @note Implicitly creates a new version of the asset.
 *
 * @param service               HTTPService to use.
 * @param asset                 Asset whose primary resource to update.
 * @param dataOrSliceCallback   Data to use for update, or callback to provide blocks for block upload.
 *                              If a callback is used, block upload will be used, regardless of size of the asset.
 * @param contentType           The content type of the resource to upload.
 * @param size                  The size of the resource in bytes.
 * @param etag                  The etag of the existing resource.
 * @param md5                   The MD5 hash of the resource, used for block upload finalize.
 * @param additionalHeaders     Additional headers to attach to HTTP requests
 */
export function updatePrimaryResource(
    svc: AdobeHTTPService | ServiceConfig,
    asset: AdobeAsset,
    dataOrSliceCallback: SliceableData | GetSliceCallback,
    contentType: string,
    size?: number,
    etag?: string,
    md5?: string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, { blockUpload?: AdobeBlockUpload }> {
    dbgl('updatePrimaryResource()');

    validateParams(
        ['service', svc, 'object'],
        ['asset', asset, 'object'],
        ['dataOrSliceCallback', dataOrSliceCallback, ['function', 'object', 'string']],
        ['contentType', contentType, 'string'],
        ['size', size, 'number', true],
        ['md5', md5, 'string', true],
    );

    assertLinksContain(asset.links, [LinkRelation.PRIMARY]);

    return _upload({
        svc,
        asset,
        dataOrSliceCallback,
        contentType,
        relation: LinkRelation.PRIMARY,
        size,
        md5,
        maybeIsNew: false,
        etag,
        additionalHeaders,
    }).catch((err) => {
        if (err.response?.statusCode === 413) {
            return _doBlockUpload({
                service: getService(svc),
                asset,
                dataOrSliceCallback,
                contentType,
                relation: LinkRelation.PRIMARY,
                size,
                md5,
                etag,
                additionalHeaders,
            });
        }
        throw err;
    });
}
