/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { ACPIndexDocument, AdobeAssetEmbedded, AdobeHTTPService, IndexDocument } from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { validateParams } from '@dcx/util';
import { ServiceConfig, constructServiceEndpoint, getService } from './Service';
import { RepoResponseResult } from './common';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { Properties } from './enum/properties';
import { deserializeAsset, deserializeRepository } from './util/serialization';
import { makeStatusValidator } from './util/validation';

const dbg = newDebug('dcx:assets:indexdocument');

/**
 * Returns the ACP Index document
 *
 * @param svc               The http service
 * @param additionalHeaders Additional headers to attach to HTTP requests
 */
export function getIndexDocument(
    svc: AdobeHTTPService | ServiceConfig,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<IndexDocument>> {
    dbg('getIndexDocument()');

    validateParams(['svc', svc, 'object']);

    const service = getService(svc);
    const indexDocumentLink = constructServiceEndpoint('/index', service);

    return service
        .invoke<'json'>(HTTPMethods.GET, indexDocumentLink, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((result) => {
            const indexDocument = deserializeIndexDocument(result.response);
            return {
                result: indexDocument,
                response: result.response,
            };
        });
}

/**
 * Deserialize an ACPIndexDocument into an IndexDocument
 *
 * @param {ACPIndexDocument} data       JSON representation of an IndexDocument
 *
 * @returns {IndexDocument}
 */
export function deserializeIndexDocument(data: ACPIndexDocument): IndexDocument {
    dbg('deserializeIndexDocument()');

    const assignedDirectories = data.children!.map((directory) => {
        const dataItem = directory[Properties.EMBEDDED][LinkRelation.REPO_METADATA];
        const asset: AdobeAssetEmbedded<'http://ns.adobe.com/adobecloud/rel/repository'> = deserializeAsset(dataItem);
        const repository = deserializeRepository(directory[Properties.EMBEDDED][LinkRelation.REPOSITORY]);
        asset.embedded = {
            RepositoryResource: repository,
        };
        return asset;
    });

    const indexDocument: IndexDocument = {
        regions: data[Properties.REPO_REGIONS],
        assignedDirectories: assignedDirectories,
        links: data['_links'],
    };
    return indexDocument;
}
