/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeBlockDownload, AdobeDCXError, AdobeHTTPService } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import AdobePromise from '@dcx/promise';
import { getLinkHref, isFunction, isObject } from '@dcx/util';
import { ServiceConfig } from './Service';
import { LinkRelation } from './enum/link';
import { getIndexLinks } from './util/link';

/**
 * Optional context.
 */
export type OptionalContext<T> = Partial<T> | undefined | void;

/**
 * Scope for leaf methods calling ops or resolve APIs.
 *
 * Allows bypassing caching while still using
 * common links that can be retained for long times.
 *
 * ie. operations endpoint and resolve API
 */
export interface AdobeOpsContext {
    opsHref?: string | (() => string) | (() => Promise<string>);
    resolveByPathHref?: string | (() => string) | (() => Promise<string>);
    resolveByIdHref?: string | (() => string) | (() => Promise<string>);
}

/**
 * Defines the possible context provided to a leaf function that may use streaming.
 *
 * If in streaming mode (responseType='stream'), and if previous asynchronous methods
 * were called before reaching the leaf method, a stream would have been created synchronously
 * and attached to the AdobePromise source object.
 *
 * If a stream exists in the context, it should be used instead of creating a new stream.
 */
export type AdobeStreamableContext = {
    blockDownload?: AdobeBlockDownload;
};

/**
 * Get ops endpoint URL from context or API.
 *
 * @this {OptionalContext<AdobeOpsContext>}
 * @param svc               The http service
 * @param additionalHeaders Additional headers to attach to HTTP Requests`
 */
export function getOpsHref(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    additionalHeaders?: Record<string, string>,
): AdobePromise<string, AdobeDCXError> {
    // First check for a bound scope.
    // This allows clients to use leaf methods more simply without a cache by calling like:
    // getOpsURL.call({ opsURL: 'https://endpoint.com' });
    if (isObject(this)) {
        if (typeof this.opsHref === 'string') {
            return AdobePromise.resolve(this.opsHref);
        }
        if (isFunction(this.opsHref)) {
            return AdobePromise.resolve(this.opsHref());
        }
    }

    // Try getting ops link from cache, or fetch it
    return getIndexLinks(svc, additionalHeaders).then((links) => {
        try {
            return getLinkHref(links, LinkRelation.REPO_OPS);
        } catch (e) {
            throw new DCXError(DCXError.UNEXPECTED, 'Could not get ops href.', e as Error);
        }
    });
}

/**
 * Get resolveByPath URL from context or API.
 *
 * @this {OptionalContext<AdobeOpsContext>}
 * @param {AdobeHTTPService} svc    The http service
 *
 * @returns {AdobePromise<string, AdobeDCXError>}
 */
export function getResolveByPathHref(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
): AdobePromise<string, AdobeDCXError> {
    // First check for a bound scope.
    // This allows clients to use leaf methods more simply without a cache by calling like:
    // getOpsURL.call({ opsURL: 'https://endpoint.com' });
    if (isObject(this)) {
        if (typeof this.resolveByPathHref === 'string') {
            return AdobePromise.resolve(this.resolveByPathHref);
        }
        if (isFunction(this.resolveByPathHref)) {
            return AdobePromise.resolve(this.resolveByPathHref());
        }
    }

    // Try getting ops link from cache, or fetch it
    return getIndexLinks(svc).then((links) => {
        try {
            return getLinkHref(links, LinkRelation.RESOLVE_BY_PATH);
        } catch (e) {
            throw new DCXError(DCXError.UNEXPECTED, 'Could not get ResolveByPath href.', e as Error);
        }
    });
}

/**
 * Get resolveById URL from context or API.
 *
 * @this {OptionalContext<AdobeOpsContext>}
 * @param {AdobeHTTPService} svc    The http service
 *
 * @returns {AdobePromise<string, AdobeDCXError>}
 */
export function getResolveByIdHref(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
): AdobePromise<string, AdobeDCXError> {
    // First check for a bound scope.
    // This allows clients to use leaf methods more simply without a cache by calling like:
    // getOpsURL.call({ opsURL: 'https://endpoint.com' });
    if (isObject(this)) {
        if (typeof this.resolveByIdHref === 'string') {
            return AdobePromise.resolve(this.resolveByIdHref);
        }
        if (isFunction(this.resolveByIdHref)) {
            return AdobePromise.resolve(this.resolveByIdHref());
        }
    }

    // Try getting ops link from cache, or fetch it
    return getIndexLinks(svc).then((links) => {
        try {
            return getLinkHref(links, LinkRelation.RESOLVE_BY_ID);
        } catch (e) {
            throw new DCXError(DCXError.UNEXPECTED, 'Could not get ResolveByID href.', e as Error);
        }
    });
}
