/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeAsset, AdobeHTTPService, AdobeResponse, BasicLink, EmbeddableResource, LinkSet } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { getLinkHrefTemplated } from '@dcx/util';
import { RepoResponse } from './common';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { Properties } from './enum/properties';
import { makeStatusValidator } from './util/validation';

const dbg = newDebug('dcx:assets:pagination');

export interface PageData {
    orderBy?: string;
    start?: string;
    next?: string;
    property?: string;
    state?: string;
    type?: string;
    count?: number;
}

export interface Page {
    page?: PageData;
    children: AdobeAsset[];
}

export interface AdobeGetPageResult<T> {
    paged: PageResource<T>;
    result: any;
}

export interface PageResource<T> {
    currentPage: PageData | undefined;
    listResource?: string;

    /**
     * Returns true if another page exists
     */
    hasNextPage(): boolean;

    /**
     * Get a page of data based on PageOptions
     * @param opts      The page options of data to load
     */
    getPage<U extends EmbeddableResource = never>(
        opts: PageOptions<U>,
    ): AdobePromise<RepoResponse<AdobeGetPageResult<T>>>;

    /**
     * Get's the next page in a pagedResource
     */
    getNextPage(): AdobePromise<RepoResponse<AdobeGetPageResult<T>>>;
}

export interface PageOptions<T extends EmbeddableResource = never> {
    /**
     * Specifies the beginning of the desired page. This is the value from which the page will start. When not specified, the first page is given.
     * @see [rel/page](https://developers.corp.adobe.com/storage-api/docs/reference/page.md#query-parameters)
     */
    start?: string;
    /**
     * Specifies a positive integer as a hint as to the maximum number of items that should be returned for a given request.
     * The actual response size may be smaller or larger, as constrained by the need to provide reliable operation of the start parameter.
     * @see [rel/page](https://developers.corp.adobe.com/storage-api/docs/reference/page.md#query-parameters)
     */
    limit?: number;
    count?: number;
    /**
     * A comma-separated, ordered list of properties by which the Resource is sorted.
     * The first property is used for primary sorting, the second property to resolve ties in primary sorting, and so on.
     * The name of a property may be prefixed with a + to indicate ascending ordering or a - to indicate
     * descending ordering by that property. If the property name is not prefixed, the result is sorted in ascending order.
     * A Resource can be sorted by the following properties: repo:name, repo:createDate, repo:modifyDate.
     * @see [rel/page](https://developers.corp.adobe.com/storage-api/docs/reference/page.md#query-parameters)
     */
    orderBy?: string;
    next?: string;
    embed?: T[];
    itemTransformer?: (item: any, svc: AdobeHTTPService) => [string, any];
    type?: string;
}

type ListResource = 'api:primary' | 'version-history';

export class PageResource<T = AdobeAsset> implements AsyncIterator<AdobeGetPageResult<T>> {
    private _items: Record<string, T> = {};
    private _nextPageLink: BasicLink | undefined;
    private _data: any;
    currentPage: PageData | undefined;
    ListResource?: ListResource;

    constructor(
        private _links: LinkSet,
        private _svc: AdobeHTTPService,
        private _transformer: (pre: any, svc: AdobeHTTPService) => [string, T],
        ListResource?: ListResource,
    ) {
        this.ListResource = ListResource;
        if (!_links || !_links[LinkRelation.PAGE]) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Asset must have links that contain a page relation.');
        }
    }

    public get items(): T[] {
        return Object.values(this._items);
    }

    public get data(): any {
        return this._data;
    }

    getPage<R extends EmbeddableResource = never>(
        opts: Omit<PageOptions<R>, 'itemTransformer'> = {},
        additionalHeaders?: Record<string, string>,
    ): AdobePromise<RepoResponse<AdobeGetPageResult<T>>> {
        dbg('getPage()');

        const { embed, ...options } = opts;
        if (embed) {
            (options as Record<string, unknown>).embed = embed.some((embeddable) => typeof embeddable === 'object') ? JSON.stringify(embed) : embed.join(',');
        }

        if (this.ListResource) {
            Object.assign(options, { resource: this.ListResource });
        }

        const href = getLinkHrefTemplated(this._links, LinkRelation.PAGE, options as Record<string, string>);
        return this._svc
            .invoke(HTTPMethods.GET, href, additionalHeaders, undefined, {
                isStatusValid: makeStatusValidator(),
                responseType: 'json',
            })
            .then((response) => {
                const data = this.parseResponse(response);
                this._data = data;
                return {
                    paged: this,
                    result: this._data,
                    response: response,
                };
            });
    }

    getNextPage(): AdobePromise<RepoResponse<AdobeGetPageResult<T>>> | undefined {
        dbg('getNextPage()');

        if (!this.hasNextPage() || !this._nextPageLink) {
            return undefined;
        }
        return this._svc
            .invoke(HTTPMethods.GET, this._nextPageLink.href, undefined, undefined, {
                isStatusValid: makeStatusValidator(),
                responseType: 'json',
            })
            .then((response) => {
                const data = this.parseResponse(response);
                this._data = data;
                return {
                    paged: this,
                    result: this._data,
                    response: response,
                };
            });
    }

    hasNextPage(): boolean {
        dbg('hasNextPage() ', this._nextPageLink !== undefined);

        return this._nextPageLink !== undefined;
    }

    *[Symbol.iterator]() {
        for (const key in this._items) {
            yield this._items[key];
        }
    }

    async *[Symbol.asyncIterator]() {
        for (const key in this._items) {
            yield this._items[key];
        }

        while (this.hasNextPage()) {
            const nextPage = await this.next();
            for (const item of nextPage.value.paged) {
                yield item;
            }
        }
    }

    async next(): Promise<IteratorResult<AdobeGetPageResult<T>>> {
        dbg('next()');

        if (!this.hasNextPage()) {
            return { done: true, value: undefined };
        }

        const nextPage = await this.getNextPage();
        if (!nextPage) {
            return { done: true, value: undefined };
        }
        return { done: false, value: nextPage };
    }

    private parseResponse(result: AdobeResponse): any {
        dbg('parseResponse()');

        this._items = {};
        const data = result.response;
        for (const i in data[Properties.CHILDREN]) {
            const item = data[Properties.CHILDREN][i];
            const [id, toSet] = this._transformer(item, this._svc);
            this._items[id] = toSet;
        }
        this._nextPageLink = data[Properties.LINKS].next;
        this.currentPage = data[Properties.PAGE];
        return data;
    }
}
