/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPAccessControlList,
    ACPRepository,
    ACPVersion,
    AdobeAsset,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeResponse,
    EffectivePrivileges,
    LinkMode,
    LinkRelationKey,
    LinkSet,
    Privilege,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { merge } from '@dcx/util';
import { Asset } from './Asset';
import { AdobeFileBase } from './FileBase';
import { RepoResponseResult } from './common';
import { AssetTypes } from './enum/asset_types';
import { VersionProperties } from './enum/properties';
import { BulkRequestDescriptor } from './util/bulk';

const dbg = newDebug('dcx:assets:version');
const dbgl = newDebug('dcx:assets:version:leaf');

export interface AdobeVersionPatchOperation {
    op: 'add' | 'replace' | 'remove';
    path: string;
    value?: AdobeVersionMilestone;
}

export interface AdobeVersionMilestone {
    label: string;
    description?: string;
}

export interface AdobeVersion extends Partial<Omit<AdobeAsset, 'name' | 'path'>> {
    milestone?: AdobeVersionMilestone;
}

export type VVersion = Partial<Omit<Asset, 'name' | 'path'>> & {
    created_at?: Date;
    updated_at?: Date;
};

export class Version extends AdobeFileBase implements AdobeVersion {
    readonly type = AssetTypes.Version;
    protected _data: AdobeVersion;

    constructor(data: AdobeVersion & ACPVersion, protected _svc: AdobeHTTPService, links?: LinkSet) {
        super(data, _svc, links);
        this._data = deserializeVersion(data);
    }

    get milestone(): AdobeVersionMilestone | undefined {
        return this._data.milestone;
    }

    /**
     * @hidden
     * Not found on version links
     */
    getRepositoryResource(): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError> {
        dbg('getRepositoryResource()');

        throw new DCXError(DCXError.NOT_IMPLEMENTED, 'Method not implemented.');
    }

    /**
     * @hidden
     * Not found on version links
     */
    getEffectivePrivileges(): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError> {
        dbg('getEffectivePrivileges()');

        throw new DCXError(DCXError.NOT_IMPLEMENTED, 'Method not implemented.');
    }

    /**
     * @hidden
     * Not found on version links
     */
    performBulkRequest(
        requests: BulkRequestDescriptor[],
        linkMode?: LinkMode,
    ): AdobePromise<RepoResponseResult<AdobeResponse[]>, AdobeDCXError> {
        dbg('performBulkRequest()');

        throw new DCXError(DCXError.NOT_IMPLEMENTED, 'Method not implemented.');
    }

    /**
     * @hidden
     * Not found on version links
     */
    checkACLPrivilege(
        privilege: Omit<Privilege, 'none'>,
        relation: LinkRelationKey,
    ): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError> {
        dbg('checkACLPrivilege()');

        throw new DCXError(DCXError.NOT_IMPLEMENTED, 'Method not implemented.');
    }

    /**
     * @hidden
     * Not found on version links
     */
    getACLPolicy(): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError> {
        dbg('getACLPolicy()');

        throw new DCXError(DCXError.NOT_IMPLEMENTED, 'Method not implemented.');
    }
}

export function adobeVersionTransformer(data: AdobeVersion & ACPVersion): [string, AdobeVersion] {
    dbgl('adobeVersionTransformer()');

    const ver = deserializeVersion(data);
    ver.links = merge({}, (data as AdobeVersion).links, data['_links']);
    return [ver.version as string, ver];
}

export function versionTransformer(data: AdobeVersion & ACPVersion, svc: AdobeHTTPService): [string, Version] {
    dbgl('versionTransformer()');

    const ver = new Version(data, svc);
    return [ver.version as string, ver];
}

export function deserializeVersion(data: AdobeVersion & ACPVersion): AdobeVersion {
    dbgl('deserializeVersion()');

    const ver: AdobeVersion = {
        version: data.version || data[VersionProperties.VERSION],
        createDate: data.created || data[VersionProperties.CREATED],
        createdBy: data.createdBy || data[VersionProperties.CREATED_BY],
        milestone: data.milestone || data[VersionProperties.MILESTONE],
        links: data['_links'],
    };
    return ver;
}
