/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { ACPVersions, AdobeHTTPService, LinkSet } from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import { merge } from '@dcx/util';
import { AdobeVersion, Version, deserializeVersion } from './Version';
import { Properties, VersionProperties } from './enum/properties';

const dbg = newDebug('dcx:assets:versionset');
const dbgl = newDebug('dcx:assets:versionset:leaf');

export interface AdobeVersionSet {
    repositoryId: string | undefined;
    assetId: string | undefined;
    versions: AdobeVersion[];
    versionCount: number | undefined;
    links?: LinkSet;
}

export class VersionSet implements AdobeVersionSet {
    /**
     * The repository Id of the asset
     */
    repositoryId: string | undefined;
    /**
     * The asset Id of the asset
     */
    assetId: string | undefined;
    /**
     * An array of versions for the asset
     */
    versions: Version[];
    /**
     * A count of versions for the asset
     */
    versionCount: number | undefined;
    /**
     * The assets links
     */
    links?: LinkSet;

    constructor(data: ACPVersions & AdobeVersionSet, protected _svc: AdobeHTTPService, links?: LinkSet) {
        this.versionCount = data.versionCount || data[VersionProperties.TOTAL_CHILDREN];
        this.repositoryId = data.repositoryId || data[Properties.REPO_REPOSITORY_ID];
        this.assetId = data.assetId || data[Properties.REPO_ASSET_ID];
        this.versions = [];
        for (const versionIndex in data.versions || data[Properties.CHILDREN]) {
            this.versions.push(
                new Version(
                    data.versions && data.versions[versionIndex]
                        ? data.versions[versionIndex]
                        : data[Properties.CHILDREN][versionIndex],
                    _svc,
                ),
            );
        }

        this.links = merge({}, (data as AdobeVersionSet).links, data['_links'], links);
    }

    /**
     * Returns a version from the version set by Id
     *
     * @param {string}versionId     The version Id
     *
     * @returns {Version | undefined}
     */
    versionByVersionId(versionId: string): Version | undefined {
        dbg('versionByVersionId()');

        return this.versions.find((version) => version.version === versionId);
    }

    /**
     * Returns all versions containing a label
     *
     * @param {string} label    The label to search for
     *
     * @returns {AdobeVersion[]}
     */
    versionsByLabel(label: string): AdobeVersion[] {
        dbg('versionsByLabel()');

        return this.versions.filter((version) => {
            if (version.milestone && version.milestone.label === label) {
                return true;
            }
            return false;
        });
    }
}

/**
 * Deserialize a raw version set into an AdobeVersionSet
 *
 * @param {ACPVersions} data  JSON raw version set
 *
 * @returns {AdobeVersionSet}
 */
export function deserializeVersionSet(data: ACPVersions): AdobeVersionSet {
    dbgl('deserializeVersionSet()');

    const ver: AdobeVersionSet = {
        versionCount: data[VersionProperties.TOTAL_CHILDREN],
        repositoryId: data[VersionProperties.REPO_ID],
        assetId: data[Properties.REPO_ASSET_ID],
        links: {},
        versions: [],
    };

    const children = data.children || data[Properties.CHILDREN];
    if (children && children.length > 0) {
        ver.versions = children.map((child) => {
            return deserializeVersion(child);
        });
    }
    ver.links = data['_links'];

    return ver;
}
