/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeAsset } from '@dcx/common-types';
import { DCXError } from '@dcx/error';

export interface AdobeGenericCache<T> {
    /**
     * Return the key for the passed asset
     * @param asset     The asset to get the key for
     */
    getKey(asset: AdobeAsset): string | undefined;

    /**
     * Returns a value for the asset from the cache
     * @param asset     The asset key
     */
    getValueWithAsset(asset: AdobeAsset): Promise<T> | undefined;

    /**
     * Sets a promise for the key that can be resolved with the value
     * @param key       The key
     * @param repoId    The repoId
     */
    setPending(key: string, repoId?: string): void;

    /**
     * Returns value for key from cache
     * @param key       Cache key
     * @param repoId    The repositoryId
     */
    get(key: string, repoId?: string): Promise<T> | undefined;

    /**
     * Sets value for asset as key, if no repoId is set on the asset the asset will be stored in the session
     * @param value     The value to set
     * @param asset     The asset to use as the key
     */
    setValueWithAsset(value: T, asset: AdobeAsset): void;

    /**
     * Sets the key value pair
     * @param value     The value
     * @param key       The key
     * @param repoId    The repoId
     */
    set(value: T, key: string, repoId?: string): void;

    /**
     * Delete entry at key
     * @param {string} key
     */
    delete(key: string): void;

    /**
     * Delete entry with asset key
     * @param {AdobeAsset} asset
     */
    deleteWithAsset(asset: AdobeAsset): void;
}

export const DEFAULT_CACHE_MAX_ENTRIES = 100000;

/**
 * A cache that holds references to a limited number of values. Each time a value is accessed, it is moved to the head of a queue.
 * When a value is added to a full cache, the value at the end of that queue is evicted.
 */
export class GenericCache<T> implements AdobeGenericCache<T> {
    private values: Record<string, Map<string, Promise<T>>> = {};
    private maxEntries = DEFAULT_CACHE_MAX_ENTRIES;
    private defaultSessionKey;
    private promiseToResolveMap: Map<Promise<T>, (res?: T) => void> = new Map();

    constructor(maxEntries: number = DEFAULT_CACHE_MAX_ENTRIES, defaultSessionKey = 'SESSION') {
        if (maxEntries <= 0) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Cache Max enteries must be great than 0.');
        }
        this.maxEntries = maxEntries;
        this.defaultSessionKey = defaultSessionKey;
    }

    /**
     * Clear cache, resolve all pending promises to undefined
     */
    public clear(): void {
        this.promiseToResolveMap.forEach((v) => {
            v.call(undefined);
        });

        for (const i in this.values) {
            this.values[i].clear();
        }
        this.values = {};
    }

    /**
     * Return the key for the passed asset
     * @param asset     The asset to get the key for
     */
    public getKey(asset: AdobeAsset): string | undefined {
        if (!asset.assetId && typeof asset === 'object') {
            return undefined;
        }

        return asset.assetId;
    }

    /**
     * Returns a value for the asset from the cache
     * @param asset     The asset key
     */
    public getValueWithAsset(asset: AdobeAsset): Promise<T> | undefined {
        if (!asset.assetId && typeof asset === 'object') {
            return;
        }

        const key = this.getKey(asset);
        if (!key) {
            return;
        }

        return this.get(key, asset.repositoryId);
    }

    /**
     * Sets a promise for the key that can be resolved with the value
     * @param key       The key
     * @param repoId    The repoId
     */
    public setPending(key: string, repoId: string = this.defaultSessionKey) {
        let resolve;

        //Initialize cache for repoId if not already done
        if (!this.values[repoId]) {
            this.values[repoId] = new Map<string, Promise<T>>();
        }

        // if already set pending, return existing promise
        const existing = this.values[repoId].get(key);
        if (existing && existing instanceof Promise) {
            return this.promiseToResolveMap.get(existing);
        }

        const promise = new Promise<T>((resolve_) => {
            resolve = resolve_;
        });

        this.values[repoId].set(key, promise);

        this.promiseToResolveMap.set(promise, resolve);

        promise
            .then(() => this.promiseToResolveMap.delete(promise))
            .catch(() => this.promiseToResolveMap.delete(promise));

        return resolve;
    }

    /**
     * Returns value for key from cache
     * @param key       Cache key
     * @param repoId    The repositoryId
     */
    public get(key: string, repoId = this.defaultSessionKey): Promise<T> | undefined {
        if (this.values[repoId] && repoId in this.values) {
            return this.values[repoId].get(key) as Promise<T>;
        }
    }

    /**
     * Sets value for asset as key, if no repoId is set on the asset the asset will be stored in the session
     * @param value     The value to set
     * @param asset     The asset to use as the key
     */
    public setValueWithAsset(value: T | undefined, asset: AdobeAsset): void {
        if (!value) {
            return;
        }

        const key = this.getKey(asset);
        if (key) {
            const repoId: string = asset.repositoryId || this.defaultSessionKey;
            this.set(value, key, repoId);
        }
    }

    /**
     * Sets the key value pair
     * @param value     The value
     * @param key       The key
     * @param repoId    The repoId
     */
    public set(value: T, key: string, repoId = this.defaultSessionKey): void {
        if (!this.values[repoId]) {
            this.values[repoId] = new Map<string, Promise<T>>();
        } else {
            if (this.values[repoId] && this.values[repoId].get(key) instanceof Promise) {
                const p = this.values[repoId].get(key) as Promise<T>;
                const resolve = this.promiseToResolveMap.get(p);
                this.promiseToResolveMap.delete(p);
                if (resolve) {
                    resolve(value);
                }
            }
        }
        if (this.values[repoId].size >= this.maxEntries) {
            // least-recently used cache eviction strategy
            const keyToDelete = this.values[repoId].keys().next().value;

            this.values[repoId].delete(keyToDelete);
        }

        this.values[repoId].set(key, Promise.resolve(value));
    }

    public delete(key: string, repoId = this.defaultSessionKey): void {
        if (!this.values[repoId]) {
            return;
        }

        this.values[repoId].delete(key);
    }

    public deleteWithAsset(asset: AdobeAsset): void {
        const key = this.getKey(asset);
        if (key) {
            this.delete(key, asset.repositoryId);
        }
    }
}
