/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { LinkSet } from '@dcx/common-types';
import { IndexRepository } from '../util/link';
import { AdobeGenericCache, DEFAULT_CACHE_MAX_ENTRIES, GenericCache } from './GenericCache';

export const MAX_CACHE_PERIOD_MS = 1000 * 3600 * 24 * 30; // 30 days

export const enum CACHE_KEYS {
    SESSION_KEY = 'SESSION',
    INDEX_KEY = 'INDEX',
    INDEX_DOCUMENT_KEY = '/Index.json',
    REPOSITORY_DOCUMENT_KEY = '/Repositories.json',
    REPOSITORIES_LINK_KEY = '/Repositories',
    ASSETS_DOCUMENT_KEY = '/Assets.json',
}

export interface AdobeRepositoryLinksCache<LinkSet> extends AdobeGenericCache<LinkSet> {
    /**
     * Set the index links
     * @param value     The value to cache
     */
    setIndexLinks(value: LinkSet): void;

    /**
     * Returns the index links, these links contain the resolve and ops links
     */
    getIndexLinks(): Promise<LinkSet> | undefined;

    /**
     * Set the index document links
     * @param value     The value to cache
     */
    setIndexRepository(values: IndexRepository): void;

    /**
     * Returns the index document links
     */
    getIndexRepository(): IndexRepository | undefined;

    /**
     * Set the repository links
     * @param value     The value to cache
     */
    setRepositoryLinks(value: LinkSet): void;

    /**
     * Get the repository links
     */
    getRepositoryLinks(): Promise<LinkSet> | undefined;
}

/**
 * A cache that holds references to a limited number of values. Each time a value is accessed, it is moved to the head of a queue.
 * When a value is added to a full cache, the value at the end of that queue is evicted.
 */
export class RepositoryLinksCache extends GenericCache<LinkSet> implements AdobeRepositoryLinksCache<LinkSet> {
    indexRepository?: IndexRepository;
    private timestampsOnLinkCreation = 0;
    private maxCachePeriodMS = 0;
    constructor(maxEntries: number = DEFAULT_CACHE_MAX_ENTRIES, maxCachePeriodMS: number = MAX_CACHE_PERIOD_MS) {
        super(maxEntries, CACHE_KEYS.SESSION_KEY);
        this.maxCachePeriodMS = maxCachePeriodMS;
    }

    private isLinkExpired(): boolean {
        return this.maxCachePeriodMS < Date.now() - this.timestampsOnLinkCreation;
    }

    /**
     * Set the index links
     * @param value     The value to cache
     */
    setIndexLinks(value: LinkSet): void {
        this.set(value, CACHE_KEYS.INDEX_KEY, CACHE_KEYS.SESSION_KEY);
        this.timestampsOnLinkCreation = Date.now();
    }

    /**
     * Returns the index links
     */
    getIndexLinks(): Promise<LinkSet> | undefined {
        if (!this.isLinkExpired()) {
            return this.get(CACHE_KEYS.INDEX_KEY, CACHE_KEYS.SESSION_KEY);
        }
    }

    /**
     * Set the index document links
     * @param value     The value to cache
     */
    setIndexRepository(value: IndexRepository): void {
        this.indexRepository = value;
    }

    /**
     * Returns the index document links
     */
    getIndexRepository(): IndexRepository | undefined {
        return this.indexRepository;
    }

    /**
     * Set the repositories links
     * @param value     The value to cache
     */
    setRepositoryLinks(value: LinkSet): void {
        this.set(value, CACHE_KEYS.REPOSITORIES_LINK_KEY, CACHE_KEYS.SESSION_KEY);
        this.timestampsOnLinkCreation = Date.now();
    }

    /**
     * Returns the repositories links
     */
    getRepositoryLinks(): Promise<LinkSet> | undefined {
        if (!this.isLinkExpired()) {
            return this.get(CACHE_KEYS.REPOSITORIES_LINK_KEY, CACHE_KEYS.SESSION_KEY);
        }
    }
}
