/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeAsset, AdobeHTTPService, AdobeResponse, JSONPatchDocument } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { getLinkHref, validateParams } from '@dcx/util';
import { RepoResponseResult } from './common';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { JSONPatchMediaType } from './enum/media_types';
import { EmbeddedMetadataMediaTypes } from './enum/properties';
import { assertLinksContain, makeStatusValidator } from './util/validation';

export type EmbeddedMetadataRepresentation = 'json' | 'xml';

/**
 * Get embedded metadata for an asset.
 *
 * @param {AdobeHTTPService}        svc                 HTTPService to use
 * @param {AdobeAsset}              asset               Asset whose embedded metadata to fetch, must contain embedded link.
 * @param {'json' | 'xml'}          format              Whether to return as JSON or XML.
 * @param {Record<string, string>}  additionalHeaders   Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<RepoResponseResult<T>>}
 */
export function getEmbeddedMetadata<T = any>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    format: EmbeddedMetadataRepresentation = 'json',
    additionalHeaders: Record<string, string> = {},
): AdobePromise<RepoResponseResult<T>> {
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['format', format, 'enum', false, ['json', 'xml']],
    );
    assertLinksContain(asset.links, [LinkRelation.EMBEDDED_METADATA]);

    const href = getLinkHref(asset.links, LinkRelation.EMBEDDED_METADATA);
    return svc
        .invoke(
            HTTPMethods.GET,
            href,
            { ...additionalHeaders, accept: EmbeddedMetadataMediaTypes[format.toUpperCase()], vary: 'accept' },
            undefined,
            {
                isStatusValid: makeStatusValidator(),
                responseType: format === 'json' ? 'json' : 'text',
            },
        )
        .then((response) => {
            return {
                result: response.response as T,
                response: response,
            };
        });
}

/**
 * Update entire embedded/XMP metadata resource.
 *
 * @note
 * Replaces existing XMP resource.
 *
 * @note
 * Currently the service requires an etag, but spec states unguarded updates should be possible.
 * see: https://jira.corp.adobe.com/browse/SYSE-7940
 * and: https://jira.corp.adobe.com/browse/SYSE-5943
 *
 * @param {AdobeHTTPService}            svc             HTTPService to use
 * @param {AdobeAsset}                  asset           Asset whose embedded metadata should be updated
 * @param {T = Record<string, unknown>} data            New embedded metadata
 * @param {string}                      [etag]          ETag of metadata resource to update.
 * @param {'json'|'xml'}                [format='json'] Defines the representation of the body, either XML or JSON.
 *                                                      If using XML, clients must pass the data as a string.
 *                                                      If using XML with TypeScript, clients must specify the generic as `string`.
 *                                                      Defaults to json.
 * @param {Record<string, string>}      additionalHeaders Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<AdobeResponse<'void'>>}
 */
export function putEmbeddedMetadata<T = Record<string, unknown>>(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    data: T,
    etag?: string,
    format: EmbeddedMetadataRepresentation = 'json',
    additionalHeaders: Record<string, string> = {},
): AdobePromise<AdobeResponse<'void'>> {
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['data', data, ['string', 'object', 'object[]']],
        ['etag', etag, 'string', true],
        ['format', format, 'enum', false, ['json', 'xml']],
    );
    assertLinksContain(asset.links, [LinkRelation.EMBEDDED_METADATA]);

    const href = getLinkHref(asset.links, LinkRelation.EMBEDDED_METADATA);
    const headers = {
        ...additionalHeaders,
        [HeaderKeys.CONTENT_TYPE]: EmbeddedMetadataMediaTypes[format.toUpperCase()],
        [HeaderKeys.IF_MATCH]: etag as string /** pruned if undefined in HTTPService */,
    };
    return svc.invoke(HTTPMethods.PUT, href, headers, typeof data === 'string' ? data : JSON.stringify(data), {
        isStatusValid: makeStatusValidator(),
    });
}

/**
 * Update embedded/XMP metadata using JSON Patch document.
 *
 * @see
 * https://tools.ietf.org/html/rfc6902#page-6
 *
 * @note
 * Currently the service requires an etag, but spec states unguarded updates should be possible.
 * see {@link https://jira.corp.adobe.com/browse/SYSE-7940}
 * and {@link https://jira.corp.adobe.com/browse/SYSE-5943}
 *
 * @param {AdobeHTTPService}                        svc       HTTPService to use
 * @param {AdobeAsset}                              asset     Asset to update, must be composite
 * @param {JSONPatchDocument | string}              data      Data to use as PATCH body
 * @param {string}                                  [etag]    ETag of the embedded metadata resource to be updated
 * @param {Record<string, string>}                  additionalHeaders   Additional headers to attach to HTTP Requests
 *
 * @returns {AdobePromise<AdobeResponse<'void'>>}
 */
export function patchEmbeddedMetadata(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    data: JSONPatchDocument | string,
    etag?: string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<AdobeResponse<'void'>> {
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['data', data, ['string', 'object', 'object[]']],
        ['etag', etag, 'string', true],
    );
    assertLinksContain(asset.links, [LinkRelation.EMBEDDED_METADATA]);

    const href = getLinkHref(asset.links, LinkRelation.EMBEDDED_METADATA);
    return svc.invoke(
        HTTPMethods.PATCH,
        href,
        Object.assign(additionalHeaders, {
            [HeaderKeys.CONTENT_TYPE]: JSONPatchMediaType,
            [HeaderKeys.IF_MATCH]: etag as string /** pruned if undefined in HTTPService */,
        }),
        typeof data === 'string' ? data : JSON.stringify(data),
        {
            isStatusValid: makeStatusValidator(),
            retryOptions: {
                pollCodes: [202],
                pollHeader: 'location',
                pollMethod: 'GET',
            },
        },
    );
}
