/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeDCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import fs from 'fs';
import os from 'os';
import path from 'path';

export const DEFAULT_LOCALSTORAGE_MAX_SIZE = 5 * 1024 * 1024;
export const KEY_FOR_EMPTY_STRING = '---.EMPTY_STRING.---';

const dbg = newDebug('dcx:assets:nodeLocalStorage');

/**
 * A `MetaKey` is meant to store the unescaped/unencoded key
 * and some metadata associated with the key (i.e. creation timestamp and amount of storage consumed).
 */
class MetaKey {
    key: string;
    index: number;
    size: number;
    createDate: number;
    constructor(
        keyInput: string,
        indexInput: number,
        sizeInput: number | undefined,
        createDatetimeInput: number | undefined,
    ) {
        this.key = keyInput;
        this.index = indexInput;
        this.size = sizeInput ? sizeInput : 0;
        this.createDate = createDatetimeInput ? createDatetimeInput : new Date().getTime();
    }
}

export class LocalStorage {
    #keys: string[];
    #location: string;
    #bytesInUse: number;
    #metaKeyMap: Record<string, MetaKey>;
    #quota: number;

    constructor(_location = os.tmpdir() + '/dcxJs/', quota = DEFAULT_LOCALSTORAGE_MAX_SIZE) {
        this.#location = _location;
        this.#quota = quota;
        this.#location = path.resolve(this.#location);
        this.#bytesInUse = 0;
        this.#keys = [];
        this.#metaKeyMap = {};
        this._init();
        return this;
    }

    private _escapeKey(key) {
        return key === '' ? KEY_FOR_EMPTY_STRING : `${key}`;
    }

    private _encodeURIComponent(key) {
        return encodeURIComponent(key)
            .replace(/[!'()]/g, escape)
            .replace(/\*/g, '%2A');
    }

    private _getStat(key) {
        key = this._escapeKey(key);
        const filename = path.join(this.#location, encodeURIComponent(key));
        try {
            return fs.statSync(filename);
        } catch (error) {
            return null;
        }
    }

    private _init() {
        try {
            this.#bytesInUse = 0;
            const keys = fs.readdirSync(this.#location);
            for (let index = 0; index < keys.length; index++) {
                const k = this.#keys[index];
                const _decodedKey = decodeURIComponent(k);
                this.#keys.push(_decodedKey);
                const fileStat = this._getStat(k);
                this.#bytesInUse += fileStat?.size || 0;
                this.#metaKeyMap[_decodedKey] = new MetaKey(k, index, fileStat?.size, fileStat?.ctimeMs);
            }
        } catch (error) {
            // If it errors, that might mean it didn't exist, so try to create it
            if (!(error instanceof Error) || ('code' in error && error.code !== 'ENOENT')) {
                const message = error instanceof Error ? error.message : 'Unknown Error';
                dbg('failed to instantiate node localstorage because ', message);
                return;
            }
            try {
                fs.mkdirSync(this.#location, {
                    recursive: true,
                });
            } catch (error2) {
                if (!(error2 instanceof Error) || ('code' in error2 && error2.code !== 'EEXIST')) {
                    const message = error instanceof Error ? error.message : 'Unknown Error';
                    dbg('failed to create directory for node localstorage because ', message);
                }
            }
        }
    }

    setItem(key, value?) {
        key = this._escapeKey(key);
        const encodedKey = this._encodeURIComponent(key);
        const filename = path.join(this.#location, encodedKey);
        const valueString = `${value}`;
        const valueStringLength = valueString.length;
        const oldLength = this.#metaKeyMap[key]?.size || 0;
        if (this.#bytesInUse - oldLength + valueStringLength > this.#quota) {
            throw new AdobeDCXError(AdobeDCXError.EXCEEDS_QUOTA, 'EXCEEDS THE QUOTA OF LOCAL STORAGE.');
        }
        fs.writeFileSync(filename, valueString);
        this.#metaKeyMap[key] = new MetaKey(encodedKey, this.#keys.push(key) - 1, valueStringLength, Date.now());
        this.#bytesInUse += valueStringLength - oldLength;
    }

    getItem(key, expirationPeriodMS) {
        key = this._escapeKey(key);
        const encodedKey = this._encodeURIComponent(key);
        const metaKey = this.#metaKeyMap[key] as MetaKey;
        const createDate = new Date(metaKey?.createDate || 0);
        if (!metaKey || Date.now() - createDate.getTime() > expirationPeriodMS) {
            return null;
        }
        const filename = path.join(this.#location, encodedKey);
        return fs.readFileSync(filename, 'utf8');
    }

    private _rm(target: string) {
        if (typeof fs.rmSync === 'function') {
            fs.rmSync(target, { force: true, recursive: true });
        } else {
            fs.rmdirSync(target, { recursive: true });
        }
    }

    removeItem(key) {
        key = this._escapeKey(key);
        const meta = this.#metaKeyMap[key];
        if (!!meta) {
            delete this.#metaKeyMap[key];
            this.#bytesInUse -= meta.size;
            this.#keys.splice(meta.index, 1);
            for (const metaKey in this.#metaKeyMap) {
                if (this.#metaKeyMap[metaKey].index > meta.index) {
                    meta.index -= 1;
                }
            }
            this._rm(path.join(this.#location, meta.key));
        }
    }

    clear() {
        this._rm(this.#location);
        fs.mkdirSync(this.#location, { recursive: true });
        this.#metaKeyMap = {};
        this.#keys = [];
        this.#bytesInUse = 0;
    }
}
