/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPCopyResourceDesignator,
    ACPProblemResponse,
    ACPRepoMetadataResource,
    AdobeAsset,
    AdobeDCXComposite,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeResponse,
    AssetWithRepoAndPathOrId,
    ComponentResourceDesignator,
    CopyResourceDesignator,
    JSONPatchDocument,
    LinkRelationKey,
    RequireAtLeastOne,
    ResourceDesignator,
} from '@dcx/common-types';
import DCXError, { _handleErrorResponsePayload, isAdobeDCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { getLinkHrefTemplated, isArray, pruneUndefined, validateParam, validateParams } from '@dcx/util';
import { AdobeOpsContext, OptionalContext, getOpsHref } from './LeafContext';
import { ServiceConfig, getService } from './Service';
import { RepoResponseResult } from './common';
import { AssetTypes } from './enum/asset_types';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { DirectoryMediaType, OperationDocumentMediaType } from './enum/media_types';
import { Properties } from './enum/properties';
import { deserializeAsset } from './util/serialization';
import {
    assertEachObjectContainsOneOf,
    assertObjectContains,
    assertObjectContainsOneOf,
    doesAssetContainLinks,
    makeStatusValidator,
} from './util/validation';

const dbg = newDebug('dcx:assets:operations');
const dbgb = newDebug('dcx:assets:operations:builder');

export type AdobeOperation = 'move' | 'copy' | 'discard' | 'restore' | 'package' | 'delete' | 'copy_resources';

interface OperationSource {
    assetId?: string;
    path?: string;
    'repo:assetId'?: string;
    'repo:path'?: string;
    'if-none-match'?: string;
    'if-match'?: string;
    'repo:baseAssetId'?: string;
    baseAssetId?: string;
    version?: string;
    etag?: string;
    format?: string;
    'dc:format'?: string;
    'repo:version'?: string;
}

type AdobeOperationSource = OperationSource &
    RequireAtLeastOne<{ repositoryId?: string; 'repo:repositoryId'?: string }, 'repositoryId' | 'repo:repositoryId'>;

export type IDBasedOperationSource = RequireAtLeastOne<AdobeOperationSource, 'assetId' | 'repo:assetId'>;

/**
 * Path or ID based source for Operation API.
 */
export type PathOrIdAssetDesignator =
    | RequireAtLeastOne<AdobeOperationSource, 'path' | 'repo:path' | 'assetId' | 'repo:assetId'>
    | AdobeDCXComposite;

export interface AdobeOperationDocument {
    op: AdobeOperation;
    target: AdobeOperationSource | AdobeOperationSource[];
    [key: string]: unknown;
}

export interface AdobeMoveCopyDocument extends AdobeOperationDocument {
    op: 'move' | 'copy';
    source: AdobeOperationSource | AdobeOperationSource[];
    intermediates?: boolean;
    asset?: AdobeAsset | ACPRepoMetadataResource;
}

export interface AdobeDiscardRestoreDocument extends AdobeOperationDocument {
    op: 'discard' | 'delete' | 'restore';
    recursive?: boolean;
    name?: string;
}

export interface AdobePackageDocument extends AdobeOperationDocument {
    op: 'package';
    source?: AdobeOperationSource | AdobeOperationSource[];
}

export interface AdobeCopyResourcesDocument extends Omit<AdobeOperationDocument, 'target'> {
    op: 'copy_resources';
    source: PathOrIdAssetDesignator;
    target: PathOrIdAssetDesignator;
    resources: CopyResourceDesignator[];
}

/**
 * @private Limit of resources that can be copied in a single COPY_RESOURCES operation.
 * {@see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/responses/service_responses.html#limit-resource-count | Limit Resource Count Problem Type}}
 * {@see  {@link https://wiki.corp.adobe.com/pages/viewpage.action?spaceKey=dcxteam&title=Composite+Service+Resource+Copy+API+Spec#CompositeServiceResourceCopyAPISpec-HTTPResponseStatusCodes | CoS Resources Copy API Spec}}
 **/
export const COPY_RESOURCES_RESOURCE_LIMIT = 30;

/**
 * Used to narrow Any Operation Document to AdobeCopyResourcesDocument
 * @param doc Any Operation Document
 * @returns true if doc is AdobeCopyResourcesDocument
 */
export const isAdobeCopyResourcesDocument = (doc: AnyOperationDocument): doc is AdobeCopyResourcesDocument =>
    doc.op === 'copy_resources';

export type AnyOperationDocument =
    | AdobeMoveCopyDocument
    | AdobeCopyResourcesDocument
    | AdobeDiscardRestoreDocument
    | AdobePackageDocument;

interface OperationDocTypeMap {
    move: AdobeMoveCopyDocument;
    copy: AdobeMoveCopyDocument;
    // eslint-disable-next-line camelcase
    copy_resources: AdobeCopyResourcesDocument;
    discard: AdobeDiscardRestoreDocument;
    delete: AdobeDiscardRestoreDocument;
    restore: AdobeDiscardRestoreDocument;
    package: AdobePackageDocument;
}

interface BuildOpDocOptions {
    asset?: ACPRepoMetadataResource | AdobeAsset;
    name?: string;
    recursive?: boolean;
    overwriteExisting?: boolean;
    createIntermediates?: boolean;
}

type OperationResultSource = Omit<OperationSource, 'assetId' | 'baseAssetId' | 'path' | 'etag' | 'format'>;
interface OperationResult {
    op: AdobeOperation;
    source?: OperationResultSource | OperationResultSource[];
    target?: OperationResultSource;
    asset?: ACPRepoMetadataResource;
    intermediates?: boolean;
    error?: ACPProblemResponse;
    resources?: ACPCopyResourceDesignator[];
}
export type AdobeOperationResult = Omit<OperationResult, 'error'> & {
    error?: AdobeDCXError<ACPProblemResponse>;
    _additionalData?: ACPProblemResponse;
};

export type CopyResourcesOperationResult = {
    source: OperationResultSource;
    target: OperationResultSource;
    resources: ACPCopyResourceDesignator[];
    asset?: AdobeAsset;
};

/**
 * Trigger server copy resources, and won't fall back to client copy on error 501
 * https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html
 *
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param svc               The http service
 * @param srcAsset          The source asset
 * @param destAsset         The destination asset
 * @param resources         List of resources designators
 * @param intermediates     Whether to create intermediate directories if missing.
 * @param manifestPatch     A JSON Patch Document to be applied to the target asset's manifest.
 * @param additionalHeaders Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>}
 */

export function copyResources(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    srcAsset: PathOrIdAssetDesignator,
    targetAsset: PathOrIdAssetDesignator,
    resources: CopyResourceDesignator[],
    intermediates?: boolean,
    manifestPatch?: JSONPatchDocument,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError> {
    dbg('copyResource()');
    return _copyResources(svc, srcAsset, targetAsset, resources, intermediates, manifestPatch, additionalHeaders)
        .then((res) => {
            return res.response;
        })
        .then(_deserializeSuccessResult);
}

/**
 * Copies an asset for the src to the destination
 * User cannot copy component resources using this method
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param svc                   The http service
 * @param srcAsset              The source asset
 * @param destAsset             The destination asset
 * @param createIntermediates   Should intermediates be created?
 * @param overwriteExisting     Should an existing asset in dest be overwritten
 * @param additionalHeaders     Additional headers to attach to HTTP requests
 * @param manifestPatch         A JSON Patch Document to be applied to the target asset's manifest.
 */
export function copyAsset(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    srcAsset: PathOrIdAssetDesignator,
    destAsset: PathOrIdAssetDesignator,
    createIntermediates: boolean,
    overwriteExisting?: boolean,
    additionalHeaders?: Record<string, string>,
    manifestPatch?: JSONPatchDocument,
): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError> {
    dbg('copyAsset()');

    validateParams(
        ['svc', svc, 'object'],
        ['srcAsset', srcAsset, 'object'],
        ['destAsset', destAsset, 'object'],
        ['createIntermediates', createIntermediates, 'boolean'],
        ['overwriteExisting', overwriteExisting, 'boolean', true],
        ['manifestPatch', manifestPatch, ['object', 'string'], true],
    );

    assertObjectContainsOneOf(srcAsset, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');
    assertObjectContainsOneOf(destAsset, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');
    const doc = _buildOperationDoc(
        'copy',
        destAsset as AdobeOperationSource,
        srcAsset as AdobeOperationSource,
        {
            overwriteExisting,
            createIntermediates,
        },
        {
            'repo:manifestPatch': JSON.stringify(manifestPatch),
        },
    );
    const service = getService(svc);
    return getOpsHref
        .call(this, svc)
        .then((opsEndpoint) => doOperation(service, opsEndpoint, doc, additionalHeaders))
        .then(_appendRepositoryId.bind(undefined, destAsset))
        .then(_deserializeAdobeAssetResult);
}

/**
 * Narrowing function for CopyResourceDesignator.
 */
const hasSourceAndTarget = (
    resourceDesignator: CopyResourceDesignator,
): resourceDesignator is {
    target: Required<ComponentResourceDesignator>;
    source: Required<ComponentResourceDesignator>;
} => {
    return resourceDesignator.hasOwnProperty('source') && resourceDesignator.hasOwnProperty('target');
};

/**
 * Normalized ResourceDesignator format.
 */
type NormalizedResourceDesignator = {
    reltype: LinkRelationKey;
    // eslint-disable-next-line camelcase
    component_id: string;
    revision: string;
};

/**
 * Normalize a CopyResource Designator, the returned format will always include both source and target, in the normalizedResource designator
 */
export const normalizeCopyResourcesDesignator = (
    resourceDesignator: CopyResourceDesignator,
): { source: NormalizedResourceDesignator; target: NormalizedResourceDesignator } => {
    if (hasSourceAndTarget(resourceDesignator)) {
        return {
            source: normalizeResourceDesignator(resourceDesignator.source),
            target: normalizeResourceDesignator(resourceDesignator.target),
        };
    }
    const rd = normalizeResourceDesignator(resourceDesignator as ResourceDesignator);
    return { source: rd, target: rd };
};

/**
 * Normalize a ResourceDesignator to the object format, which always includes a reltype member.
 */
const normalizeResourceDesignator = (resourceDesignator: ResourceDesignator): NormalizedResourceDesignator => {
    return typeof resourceDesignator === 'string'
        ? { reltype: resourceDesignator, component_id: '', revision: '' }
        : (resourceDesignator as NormalizedResourceDesignator);
};

/**
 * Moves an asset from src to dest
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        The http service
 * @param {PathOrIdAssetDesignator} srcAsset       The source asset
 * @param {PathOrIdAssetDesignator} destAsset      The destination asset
 * @param {boolean} createIntermediates                 Should intermediates be created?
 * @param {boolean} overwriteExisting                   Should an existing asset in dest be overwritten
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
 */
export function moveAsset(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    srcAsset: PathOrIdAssetDesignator,
    destAsset: PathOrIdAssetDesignator,
    createIntermediates: boolean,
    overwriteExisting?: boolean,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError> {
    dbg('moveAsset()');

    validateParams(
        ['svc', svc, 'object'],
        ['srcAsset', srcAsset, 'object'],
        ['destAsset', destAsset, 'object'],
        ['createIntermediates', createIntermediates, 'boolean'],
        ['overwriteExisting', overwriteExisting, 'boolean', true],
    );

    assertObjectContainsOneOf(srcAsset, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');
    assertObjectContainsOneOf(destAsset, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');

    const doc = _buildOperationDoc('move', destAsset as AdobeOperationSource, srcAsset as AdobeOperationSource, {
        createIntermediates,
        overwriteExisting,
    });
    const service = getService(svc);
    return getOpsHref
        .call(this, svc)
        .then((opsEndpoint) => doOperation(service, opsEndpoint, doc, additionalHeaders))
        .then(_appendRepositoryId.bind(undefined, destAsset))
        .then(_deserializeAdobeAssetResult);
}

/**
 * Discards an asset
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        The http service
 * @param {PathOrIdAssetDesignator} asset          The asset to be discarded
 * @param {string} etag                                 The etag of the asset to be discarded
 * @param {boolean} recursive                           If asset has children, discard recursively?
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError>}
 */
export function discardAsset(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    asset: PathOrIdAssetDesignator,
    etag?: string,
    recursive?: boolean,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError> {
    dbg('discardAsset()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['etag', etag, 'string', true],
        ['recursive', recursive, 'boolean', true],
    );

    const service = getService(svc);

    // TODO: allow using discard link - currently returns 501 not implemented
    // if (doesAssetContainLinks(asset, [LinkRelation.DISCARD])) {
    //     return service.invoke('POST', getLinkHrefTemplated(asset, LinkRelation.DISCARD, {}));
    // }

    assertObjectContainsOneOf(asset, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');
    const doc = _buildOperationDoc('discard', { ...asset, etag } as AdobeOperationSource, undefined, {
        recursive,
    });
    return getOpsHref
        .call(this, svc)
        .then((opsEndpoint) => doOperation(service, opsEndpoint, doc, additionalHeaders))
        .then(_deserializeSuccessResult);
}

/**
 * Delete an asset.
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        HTTPService to use.
 * @param {PathOrIdAssetDesignator} asset          Asset to delete.
 * @param {string} etag                                 Optional etag. If not provided will use "*", an unconditional delete.
 * @param {boolean} recursive                           Whether to delete the asset recursively. Required for directory assets.
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError>}
 */
export function deleteAsset(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    asset: PathOrIdAssetDesignator,
    etag = '*',
    recursive?: boolean,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError> {
    dbg('deleteAsset()');

    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['etag', etag, 'string', true],
        ['recursive', recursive, 'boolean', true],
    );

    const service = getService(svc);

    if (asset.format === DirectoryMediaType && recursive == null) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Recursive flag is required for directory assets.');
    }

    if (!recursive && doesAssetContainLinks(asset, [LinkRelation.REPO_METADATA])) {
        const href = getLinkHrefTemplated(asset, LinkRelation.REPO_METADATA, {});
        return service
            .invoke('DELETE', href, { [HeaderKeys.IF_MATCH]: etag }, undefined, {
                isStatusValid: makeStatusValidator(),
            })
            .then(_deserializeSuccessResult);
    }

    assertObjectContainsOneOf(asset, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');

    const opSource = Object.create(Object.getPrototypeOf(asset), Object.getOwnPropertyDescriptors(asset));
    opSource['etag'] = etag;
    const doc = _buildOperationDoc('delete', opSource as AdobeOperationSource, undefined, {
        recursive,
    });

    return getOpsHref
        .call(this, svc)
        .then((opsEndpoint) => doOperation(service, opsEndpoint, doc, additionalHeaders))
        .then(_deserializeSuccessResult);
}

/**
 * Restore a discarded asset
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        HTTPService to use.
 * @param {IDBasedOperationSource} asset                The asset to restore
 * @param {Record<string, string>} additionalHeaders    Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
 */
export function restoreAsset(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    asset: IDBasedOperationSource,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError> {
    dbg('restoreAsset()');

    validateParams(['svc', svc, 'object'], ['asset', asset, 'object']);

    const service = getService(svc);

    // TODO: allow using restore link - currently returns 501 not implemented
    // if (doesAssetContainLinks(asset, [LinkRelation.RESTORE])) {
    //     return service.invoke('POST', getLinkHrefTemplated(asset, LinkRelation.RESTORE, {}));
    // }

    // restore can only be done by ID
    assertObjectContainsOneOf(asset, ['assetId', 'repo:assetId'], 'string');
    const doc = _buildOperationDoc('restore', asset as AdobeOperationSource);
    return getOpsHref
        .call(this, svc)
        .then((opsEndpoint) => doOperation(service, opsEndpoint, doc, additionalHeaders))
        .then(_appendRepositoryId.bind(undefined, asset))
        .then(_deserializeAdobeAssetResult);
}

/**
 * Package an asset (zip) and store at destination
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc                                    HTTPService to use.
 * @param {PathOrIdAssetDesignator | PathOrIdAssetDesignator[]} sources   Source(s) to package
 * @param {PathOrIdAssetDesignator} destination                                Destination for zipped package
 * @param {boolean} createIntermediates                                             Should intermediates be created?
 * @param {boolean} overwriteExisting                                               Should an existing asset in dest be overwritten
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError>}
 */
export function packageAssets(
    this: OptionalContext<AdobeOpsContext>,
    svc: AdobeHTTPService | ServiceConfig,
    sources: PathOrIdAssetDesignator | PathOrIdAssetDesignator[],
    destination: PathOrIdAssetDesignator,
    createIntermediates: boolean,
    overwriteExisting: boolean,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError> {
    dbg('packageAssets()');

    validateParams(['svc', svc, 'object'], ['destination', destination, 'object']);
    validateParam('sources', sources, ['object', 'object[]']);

    sources = isArray(sources) ? sources : [sources];

    assertEachObjectContainsOneOf(sources, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');
    assertObjectContainsOneOf(destination, ['repo:path', 'path', 'assetId', 'repo:assetId'], 'string');
    const doc = _buildOperationDoc('package', destination as AdobeOperationSource, sources as AdobeOperationSource[], {
        createIntermediates,
        overwriteExisting,
    });
    const service = getService(svc);

    return getOpsHref
        .call(this, svc)
        .then((opsEndpoint) => doOperation(service, opsEndpoint, doc, additionalHeaders))
        .then(_appendRepositoryId.bind(undefined, destination))
        .then(_deserializeSuccessResult);
}

/**
 * Execute a POST on the ops href provided using the operation document as body.
 * Also split the response into an array of results, creating a DCX error for each failed operation.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc                                        HTTPService to use.
 * @param {string} opsEndpoint                                                          Operations endpoint
 * @param {AnyOperationDocument | AnyOperationDocument[] | string} operationDocument    Operations document
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 */
export function doBatchOperation(
    svc: AdobeHTTPService,
    opsEndpoint: string,
    operationDocument: AnyOperationDocument | AnyOperationDocument[] | string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<AdobeOperationResult[], 'json'>, AdobeDCXError> {
    return doOperation(svc, opsEndpoint, operationDocument, additionalHeaders).then(_parseBatchErrors);
}

/**
 * @private
 *
 * @param response
 */
function _parseBatchErrors(response: AdobeResponse<'json'>): RepoResponseResult<AdobeOperationResult[], 'json'> {
    // batch may not actually be called with an array of operations,
    // so the results may just be an object.
    // still coerce it into an array of results.
    const resp = !isArray<OperationResult>(response.response) ? [response.response] : response.response;

    const result = resp.map(_translateResultError) as AdobeOperationResult[];

    return {
        result,
        response,
    };
}

/**
 * Turns an operation error into a DCXError.
 * Uses a copy of the result object to avoid modifying the original response object.
 *
 * @private
 * @param result
 */
function _translateResultError(result: OperationResult): AdobeOperationResult {
    if (!result.error) {
        return result as unknown as AdobeOperationResult;
    }

    const parsed = { ...result } as unknown as AdobeOperationResult;
    const isValid = makeStatusValidator()(result.error.status) as AdobeDCXError;
    parsed.error = isAdobeDCXError<ACPProblemResponse>(isValid)
        ? isValid
        : new DCXError(DCXError.UNEXPECTED, 'Unexpected response');

    // set additionalData to the original ACP problem
    parsed._additionalData = result.error;

    // use original title as error message, may not be what we want
    (parsed.error as any)._message = result.error.title;
    return parsed;
}

/**
 * Execute a POST on the ops href provided using the operation document as body.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc                                        HTTPService to use.
 * @param {string} opsEndpoint                                                          Operations endpoint
 * @param {AnyOperationDocument | AnyOperationDocument[] | string} operationDocument    Operations document
 * @param {Record<string, string>} additionalHeaders                                          Additional headers to include in the POST
 *
 * @returns {AdobePromise<AdobeResponse<'json'>, AdobeDCXError>}
 */
export function doOperation(
    svc: AdobeHTTPService,
    opsEndpoint: string,
    operationDocument: AnyOperationDocument | AnyOperationDocument[] | string,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeResponse<'json'>, AdobeDCXError> {
    dbg('doOperation()');

    validateParams(
        ['svc', svc, 'object'],
        ['opsEndpoint', opsEndpoint, 'string'],
        ['operationDocument', operationDocument, ['string', 'object']],
        ['additionalHeaders', additionalHeaders, 'object', true],
    );
    return _doOperation(svc, opsEndpoint, operationDocument, additionalHeaders).then((res) => {
        res.response = res.response || {};

        // response body on 200 code may have "error" property describing the error
        if (res.response.error) {
            throw new DCXError(
                DCXError.UNEXPECTED_RESPONSE,
                res.response.type || 'Operation failed.',
                new Error(res.response.title),
                res,
            );
        }

        return res;
    });
}

type OperationSourceType = 'pathAndBaseAssetId' | 'path' | 'id' | 'idBasedHref' | 'pathBasedHref';
/**
 * OperationDocumentBuilder
 */
export class OperationDocumentBuilder {
    public readonly opBatchLimit = 100;

    private _docs: AnyOperationDocument[] = [];

    // identifies the source and target types for the current document
    // automatically set from the first operation that is added to the batch
    // if subsequent operations are added that use a different type, it will throw
    private _batchSourceType: OperationSourceType | undefined;
    private _batchTargetType: OperationSourceType | undefined;

    getDocumentEntry<T extends AdobeOperation = AdobeOperation>(id: number): OperationDocTypeMap[T] {
        return this._docs[id] as OperationDocTypeMap[T];
    }

    getDocument(): AnyOperationDocument[] {
        return this._docs;
    }

    get entryCount(): number {
        return this._docs.length;
    }

    copyResources(
        srcAsset: PathOrIdAssetDesignator,
        targetAsset: PathOrIdAssetDesignator,
        resources: CopyResourceDesignator[],
        intermediates?: boolean,
        manifestPatch?: JSONPatchDocument,
        additionalHeaders?: Record<string, string>,
    ): OperationDocumentBuilder {
        dbgb('copyResource()');
        this._assertUnderLimit();

        this._checkSourceType(srcAsset);
        this._checkTargetType(targetAsset);
        // when performing batch operations we cannot submit multiple documents that use the same
        // source and target. Instead we must combine them into a single document.
        const existingDoc = this._docs.find(_findExistingCopyResourcesDocument(srcAsset, targetAsset));

        if (existingDoc) {
            // Extend the existing doc if it exists
            existingDoc.resources = existingDoc.resources.concat(resources);

            if (manifestPatch) {
                existingDoc['repo:manifestPatch'] = JSON.stringify(manifestPatch);
            }

            if (intermediates !== undefined) {
                existingDoc.intermediates = intermediates;
            }

            return this;
        }

        const doc = _buildOperationDoc(
            'copy_resources',
            targetAsset as AdobeOperationSource,
            srcAsset as AdobeOperationSource,
            { createIntermediates: intermediates },
            {
                resources,
                'repo:manifestPatch': JSON.stringify(manifestPatch),
                additionalHeaders,
            },
        );
        this._docs.push(doc);

        return this;
    }

    copy(
        srcAsset: PathOrIdAssetDesignator,
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting?: boolean,
        addlFields?: Record<string, unknown>,
    ): OperationDocumentBuilder {
        dbgb('copy()');

        this._assertUnderLimit();

        this._checkSourceType(srcAsset);
        this._checkTargetType(destAsset);

        const doc = _buildOperationDoc(
            'copy',
            destAsset as AdobeOperationSource,
            srcAsset as AdobeOperationSource,
            { createIntermediates, overwriteExisting },
            addlFields,
        );
        this._docs.push(doc);

        return this;
    }

    move(
        srcAsset: PathOrIdAssetDesignator,
        destAsset: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting?: boolean,
        addlFields?: Record<string, unknown>,
    ): OperationDocumentBuilder {
        dbgb('move()');

        this._assertUnderLimit();

        this._checkSourceType(srcAsset);
        this._checkTargetType(destAsset);

        const doc = _buildOperationDoc(
            'move',
            destAsset as AdobeOperationSource,
            srcAsset as AdobeOperationSource,
            { createIntermediates, overwriteExisting },
            addlFields,
        );
        this._docs.push(doc);

        return this;
    }

    package(
        sources: PathOrIdAssetDesignator | PathOrIdAssetDesignator[],
        destination: PathOrIdAssetDesignator,
        createIntermediates: boolean,
        overwriteExisting?: boolean,
        addlFields?: Record<string, unknown>,
    ): OperationDocumentBuilder {
        dbgb('package()');

        this._assertUnderLimit();

        sources = isArray(sources) ? sources : [sources];

        sources.map((s) => {
            this._checkSourceType(s);
        });

        this._checkTargetType(destination);

        const doc = _buildOperationDoc(
            'package',
            destination as AdobeOperationSource,
            sources as AdobeOperationSource | AdobeOperationSource[],
            { createIntermediates, overwriteExisting },
            addlFields,
        );
        this._docs.push(doc);

        return this;
    }

    discard(
        target: PathOrIdAssetDesignator,
        recursive: boolean,
        addlFields?: Record<string, unknown>,
    ): OperationDocumentBuilder {
        dbgb('discard()');

        this._assertUnderLimit();

        this._checkTargetType(target);

        const doc = _buildOperationDoc('discard', target as AdobeOperationSource, undefined, { recursive }, addlFields);
        this._docs.push(doc);
        return this;
    }

    restore(target: IDBasedOperationSource, addlFields?: Record<string, unknown>): OperationDocumentBuilder {
        dbgb('restore()');

        this._assertUnderLimit();

        // assert href, assetId or repo:assetId exist since restore can only be done with those identifiers
        assertObjectContainsOneOf(target, ['assetId', 'repo:assetId'], 'string');
        this._checkTargetType(target);

        const doc = _buildOperationDoc('restore', target as AdobeOperationSource, undefined, undefined, addlFields);
        this._docs.push(doc);
        return this;
    }

    delete(
        target: PathOrIdAssetDesignator,
        recursive: boolean,
        addlFields?: Record<string, unknown>,
    ): OperationDocumentBuilder {
        dbgb('delete()');

        this._assertUnderLimit();

        this._checkTargetType(target);

        const doc = _buildOperationDoc('delete', target as AdobeOperationSource, undefined, { recursive }, addlFields);
        this._docs.push(doc);
        return this;
    }

    private _assertUnderLimit() {
        if (this._docs.length >= this.opBatchLimit) {
            throw new DCXError(
                DCXError.INVALID_STATE,
                `Exceeds limit of ${this.opBatchLimit} operations in a single batch.`,
            );
        }
    }

    private _checkTargetType(target: PathOrIdAssetDesignator | IDBasedOperationSource | AssetWithRepoAndPathOrId) {
        this._checkSourceOrTargetType(target, 'Target');
    }

    private _checkSourceType(source: PathOrIdAssetDesignator | IDBasedOperationSource | AssetWithRepoAndPathOrId) {
        this._checkSourceOrTargetType(source, 'Source');
    }

    private _checkSourceOrTargetType(
        sourceOrTarget: PathOrIdAssetDesignator | IDBasedOperationSource | AssetWithRepoAndPathOrId,
        identifier: 'Source' | 'Target',
    ) {
        // only allow a single type to be defined in each source or target.
        const existing = [
            !!sourceOrTarget.assetId || !!sourceOrTarget['repo:assetId'],
            !!sourceOrTarget.path || !!sourceOrTarget['repo:path'],
        ];
        if (existing.filter((e) => e).length === 0) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                `${identifier} identifier is underspecified. Exactly one of [href, repo:path, repo:assetId] required.`,
            );
        }

        const sourceOrTargetType = this._getSourceType(sourceOrTarget);
        const expectedType = identifier === 'Source' ? this._batchSourceType : this._batchTargetType;

        // first time it's always a valid choice, as it defines the restriction for later entries.
        if (!expectedType) {
            if (identifier === 'Source') {
                this._batchSourceType = sourceOrTargetType;
            } else {
                this._batchTargetType = sourceOrTargetType;
            }
            return;
        }

        // but subsequently, all source types must match
        if (sourceOrTargetType !== expectedType) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                `Operation ${identifier.toLowerCase()} types must all be the same type. ` +
                    `Expected ${expectedType}, encountered ${sourceOrTargetType}.`,
            );
        }
    }

    private _getSourceType(
        source: PathOrIdAssetDesignator | IDBasedOperationSource | AssetWithRepoAndPathOrId,
    ): OperationSourceType | undefined {
        if (source.assetId || source['repo:assetId']) {
            return 'id';
        } else if (source.path || source['repo:path']) {
            if (source.baseAssetId || source['repo:baseAssetId']) {
                // has base asset ID
                return 'pathAndBaseAssetId';
            }
            return 'path';
        }
    }
}

export function newOperationDocBuilder(): OperationDocumentBuilder {
    return new OperationDocumentBuilder();
}

/**
 * ===================
 * Internal methods
 * ===================
 */

/**
 * This method is used to determine if the asset designator provided by a user matches an existing asset designator
 * @param docAssetDesignator Asset designator from an operation document (source or target)
 * @param assetDesignator Asset designator provided by a user
 * @returns true if the asset designator from the operation document matches the asset designator provided by the user
 */
function _isMatchingAsset(
    docAssetDesignator: PathOrIdAssetDesignator,
    assetDesignator: PathOrIdAssetDesignator,
): boolean {
    return (
        docAssetDesignator[Properties.REPO_ASSET_ID] ===
            (assetDesignator[Properties.REPO_ASSET_ID] ?? assetDesignator.assetId) &&
        docAssetDesignator[Properties.REPO_REPOSITORY_ID] ===
            (assetDesignator[Properties.REPO_REPOSITORY_ID] ?? assetDesignator.repositoryId)
    );
}

/**
 * Creates a callback to be provided to `Array.find` which will narrow the type of the document to an AdobeCopyResourcesDocument
 * @param sourceAsset
 * @param targetAsset
 * @returns callback to be provided to `this._docs.find`
 */
function _findExistingCopyResourcesDocument(
    sourceAsset: PathOrIdAssetDesignator,
    targetAsset: PathOrIdAssetDesignator,
) {
    return function _generatedFindCallback(doc: AnyOperationDocument): doc is AdobeCopyResourcesDocument {
        return (
            doc.op === 'copy_resources' &&
            _isMatchingAsset(doc.source, sourceAsset) &&
            _isMatchingAsset(doc.target, targetAsset)
        );
    };
}

function _doOperation(
    svc: AdobeHTTPService,
    opsEndpoint: string,
    opDocument: AnyOperationDocument | AnyOperationDocument[] | string,
    additionalHeaders: Record<string, string> = {},
): AdobePromise<AdobeResponse<'json'>, AdobeDCXError> {
    return svc.invoke(
        HTTPMethods.POST,
        opsEndpoint,
        Object.assign({ [HeaderKeys.CONTENT_TYPE]: OperationDocumentMediaType }, additionalHeaders),
        typeof opDocument === 'string' ? opDocument : JSON.stringify(opDocument),
        {
            isStatusValid: makeStatusValidator(),
            responseType: 'json',
            retryOptions: {
                pollCodes: [202],
                pollHeader: 'location',
                pollMethod: 'get',
            },
        },
    );
}

function _appendRepositoryId(
    targetWithRepoId: { repositoryId?: string; 'repo:repositoryId'?: string },
    response: AdobeResponse<'json'>,
): AdobeResponse<'json'> {
    response.response.asset = {
        ...(response.response.asset || {}),
        repositoryId: targetWithRepoId.repositoryId || targetWithRepoId['repo:repositoryId'],
    };
    return response;
}

function _deserializeAdobeAssetResult(response: AdobeResponse<'json'>): RepoResponseResult<AdobeAsset, 'json'> {
    return {
        response,
        result: deserializeAsset(response.response.asset),
    };
}

function _deserializeSuccessResult(response: AdobeResponse<'json'>): RepoResponseResult<{ success: boolean }, 'json'> {
    const success = response.statusCode > 199 && response.statusCode < 400;
    return {
        response,
        result: { success },
    };
}

/**
 * Convert incoming AdobeAsset compatible properties to ACP-compatible properties.
 * Copies objects, since they may have additional fields to pass along, but also may
 * be Asset class or sub-class instances.
 *
 * @param sourceOrTarget
 */
function _convertToACPSource(
    type: 'source' | 'target',
    sourceOrTarget: AdobeOperationSource | undefined,
    overwriteExisting?: boolean,
): AdobeOperationSource | undefined {
    dbg('_convertToACPSource()');

    /* istanbul ignore if */
    if (typeof sourceOrTarget !== 'object') {
        return;
    }

    const out: Record<string, string | undefined> = {
        'repo:repositoryId': sourceOrTarget.repositoryId || sourceOrTarget['repo:repositoryId'],
        'repo:path': sourceOrTarget.path || sourceOrTarget['repo:path'],
        'repo:assetId': sourceOrTarget.assetId || sourceOrTarget['repo:assetId'],
        'repo:baseAssetId': sourceOrTarget.baseAssetId || sourceOrTarget['repo:baseAssetId'],
    };

    // remove extra identifiers
    if (typeof out['href'] === 'string') {
        // prefer href over assetId
        delete out['repo:path'];
        delete out['repo:assetId'];
        delete out['repo:baseAssetId'];
    } else if (typeof out['repo:assetId'] === 'string') {
        // prefer assetId over path
        delete out['repo:path'];
        delete out['repo:baseAssetId'];
    }

    if (type === 'target') {
        if (overwriteExisting === true) {
            // copy or move, overwrite

            // If-Match is not supported on Directory
            out[HeaderKeys.IF_MATCH] =
                sourceOrTarget.format !== AssetTypes.Directory && sourceOrTarget['dc:format'] !== AssetTypes.Directory
                    ? sourceOrTarget.etag || '*'
                    : '*';
        } else if (overwriteExisting === false) {
            // copy or move, don't overwrite
            out[HeaderKeys.IF_NONE_MATCH] = '*';
        } else if (
            sourceOrTarget.format !== AssetTypes.Directory &&
            sourceOrTarget['dc:format'] !== AssetTypes.Directory
        ) {
            // copy or move, not directory, use etag if exists
            out[HeaderKeys.IF_MATCH] = sourceOrTarget.etag;
        }
    } else if (sourceOrTarget.format !== AssetTypes.Directory && sourceOrTarget['dc:format'] !== AssetTypes.Directory) {
        // no if-match with directory
        out[HeaderKeys.IF_MATCH] = sourceOrTarget.etag || '*';
    }

    if (sourceOrTarget.version) {
        out['repo:version'] = sourceOrTarget.version;
    }

    if (out['repo:path']) {
        assertObjectContains(out, 'repo:repositoryId', 'string');
    }

    dbg('_cTACPS() out', out);

    return pruneUndefined(out);
}

/**
 * Converts arguments into a single formatted operation document.
 * Does not validate arguments, assumes they were validated by exported methods.
 *
 * @param op
 * @param source
 * @param target
 */
function _buildOperationDoc<T extends AdobeOperation = any>(
    op: T,
    target: AdobeOperationSource,
    source?: AdobeOperationSource | AdobeOperationSource[],
    options: BuildOpDocOptions = {},
    additionalFields: Record<string, unknown> = {},
): OperationDocTypeMap[T] {
    dbg('_buildOperationDoc()');

    const doc = pruneUndefined({
        op,
        target,
        source: isArray(source) ? [] : source ? {} : undefined,
    }) as OperationDocTypeMap[T];

    const { overwriteExisting, createIntermediates, recursive } = options;

    // create source
    if (doc.source) {
        doc.source = isArray(source)
            ? source.map((s) => _convertToACPSource('source', s, overwriteExisting)).filter((e) => e != null)
            : _convertToACPSource('source', source, overwriteExisting);
    }

    // create target
    /* istanbul ignore else */
    if (typeof target === 'object') {
        const t = _convertToACPSource('target', target, overwriteExisting);
        if (t) {
            doc.target = t;
        }
    }
    // only set createIntermediates if using path based target
    if ((doc.target as AdobeOperationSource)['repo:assetId'] == null && createIntermediates != null) {
        doc.intermediates = createIntermediates;
    }

    if (recursive != null) {
        doc.recursive = recursive;
    }

    Object.assign(doc, additionalFields);

    dbg('_OD() doc', doc);

    return doc as OperationDocTypeMap[T];
}

export function _copyResources(
    svc: AdobeHTTPService | ServiceConfig,
    sourceAsset: PathOrIdAssetDesignator,
    targetAsset: PathOrIdAssetDesignator,
    resources: CopyResourceDesignator[],
    intermediates?: boolean,
    manifestPatch?: JSONPatchDocument,
    additionalHeaders?: Record<string, string>,
): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError> {
    validateParams(['resources', resources, 'array', false]);
    const doc = newOperationDocBuilder()
        .copyResources(sourceAsset, targetAsset, resources, intermediates, manifestPatch)
        .getDocument();
    return getOpsHref(svc)
        .then((opsHref) => {
            return doOperation(getService(svc), opsHref, doc, additionalHeaders);
        })
        .then(_handleErrorResponsePayload)
        .then((response) => {
            const { asset, source, target, resources } = response.response[0];
            const result: CopyResourcesOperationResult = {
                source: deserializeAsset(source),
                target: deserializeAsset(target),
                resources,
                asset: asset ? deserializeAsset(asset) : undefined,
            };

            return {
                result,
                response,
            };
        });
}
