/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeAsset,
    AdobeDCXError,
    AdobeHTTPService,
    AdobeResponse,
    AdobeResponseType,
    LinkRelationKey,
    RepoDownloadStreamableReturn,
    SliceableData,
} from '@dcx/common-types';
import { DCXError, ProblemTypes } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { getLinkHrefTemplated, isFunction, isObject, pruneUndefined, validateParams } from '@dcx/util';
import { _doBlockDownload } from './BlockTransfer/BlockDownload';
import { AdobeStreamableContext, OptionalContext } from './LeafContext';
import { HeaderKeys } from './enum/header_keys';
import { HTTPMethods } from './enum/http_methods';
import { LinkRelation } from './enum/link';
import { doLinksContain, makeStatusValidator } from './util/validation';
const dbg = newDebug('dcx:assets:private');

/**
 * A default slice callback used if clients to not provide one and block transfer is required
 * @internal
 * @data    {Buffer | Blob | ArrayBuffer | string}               Bound this scope containing the buffer to be sliced
 */
export function getDefaultSliceCallback(data: SliceableData): (start: number, end: number) => Promise<SliceableData> {
    if (!isFunction(data.slice)) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Data cannot be sliced');
    }
    return async function defaultSliceCallback(start: number, end: number) {
        return data.slice(start, end);
    };
}

/**
 * Fetch a URL that may return a 400 with Response Too Large problem type.
 * If that occurs, get the Location header from the response and fetch that.
 * Redact authorization from that call, as it goes directly to another origin.
 *
 * @this {OptionalContext<AdobeStreamableContext>}
 * @param {AdobeHTTPService} svc
 * @param {string} url
 * @param {AdobeResponseType} [responseType = 'buffer']
 * @param {Record<string, string>} [additionalHeaders = {}]
 * @param {boolean} [skipBlockDownload = false]
 */
export function _getUrlFallbackDirect<T extends AdobeResponseType = 'defaultbuffer'>(
    this: OptionalContext<AdobeStreamableContext>,
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    url: string,
    resource: LinkRelationKey,
    responseType: T = 'defaultbuffer' as T,
    componentId?: string,
    version?: string,
    additionalHeaders: Record<string, string> = {},
    skipBlockDownload = false,
): RepoDownloadStreamableReturn<T> {
    dbg('_getUrlFallbackDirect()');

    let response: AdobeResponse;
    const ctx = typeof this !== 'undefined' ? this : {};
    return AdobePromise.resolve(undefined, ctx as RepoDownloadStreamableReturn<T>)
        .then(() =>
            // DCX-11103 requests that the default for all block transfers is to request transfer acceleration
            // Because the `repo:accelerated` parameter is suggested to be deprecated as part of CA-967,
            // we are opting for setting the priority header on the block upload init request to trigger
            // RAPI to utilize EALinks, which also achieve the same transfer acceleration per CSA-374.
            // Consumers are still able to override the priority value by providing one of their own.
            // This is being done on what is presumed to be the rel/download link to ensure that the
            // presigned URL that is returned in the event of a 400 includes Edge Acceleration
            svc.invoke(HTTPMethods.GET, url, { priority: 'u=1', ...additionalHeaders }, undefined, {
                responseType: responseType as AdobeResponseType,
                isStatusValid: makeStatusValidator([400]),
            }),
        )
        .then<AdobeResponse | { status?: number; type?: string }>((resp) => {
            dbg('_gUFD() status code', resp.statusCode);

            response = resp; // save this for later, in case we don't attempt direct download
            if (resp.statusCode === 400 && resp.xhr) {
                return resp.xhr.getResponseDataAsJSON() as Promise<{ status?: number; type?: string }>;
            }

            return resp;
        })
        .then<boolean>((resp) => {
            // got a 400 on the original request
            // and the body returned had a "status" property equal to 400
            // and the body also contained a "too large" problem code
            const tryDirect =
                isObject(resp) &&
                (response.statusCode === 400 || resp.status === 400) &&
                resp.type === ProblemTypes.RESPONSE_TOO_LARGE;
            dbg('_gUFD() do direct', tryDirect);

            // got 400 and not trying direct, throw the error
            if (!tryDirect && response.statusCode === 400) {
                throw new DCXError(DCXError.UNEXPECTED_RESPONSE, 'Unexpected response', undefined, response);
            }

            return tryDirect;
        })
        .then<AdobeResponse>((tryDirectDownload) => {
            if (!tryDirectDownload) {
                return response;
            }

            if (!('location' in response.headers) || typeof response.headers.location !== 'string') {
                // missing or invalid location, fetch it
                /* istanbul ignore if */
                if (!doLinksContain(asset.links, [LinkRelation.BLOCK_DOWNLOAD])) {
                    throw new DCXError(DCXError.INVALID_DATA, 'Resource too large and missing download link.');
                }
                const resourceDesignator = pruneUndefined({
                    reltype: resource,
                    component_id: componentId,
                    revision: version,
                });
                const bdlHref = getLinkHrefTemplated(asset.links, LinkRelation.BLOCK_DOWNLOAD, {
                    resource: resource ? JSON.stringify(resourceDesignator) : undefined,
                });

                if (skipBlockDownload) {
                    return {
                        statusCode: 200,
                        headers: pruneUndefined({
                            [HeaderKeys.CONTENT_TYPE]: response.headers['content-type'],
                            [HeaderKeys.CONTENT_LENGTH]: response.headers['content-length'],
                        }),
                        responseType,
                        response: { href: bdlHref },
                        message: 'OK',
                    } as AdobeResponse;
                }

                return _doBlockDownload.call(
                    ctx,
                    svc,
                    bdlHref,
                    undefined,
                    undefined,
                    responseType,
                    false,
                    undefined,
                    additionalHeaders,
                );
            }

            if (skipBlockDownload) {
                return {
                    statusCode: 200,
                    headers: pruneUndefined({
                        [HeaderKeys.CONTENT_TYPE]: response.headers['content-type'],
                        [HeaderKeys.CONTENT_LENGTH]: response.headers['content-length'],
                    }),
                    responseType,
                    response: { href: response.headers.location },
                    message: 'OK',
                } as AdobeResponse;
            }

            return _doBlockDownload.call(
                ctx,
                svc,
                response.headers.location,
                undefined,
                undefined,
                responseType,
                true,
                undefined,
                additionalHeaders,
            );
        });
}

/**
 * @internal
 * @private
 */
type _DirectUploadParams = {
    additionalHeaders?: Record<string, string>;
    asset: AdobeAsset;
    contentType?: string;
    data: SliceableData;
    etag?: string;
    headHref: string;
    href: string;
    maybeIsNew?: boolean;
    relation: LinkRelationKey;
    service: AdobeHTTPService;
};

/**
 * Perform a direct upload to RAPI
 * @internal
 * @private
 */
export function _directUpload(
    {
        additionalHeaders = {},
        asset,
        contentType,
        data,
        etag,
        headHref,
        href,
        maybeIsNew,
        relation,
        service,
    }: _DirectUploadParams,
    isRetry = false,
): AdobePromise<AdobeResponse, AdobeDCXError> {
    dbg('_doUpload()');

    validateParams(
        ['service', service, 'object'],
        ['asset', asset, 'object'],
        ['href', href, 'string'],
        ['headHref', headHref, 'string'],
        ['contentType', contentType, 'string', true],
        ['maybeIsNew', maybeIsNew, 'boolean', true],
        ['etag', etag, 'string', true],
        ['isRetry', isRetry, 'boolean', true],
        ['additionalHeaders', additionalHeaders, 'object', true],
    );

    // If we don't know if the target is new
    if (maybeIsNew == null) {
        // Perform HEAD operation to check if asset exists
        return service
            .invoke(HTTPMethods.HEAD, headHref, additionalHeaders, undefined, {
                isStatusValid: makeStatusValidator([404]),
            })
            .then((response) => {
                const isNew = response.statusCode !== 200;
                // TODO: Determine if we should cache links??
                return _directUpload(
                    {
                        additionalHeaders,
                        asset,
                        contentType,
                        data,
                        etag,
                        headHref,
                        href,
                        maybeIsNew: isNew,
                        relation,
                        service,
                    },
                    isRetry,
                );
            });
    }

    const isNew = maybeIsNew;

    // If this is an existing asset, use the etag provided or fallback to wildcard If-Match header
    if (!isNew) {
        additionalHeaders[HeaderKeys.IF_MATCH] = etag || '*';
    } else {
        // If we are in a retry after a failed update attempt (412). if-match header may be set, clear it
        delete additionalHeaders[HeaderKeys.IF_MATCH];
    }

    if (contentType) {
        additionalHeaders[HeaderKeys.CONTENT_TYPE] = contentType;
    }

    return service
        .invoke(HTTPMethods.PUT, href, additionalHeaders, data, {
            isStatusValid: makeStatusValidator([404, 409, 412]),
            retryOptions: { pollHeader: 'location', pollCodes: [202] },
        })
        .then((response) => {
            const statusCode = response.statusCode;
            // If we got back an error and are not retrying
            if (statusCode > 400 && !isRetry) {
                // Only retry if no etag provided
                if (etag != null) {
                    throw makeStatusValidator()(statusCode, response);
                }

                // Did we think it was an existing component but got a 404?
                if (!isNew && statusCode === 404) {
                    // Component should have already existed but 404 tells a different story.
                    // Retry and assume new
                    return _directUpload(
                        {
                            additionalHeaders,
                            asset,
                            contentType,
                            data,
                            etag,
                            headHref,
                            href,
                            maybeIsNew: true,
                            relation,
                            service,
                        },
                        true,
                    );
                } else if (statusCode === 404) {
                    throw new DCXError(DCXError.NOT_FOUND, 'Unexpected response', undefined, response);
                }

                // Did the server return 409 (Conflict) or 412 (Precondition Failed)
                if (statusCode === 409 || statusCode === 412) {
                    // Whoops, it's not new, make a head request to find out whether the asset exists and retry once
                    return _directUpload(
                        {
                            additionalHeaders,
                            asset,
                            contentType,
                            data,
                            etag,
                            headHref,
                            href,
                            maybeIsNew: undefined,
                            relation,
                            service,
                        },
                        true,
                    );
                }
            }

            return response;
        });
}
