/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

// import { sleep } from '@dcx/common-int';
import {
    AdobeAsset,
    AdobeBlockDownload,
    AdobeBlockUpload,
    AdobeComponentUploadRecord,
    AdobeResponse,
    GetSliceCallback,
    SliceableData,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import { AdobePromise } from '@dcx/promise';
import {
    getLinkProperty,
    isFunction,
    isNodeReadableStream,
    isObject,
    isWHATWGReadableStream,
    parseHeaders,
    stringToBuffer,
} from '@dcx/util';
import { LinkRelation } from '../enum/link';
import { BlockTransferProperties } from '../enum/properties';
import { parseLinkString } from './link';
import { doLinksContain } from './validation';
const dbg = newDebug('dcx:assets:block_transfer');

export const DEFAULT_BLOCK_UPLOAD_THRESHOLD = 10485760; // 10MB
export const DEFAULT_BLOCK_DOWNLOAD_THRESHOLD = 52428800; // 50MB
export const DEFAULT_MAX_CONCURRENT_REQUESTS = 4;
export const DEFAULT_STRING_LENGTH_CHECK = DEFAULT_BLOCK_UPLOAD_THRESHOLD / 4; // 2.5MB
let currentBlockDownloadThreshold = DEFAULT_BLOCK_DOWNLOAD_THRESHOLD;

const _shouldUseBlockTransfer = (size: number, threshold: number) => {
    return size >= threshold;
};

/**
 * @internal
 */
export function shouldUseBlockTransferForUpload(asset: AdobeAsset, size: number) {
    const minBlockTransferSize = getMinBlockTransferSize(asset);
    if (minBlockTransferSize && size < minBlockTransferSize) {
        // Must use single transfer if asset is smaller than repo defined minBlockTransferSize
        return false;
    }

    const maxSingleTransferSize = getMaxSingleTransferSize(asset);
    if (maxSingleTransferSize && size > maxSingleTransferSize) {
        // Must use block transfer if asset is larger than repo defined maxSingleTransferSize
        return true;
    }

    // Otherwise use a threshold to
    return size > DEFAULT_BLOCK_UPLOAD_THRESHOLD;
}

function getBlockUploadInitLinkProperty(asset: AdobeAsset, blockUploadProperty: string) {
    if (!doLinksContain(asset.links, [LinkRelation.BLOCK_UPLOAD_INIT])) {
        return;
    }

    const value = getLinkProperty(asset.links, LinkRelation.BLOCK_UPLOAD_INIT, blockUploadProperty);

    return value ? parseInt(value) : undefined;
}

function getMinBlockTransferSize(asset: AdobeAsset) {
    return getBlockUploadInitLinkProperty(asset, BlockTransferProperties.REPO_MIN_BLOCK_TRANSFER_SIZE);
}

function getMaxSingleTransferSize(asset: AdobeAsset) {
    return getBlockUploadInitLinkProperty(asset, BlockTransferProperties.MAX_SINGLE_TRANSFER_SIZE);
}

/**
 * Check whether block download should be used
 *
 * @param {number} size - size of asset in bytes
 * @param {number} [threshold = 52428800] - threshold at which to use block download
 */
export const shouldUseBlockTransferForDownload = (
    size: number,
    threshold: number = DEFAULT_BLOCK_DOWNLOAD_THRESHOLD,
) => {
    return _shouldUseBlockTransfer(size, threshold);
};

interface AsyncData<T extends SliceableData = SliceableData> {
    getSlice: GetSliceCallback<T>;
    size: number;
}

type UploadableTypes = SliceableData | AsyncData;

/**
 * Returns the size in bytes of the data param
 * @internal
 * @param data      The data to get the size for
 * @returns         The size of the data
 */

export const getDataLength = (data: UploadableTypes): number => {
    if (typeof data === 'string') {
        // This check will only needed to be done when absolute nessesary as in case of length of greater than eq to 2.5M
        if (data.length >= DEFAULT_STRING_LENGTH_CHECK) {
            // This is memory intensive code so we are trying to minisize its use
            return stringToBuffer(data).byteLength;
        }

        // Return the length if length is less than 2.5M
        // there is possibility that in this case length of string will not be equal to size of string but
        // in worst case if each char is of 4 Byte (UTF-8) the size will not exceed the default block upload limit
        return data.length;
    }

    return 'size' in data ? data.size : data.byteLength;
};

/**
 * @internal
 * @param data      The data to parse
 * @param size      The size of the data
 * @returns         The parsed data
 */
export const _parseUploadableData = (data: SliceableData | GetSliceCallback, size?: number): UploadableTypes => {
    if (!isFunction(data)) {
        // not GetSliceCallback
        return data;
    }

    // GetSliceCallback has to have at lease 2 parameters
    if (data.length < 2) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'GetSliceCallback is expected to accept 2 parameters');
    }
    if (size === undefined || isNaN(size) || size < 0) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            'Size parameter should indicate total number of bytes to be read from GetSliceCallback',
        );
    }
    return {
        getSlice: data,
        size,
    };
};

/**
 * Returns the current block download threshold
 * @returns the current block download threshold
 */
export const getBlockDownloadThreshold = (): number => {
    return currentBlockDownloadThreshold;
};

/**
 * Sets the current block download threshold
 * @param threshold     The new threshold
 */
export const setBlockDownloadThreshold = (threshold: number): void => {
    if (Number.isNaN(threshold) || typeof threshold !== 'number' || threshold <= 0) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid block download threshold, must be positive integer');
    }
    currentBlockDownloadThreshold = threshold;
};
/**
 * Convert a stream to GetSliceCallback. This should be seen as a temporary placeholder until request body streams are available.
 * See (Send ReadableStream in request body) {@link https://developer.mozilla.org/en-US/docs/Web/API/Request#browser_compatibility}
 * See {@link https://web.dev/fetch-upload-streaming/}
 */
/* istanbul ignore next */
export function streamToGetSliceCallback(
    stream: AdobeResponse<'stream'>['response'],
    resourceSize: number,
    targetAsset: AdobeAsset,
    maxOutstandingTransfers = DEFAULT_MAX_CONCURRENT_REQUESTS,
): GetSliceCallback {
    const maxSingleTransferSize = getMaxSingleTransferSize(targetAsset) ?? DEFAULT_BLOCK_UPLOAD_THRESHOLD;
    // bufferSize is limited to the maximum number of concurrent requests allowed multiplied by the size of those requests
    // In all observed cases so far, RAPI has returned a maxSingleTransferSize.
    // The specification states that it might be optional, so we utilize the default
    // Block Upload Threshold in the event that it is not returned by the server.
    const bufferSize = maxSingleTransferSize * maxOutstandingTransfers;
    const arrayBuffer = new ArrayBuffer(bufferSize);
    const resourceBuffer = new Uint8Array(arrayBuffer);
    let bytesRead = 0;
    const reader = isWHATWGReadableStream(stream)
        ? stream.pipeThrough(chunkSizeLimiter(maxSingleTransferSize, maxOutstandingTransfers)).getReader()
        : stream;
    let streamEnded = false;
    // promiseChain is used to ensure slice reads occur in-order and only after previous reads have completed
    let promiseChain = Promise.resolve();
    async function getSliceCallback(startByte: number, endByte: number) {
        // This is an early exit case to instruct the BlockUpload class to
        // no longer request any more blocks. It relies on an empty block to determine
        // whether or not the transfer document should continue to be extended (and more data should be requested)
        // An empty block signifies that there is no more data to be uploaded.
        if (streamEnded) {
            return new Uint8Array([]);
        }
        try {
            // Since multiple invocations of `getSliceCallback` are expected to occur concurrently,
            // we want to ensure that stream reads occur properly in sequence
            // This function will trigger the recursive async `read` loop which will resolve once the requested
            // slice has been cached in memory.
            let startSliceRead;
            // This promise is expected to resolve with the requested slice
            const slicePromise = new Promise<Uint8Array>(async (resolve, reject) => {
                // Process a single chunk of data from either IncomingMessage (Readable) or ReadableStream
                function processChunk(chunk: Uint8Array) {
                    const offset = bytesRead % bufferSize;
                    if (chunk.byteLength + offset > bufferSize) {
                        // The chunk will be split in 2 parts at the sliceIndex
                        // The first part will fill up the remainder of the buffer starting from the offset
                        // The remainder of the chunk will start from the beginning of the buffer and take up as much space as it needs.
                        // Chunks are limited to a max byteLength of DEFAULT_BLOCK_UPLOAD_THRESHOLD
                        // [{part2}---------offset-{part1}]
                        const sliceIndex = bufferSize - offset;
                        resourceBuffer.set(chunk.subarray(0, sliceIndex), offset); // [-------part1]
                        resourceBuffer.set(chunk.subarray(sliceIndex), 0); // [part2--------]
                    } else {
                        resourceBuffer.set(chunk, offset); // [----chunk----]
                    }
                    bytesRead += chunk.byteLength;
                    dbg(`chunk ${startByte}-${endByte} progress: ${bytesRead - startByte}/${endByte - startByte}`);
                    if (bytesRead >= endByte) {
                        // Resolve with the requested slice of data
                        const slice = resourceBuffer.subarray(startByte % bufferSize, endByte % bufferSize || endByte);
                        dbg(`resolving slice - start:  ${startByte}, end: ${endByte}, length: ${slice.byteLength}`);
                        resolve(slice);
                    }
                }
                // We handle 2 types of streams an IncomingMessage (Readable) and a ReadableStream
                // The if condition covers node.js Readable while the else portion covers ReadableStream
                // the if section only runs in the node environment, while the else section may run in either environment.
                // In each scenario we are defining the function that will read from the provided stream type as well as a "startSliceRead" function.
                // The `startSliceRead` function is scheduled to be invoked after the previous slice has resolved since `getSliceCallback` is invoked
                // in an asynchronous manner. With this scheduling, we ensure that multiple readers are not competing for data on a single stream source.
                if (isNodeReadableStream(reader)) {
                    async function readFromReadable() {
                        const chunk: any = reader.read(endByte - bytesRead);
                        // node may return null if it has not yet buffered more/enough data to be read.
                        if (chunk === null) {
                            // return to wait for more data
                            return;
                        }
                        processChunk(chunk);
                    }
                    const streamEndHandler = () => {
                        // This is the last block return only the remaining bytes
                        streamEnded = true;
                        resolve(resourceBuffer.subarray(startByte % bufferSize, bytesRead % bufferSize || bufferSize));
                    };
                    const cleanupListeners = () => {
                        reader.off('readable', readFromReadable);
                        reader.off('end', streamEndHandler);
                        reader.off('error', reject);
                    };
                    const attachListeners = () => {
                        reader.on('readable', readFromReadable);
                        reader.on('end', streamEndHandler);
                        reader.on('error', reject);
                    };
                    startSliceRead = async () => {
                        // assigning the readable listener switches the stream to "flowing" mode.
                        attachListeners();
                        await readFromReadable();
                        slicePromise.finally(cleanupListeners);
                    };
                } else {
                    // WHATWG ReadableStream handling (browser/ node 18+)
                    async function readFromReadableStream() {
                        const { value, done } = await reader.read();
                        if (done) {
                            streamEnded = true;
                            // There is no data left to read, resolve with the bytes read since the startByte.;
                            dbg('End of stream', startByte, bytesRead);
                            return resolve(
                                resourceBuffer.subarray(startByte % bufferSize, bytesRead % bufferSize || bufferSize),
                            );
                        }
                        processChunk(value);
                        if (bytesRead <= endByte) {
                            return await readFromReadableStream();
                        }
                    }
                    startSliceRead = () => {
                        dbg('starting read for slice', startByte, endByte);
                        return readFromReadableStream();
                    };
                }
                // getSliceCallback is invoked asynchronously in sequential order. To prevent the multiple asynchronous
                // invocations from reading from the underlying stream at the same time, we schedule each invocation
                // to only `startSliceRead` after the previous `slicePromise` has resolved. promiseChain is updated after
                // each invocation of `getSliceCallback`. In essence this converts the "eager" interface of the block uploader
                // which will attempt to concurrently saturate maxOutStandingRequests with slice data, to the sequential+flowing stream source.
                promiseChain = promiseChain.then(async () => {
                    startSliceRead();
                    await slicePromise;
                    return;
                });
            });
            return slicePromise;
        } catch (err: any) {
            throw new DCXError(DCXError.UNEXPECTED, err.message, err);
        }
    }
    return getSliceCallback;
}

/**
 * Limits the size of chunks that are emitted by a given stream.
 * If an incoming chunk is larger than expected, the incoming chunk is en
 * @param maxChunkSize the maximum size of a single chunk of data
 */
function chunkSizeLimiter(maxChunkSize: number, maxOutstanding: number) {
    let controller;
    const queueingStrategy = new ByteLengthQueuingStrategy({
        highWaterMark: maxChunkSize * maxOutstanding,
    });
    return {
        writable: new WritableStream(
            {
                write(chunk: Uint8Array) {
                    dbg('Chunk Received:', chunk.byteLength);
                    if (chunk.byteLength < maxChunkSize) {
                        controller.enqueue(chunk);
                    } else {
                        const chunkCount = Math.ceil(chunk.byteLength / maxChunkSize);
                        const chunks = [...Array(chunkCount).keys()].map((i) =>
                            chunk.subarray(i * maxChunkSize, Math.min((i + 1) * maxChunkSize, chunk.byteLength)),
                        );
                        for (const smallerChunk of chunks) {
                            dbg('enqueue smaller chunk:', smallerChunk.byteLength);
                            controller.enqueue(smallerChunk);
                        }
                    }
                },
                close() {
                    controller.close();
                },
            },
            queueingStrategy,
        ),
        readable: new ReadableStream(
            {
                start(_controller) {
                    controller = _controller;
                },
            },
            queueingStrategy,
        ),
    };
}

export function deserializeUploadComponentRecord(
    type: string,
    id: string,
    length: number,
    data: string,
): AdobeComponentUploadRecord {
    const result = parseHeaders(data);
    const record: AdobeComponentUploadRecord = {
        id,
        length,
        type,
        links: parseLinkString(result['link']),
        etag: result['etag'],
        location: result['location'],
        version: result['version'],
        revision: result['revision'],
        md5: result['content-md5'],
    };

    return record;
}

export function maybeGetBlockTransfer(
    promise: AdobePromise,
): Promise<AdobeBlockUpload | AdobeBlockDownload | undefined> {
    return new Promise((resolve) => {
        let timer;
        promise.finally(() => {
            clearTimeout(timer);
            resolve(undefined);
        });

        function checkForBlockTransfer() {
            ['blockUpload', 'blockDownload'].forEach((transferType) => {
                if (isObject(promise[transferType])) {
                    clearTimeout(timer);
                    return resolve(promise[transferType]);
                }
                clearTimeout(timer);
                timer = setTimeout(checkForBlockTransfer, 1);
            });
        }
        checkForBlockTransfer();
    });
}

export async function pauseBlockTransfer(promise: AdobePromise, stateChangeCb?: (state) => void) {
    const blockTransfer = await maybeGetBlockTransfer(promise);
    if (!blockTransfer) {
        return;
    }
    if (stateChangeCb) {
        (blockTransfer as any).on('stateChanged', stateChangeCb);
    }
    await blockTransfer.pause();
    return blockTransfer;
}
