/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeAsset, AdobeHTTPService, AdobeResponse, HTTPMethod, LinkMode } from '@dcx/common-types';
import { AdobeDCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import {
    arrayBufferToString,
    checkRetriable,
    concatUint8Arrays,
    getLinkHref,
    isArrayBuffer,
    parseHeaders,
    stringToBuffer,
    validateParams,
} from '@dcx/util';
import { getService } from '../Service';
import { RepoResponseResult } from '../common';
import { HeaderKeys } from '../enum/header_keys';
import { HTTPMethods } from '../enum/http_methods';
import { LinkRelation } from '../enum/link';
import { JSONProblemMediaType } from '../enum/media_types';
import { isBufferLike } from './duck_type';
import { assertLinksContain, makeStatusValidator } from './validation';

const dbg = newDebug('dcx:assets:bulk');

export interface BulkRequestDescriptor {
    method: HTTPMethod;
    href: string;
    headers?: Record<string, string | number>;
    body?: string | Buffer | ArrayBuffer;
}

/**
 * Constructs a Multipart request body given a list of reuqests
 * @param requests List of requests to put in the bulk request body
 * @param boundary Boundary token
 */
export function constructMultipartRequestBody(requests: BulkRequestDescriptor[], boundary: string): Uint8Array {
    const kAdobeMultipartCRLF = '\r\n';
    let multipartRequest = Uint8Array.from([]);

    for (let i = 0; i < requests.length; i++) {
        const req = requests[i];
        if (i === 0) {
            multipartRequest = concatUint8Arrays(
                multipartRequest,
                stringToBuffer(`--${boundary}${kAdobeMultipartCRLF}`),
            );
        } else {
            multipartRequest = concatUint8Arrays(
                multipartRequest,
                stringToBuffer(`${kAdobeMultipartCRLF}--${boundary}${kAdobeMultipartCRLF}`),
            );
        }

        multipartRequest = concatUint8Arrays(
            multipartRequest,
            stringToBuffer(`${[HeaderKeys.CONTENT_TYPE]}: application/http${kAdobeMultipartCRLF}`),
        );
        multipartRequest = concatUint8Arrays(
            multipartRequest,
            stringToBuffer(`${kAdobeMultipartCRLF}${req.method} ${req.href}`),
        );

        let contentLengthSet = false;
        for (const key in req.headers) {
            const lowerKey = key.toLowerCase();
            if (lowerKey === 'content-length') {
                contentLengthSet = true;
            }
            multipartRequest = concatUint8Arrays(
                multipartRequest,
                stringToBuffer(`${kAdobeMultipartCRLF}${lowerKey}: ${req.headers[key]}`),
            );
        }

        if (req.body) {
            const uIntArrayBody = bodyToUint8Array(req.body);

            if (!contentLengthSet) {
                multipartRequest = concatUint8Arrays(
                    multipartRequest,
                    stringToBuffer(`${kAdobeMultipartCRLF}content-length: ${uIntArrayBody.length}`),
                );
            }

            multipartRequest = concatUint8Arrays(
                multipartRequest,
                stringToBuffer(`${kAdobeMultipartCRLF}${kAdobeMultipartCRLF}`),
            );

            multipartRequest = concatUint8Arrays(multipartRequest, uIntArrayBody);
        }
    }

    multipartRequest = concatUint8Arrays(
        multipartRequest,
        stringToBuffer(`${kAdobeMultipartCRLF}--${boundary}--${kAdobeMultipartCRLF}`),
    );

    return multipartRequest;
}

export function bodyToUint8Array(body: string | ArrayBuffer | Buffer): Uint8Array {
    if (typeof body === 'string') {
        return stringToBuffer(body) as Uint8Array;
    } else if (isArrayBuffer(body)) {
        return new Uint8Array(body);
    } else if (isBufferLike(body)) {
        return body;
    }

    throw new AdobeDCXError(
        AdobeDCXError.INVALID_PARAMS,
        'Bulk subrequest body expecting string | ArrayBuffer | Buffer',
    );
}

/**
 * Parses a multipart response object and returns an array of AdobeResponse objects
 * @param result            The response object
 * @param expectedParts     The number of expected parts
 */
export function parseMultipartResponseParts(result: AdobeResponse, expectedParts?: number): AdobeResponse[] {
    const boundary = result.headers[HeaderKeys.CONTENT_TYPE] as string;
    if (!boundary) {
        throw new AdobeDCXError(AdobeDCXError.UNEXPECTED_RESPONSE, 'Missing boundary header in multipart response');
    }

    const boundaryValue = boundary.split('=')[1];

    const parts = multiPartParser(result.response, boundaryValue);

    if (expectedParts && parts.length !== expectedParts) {
        throw new AdobeDCXError(
            AdobeDCXError.UNEXPECTED_RESPONSE,
            `Unexpected number of parts; Expected ${expectedParts}, Received ${parts.length}`,
        );
    }
    return parts;
}

/**
 * Uses a Boyer Moore search algorithm to search for a sequence of bytes within a byte array.
 * Returns an array of indices for the start of each byte pattern present in the supplied byte array.
 * @param byteArray -- haystack to search
 * @param sequence -- needle to find (may appear multiple times in haystack)
 * @returns array of starting indices for each time the pattern appears in the byteArray
 */
function boyerMooreSearch(byteArray: Uint8Array, sequence: Uint8Array) {
    const badchar = new Array(256).fill(-1);
    const shiftIndices: number[] = [];
    for (let i = 0; i < sequence.length; i++) {
        badchar[sequence[i]] = i;
    }

    let shift = 0;
    while (shift <= byteArray.length - sequence.length) {
        let index = sequence.length - 1;

        while (index >= 0 && sequence[index] === byteArray[shift + index]) {
            index--;
        }

        if (index < 0) {
            shiftIndices.push(shift);
            shift +=
                shift + sequence.length < byteArray.length
                    ? sequence.length - badchar[byteArray[shift + sequence.length]]
                    : 1;
        } else {
            shift += Math.max(1, index - badchar[byteArray[shift + index]]);
        }
    }
    return shiftIndices;
}

/**
 * Given a provided byte array, delimiter indices array, and delimiter size, returns an
 * array of subarrays that are split at the delimiter indices with delimiter size bytes skipped
 * after each delimiter. The last parameter, `isInclusive` determines whether or not the first and last
 * byte ranges are included. the default value is false, which will only retrieve the byte arrays between
 * each range defined by the delimiterIndices
 *
 * ```
 * const bytes = new Uint8Array([22,13,10,22,26,13,10,14,02]);
 * const delimiterIndices = [1, 5];
 * const delimiterSize = 2;
 * const result = splitParts(bytes, delimiterIndices, delimiterSize);
 * console.log(result); // [Uint8Array(2) [22,26]]
 *
 * const inclusiveResult = splitParts(bytes,delimiterIndices, delimiterSize, true);
 * console.log(result); // [Uint8Array(1) [22], Uint8Array(2) [22,26], Uint8Array(2) [14, 02]]
 * ```
 * @param byteArray byteArray to split
 * @param delimiterIndices An array of indices where a delimiter would appear
 * @param delimiterSize The size of the delimiter (delimiter bytes are excluded from the result based on this value)
 * @param isInclusive Whether or not to include the byte ranges outside of the provided indices. When `true`, the byte
 * range from the start of the byte array to the first delimiter index AND the last delimiter index to the end of the
 * byte array are included in the response. If false, those two ranges are excluded. *Default:* `false`
 * @returns requested sub arrays for provided byteArray
 */
function splitParts(
    byteArray: Uint8Array,
    delimiterIndices: number[],
    delimiterSize: number,
    isInclusive = false,
): Uint8Array[] {
    let lastIndex = isInclusive ? 0 : undefined;
    const responses: Uint8Array[] = [];
    for (const index of delimiterIndices) {
        if (lastIndex !== undefined) {
            responses.push(byteArray.subarray(lastIndex, index));
        }
        lastIndex = index + delimiterSize;
    }
    if (isInclusive) {
        // get from the last delimiter until the end of the array
        responses.push(byteArray.subarray(lastIndex));
    }
    return responses;
}

/**
 * Will parse the provided httpContent into an object containing the headers and data (does not parse out data into requested content-type)
 * @param httpContent
 * @param isMultiPartResponsePart
 * @returns
 * @see {@link https://tools.ietf.org/html/rfc7230#section-3 RFC-7230}
 */
export function parseHttpResponseContent(
    httpContent: Uint8Array,
    isMultiPartResponsePart: boolean | undefined = false,
) {
    // locate all indices where the end of a section of a message is expected to have occurred within the HTTP Content.
    const crlf = stringToBuffer('\r\n\r\n');
    const crlfIndices = boyerMooreSearch(httpContent, crlf).slice(0, 2);
    // We are currently matching both CRLF and LF since it is possible that the server is sending both
    // Depending on the interpretation of https://datatracker.ietf.org/doc/html/rfc9112#section-2.2
    // we may be able to remove the LF search.
    // Due to a difference in the response of the block/finalize with repoMetaPatch and respondWith parameters,
    // we must also accept a double LF as a delimiter. We will only perform the search if the CRLF search
    // did not result in any matches.
    const lf = stringToBuffer('\n\n');
    const splits: [number[], number] =
        crlfIndices.length > 0 ? [crlfIndices, crlf.byteLength] : [boyerMooreSearch(httpContent, lf).slice(0, 2), lf.byteLength];
    // split the httpContent into parts based on the newLineIndices
    const parts = splitParts(httpContent, ...splits, true);
    // parse the headers from the first part (or first two parts if multipart))
    // The multipart response will have an extra header section that includes
    const headers = parts
        .slice(0, isMultiPartResponsePart || parts.length > 1 ? 2 : 1)
        .reduce((hdrs, partString) => Object.assign(hdrs, parseHeaders(arrayBufferToString(partString))), {});
    const data = parts.length > 1 ? parts[parts.length - 1] : undefined;
    const statusCode = parseInt(
        arrayBufferToString(parts[isMultiPartResponsePart ? 1 : 0])
            .split('\r\n', 1)[0]
            .split(' ')[1],
    );
    const contentLength = parseInt(headers['content-length'], 10);

    let content: Uint8Array | undefined;
    if (isNaN(contentLength)) {
        content = data?.length === 0 ? new Uint8Array([]) : data;
    } else {
        content = contentLength === 0 ? new Uint8Array([]) : data?.subarray(0, contentLength);
    }

    return {
        headers,
        response:
            headers[HeaderKeys.CONTENT_TYPE] === JSONProblemMediaType && content !== undefined
                ? JSON.parse(arrayBufferToString(content))
                : content,
        statusCode,
    } as AdobeResponse;
}

function multiPartParser(responseData: ArrayBuffer, boundary: string): AdobeResponse[] {
    const bufferView = new Uint8Array(responseData);
    const boundaryBytes = stringToBuffer(`--${boundary}`);
    const parts = splitParts(bufferView, boyerMooreSearch(bufferView, boundaryBytes), boundaryBytes.byteLength);
    return parts.map((part) => parseHttpResponseContent(part, true));
}

export function assertValidBulkRequest(requests: BulkRequestDescriptor[]): void {
    if (requests.length > 10) {
        throw new AdobeDCXError(
            AdobeDCXError.INVALID_PARAMS,
            'A single bulk request can only contain a maximum of 10 sub-requests.',
        );
    }

    const { writeOperations, readOperations } = requests.reduce<{
        readOperations: BulkRequestDescriptor[];
        writeOperations: BulkRequestDescriptor[];
    }>(
        (prev, req) => {
            if (typeof req.href !== 'string') {
                throw new AdobeDCXError(
                    AdobeDCXError.INVALID_PARAMS,
                    'A sub-request of the bulk operation is missing an href',
                );
            }
            if (typeof req.method !== 'string') {
                throw new AdobeDCXError(
                    AdobeDCXError.INVALID_PARAMS,
                    'A sub-request of the bulk operation is missing the HTTP method',
                );
            }
            const method = req.method.toUpperCase();
            if (!Object.values(HTTPMethods).includes(method as HTTPMethods)) {
                throw new AdobeDCXError(
                    AdobeDCXError.INVALID_PARAMS,
                    'A sub-request of the bulk operation includes an invalid HTTP method',
                );
            }
            if ([HTTPMethods.GET, HTTPMethods.HEAD].includes(method as HTTPMethods)) {
                prev.readOperations.push(req);
            } else {
                prev.writeOperations.push(req);
            }
            return prev;
        },
        { readOperations: [], writeOperations: [] },
    );

    if (writeOperations.length > 0 && readOperations.length > 0) {
        throw new AdobeDCXError(
            AdobeDCXError.INVALID_PARAMS,
            'Cannot mix READ and WRITE operations in bulk sub requests.',
        );
    }
}

export function performBulkRequest(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    requests: BulkRequestDescriptor[],
    linkMode: LinkMode = 'id',
    additionalHeaders: Record<string, string> = {},
    retrySubrequests = false,
): AdobePromise<RepoResponseResult<AdobeResponse[]>, AdobeDCXError> {
    dbg('performBulkRequest()');
    validateParams(
        ['svc', svc, 'object'],
        ['asset', asset, 'object'],
        ['requests', requests, 'array'],
        ['linkMode', linkMode, 'string', true, ['id', 'path']],
    );
    assertLinksContain(asset.links, [LinkRelation.BULK_REQUEST]);
    assertValidBulkRequest(requests);

    return _doBulkRequest(svc, asset, requests, linkMode, additionalHeaders).then(
        async ({ response, subresponses }) => {
            const retriableRequestDescriptors = getRetriableSubrequestDescriptors(subresponses, requests);
            const result = retrySubrequests
                ? await retryBulkSubrequests(
                      svc,
                      asset,
                      retriableRequestDescriptors,
                      linkMode,
                      additionalHeaders,
                      subresponses,
                  )
                : subresponses;
            return { result, response };
        },
    );
}

function _doBulkRequest(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    requests: BulkRequestDescriptor[],
    linkMode: LinkMode = 'id',
    additionalHeaders: Record<string, string> = {},
) {
    const boundary = `boundary-${Date.now()}`;
    const bulkRequestBody = constructMultipartRequestBody(requests, boundary);
    const headers = { ...additionalHeaders, [HeaderKeys.CONTENT_TYPE]: `multipart/mixed;boundary=${boundary}` };
    const bulkLink = getLinkHref(asset.links, LinkRelation.BULK_REQUEST, linkMode);
    return svc
        .invoke(HTTPMethods.POST, bulkLink, headers, bulkRequestBody, {
            isStatusValid: makeStatusValidator(),
            responseType: 'defaultbuffer',
            retryOptions: { pollHeader: 'location', pollCodes: [202], pollMethod: HTTPMethods.GET },
        })
        .then((response) => ({ response, subresponses: parseMultipartResponseParts(response, requests.length) }));
}

function getRetriableSubrequestDescriptors(
    responses: AdobeResponse[],
    requests: BulkRequestDescriptor[],
): BulkRequestDescriptor[] {
    return (
        responses
            .filter(({ statusCode }) => checkRetriable(statusCode))
            .map((response) => requests.find(({ href }) => href === response.headers['content-id']))
            // We anticipate that we should be able to match each of the retriable failed responses to their requests
            // But in the event that we are not able to do so, this filter will safely remove `undefined` from the list
            .filter((defined) => defined) as BulkRequestDescriptor[]
    );
}
async function retryBulkSubrequests(
    svc: AdobeHTTPService,
    asset: AdobeAsset,
    requests: BulkRequestDescriptor[],
    linkMode: LinkMode = 'id',
    additionalHeaders: Record<string, string> = {},
    responses: AdobeResponse[],
    retriesLeft = 5,
): Promise<AdobeResponse[]> {
    if (requests.length === 0 || retriesLeft <= 0) {
        return responses;
    }

    if (requests.length === 1) {
        const [requestDesc] = requests;
        // This will fall to default backoff retry cycle since it is a single request
        const directResult = await getService(svc).invoke(
            requestDesc.method,
            requestDesc.href,
            requestDesc.headers as Record<string, string>,
            requestDesc.body,
            {
                isStatusValid: makeStatusValidator(),
                responseType: 'defaultbuffer',
                retryOptions: { pollHeader: 'location', pollCodes: [202], pollMethod: HTTPMethods.GET },
            },
        );
        // Downstream handlers may still utilize the `content-id` header for additional bulk response handling
        directResult.headers['content-id'] = requestDesc.href;
        // retusn existing responses while replacing the failed response with the successful direct result
        return responses.map((response) =>
            'content-id' in response.headers && response.headers['content-id'] === directResult.headers['content-id']
                ? directResult
                : response,
        );
    }

    // retry the failed bulk requests
    const retriedResponses = await _doBulkRequest(svc, asset, requests, linkMode, additionalHeaders).then(
        async (retriedResponse) => {
            const retriableRequestDescriptors = getRetriableSubrequestDescriptors(
                retriedResponse.subresponses,
                requests,
            );
            if (retriableRequestDescriptors.length) {
                return await retryBulkSubrequests(
                    svc,
                    asset,
                    retriableRequestDescriptors,
                    linkMode,
                    additionalHeaders,
                    retriedResponse.subresponses,
                    retriesLeft - 1,
                );
            }
            return retriedResponse.subresponses;
        },
    );

    return responses.map(
        (response) =>
            retriedResponses.find(
                (retriedResponse) => retriedResponse.headers['content-id'] === response.headers['content-id'],
            ) || response,
    );
}
