/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeDCXBranch,
    AdobeDCXComposite,
    AdobeHTTPService,
    AdobeMinimalAsset,
    AdobeResponse,
    AssetWithRepoAndPathOrId,
    BlockTransferDocument,
    ComponentResourceDesignator,
} from '@dcx/common-types';
import { isAnyFunction, isObject } from '@dcx/util';
import { ServiceConfig } from '../Service';
import { RepoResponse, RepoResponseResult } from '../common';
import { LinkRelation } from '../enum/link';
import { Properties } from '../enum/properties';

/**
 * Returns true if the object contains the properties: headers, responseType, xhr, and statusCode
 * @param {unknown} p
 */
export function isAdobeResponseLike(p: unknown): p is AdobeResponse {
    return isObject(p) && ['headers', 'responseType', 'statusCode', 'xhr'].every((key) => key in p);
}

/**
 * Returns true if object has response property that is AdobeResponseLike
 * @param {unknown} p
 */
export function isRepoResponseLike(p: unknown): p is RepoResponse {
    return isObject(p) && isAdobeResponseLike(p.response);
}

/**
 * Returns true if object has component_id property that is ComponentResourceDesignator
 * @param {unknown} p
 */
export function isComponentResourceDesignator(p: unknown): p is ComponentResourceDesignator {
    return isObject(p) && 'component_id' in p;
}

/**
 * Returns true if object is RepoResponseLike and contains property `result` of any type.
 * @param {unknown} p
 */
export function isRepoResponseResultLike(p: unknown): p is RepoResponseResult {
    return isObject(p) && isRepoResponseLike(p) && 'result' in p;
}

export function isBufferLike(p: unknown): p is Buffer {
    return isObject(p) && isAnyFunction(p.slice);
}

export function isAdobeDCXCompositeLike(asset: unknown): asset is AdobeDCXComposite {
    return isObject(asset) && 'resolvePullWithBranch' in asset;
}

export function isAdobeDCXBranchLike(obj: unknown): obj is AdobeDCXBranch {
    return isObject(obj) && 
    (
        ['compositeId', 'compositeAssetId', 'compositeRepositoryId'].some((key) => key in obj) || isObject(obj._core)
    );
}
/**
 * Returns true if the asset is minimal.
 * Contains either:
 * 1. Links
 * 2. Repository ID + {assetId || path}
 *
 * @param {unknown} asset
 */
export function isMinimalAdobeAsset(asset: unknown): asset is AdobeMinimalAsset {
    return isObject(asset) && 
    (
        isObject(asset.links) ||
        (typeof asset.repositoryId === 'string' &&
            (typeof asset.path === 'string' || typeof asset.assetId === 'string'))
    );
}

export function isBlob(thing: unknown): thing is Blob {
    return typeof Blob !== 'undefined' && thing instanceof Blob;
}

/**
 * Returns true if the asset is resolvable.
 * Contains Repository ID + {assetId || path}
 *
 * @param {unknown} asset
 */
export function isResolvableAsset(asset: unknown): asset is AssetWithRepoAndPathOrId {
    return isObject(asset) && 
    (
        (typeof asset.repositoryId === 'string' && typeof asset.path === 'string') || typeof asset.assetId === 'string'
    );
}

export function isHttpService(o: unknown): o is AdobeHTTPService {
    // duck type, maybe not named AdobeHTTPService if clients implement their own,
    // but Service#invoke is always required.
    return isObject(o) && (o.name === 'AdobeHTTPService' || isAnyFunction(o.invoke));
}

export function isServiceConfig(o: unknown): o is ServiceConfig {
    return isObject(o) && isHttpService(o.service);
}

export function isTransferDocument(transferDoc: unknown): transferDoc is BlockTransferDocument {
    if (!isObject(transferDoc)) {
        return false;
    }

    const links = transferDoc[Properties.LINKS];
    if (!isObject(links)) {
        return false;
    }

    const blockLink = links[LinkRelation.BLOCK_TRANSFER];
    return !!(blockLink && links[LinkRelation.BLOCK_EXTEND] && links[LinkRelation.BLOCK_FINALIZE]);
}
