/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeAsset,
    AdobeHTTPService,
    AdobeResponse,
    BasicLink,
    LinkRelation,
    LinkSet,
    RenditionLink,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import AdobePromise from '@dcx/promise';
import { parse, pruneUndefined } from '@dcx/util';
import { ServiceConfig, constructServiceEndpoint, getReposityLinksCache, getService } from '../Service';
import { CACHE_KEYS } from '../cache/RepositoryLinksCache';
import { HTTPMethods } from '../enum/http_methods';
import { BlockTransferProperties, Properties } from '../enum/properties';
import { headHTTPResource } from './http';
import { deserializeAsset } from './serialization';
import { makeStatusValidator } from './validation';

const dbg = newDebug('dcx:assets:util:link');
// const dbg = (...args) => console.debug('dcx:assets:util:link ', ...args);

export interface IndexRepository {
    indexLinks: LinkSet;
    repositoryLinks: LinkSet;
    assetLinks: LinkSet;
}

export function getIndexLinks(
    svc: AdobeHTTPService | ServiceConfig,
    additionalHeaders?: Record<string, string>,
): AdobePromise<LinkSet> {
    dbg('getIndexLinks()');
    const service = getService(svc);
    const cache = getReposityLinksCache(svc);

    if (cache) {
        const indexLinks = cache.getIndexLinks();
        if (indexLinks) {
            return AdobePromise.resolve<LinkSet>(indexLinks);
        }

        cache.setPending(CACHE_KEYS.INDEX_KEY);
    }

    const url = constructServiceEndpoint('/', service);
    return headHTTPResource(service, url, additionalHeaders)
        .then<LinkSet>((response) => {
            return parseLinksFromResponseHeader(response);
        })
        .then<LinkSet>((linkSet: LinkSet) => {
            if (cache) {
                cache.setIndexLinks(linkSet);
            }
            return linkSet;
        })
        .catch((e) => {
            /* istanbul ignore if */
            if (cache) {
                cache.delete(CACHE_KEYS.INDEX_KEY);
            }
            throw e;
        });
}

export function getIndexRepository(
    svc: AdobeHTTPService | ServiceConfig,
    additionalHeaders?: Record<string, string>,
): AdobePromise<IndexRepository> {
    dbg('getIndexDocument()');
    const service = getService(svc);
    const cache = getReposityLinksCache(svc);

    if (cache) {
        const indexRepository = cache.getIndexRepository();
        if (indexRepository) {
            return AdobePromise.resolve(indexRepository);
        }
    }

    const url = constructServiceEndpoint('/', service);
    return service
        .invoke<'json'>(HTTPMethods.GET, url, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then((response) => {
            const indexLinks = parseLinksFromResponseHeader(response);
            const indexDocument = response.response;

            const result = <IndexRepository>{};
            for (const documentIndex in indexDocument.children) {
                const document = indexDocument.children[documentIndex];
                const documentLinks = document[Properties.LINKS];
                switch (document[Properties.REPO_PATH]) {
                    case CACHE_KEYS.INDEX_DOCUMENT_KEY:
                        result.indexLinks = documentLinks;
                        break;
                    case CACHE_KEYS.ASSETS_DOCUMENT_KEY:
                        result.assetLinks = documentLinks;
                        break;
                    case CACHE_KEYS.REPOSITORY_DOCUMENT_KEY:
                        result.repositoryLinks = documentLinks;
                        break;
                }
            }
            if (cache) {
                //Index links contain the resolve and ops links
                cache.setIndexLinks(indexLinks);

                //Index Document links contain the primary and page links for the Index, Asset and Repository Documents
                cache.setIndexRepository(result);
            }

            return result;
        });
}

export function parseLinksFromResponseHeader(response: AdobeResponse): LinkSet {
    if (!response.headers || !response.headers.link) {
        throw new DCXError(DCXError.INVALID_DATA, 'Failed to parse, missing link header');
    }

    return parseLinkString(response.headers.link as string);
}

export function parseLinkString(linkString: string): LinkSet {
    try {
        const links = parse(linkString);
        const linkSet: LinkSet = {};
        for (const linkKey in links.refs) {
            // Pluck desired properties from the parsed link
            const { rel, uri, templated, type, width, height, ...attrs } = links.refs[linkKey];
            // Format the output Link for the LinkSet object
            const _link: RenditionLink | BasicLink = pruneUndefined({
                href: uri,
                templated: templated ? templated === 'true' : undefined,
                type,
                // rendition link properties
                // https://git.corp.adobe.com/pages/caf/api-spec/chapters/common_resources/renditions_resources.html#rendition-links
                width,
                height,
                // Block Transfer Properties
                [BlockTransferProperties.MAX_SINGLE_TRANSFER_SIZE]:
                    attrs[BlockTransferProperties.MAX_SINGLE_TRANSFER_SIZE.toLowerCase()],
                [BlockTransferProperties.REPO_MIN_BLOCK_TRANSFER_SIZE]:
                    attrs[BlockTransferProperties.REPO_MIN_BLOCK_TRANSFER_SIZE.toLowerCase()],
            });
            // width and height are always expected to be numbers if they exist, ensure they are parsed as such
            (['width', 'height'] as const)
                .filter((property) => property in _link)
                .forEach((property) => {
                    const parsed = parseInt(_link[property], 10);
                    if (!isNaN(parsed)) {
                        _link[property] = parsed;
                    }
                });

            const existingData = linkSet[rel];
            if (Array.isArray(existingData)) {
                existingData.push(_link);
            } else if (existingData) {
                linkSet[rel] = [existingData, _link];
            } else {
                linkSet[rel] = _link;
            }
        }

        return <LinkSet>{ ...linkSet };
    } catch (e) {
        throw new DCXError(DCXError.INVALID_DATA, 'Failed to parse, invalid link header', e as Error);
    }
}

export function getPubsAsset(
    svc: AdobeHTTPService | ServiceConfig,
    additionalHeaders?: Record<string, string>,
): AdobePromise<AdobeAsset> {
    dbg('getPubsAsset()');
    const service = getService(svc);
    const url = constructServiceEndpoint('/index-pubs', service);
    return service
        .invoke<'json'>(HTTPMethods.GET, url, additionalHeaders, undefined, {
            responseType: 'json',
            isStatusValid: makeStatusValidator(),
        })
        .then<AdobeAsset>((response) => {
            const pubsAsset = deserializeAsset(
                response.response['newPubsParentDir'][Properties.EMBEDDED][LinkRelation.REPO_METADATA],
            );
            const cache = getReposityLinksCache(svc);
            if (cache) {
                cache.setValueWithAsset(pubsAsset.links!, pubsAsset);
            }
            return pubsAsset;
        });
}
