/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    ACPRepoMetadataResource,
    ACPRepository,
    AdobeAsset,
    AdobeAssetEmbedded,
    AdobeRepository,
    EffectivePrivileges,
} from '@dcx/common-types';
import { newDebug } from '@dcx/logger';
import { merge, pruneUndefined } from '@dcx/util';
import { LinkRelation } from '../enum/link';
import { Properties } from '../enum/properties';

const dbg = newDebug('dcx:assets:util:serialization');

export function deserializeAsset(
    data: AdobeAsset &
        ACPRepoMetadataResource & {
            _embedded?: {
                'http://ns.adobe.com/rel/ac/effective'?: EffectivePrivileges;
                'http://ns.adobe.com/rel/repository'?: AdobeRepository;
            };
        },
    embedded?: {
        'http://ns.adobe.com/rel/ac/effective'?: EffectivePrivileges;
        'http://ns.adobe.com/rel/repository'?: AdobeRepository;
        'http://ns.adobe.com/adobecloud/rel/metadata/application'?: string | Record<string, unknown>;
    },
): AdobeAsset {
    dbg('deserializeAsset()');

    const asset: AdobeAsset = {};

    asset.repositoryId = data.repositoryId || data[Properties.REPO_REPOSITORY_ID];
    asset.assetId = data.assetId || data[Properties.REPO_ASSET_ID];
    asset.name = data.name || data[Properties.REPO_NAME];

    asset.size = data.size != null ? data.size : data[Properties.REPO_SIZE];

    asset.path = data.path || data[Properties.REPO_PATH];
    asset.assetClass = data.etag || data[Properties.REPO_ASSET_CLASS];
    asset.etag = data.etag || data[Properties.REPO_ETAG];
    asset.version = data.version || data[Properties.REPO_VERSION];
    asset.format = data.format || data[Properties.DC_FORMAT];
    asset.md5 = data.md5;

    asset.createDate = data.createDate || data[Properties.REPO_CREATE_DATE];
    asset.modifyDate = data.modifyDate || (data as any).modifiedDate || data[Properties.REPO_MODIFY_DATE];
    asset.discardDate = data.discardDate || data[Properties.REPO_DISCARD_DATE];

    asset.createdBy = data.createdBy || data[Properties.REPO_CREATED_BY];
    asset.modifiedBy = data.modifiedBy || data[Properties.REPO_MODIFIED_BY];
    asset.discardedBy = data.discardedBy || data[Properties.REPO_DISCARDED_BY];

    asset.deviceCreateDate = data.deviceCreateDate || data[Properties.REPO_DEVICE_CREATE_DATE];
    asset.deviceModifyDate = data.deviceModifyDate || data[Properties.REPO_DEVICE_MODIFY_DATE];
    asset.defaultScheduledDeletionDuration =
        data.defaultScheduledDeletionDuration || data[Properties.REPO_DEFAULT_SCHEDULED_DELETION_DURATION];
    asset.scheduledDeletionDate = data.scheduledDeletionDate || data[Properties.REPO_SCHEDULED_DELETION_DATE];

    asset.baseAssetId = data.baseAssetId || data[Properties.REPO_BASE_ASSET_ID];
    asset.state = data.state || data[Properties.REPO_STATE];
    asset.links = data.links || data[Properties.LINKS];

    asset.width = data.width || data[Properties.IMAGE_WIDTH];
    asset.length = data.length || data[Properties.IMAGE_LENGTH];

    const flatEmbeddables = [embedded, data._embedded].flat();
    (asset as AdobeAssetEmbedded).embedded = Object.entries({
        EffectivePrivileges: LinkRelation.EFFECTIVE_PRIVILAGES,
        RepositoryResource: LinkRelation.REPOSITORY,
        AppMetadata: LinkRelation.APP_METADATA,
    }).reduce((result, [key, reltype]) => {
        flatEmbeddables
            .filter((embeddableData) => embeddableData && reltype in embeddableData)
            .forEach((embeddableData) =>
                merge(result, {
                    [key]: reltype === LinkRelation.REPOSITORY ? deserializeRepository(embeddableData) : embeddableData,
                }),
            );
        return result;
    }, {});
    return pruneUndefined(asset);
}

export function convertToACPRepoMetadataResource(data: AdobeAsset & ACPRepoMetadataResource): ACPRepoMetadataResource {
    dbg('convertToACPRepoMetadataResource()');

    const resource: ACPRepoMetadataResource = {};

    resource[Properties.REPO_REPOSITORY_ID] = data[Properties.REPO_REPOSITORY_ID] || data.repositoryId;
    resource[Properties.REPO_ASSET_ID] = data[Properties.REPO_ASSET_ID] || data.assetId;
    resource[Properties.REPO_NAME] = data[Properties.REPO_NAME] || data.name;
    resource[Properties.REPO_SIZE] = data[Properties.REPO_SIZE] != null ? data[Properties.REPO_SIZE] : data.size;
    resource[Properties.REPO_PATH] = data[Properties.REPO_PATH] || data.path;
    resource[Properties.REPO_ASSET_CLASS] = data[Properties.REPO_ASSET_CLASS] || data.etag;
    resource[Properties.REPO_ETAG] = data[Properties.REPO_ETAG] || data.etag;
    resource[Properties.REPO_VERSION] = data[Properties.REPO_VERSION] || data.version;
    resource[Properties.DC_FORMAT] = data[Properties.DC_FORMAT] || data.format;
    resource[Properties.REPO_CREATE_DATE] = data[Properties.REPO_CREATE_DATE] || data.createDate;
    resource[Properties.REPO_MODIFY_DATE] =
        data[Properties.REPO_MODIFY_DATE] || data.modifyDate || (data as any).modifiedDate;
    resource[Properties.REPO_DISCARD_DATE] = data[Properties.REPO_DISCARD_DATE] || data.discardDate;
    resource[Properties.REPO_CREATED_BY] = data[Properties.REPO_CREATED_BY] || data.createdBy;
    resource[Properties.REPO_MODIFIED_BY] = data[Properties.REPO_MODIFIED_BY] || data.modifiedBy;
    resource[Properties.REPO_DISCARDED_BY] = data[Properties.REPO_DISCARDED_BY] || data.discardedBy;
    resource[Properties.REPO_DEVICE_CREATE_DATE] = data[Properties.REPO_DEVICE_CREATE_DATE] || data.deviceCreateDate;
    resource[Properties.REPO_DEVICE_MODIFY_DATE] = data[Properties.REPO_DEVICE_MODIFY_DATE] || data.deviceModifyDate;
    resource[Properties.REPO_DEFAULT_SCHEDULED_DELETION_DURATION] =
        data[Properties.REPO_DEFAULT_SCHEDULED_DELETION_DURATION] || data.defaultScheduledDeletionDuration;
    resource[Properties.REPO_SCHEDULED_DELETION_DATE] =
        data[Properties.REPO_SCHEDULED_DELETION_DATE] || data.scheduledDeletionDate;
    resource[Properties.REPO_BASE_ASSET_ID] = data[Properties.REPO_BASE_ASSET_ID] || data.baseAssetId;
    resource[Properties.REPO_STATE] = data[Properties.REPO_STATE] || data.state;
    resource[Properties.LINKS] = data[Properties.LINKS] || data.links;
    resource[Properties.IMAGE_WIDTH] = data[Properties.IMAGE_WIDTH] || data.width;
    resource[Properties.IMAGE_LENGTH] = data[Properties.IMAGE_LENGTH] || data.length;

    return pruneUndefined(resource);
}

export function deserializeRepository(data: Partial<ACPRepository> = {}): Partial<AdobeRepository> {
    dbg('deserializeRepository()');

    const key = data[LinkRelation.REPOSITORY] ? data[LinkRelation.REPOSITORY] : data;
    return {
        repositoryId: key[Properties.REPO_REPOSITORY_ID],
        repositoryType: key[Properties.REPO_REPOSITORY_TYPE],
        owner: key[Properties.REPO_OWNER],
        createDate: key[Properties.REPO_CREATE_DATE],
        title: key[Properties.DC_TITLE],
        availableRegions: key[Properties.REPO_AVAILABLE_REGIONS],
    };
}
