/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeAsset,
    AdobeAssetWithLinks,
    AdobeDCXError,
    AdobeResponse,
    KeysOfUnion,
    LinkRelationKey,
    LinkSet,
} from '@dcx/common-types';
import DCXError, { ErrorCodes, HTTP_STATUS_ERROR_MAP, _defaultStatusValidator, isAdobeDCXError } from '@dcx/error';
import { ExpectType, getLinkProperty, isObject, isValidPath, validateParam } from '@dcx/util';
import { isResolvableAsset } from './duck_type';

export const makeStatusValidator = (
    validCodes?: number[],
    statusToErrMsg?: Record<number, string>,
    statusToErrCode?: Record<number, keyof typeof ErrorCodes>,
): ((statusCode?: number, response?: AdobeResponse) => AdobeDCXError | boolean) => {
    if (statusToErrMsg == null && statusToErrCode == null && validCodes == null) {
        return _defaultStatusValidator;
    }
    const _statusToErrCode = statusToErrCode || {};
    const _statusToErrMsg = statusToErrMsg || {};
    const _validCodes = validCodes || [];

    return (statusCode?: number, response?: AdobeResponse) => {
        if (!statusCode || !response) {
            return new DCXError(ErrorCodes.NETWORK_ERROR, 'Invalid or missing status code', undefined, response);
        }
        if (_validCodes.includes(statusCode)) {
            return true;
        }

        const msg =
            _statusToErrMsg[statusCode] || HTTP_STATUS_ERROR_MAP.get(statusCode)?.message || 'Unexpected response';
        const errCode = _statusToErrCode[statusCode] || HTTP_STATUS_ERROR_MAP.get(statusCode)?.code;

        // unless explicitly marked an error, 200-299 is valid
        const validOrErr = _defaultStatusValidator(statusCode, response);
        if (validOrErr === true && errCode == null) {
            return true;
        }

        if (isAdobeDCXError(validOrErr)) {
            return validOrErr;
        }

        if (errCode) {
            return new DCXError(errCode, msg, undefined, response);
        }

        return false;
    };
};

export const assertLinksContain = (
    links: LinkSet | undefined,
    requiredLinkTypes: LinkRelationKey[] = [],
    errorCode?: string,
    errorMessage?: string,
) => {
    if (!isObject(links)) {
        throw new DCXError(errorCode || DCXError.INVALID_PARAMS, errorMessage || 'Missing or invalid links on Asset');
    }

    requiredLinkTypes.map((linkKey) => {
        if (!(linkKey in links) || !isObject(links[linkKey])) {
            throw new DCXError(
                errorCode || DCXError.INVALID_PARAMS,
                errorMessage || `Missing required link: ${linkKey}`,
            );
        }
    });
};

// Not used right now, ignore for coverage
/* istanbul ignore next */
export const assertLinksContainAny = (
    links: LinkSet = {},
    anyOneLinkRequiredInPreferredOrder: LinkRelationKey[] = [],
): LinkRelationKey => {
    if (anyOneLinkRequiredInPreferredOrder.length === 0) {
        return undefined as unknown as LinkRelationKey;
    }

    for (let i = 0; i < anyOneLinkRequiredInPreferredOrder.length; i++) {
        const linkKey = anyOneLinkRequiredInPreferredOrder[i];
        if (linkKey in links && isObject(links[linkKey])) {
            return linkKey;
        }
    }

    throw new DCXError(
        DCXError.INVALID_PARAMS,
        `Missing links, one required: ${anyOneLinkRequiredInPreferredOrder.join(', ')}`,
    );
};

export const assertLinkProperty = (links: LinkSet, linkRelation: LinkRelationKey, property: string, value: string) => {
    const propValue = getLinkProperty(links, linkRelation, property);
    if (!propValue) {
        throw new DCXError(DCXError.INVALID_DATA, `Missing ${property} param on Link`);
    }

    if (propValue !== value) {
        throw new DCXError(DCXError.INVALID_DATA, `Invalid ${property} param on Link, expected ${value}`);
    }
};

export const doLinksContain = (links: LinkSet | undefined, linksToCheck: LinkRelationKey[] = []) => {
    try {
        assertLinksContain(links, linksToCheck);
    } catch (_) {
        return false;
    }
    return true;
};

export const doesAssetContainLinks = (
    asset: unknown,
    linksToCheck: LinkRelationKey[] = [],
): asset is AdobeAssetWithLinks => {
    if (!isObject(asset)) {
        return false;
    }
    return doLinksContain((asset.links || asset._links) as LinkSet, linksToCheck);
};

export const assertPathIsValid = (path: string, isAbsolute = false) => {
    if (isValidPath(path, isAbsolute)) {
        return true;
    }
    throw new DCXError(DCXError.INVALID_PARAMS, 'Invalid path');
};

export function assertAssetHasIdOrPath(asset: AdobeAsset): void {
    if (!asset.assetId && !asset.path) {
        throw new DCXError(DCXError.INVALID_PARAMS, 'Asset must contain an assetId or path');
    }
}

export function assertAssetIsResolvable(asset: AdobeAsset): void {
    if (!isResolvableAsset(asset)) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            'Asset must contain links or repositoryId + path or assetId to be resolved.',
        );
    }
}

export function assertObjectContains<T = Record<string, unknown>>(
    o: T,
    requiredPropKey: KeysOfUnion<T>,
    requiredPropType?: ExpectType,
): void {
    if (!isObject(o)) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            `Invalid parameter. Expected object, encountered "${o === null ? 'null' : typeof o}".`,
        );
    }

    if (!((requiredPropKey as string) in o)) {
        throw new DCXError(
            DCXError.INVALID_PARAMS,
            `Invalid parameter object. Expected object containing key "${String(requiredPropKey)}".`,
        );
    }

    if (requiredPropType) {
        try {
            validateParam(requiredPropKey as string, o[requiredPropKey], requiredPropType);
        } catch (_) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                `Invalid parameter object. Expected object containing key "${String(requiredPropKey)}" ` +
                    `with type "${requiredPropType}", encountered type "${typeof o[requiredPropKey]}".`,
            );
        }
    }
}

export function assertObjectContainsOneOf<T = Record<string, unknown>>(
    o: T,
    someRequiredKeys: KeysOfUnion<T>[],
    requiredPropType?: ExpectType,
): void {
    if (someRequiredKeys.length < 1) {
        return;
    }

    for (const k of someRequiredKeys) {
        try {
            assertObjectContains(o, k, requiredPropType);
            return;
        } catch (_) {
            // no op
        }
    }

    throw new DCXError(
        DCXError.INVALID_PARAMS,
        `Invalid parameter object. Expected object containing one of [${someRequiredKeys.join(', ')}]` +
            (requiredPropType
                ? ` with type ${requiredPropType}, encountered types [${someRequiredKeys
                      .map((k) => typeof o[k])
                      .join(', ')}].`
                : '.'),
    );
}

export function assertEachObjectContainsOneOf<T = Record<string, unknown>[]>(
    arr: T[],
    someRequiredKeys: KeysOfUnion<T>[],
    requiredPropType?: ExpectType,
): void {
    arr.map((o) => assertObjectContainsOneOf(o, someRequiredKeys, requiredPropType));
}
