/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { ACPAccessControlList, ACPRepoMetadataResource, ACPRepository, AdobeAsset, AdobeDCXError, AdobeHTTPService, AdobeMinimalAsset, AdobeRepoMetadata, AdobeResponse, AdobeResponseType, AssetWithRepoAndPathOrId, CopyResourceDesignator, EffectivePrivileges, JSONPatchDocument, Link, LinkedResource, LinkMode, LinkSet, Privilege, RepoDownloadStreamableReturn, RequestDescriptor, ResolvableResourceRelations, ResourceDesignator } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { AdobeRepositoryLinksCache } from './cache/RepositoryLinksCache';
import { RepoResponseResult, RepoResponseSuccessResult } from './common';
import { AssetType } from './enum/asset_types';
import { LinkRelationKey } from './enum/link';
import { AdobeStreamableContext, OptionalContext } from './LeafContext';
import { CopyResourcesOperationResult, PathOrIdAssetDesignator } from './operations';
import { ServiceConfig } from './Service';
import { BulkRequestDescriptor } from './util/bulk';
export { ACPRepoMetadataResource, AdobeAsset } from '@dcx/common-types';
export interface Asset<T extends AdobeAsset> extends AdobeAsset {
    readonly type: AssetType;
    /**
     * Performs a HEAD operation on the Asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    headPrimaryResource(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Downloads an asset's primary resource
     * @param responseType - Type to tranform response into
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getPrimaryResource<R extends AdobeResponseType>(responseType?: R, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<R>;
    /**
     * Retrieves the repository metadata resource for the asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getRepoMetadata(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeRepoMetadata>, AdobeDCXError>;
    /**
     * @hidden
     * Retrieves the repository metadata for the base directory of the asset
     */
    getBaseDirectoryMetadata(): AdobePromise<AdobeRepoMetadata>;
    /**
     * Fetches, Hydrates and returns the assets links
     *
     * @returns {AdobePromise<LinkSet>}
     */
    getLinks(additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;
    /**
     * Get the repository resource for an asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getRepositoryResource(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError>;
    /**
     * Performs a HEAD request on the application metadata resource.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    headAppMetadata(additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError>;
    /**
     * Returns the application metadata associated with the asset.
     *
     * @example
     * ```ts
     * const { result, etag, response } = await asset.getAppMetadata(etag);
     * // result is type object, the app metadata (undefined if etag was provided and not modified)
     * // etag is type string, the app metadata's etag
     * // response is type AdobeResponse
     * ```
     *
     * @note
     * Browser clients may prefer *not* using the etag parameter, instead relying on
     * the browser's HTTP cache to add the header, unless you are managing a cache yourself.
     *
     * @note
     * If etag provided and resource has not been modified, result will be `null`.
     *
     * @param etag    If specified, will be passed as the If-None-Match header value.
     *                  Note: this etag refers to the application metadata resource.
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getAppMetadata<R = any>(etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<R> & {
        etag: string;
    }, AdobeDCXError>;
    /**
     * Creates application metadata for an asset.
     * @example
     * ```ts
     * const { result, response } = await asset.putAppMetadata(obj, etag);
     * // result is type object, shape: { etag: string }, with etag being the
     * //                        value of the new app metadata resource's etag
     * // response is type AdobeResponse
     * ```
     *
     * @note Replaces the entire resource.
     *
     * @param metadata  New application metadata JSON object
     * @param etag      If specified, will be passed as the If-Match header value.
     *                      Note: this etag refers to the application metadata resource.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    putAppMetadata<R = string | Record<string, unknown>>(metadata: R, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
        etag: string;
    }>, AdobeDCXError>;
    /**
     * Update application metadata associated with the asset.
     *
     * @param patchDoc              JSON Patch Document to apply.
     *                              {@link https://git.corp.adobe.com/pages/caf/api-spec/#patch|RFC 6902}
     * @param etag                  ETag of the application metadata resource.
     *                              Used as the If-Match header value.
     *                              For unconditional update, use wildcard "*".
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     * @returns {AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError>}
     */
    patchAppMetadata<R = string | JSONPatchDocument>(patchDoc: R, etag: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
        etag: string;
    }>, AdobeDCXError>;
    /**
     * Retrieves the effective privileges for the various resources of the asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getEffectivePrivileges(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError>;
    /**
     * Executes a bulk request that allows clients to combine multiple read or write operations into a single HTTP request.
     *
     * Bulk Requests have the limitations below:
     * 1) Bulk requests must pertain to the Resources of a single Asset.
     * 2) User agents can either read multiple Resources (with HEAD and GET requests) or write multiple Resources
     *    (with POST, PUT, PATCH, and DELETE requests), but cannot mix these.
     * 3) Bulk requests may be issued using both ID-based and path-based Links, but these may not be mixed in
     *    a single request. The addressing mode used in each sub-request must match that of the bulk request.
     *
     * @note Currently only supports bulk READ Operations
     *
     * @param requests      A list of BulkRequestDescriptor's to be included as part of the bulk request
     * @param linkMode      The link mode used by the requests
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    performBulkRequest(requests: BulkRequestDescriptor[], linkMode?: LinkMode, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeResponse[]>, AdobeDCXError>;
    /**
     * Fetch's the links for the asset if the optionally provded link relation is missing from the instance
     * @param linksToPopulate       The link relation that is needed
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    fetchLinksIfMissing(linksToPopulate: LinkRelationKey[], additionalHeaders?: Record<string, string>): AdobePromise<Asset, AdobeDCXError, RequestDescriptor>;
    /**
     * Retrieves the ACL resource for an asset
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    getACLPolicy(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;
    /**
     * Checks whether the current user has the requested Privilege on the specified Resource of an Asset.
     * @param privilege             The Privilege to be checked. Legal values are read, write, delete, and ack.
     * @param relation              The LinkRelation type of the Resource whose Privilege will be checked.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     */
    checkACLPrivilege(privilege: Omit<Privilege, 'none'>, relation: LinkRelationKey, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError>;
    /**
     * Patch the ACL policy for this asset using a JSON Patch Document or stringified representation.
     * @param policy A JSON Patch Document in string on JSON format representing the patch operations to perform on the ACL
     * @param etag Optional etag for the ACL policy. If supplied, the patch will only be performed if the remote policy's etag matches the one provided.
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     * @see {@link https://acpcs.corp.adobe.com/apis/?choose=aclfrontservice#operation/jsonPatchPolicy patch documentation}
     */
    patchACLPolicy(policy: JSONPatchDocument | string, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;
    /**
     * Issues a DELETE request against an asset's ACL Policy removing all ACEs.
     *
     * @param additionalHeaders Additional headers to be applied to HTTP requests
     * @returns {AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>}
     */
    deleteACLPolicy(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>;
    /**
     * ******************************************************************************
     * Operations
     * ******************************************************************************
     */
    /**
     * Copy the asset to some destination.
     *
     * @note
     * To copy to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param destAsset             Asset containing either path or assetId
     * @param createIntermediates   Whether to create intermediate directories if missing.
     * @param overwriteExisting     Whether to overwrite an existing asset.
     * @param additionalHeaders     Additional headers to apply to HTTP Requests
     * @param manifestPatch         A JSON Patch Document to be applied to the target asset's manifest.
     */
    copy(destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>, manifestPatch?: JSONPatchDocument): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>;
    /**
     * Move the asset, uses the operation endpoint.
     *
     * @note
     * To move to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destAsset - Asset containing either path or assetId
     * @param {boolean} createIntermediates - Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting - Whether to overwrite the existing asset.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    move(destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>;
    /**
     * Deletes the asset, irreversible.
     * If the asset is a directory, recursive is a required parameter.
     *
     * @param {string}  [etag]              Etag of asset to delete, if undefined will delete unconditionally.
     * @param {boolean} [recursive]         Whether the delete should delete subdirectories recursively, required for directory.
     * @param {Record<string, string>} additionalHeaders    Additional headers to apply to HTTP Requests.
     *
     * @returns AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>
     */
    delete(etag?: string, recursive?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;
    /**
     * Discards the asset, can be reversed with restore.
     *
     * @param {boolean} [etag]              Etag of the asset to discard, if undefined will discard unconditionally.
     * @param {boolean} [recursive]         Whether the delete should delete subdirectories recursively, required for directory.
     * @param {Record<string, string>} [additionalHeaders]    Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>}
     */
    discard(etag?: string, recursive?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;
    /**
     * Restores the asset, can only be performed on discarded assets.
     *
     * @param {Record<string, string>} [additionalHeaders]    Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    restore(additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>;
    /**
     * Package the asset into a zip file.
     *
     * @note
     * To package to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be packaged to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destAsset - Asset containing either path or assetId.
     * @param {boolean} createIntermediates - Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting - Whether to overwrite the existing asset.
     * @param {Record<string, string>} additionalHeaders - Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>}
     */
    package(destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseSuccessResult<'json'>, AdobeDCXError>;
    /**
     * Copies resources from source to target asset using the COPY_RESOURCES operation.
     * See {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html}
     *
     *
     * @param {PathOrIdAssetDesignator} sourceAsset                     The source asset
     * @param {PathOrIdAssetDesignator} targetAsset                     The destination asset
     * @param {CopyResourceDesignator[]} resources                      An array of resource designators to be copied from the source to the target asset.
     * @param {boolean} [intermediates]                                 Whether to create intermediate directories if missing.
     * @param {JSONPatchDocument} [manifestPatch]                       A JSON Patch Document to be applied to the target asset's manifest.
     * @param {Record<string, string>} [additionalHeaders]              Additional headers to apply to HTTP Requests.
     *
     * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
     */
    copyResources(targetAsset: PathOrIdAssetDesignator, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>;
}
export declare class Asset<T extends AdobeAsset = AdobeAsset> implements Asset<T>, LinkedResource {
    readonly type: AssetType;
    /**
     * Underlying Data Source
     */
    protected _data: T;
    /**
     * Service Instance
     */
    protected _svc: AdobeHTTPService;
    /**
     * Links Cache
     */
    protected _cache: AdobeRepositoryLinksCache<LinkSet> | undefined;
    /**
     * The assets links
     */
    private _links;
    constructor(data: T | ACPRepoMetadataResource, svcOrSvcConfig: AdobeHTTPService | ServiceConfig, links?: LinkSet);
    /**
     * ******************************************************************************
     * LinkedResource Interface Implementation
     * ******************************************************************************
     */
    /**
     * Set links on asset
     * @param links
     */
    setLinks(links: LinkSet): void;
    /**
     * Returns the links for this asset if they have been hydrated
     *
     * @returns {LinkSet}
     */
    get links(): LinkSet;
    set links(links: LinkSet);
    /**
     * Sets the links
     * @param {string} relationship     The link relation key
     * @param {Link} link               The link
     */
    setLink(relationship: string, link: Link): void;
    /**
     * Returns a Link by relation
     * @param {stirng} relationship     The link relation key
     *
     * @returns {Link}
     */
    getLink(relationship: string): Link;
    /**
     * Removes a link
     * @param relationship      The relation key of the link to remove
     *
     * @returns {void}
     */
    removeLink(relationship: string): void;
    /**
     * @private
     *
     * Updates the cache with the assets links
     *
     * @returns {void}
     */
    private _updateCachedLinks;
    /**
     * Gets a property for a link relationship
     * @param relationship      The link relation key
     * @param property          The property to fetch
     * @param linkMode          The link mode
     *
     * @returns {string | undefined}
     */
    getLinkProperty(relationship: string, property: string, linkMode?: LinkMode): string | undefined;
    /**
     * Returns a Link Href for a templated link
     * @param {string} relationship                                                                                   The link relation key
     * @param {Record<string, (string | number) | (string | number)[] | Record<string, string | number>>} values      The values to populate the template with
     * @param {LinkMode} linkMode                                                                                     The link mode
     *
     * @returns {string | undefined}
     */
    getLinkHrefTemplated(relationship: string, values: Record<string, (string | number) | (string | number)[] | Record<string, string | number>>, linkMode?: LinkMode): string | undefined;
    /**
     * Returns the href of a link
     * @param relationship      The link relation key
     * @param linkMode          The link mode
     *
     * @returns {string | undefined}
     */
    getLinkHref(relationship: string, linkMode?: LinkMode): string | undefined;
    /**
     * ******************************************************************************
     * Getters/Setters
     * ******************************************************************************
     */
    /**
     * Returns an AdobeAsset representation of this Asset instance
     *
     * @returns {AdobeAsset}
     */
    get asset(): AdobeAsset;
    /**
     * Creates a ServiceConfig object
     *
     * @returns {ServiceConfig}
     */
    get serviceConfig(): ServiceConfig;
    /**
     * The Repository ID of the Repository storing the Asset.
     *
     * @returns {string | undefined}
     */
    get repositoryId(): string | undefined;
    /**
     * Sets the repositoryId of the asset
     *
     * @returns {void}
     */
    set repositoryId(val: string | undefined);
    /**
     * A unique identifier given to every addressable Asset in a given Repository.
     */
    get assetId(): string | undefined;
    set assetId(val: string | undefined);
    /**
     * An Asset's location in the Directory hierarchy.
     */
    get path(): string | undefined;
    set path(val: string | undefined);
    /**
     * The name of the Asset.
     */
    get name(): string | undefined;
    /**
     * An ETag is an HTTP response header returned by an HTTP/1.1-compliant web server, used to determine change in content of a Resource at a given URL. This property is required if the Asset has a Primary Resource.
     */
    get etag(): string | undefined;
    set etag(val: string | undefined);
    /**
     * Identifier of the head version of the Asset. Not present for an Asset which does not yet have a version.
     */
    get version(): string | undefined;
    set version(val: string | undefined);
    /**
     * The media type of the Resource.
     */
    get format(): string | undefined;
    set format(val: string | undefined);
    /**
     * The class of an Asset is inferred from its Media Type at the time the Asset is created.
     */
    get assetClass(): string | undefined;
    /**
     * The server date and time when the Resource was created in the Repository, such as when a File is first uploaded or a Directory is created by the server as the parent of a new Asset.
     */
    get createDate(): string | undefined;
    /**
     * The server date and time when the Resource was last modified in the Repository, such as when a new version of an Asset is uploaded or a Directory's child Resource is added or removed.
     */
    get modifyDate(): string | undefined;
    /**
     * Time the asset was discarded directly or by inheritance. Does not exist for active assets.
     */
    get discardDate(): string | undefined;
    /**
     * The ID of the user who initiated the action that caused the Resource to be created in the Repository
     */
    get createdBy(): string | undefined;
    /**
     * The ID of the user who initiated the action that most recently caused the Resource to be modified in the Repository.
     */
    get modifiedBy(): string | undefined;
    /**
     * Identifier of the user that discarded the asset directly or by inheritance. Does not exist for active assets.
     */
    get discardedBy(): string | undefined;
    /**
     * A timestamp capturing the time at which this Asset was created on the client, and which may therefore precede the time at which it was created in the Repository. (It can also be later, due to clock skew.) Can be set only at the time of Asset creation and, if not set, defaults to repo:createDate.
     */
    get deviceCreateDate(): string | undefined;
    /**
     * A timestamp capturing the time at which this Asset was last modified on the client, and which may therefore precede the time at which the corresponding modification was uploaded to the Repository. (It can also be later, due to clock skew.) If not specified with any request that creates a new version of the Asset, it will be updated to repo:modifyDate.
     */
    get deviceModifyDate(): string | undefined;
    /**
     * ID of the root asset
     */
    get baseAssetId(): string | undefined;
    set baseAssetId(val: string | undefined);
    /**
     * The Asset's state, indicating whether the Asset is active, discarded, or deleted.
     */
    get state(): string | undefined;
    /**
     * The size of the Asset in bytes.
     */
    get size(): number | undefined;
    set size(val: number | undefined);
    get md5(): string | undefined;
    set defaultScheduledDeletionDuration(val: number | undefined);
    get defaultScheduledDeletionDuration(): number | undefined;
    set scheduledDeletionDate(val: string | undefined);
    get scheduledDeletionDate(): string | undefined;
    set width(val: number | undefined);
    get width(): number | undefined;
    set length(val: number | undefined);
    get length(): number | undefined;
    useLinkOrResolveResource<R extends AdobeResponseType = AdobeResponseType>(resource: ResolvableResourceRelations, responseType?: R): AdobePromise<RepoResponseResult<AdobeAsset, R>, AdobeDCXError>;
    /**
     * Update internal data object with response from mutating API call.
     *
     * @param response
     * @returns {AdobeResponse}
     */
    protected _updateDataWithResponse(response: AdobeResponse): AdobeResponse;
    protected _withSourcePromise<T extends Record<string | number | symbol, unknown>>(source: T): AdobePromise<void, AdobeDCXError, T>;
}
/**
 * Downloads an asset's primary resource
 * @this [OptionalContext<AdobeStreamableContext>] Optional this binding for AdobeStreamableContext
 * @param svc               HTTPService
 * @param asset             The asset
 * @param responseType      Type to tranform response into
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export declare function getPrimaryResource<T extends AdobeResponseType = 'defaultbuffer'>(this: OptionalContext<AdobeStreamableContext>, svc: AdobeHTTPService, asset: AdobeAsset, responseType?: T, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<T>;
/**
 * Performs a HEAD operation on the Asset
 *
 * @param svc                Service or service config
 * @param asset              The asset
 * @param additionalHeaders  Additional headers to apply to HTTP Requests
 */
export declare function headPrimaryResource(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError>;
/**
 * Returns the resolve link for an Asset
 *
 * @note Resolve link is classified as well known by CA-771 https://jira.corp.adobe.com/browse/CA-771?src=confmacro
 *
 * @param svc               Service or service config
 * @param asset             The asset
 * @param mode              The resolve by mode to return
 * @param resource          Optional resource
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export declare function getResolveLinkForAsset(svc: AdobeHTTPService | ServiceConfig, asset: AssetWithRepoAndPathOrId, mode?: 'id' | 'path', resource?: ResourceDesignator, additionalHeaders?: Record<string, string>): AdobePromise<string>;
/**
 * Call resolveByID or resolveByPath on an asset.
 * If resource is specified (and valid) the additional resource will be included in the response object.
 *
 * @note
 * This method currently only takes a LinkRelationKey as the resource argument.
 * It will be updated with support for Resource Designator objects with https://jira.corp.adobe.com/browse/DCX-4232.
 *
 * @param svc               Service or service config
 * @param asset             The asset to resolve, must be resolvable
 * @param mode              Whether to return path- or id-based links
 * @param resource          Additional resource to include in the response
 * @param responseType      Optional parsing of additional resource, defaults to 'json'
 * @param additionalHeaders Additional headers to apply to HTTP Requests
 */
export declare function resolveAsset<T extends AdobeResponseType = 'json'>(svc: AdobeHTTPService | ServiceConfig, asset: AssetWithRepoAndPathOrId, mode?: 'path' | 'id', resource?: ResourceDesignator, responseType?: T, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset, T | 'void'>, AdobeDCXError>;
/**
 * Same as `fetchLinksForAssetWithResponse()`, but only returns LinkSet object.
 *
 * @see fetchLinksForAssetWithResponse
 *
 * @param svc        Service or service config
 * @param asset                     The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function fetchLinksForAsset(svc: AdobeHTTPService | ServiceConfig, asset: AdobeMinimalAsset, additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;
/**
 * Fetch asset's Repository Metadata using one of (in order of preference):
 * 1. HEAD ResolveBy* API
 * 2. HEAD api:id
 * 3. HEAD api:repo_metadata
 * 4. HEAD api:primary
 * 5. HEAD api:path
 *
 * @note
 * Always makes an API call. Does not fetch links from cache.
 * To fetch links from cache, use `fetchLinksIfMissing()` and provide
 * a list of LinkRelations required.
 *
 * @param svc        Service or service config
 * @param asset              The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function fetchLinksForAssetWithResponse(svc: AdobeHTTPService | ServiceConfig, asset: AdobeMinimalAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset & Required<Pick<AdobeAsset, 'links'>>, 'void'>>;
/**
 * Returns cached links or fetches if none cached.
 * If partial links are cached, will return them.
 *
 * @param svc        Service or service config
 * @param asset                            The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function getLinksForAsset(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<LinkSet>;
/**
 * Retrieves the repository metadata resource for the asset
 *
 * @param svc        Service or service config
 * @param asset            The asset
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function getRepoMetadata(svc: AdobeHTTPService, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeRepoMetadata>, AdobeDCXError>;
export declare function getBaseDirectoryMetadata(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset): AdobePromise<AdobeRepoMetadata, AdobeDCXError>;
/**
 * Get the repository resource for an asset
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        Asset whose links to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function getRepositoryResource(svc: AdobeHTTPService, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPRepository>, AdobeDCXError>;
/**
 * Performs a HEAD request on the application metadata resource.
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        The asset to HEAD the app metadata for
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function headAppMetadata(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>, AdobeDCXError>;
/**
 * Returns the application metadata associated with the asset.
 *
 * @note
 * Browser clients may prefer *not* using the etag parameter, instead relying on
 * the browser's HTTP cache to add the header, unless you are managing a cache yourself.
 *
 * @note
 * If etag provided and resource has not been modified, result will be `null`.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             The asset to fetch the app metadata for
 * @param etag              If specified, will be passed as the If-None-Match header value.
 *                             Note: this etag refers to the application metadata resource.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function getAppMetadata<T = any>(svc: AdobeHTTPService, asset: AdobeAsset, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<T, 'json'> & {
    etag: string;
}, AdobeDCXError>;
/**
 * Update application metadata associated with the asset.
 *
 * @example
 * ```ts
 * const { result, response } = await asset.putAppMetadata(obj, etag);
 * // result is type object, shape: { etag: string }, with etag being the
 * //                        value of the new app metadata resource's etag
 * // response is type AdobeResponse
 * ```
 *
 * @note Replaces the entire resource.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             Asset whose links to retrieve
 * @param metadata          New application metadata JSON object
 * @param etag              If specified, will be passed as the If-None-Match header value.
 *                              Note: this etag refers to the application metadata resource.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function putAppMetadata<T = any>(svc: AdobeHTTPService, asset: AdobeAsset, metadata: T, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
    etag: string;
}>, AdobeDCXError>;
/**
 * Update application metadata associated with the asset.
 *
 * @param {AdobeHTTPService} svc    HTTP Service or ServiceConfig
 * @param {AdobeAsset} asset        Asset whose links to retrieve
 * @param patchDoc                  JSON Patch Document to apply.
 *                                  {@link https://git.corp.adobe.com/pages/caf/api-spec/#patch|RFC 6902}
 * @param etag                      ETag of the application metadata resource.
 *                                    Used as the If-Match header value.
 *                                    For unconditional update, use wildcard "*".
 * @param additionalHeaders         Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<RepoResponseResult<{ etag: string }>, AdobeDCXError>}
 */
export declare function patchAppMetadata<T = string | JSONPatchDocument>(svc: AdobeHTTPService, asset: AdobeAsset, patchDoc: T, etag: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
    etag: string;
}>, AdobeDCXError>;
/**
 * Retrieves the effective privileges for the various resources of the asset
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        Asset whose links to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function getEffectivePrivileges(svc: AdobeHTTPService, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<EffectivePrivileges>, AdobeDCXError>;
/**
 * Retrieves the ACL resource for an asset
 *
 * @param svc    HTTP Service or ServiceConfig
 * @param asset        Asset whose links to retrieve
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function getACLPolicy(svc: AdobeHTTPService, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;
/**
 * Checks whether the current user has the requested Privilege on the specified Resource of an Asset.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             Asset to check the ACL privilege against
 * @param privilege         The Privilege to be checked. Legal values are read, write, delete, and ack.
 * @param relation          The LinkRelation type of the Resource whose Privilege will be checked.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function checkACLPrivilege(svc: AdobeHTTPService, asset: AdobeAsset, privilege: Omit<Privilege, 'none'>, relation: LinkRelationKey, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<boolean>, AdobeDCXError>;
/**
 * Patch the ACL policy for this asset using a JSON Patch Document or stringified representation.
 * @param svc HTTP Service or ServiceConfig
 * @param asset Asset to issue the ACL PATCH on
 * @param policy A JSON Patch Document representing the patch operations to perform on the ACL
 * @param etag Optional etag for the ACL policy. If supplied, the patch will only be performed if the remote policy's etag matches the one provided.
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 * @see {@link https://acpcs.corp.adobe.com/apis/?choose=aclfrontservice#operation/jsonPatchPolicy patch documentation}
 */
export declare function patchACLPolicy(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, policy: JSONPatchDocument | string, etag?: string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ACPAccessControlList>, AdobeDCXError>;
/**
 * Issues a DELETE request against an asset's ACL Policy removing all ACEs.
 *
 * @param {AdobeHTTPService} svc                        HTTP Service or ServiceConfig
 * @param {AdobeAsset} asset                            Asset object
 * @param {Record<string, string>} additionalHeaders    Additional headers to be applied to HTTP requests
 *
 * @returns {AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>}
 */
export declare function deleteACLPolicy(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<'json'>, AdobeDCXError>;
/**
 * Issues GET on a resource link if it exists in either the asset object
 * or in the links cache. If neither exists, use the resolve API to fetch
 * the resource in a single request.
 *
 * @note The list of relations that can be fetched using the resolve API will likely
 * change over time. `ResolvableResourceRelations` will have to be updated with
 * each new valid resource.
 *
 * @param svc               HTTP Service or ServiceConfig
 * @param asset             Asset object
 * @param resource
 * @param additionalHeaders Additional headers to be applied to HTTP requests
 */
export declare function useLinkOrResolveResource<T extends AdobeResponseType = AdobeResponseType>(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, resource: ResolvableResourceRelations, responseType?: T, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset, T>, AdobeDCXError>;
/**
 * Fetch set of links if required link array does not exist in links property or cache.
 *
 * @param svc                   Service or service config
 * @param asset                 Asset with or without links
 * @param linksToPopulate       Required links
 * @param suppressMissingErrors Suppress error when links fetched but not found
 * @param additionalHeaders     Additional headers to be applied to HTTP requests
 */
export declare function fetchLinksIfMissing(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, linksToPopulate?: LinkRelationKey[], suppressMissingErrors?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<LinkSet, AdobeDCXError, RequestDescriptor>;
/**
 * The Links API utilizes a well-known path and cannot return some reltypes.
 * This is the list of reltypes that are not returned by the Links API.
 */
export declare const _linksThatRequireFullResolve: Set<LinkRelationKey>;
/**
 * Fetches an assets links from the provided cache
 *
 * @param {AdobeAsset} asset                                            Asset object that identifies the composite asset.
 * @param {AdobeRepositoryLinksCache<LinkSet> | undefined} cache        The cache
 * @param {LinkRelationKey[]} requiredLinks                             The expected links
 * @param {boolean} setPending                                          Put the asset key in the cache in a pending state to prevent it being loaded multiple times.
 *
 * @returns {AdobePromise<LinkSet | undefined, AdobeDCXError, RequestDescriptor>}
 */
export declare function getLinksFromCache(asset: AdobeAsset, cache: AdobeRepositoryLinksCache<LinkSet> | undefined, requiredLinks?: LinkRelationKey[], setPending?: boolean): AdobePromise<LinkSet | undefined, AdobeDCXError, RequestDescriptor>;
export declare function assetTransformer(data: ACPRepoMetadataResource): [string, AdobeAsset];
export declare function copyResources(svc: AdobeHTTPService | ServiceConfig, sourceAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>;
//# sourceMappingURL=Asset.d.ts.map