/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2021 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeBlockDownload, AdobeDCXError, AdobeHTTPService, AdobeResponseType, BlockTransferState, RepoDownloadStreamableReturn } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { EventEmitter } from '@dcx/util';
import { AdobeStreamableContext, OptionalContext } from '../LeafContext';
export interface BlockDownloadOptions {
    startByte?: number;
    endByte?: number;
    blockSize?: number;
    url?: string;
    totalSize?: number;
    maxConcurrentRequests?: number;
}
export interface BlockDownloadEvents {
    stateChanged: (state: BlockTransferState, blockDownload: AdobeBlockDownload) => void;
    [key: string]: (...args: any[]) => unknown;
}
export declare class BlockDownload extends EventEmitter<BlockDownloadEvents> implements AdobeBlockDownload {
    private _contentType?;
    private _dbgId;
    /** Current state of the transfer. */
    private _state;
    /** Error object when in ERROR state. */
    private _error;
    /** The size for each individual block request. */
    private _blockSize;
    /** Collected bytes, only used for buffer responseType */
    private _bytes;
    /** Blocks not yet handled due to arriving out of order. */
    private _cachedBlocks;
    /** Index of block requests made */
    private _blockRequestIndex;
    /** Index of block data pushed to stream or added to buffer. */
    private _blockHandledIndex;
    /** First byte to download. */
    private _startByte?;
    /** Last byte to download. If not provided, becomes totalSize. */
    private _endByte?;
    /** The byte range to use for the next request. */
    private _currentByteRange;
    /** Number of concurrent requests to issue. */
    private _maxConcurrentRequests;
    /** Total size of the asset. */
    private _totalSize?;
    /** Block download presigned URL. */
    private _url?;
    /** Pending requests. */
    private _pending;
    private _resolve;
    private _reject;
    private _promise;
    private _service;
    constructor(svc: AdobeHTTPService, responseType: 'buffer', options?: BlockDownloadOptions);
    /**
     * The content type of the retrieved asset.
     */
    get contentType(): string;
    /**
     * The size of the content or undefined if not known.
     */
    get totalSize(): number | undefined;
    /**
     * The download byte array.
     *
     * @type {Uint8Array}
     */
    get buffer(): Uint8Array;
    /**
     * The current state of the transfer.
     */
    get state(): BlockTransferState;
    /**
     * Get promise that resolves when the BlockDownload completes.
     */
    get promise(): AdobePromise<AdobeBlockDownload>;
    /**
     * Make a request for a single block.
     *
     * @param {number | undefined}  startByte    - First byte to download.
     *                                             May be undefined if downloading last N bytes, or entire asset.
     * @param {number}              endByte      - Last byte to download.
     *                                             If negative, downloads the last <lastByte> bytes.
     * @param {number}              blockIndex   - The order index of the current block.
     * @param {number}              laneIndex    - The download lane to use.
     */
    private _requestBlock;
    /**
     * Initialize block download.
     * If provided with all required properties during construction,
     * this method only moves to the next state.
     *
     * It's possible a BlockDownload is constructed without a totalSize,
     * in which case this method will download the first block (if possible)
     * to determine the total size.
     *
     * @param {string} url          - Presigned direct download URL
     * @param {number} totalSize    - Total size of the asset to download
     */
    init(url?: string, totalSize?: number): AdobePromise<AdobeBlockDownload>;
    /**
     * Start download loop.
     */
    start(): AdobePromise<BlockDownload, AdobeDCXError>;
    pause(): AdobePromise<BlockDownload, AdobeDCXError>;
    resume(): BlockDownload;
    cancel(): AdobePromise<BlockDownload, AdobeDCXError>;
    /**
     * Set current state to waiting.
     * Done by the BlockTransferManager.
     *
     * @internal
     */
    _setWaiting(): void;
    /**
     * Start main loop.
     */
    private _start;
    /**
     * Check if transfer is complete.
     */
    private get _loopShouldContinue();
    /**
     * Main control loop.
     *
     * Loop through requests until no more data to read.
     */
    private _loop;
    private _nextBlockData;
    /**
     * Initialize the current byte range, total size, and start/end byte.
     *
     * startByte == undefined && endByte == undefined
     *  => download entire asset
     *
     * startByte == N && endByte == undefined
     *  => download N-end
     *
     * startByte == undefined && endByte == N
     *  => download first N bytes
     *
     * startByte == undefined && endByte == -N
     *  => download last N bytes
     *
     * @param totalSize
     */
    private _initByteRange;
    /**
     * Move to complete state after all pending blocks are complete.
     */
    private _finalize;
    /**
     * Attempt to parse and set totalSize from the Content-Range header value.
     *
     * @param blockRes - The block response
     * @returns {BlockResolution}
     */
    private _updateTotalSize;
    /**
     * Handle async error.
     *
     * @param err
     */
    private _handleError;
    /**
     * Handle error by rejecting outer promise and pass along to next catch.
     *
     * @param err
     */
    private _handleErrorAndThrow;
    /**
     * Handle a block response.
     * Add the data to the collector, or push to stream if it is the current block
     * Otherwise cache the block to be handled in-order at a later time.
     *
     * @param blockResponse
     * @returns
     */
    private _handleBlock;
    /**
     * Increment _blockHandledIndex and handle that block if it is cached.
     */
    private _markCurrentBlockHandled;
    /**
     * If the current block is cached, handle it.
     */
    private _checkCachedBlocks;
    /**
     * Push downloaded block to stream or buffer.
     *
     * @param {AdobeResponse} response
     */
    private _pushBlockData;
    /**
     * Move to a new state, finalize if needed.
     * Skips state change if already in complete or error states.
     *
     * @param newState
     */
    private _shiftState;
    /**
     * Check current state is valid for method call.
     *
     * @throws {AdobeDCXError}
     *
     * @param method
     * @param currentState
     */
    private _assertStateIsValid;
}
export declare function newBlockDownload(svc: AdobeHTTPService, responseType: 'buffer', options?: BlockDownloadOptions): BlockDownload;
/**
 * Wait for a transfer that may use BlockDownload.
 * Resolves when either the download is complete, or the blockDownload property becomes available.
 * @deprecated - waitForBlockDownloadToStart has deprecated using maybeGetBlockTransfer instead.
 * @param res
 */
export declare function waitForBlockDownloadToStart<T extends RepoDownloadStreamableReturn<any>>(res: T): Promise<void>;
/**
 * Block download an asset.
 *
 * @this {OptionalContext<AdobeStreamableContext>}
 *
 * @param svc
 * @param url
 * @param startByte
 * @param endByte
 * @param responseType
 * @param isPresignedUrl
 * @param totalSize
 * @returns
 */
export declare function _doBlockDownload<T extends AdobeResponseType = 'defaultbuffer'>(this: OptionalContext<AdobeStreamableContext>, svc: AdobeHTTPService, url: string, startByte?: number, endByte?: number, responseType?: T, isPresignedUrl?: boolean, totalSize?: number, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<T>;
/**
 * Build a range request header object
 *
 * @example
 * const headers = buildRangeHeader(10, 100);
 * // headers == { range: bytes=10-100 }
 *
 * const headers = buildRangeHeader(undefined, 100);
 * // headers == { range: bytes=0-100 }
 *
 * const headers = buildRangeHeader(10, undefined);
 * // headers == { range: bytes=10- }
 *
 * const headers = buildRangeHeader(undefined, -10);
 * // headers == { range: bytes=-10 }
 *
 * @param startByte
 * @param endByte
 * @returns
 */
export declare function buildRangeHeader(startByte?: number, endByte?: number): Record<string, string>;
//# sourceMappingURL=BlockDownload.d.ts.map