/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { ACPTransferDocument, AdobeAsset, AdobeBlockUpload, AdobeComponentUploadRecord, AdobeDCXError, AdobeHTTPService, AdobeRepoUploadResult, AdobeResponse, BlockTransferDocument, BlockTransferState, GetSliceCallback, LinkRelationKey, RepoMetaPatch, ResourceDesignator, SliceableData, UploadProgressCallback } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { EventEmitter } from '@dcx/util';
import { ServiceConfig } from '../Service';
import { RepoResponseResult } from '../common';
/**
 * The BlockUpload class manages block transfers of large documents
 */
export declare class BlockUpload extends EventEmitter<BlockUploadEvents> implements AdobeBlockUpload {
    /**
     * Contains the result of the BlockUpload
     */
    uploadRecord: AdobeComponentUploadRecord;
    createdAsset?: AdobeAsset;
    /**
     * Contains the result of the BlockUpload
     */
    finalizeResponse: AdobeResponse | undefined;
    /**
     * An internal id assigned to every instance of this class.
     */
    private _internalBlockUploadId;
    /**
     * The HTTPService to make requests
     */
    private _service;
    /**
     * The promise resolve method that indicates completion for all blocks
     */
    private _resolve;
    /**
     * The promise rejection method that indicates an error during upload
     */
    private _reject;
    /**
     * The overall promise which resolves when all blocks are uploaded
     */
    private _promise;
    /**
     * Current state of block upload
     */
    private _state;
    /**
     * Holds the index of the next block to be requested
     */
    private _currentBlockIndex;
    /**
     * The BlockTransferDocument which specifies things like the blockSize and transfer urls
     * Links are expected to be on the transfer document once initialized. During an assetmoved
     * operation, the links are removed from an existing transfer document so that the
     * transfer document can be re-used while only the links are re-generated.
     */
    private _blockTransferDocument;
    /**
     * A accessor convenience for storing the block transfer urls
     */
    private _transferBlockLinks;
    /**
     * An array of the currently active block upload requests
     */
    private _pendingBlockRequests;
    /**
     * A user provided callback responsible for slicing the buffer in chunks for block upload
     */
    private _getSliceCallback;
    /**
     * The number of bytes that have been currently uploaded
     */
    private _bytesUploaded;
    /**
     * The total number of blocks that were uploaded. Used to trim unused transfer url's from finalize body
     */
    private _totalBlocksUploaded;
    /**
     * Transfer is indeterminate if the TransferDocument has been extended
     */
    private _indeterminateTransfer;
    /**
     * Maximum number of Concurrent upload block requests
     */
    private _maxConcurrentRequests;
    /**
     * On progress callback handler, if transfer is extended going into an indeterminate state
     */
    onProgress?: UploadProgressCallback;
    /**
     * Optional Params required when clients don't call blockInit themselves
     */
    private _asset;
    private _dataSize;
    private _relationType;
    private _contentType;
    private _componentId;
    private _md5?;
    private _ifMatch?;
    private _relPath?;
    private _createIntermediates?;
    private _respondWith?;
    private _repoMetaPatch?;
    /**
     * This class can be initialized with an existing blockTransferDocument acquired via the HLA or leaf methods
     * @param {AdobeHTTPService} service                            The HTTPService
     * @param {GetSliceCallback} getSliceCallback               The callback to slice the data buffer
     * @param {BlockTransferDocument} blockTransferDocument     The BlockTransferDocument which defined the blocks and block size
     */
    constructor(service: AdobeHTTPService, getSliceCallback: GetSliceCallback, blockTransferDocument: BlockTransferDocument);
    /**
     * This class can be initialized with an existing blockTransferDocument acquired via the HLA or leaf methods with additional parameters
     * @param {AdobeHTTPService} service                The HTTPService
     * @param {GetSliceCallback} getSliceCallback   The callback to slice the data buffer, data sh
     * @param {Required<BlockTransferDocument>} blockTransferDocument     The BlockTransferDocument which defined the blocks and block size
     * @param {LinkRelation} relationType           (optional) The block upload relation type
     * @param {number} dataSize                     (optional) The total size of the component
     * @param {string} contentType                  (optional) The content type of the component being uploaded
     * @param {string} componentId                  (optional) The componentId to upload the component against
     * @param {string} md5                          (optional) MD5 of the data being uploaded
     * @param relPath                               (optional) the relative path must be specified.
     * @param createIntermediates                   (optional) should intermediate directories be automatically created.
     * @param respondWith                           (optional) Resource of the newly-created Asset to return in the response, if any.
     * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/advanced/block.html#completing-a-block-transfer Completing a Block Transfer}
     */
    constructor(service: AdobeHTTPService, getSliceCallback: GetSliceCallback, blockTransferDocument: Required<BlockTransferDocument>, relationType?: LinkRelationKey, dataSize?: number, contentType?: string, componentId?: string, md5?: string, etag?: string, relPath?: string, createIntermediates?: boolean, respondWith?: ResourceDesignator, repoMetaPatch?: RepoMetaPatch, maxConcurrentRequests?: number);
    /**
     * This class can also be initialized with the parameters required to initialize the block transfer. The init method
     * will handle calling blockInit for clients.
     * @param {AdobeHTTPService} service                The HTTPService
     * @param {GetSliceCallback} getSliceCallback   The callback to slice the data buffer, data sh
     * @param {AdobeAsset} asset                    The asset associated with the block upload
     * @param {LinkRelation} relationType           The block upload relation type
     * @param {number} dataSize                     The total size of the component
     * @param {string} contentType                  The content type of the component being uploaded
     * @param {string} componentId                  (optional) The componentId to upload the component against
     * @param {string} md5                          (optional) MD5 of the data being uploaded
     * @param relPath                               (optional) When creating an asset, the relative path must be specified.
     * @param createIntermediates                   (optional) When creating an asset, should intermediate directories be automatically created.
     * @param respondWith                           (optional) When creating an asset, Resource of the newly-created Asset to return in the response, if any.
     * @param repoMetaPatch                         (optional) When creating an asset, it is possible to patch select fields from the Repository Metadata Resource.
     * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/advanced/block.html#completing-a-block-transfer Completing a Block Transfer}
     */
    constructor(service: AdobeHTTPService, getSliceCallback: GetSliceCallback, asset: AdobeAsset, relationType: LinkRelationKey, dataSize: number, contentType: string, componentId?: string, md5?: string, etag?: string, relPath?: string, createIntermediates?: boolean, respondWith?: ResourceDesignator, repoMetaPatch?: RepoMetaPatch, maxConcurrentRequests?: number);
    /**
     * Initializes the block transfer, only needs to be called if a BlockTransferDocument is not provided to the constructor
     */
    init(additionalHeaders?: Record<string, string>): AdobePromise<AdobeBlockUpload, AdobeDCXError>;
    /**
     * Getter for the current state of the block_transfer
     */
    get state(): BlockTransferState;
    get promise(): AdobePromise<AdobeBlockUpload>;
    /**
     * Called by clients when they wish to start the block transfer
     */
    start(): AdobePromise<AdobeBlockUpload>;
    /**
     * Pauses the block transfers
     */
    pause(): AdobePromise<AdobeBlockUpload>;
    /**
     * Resumes the block transfers
     */
    resume(): AdobeBlockUpload;
    /**
     * Cancels the block requests
     */
    cancel(): void;
    /**
     * Set current state to waiting.
     * Done by the BlockTransferManager.
     *
     * @internal
     */
    _setWaiting(): void;
    /**
     * Uploads the next block and adds to pendingBlockRequests
     */
    uploadNextBlock(data: SliceableData): AdobePromise<AdobeResponse>;
    /**
     * Returns the number of currently active transfer requests across any BlockUpload instance
     */
    private get _pendingBlocksCount();
    /**
     * Returns the blockLockPromise which indicates if we are blocked from any new requests
     */
    private _nextBlockLock;
    /**
     * Handles the assetmoved problem type returned by ACP when an asset has migrated from one
     * region to another before the block upload process has finished. When the assetmoved
     * problem type is encountered, it is necessary to repeat the entire block upload process,
     * including initialization. This is being done within the block upload class to maintain
     * priority order with the block transfer manager and re-use any existing objects created
     * on the heap for the purposes of this upload.
     */
    private _handleAssetMoved;
    /**
     * Recursive function which attempts to add the next block for upload.
     */
    private _uploadLoop;
    /**
     * Returns the buffer slice for the block at an index
     * @param blockIndex    The block index to get the slice for
     */
    private _getBlockAtIndex;
    /**
     * Uploads the block
     * @param body    Data frame to upload
     * @param href      The upload href
     */
    private _uploadBlock;
    /**
     * Returns true if the buffer passed is empty or invalid
     * @param block     The data to check for validity
     */
    private _isEmptyBlock;
    /**
     * Extends the block transfer by 150%
     */
    private _extend;
    /**
     * Adds the blockRequest to the pending array and returns a cleanup method
     * @param blockRequest     The block request to add
     */
    private _pushPendingBlockRequest;
    /**
     * Send progress event.
     *
     * If incrementOrDone is true, send the final progress event with 100% completion.
     * If it's a number, add it to bytes sent.
     *
     * @param incrementOrDone
     *
     */
    private _updateProgress;
    /**
     * Private cancel method
     */
    private _cancel;
    /**
     * Asserts if the current state is valid
     * @param method    The method asserting
     */
    private _assertStateIsValid;
    private _continueBlockUploads;
    /**
     * Finalize the blockupload, calls finalize and polls for result
     */
    private _finalize;
    /**
     * Move to a new state, finalize if needed.
     * Skips state change if already in complete or error states.
     *
     * @param newState
     */
    private _shiftState;
}
/**
 * Initializes a control message (via the Block Initiate URL), in which the user agent
 * tells the service that it wishes to transfer content, and the service tells the user
 * agent how to go about doing that.
 * @param svc                       HTTPService instance
 * @param asset                     The asset the component is related to
 * @param transferDocument          The transfer document which is used to initialize, process, and complete block uploads.
 * @param additionalHeaders         (Optional) Any additional header to include in the request
 */
export declare function initBlockUpload(svc: AdobeHTTPService, asset: AdobeAsset, transferDocument: ACPTransferDocument, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<Required<BlockTransferDocument>>, AdobeDCXError>;
/**
 * @internal
 * @private
 */
type _UploadParams = {
    additionalHeaders?: Record<string, string>;
    asset: AdobeAsset;
    componentId?: string;
    contentType: string;
    dataOrSliceCallback: SliceableData | GetSliceCallback;
    etag?: string;
    maybeIsNew?: boolean;
    md5?: string;
    progressCb?: UploadProgressCallback;
    relation: LinkRelationKey;
    size?: number;
    svc: AdobeHTTPService | ServiceConfig;
    blockSize?: number;
    maxConcurrentRequests?: number;
};
/**
 * Perform an upload, Will attempt to use block transfer if the asset size is greater than the block transfer threshold.
 * @internal
 * @private
 */
export declare function _upload({ additionalHeaders, asset, componentId, contentType, dataOrSliceCallback, etag, maybeIsNew, md5, progressCb, relation, size, svc, blockSize, maxConcurrentRequests, }: _UploadParams): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
    blockUpload?: AdobeBlockUpload;
}>;
/**
 * @internal
 * @private
 */
type _DoBlockUploadParams = {
    service: AdobeHTTPService;
    asset: AdobeAsset;
    additionalHeaders?: Record<string, string>;
    dataOrSliceCallback: SliceableData | GetSliceCallback;
    contentType: string;
    progressCb?: UploadProgressCallback;
    relation: LinkRelationKey;
    size?: number;
    componentId?: string;
    md5?: string;
    etag?: string;
    relPath?: string;
    createIntermediates?: boolean;
    respondWith?: ResourceDesignator;
    blockSize?: number;
    repoMetaPatch?: RepoMetaPatch;
    maxConcurrentRequests?: number;
};
/**
 * Perform a block upload.
 * @internal
 * @private
 */
export declare function _doBlockUpload({ service, asset, additionalHeaders, dataOrSliceCallback, contentType, progressCb, relation, size, componentId, md5, etag, relPath, createIntermediates, respondWith, blockSize, repoMetaPatch, maxConcurrentRequests, }: _DoBlockUploadParams): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
    blockUpload: AdobeBlockUpload;
}>;
export type BlockUploadEvents = {
    stateChanged: (state: BlockTransferState) => void;
};
export {};
//# sourceMappingURL=BlockUpload.d.ts.map