/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { ACPRepoMetadataResource, AdobeAsset, AdobeAssetEmbedded, AdobeHTTPService, EmbeddableResource, GetSliceCallback, LinkSet, ProgressCallback, RepoMetaPatch, RequestDescriptor, ResourceDesignator, SliceableData, UploadProgressCallback } from '@dcx/common-types';
import AdobeDCXError from '@dcx/error';
import AdobePromise from '@dcx/promise';
import { Asset } from './Asset';
import { AdobeGetPageResult, Page, PageOptions } from './Pagination';
import { ServiceConfig } from './Service';
import { RepoResponse, RepoResponseResult } from './common';
import { AssetType } from './enum/asset_types';
import { PathOrIdAssetDesignator } from './operations';
export type AdobeDirectoryData = AdobeAsset & Page;
export interface AdobeGetDirectoryDataResult {
    result: AdobeDirectoryData;
}
/**
 * Instance type is returned (From instance method or session)
 */
export interface AdobeGetDirectoryResult {
    result: Directory;
}
export interface AdobeDirectory extends Asset<AdobeDirectoryData> {
    children: AdobeAsset[];
    /**
     * Returns a paged set of results for the directory listing
     * @param {PageOptions} pageOpts              Page options
     */
    getPagedChildren<T extends EmbeddableResource = never>(pageOpts?: PageOptions<T>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAsset>>>;
    /**
     * Create an asset relative to the current directory.2
     *
     * @param {string} relPath                                      Path2 relative to current directory.
     * @param {boolean} createIntermediates                         Whether to create intermediate directories if they don't exist.
     * @param {string} contentType                                  Content type of the new asset.
     * @param {ResourceDesignator} [resourceDesignator]             Resource to be included in response.
     * @param {Record<string, string[]>} [additionalHeaders = {}]   Additional headers for the request.
     */
    createAsset(relPath: string, createIntermediates: boolean, contentType: string, resourceDesignator?: ResourceDesignator, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<Asset>, AdobeDCXError>;
    /**
     * Copy the asset to some destination.
     *
     * @note
     * To copy to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destAsset            Asset containing either path or assetId
     * @param {boolean} createIntermediates     Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting       Whether to overwrite an existing asset.
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    copy(destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean): AdobePromise<RepoResponseResult<Directory, 'json'>, AdobeDCXError>;
}
export declare class Directory extends Asset<AdobeDirectoryData> implements AdobeDirectory {
    readonly type: AssetType;
    children: AdobeAsset[];
    constructor(data: Partial<AdobeDirectoryData | ACPRepoMetadataResource>, svc: AdobeHTTPService | ServiceConfig, links?: LinkSet);
    /**
     * Returns a paged set of results for the directory listing
     * @param pageOpts          Page options
     * @param additionalHeaders Additional headers to be provided with HTTP requests
     */
    getPagedChildren<T extends EmbeddableResource = never>(pageOpts?: PageOptions<T>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>>;
    /**
     *
     * Create an asset relative to the current directory.
     *
     * @param {string} relPath                                      Path relative to current directory.
     * @param {boolean} createIntermediates                         Whether to create intermediate directories if they don't exist.
     * @param {string} contentType                                  Content type of the new asset.
     * @param {ResourceDesignator} [resourceDesignator]             Resource to be included in response.
     * @param {Record<string, string[]>} [additionalHeaders = {}]   Additional headers for the request.
     */
    createAsset(relPath: string, createIntermediates: boolean, contentType: string, resourceDesignator?: ResourceDesignator, additionalHeaders?: Record<string, string>, primaryResource?: SliceableData | GetSliceCallback, primaryResourceSize?: number, repoMetaPatch?: RepoMetaPatch, progressCb?: UploadProgressCallback): AdobePromise<RepoResponseResult<Asset>, AdobeDCXError>;
    /**
     * Move the asset, uses the operation endpoint.
     *
     * @note
     * To move to a new repositoryId, it must be specified in the destAsset.
     * If no repositoryId is specified, it will be moved to the same repository as the source asset instance.
     *
     * @param {AdobeAsset} destination          Asset containing either path or assetId
     * @param {boolean} createIntermediates     Whether to create intermediate directories if missing.
     * @param {boolean} overwriteExisting       Whether to overwrite the existing asset.
     *
     * @returns {AdobePromise<RepoResponseResult<Asset, 'json'>, AdobeDCXError>}
     */
    copy(destination: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean): AdobePromise<RepoResponseResult<Directory, 'json'>, AdobeDCXError>;
}
/**
 * Transforms a raw Directory JSON representation into an AdobeAsset
 *
 * @param {ACPRepoMetadataResource} data        JSON representation of a directory
 *
 * @returns {[string, AdobeAsset]}
 */
export declare function directoryTransformer(data: ACPRepoMetadataResource): [string, AdobeAsset];
/**
 * Returns a Directory object
 *
 * @param {AdobeHTTPService} svc        HTTP Service
 * @param {AdobeAsset} dirAsset         The asset
 *
 * @returns {AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>}
 */
export declare function getDirectory(svc: AdobeHTTPService | ServiceConfig, dirAsset: AdobeAsset): AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>;
/**
 * Returns a Directory object by URL
 *
 * @param {AdobeHTTPService} svc        HTTP Service
 * @param {AdobeAsset} dirAsset         The asset
 *
 * @returns {AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>}
 */
export declare function getDirectoryByURL(svc: AdobeHTTPService, url: string): AdobePromise<RepoResponse<AdobeGetDirectoryDataResult>, AdobeDCXError>;
/**
 * Returns a PageResource object that allows for traversing a paged set of results
 *
 * @param {AdobeHTTPService} svc        HTTP Service
 * @param {AdobeAsset} dirAsset         The asset
 * @param {PageOptions} pageOpts        Paging options
 *
 * @returns {AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>>}
 */
export declare function getPagedChildren<T extends EmbeddableResource = never>(svc: AdobeHTTPService, dirAsset: AdobeAsset, pageOpts?: Omit<PageOptions<T>, 'itemTransformer'>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponse<AdobeGetPageResult<AdobeAssetEmbedded<T>>>>;
/**
 * Create an asset relative to the current directory.
 *
 * @param service               The HTTPService
 * @param parentDir             The parent directory to create the asset in
 * @param relPath               Path relative to current directory.
 * @param createIntermediates   Whether to create intermediate directories if they don't exist.
 * @param contentType           Content type of the new asset.
 * @param respondWith           Resource to be included in response.
 * @param additionalHeaders     Additional headers for the request.
 * @param primaryResource       The primary resource to be uploaded.
 * @param primaryResourceSize   The size of the primary resource.
 * @param repoMetaPatch         Patch to apply to repository metadata immediately after create.
 * @see {@link https://git.corp.adobe.com/pages/caf/api-spec/single.html#repository-metadata-patch-support Repository Metadata Patch Support}
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError, RequestDescriptor>}
 */
export declare function createAsset(svc: AdobeHTTPService | ServiceConfig, parentDir: AdobeAsset, relPath: string, createIntermediates: boolean, contentType: string, respondWith?: ResourceDesignator, additionalHeaders?: Record<string, string>, primaryResource?: SliceableData | GetSliceCallback, primaryResourceSize?: number, repoMetaPatch?: RepoMetaPatch, progressCb?: ProgressCallback): AdobePromise<RepoResponseResult<AdobeAsset>, AdobeDCXError, RequestDescriptor>;
//# sourceMappingURL=Directory.d.ts.map