/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeAsset, AdobeBlockUpload, AdobeDCXError, AdobeHTTPService, AdobeRepoUploadResult, AdobeResponse, AdobeResponseType, GetSliceCallback, JSONPatchDocument, LinkRelationKey, LinkSet, RenditionOptions, RepoDownloadStreamableReturn, ResponseTypeMap, SliceableData } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { LinkProvider } from '@dcx/util';
import { Asset } from './Asset';
import { AdobeStreamableContext, OptionalContext } from './LeafContext';
import { ServiceConfig } from './Service';
import { RepoResponseResult } from './common';
import { EmbeddedMetadataRepresentation } from './embedded_metadata';
export declare const VersionContentType = "application/vnd.adobe.versions+json";
export interface AdobeFileData extends AdobeAsset {
    format?: string;
    version?: string;
    width?: number;
    height?: number;
    renderable?: boolean;
}
export interface AdobeFileBase extends Asset<AdobeFileData> {
    getRendition<T extends AdobeResponseType>(opts?: RenditionOptions, responseType?: T, linkProvider?: LinkProvider<RenditionOptions>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<T>>;
    blockDownload<T extends AdobeResponseType>(startByte?: number, endByte?: number, resource?: LinkRelationKey, componentId?: string, version?: string, responseType?: T): RepoDownloadStreamableReturn<T>;
    /**
     * ******************************************************************************
     * Embedded metadata APIs
     * ******************************************************************************
     */
    /**
     * Get embedded metadata for the composite.
     *
     * @param {'json' | 'xml'}          format              Whether to return as JSON or XML.
     * @param {Record<string, string>}  additionalHeaders   Additional headers to be applied to HTTP requests
     */
    getEmbeddedMetadata<T = any>(format?: EmbeddedMetadataRepresentation, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<T>>;
    /**
     * Update entire embedded/XMP metadata resource.
     *
     * @note
     * Replaces existing XMP resource.
     *
     * @note
     * Currently only supported on composites.
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * The argument `etag` will be updated to optional when the service supports it.
     * See {@link https://jira.corp.adobe.com/browse/SYSE-7940|ticket}
     * and {@link https://jira.corp.adobe.com/browse/SYSE-5943|another ticket}
     *
     * @param {T = Record<string, unknown>} data            New embedded metadata
     * @param {string}                      etag            ETag of metadata resource to update.
     * @param {'json'|'xml'}                [format='json'] Defines the representation of the body, either XML or JSON.
     *                                                      If using XML, clients must pass the data as a string.
     *                                                      If using XML with TypeScript, clients must specify the generic as `string`.
     *                                                      Defaults to json.
     * @param {Record<string, string>}      additionalHeaders     Additional headers to attach to HTTP Requests
     */
    putEmbeddedMetadata<T = Record<string, unknown>>(data: T, etag: string /** TODO: Revisit optional etag, see: https://jira.corp.adobe.com/browse/SYSE-5943 */, format?: EmbeddedMetadataRepresentation, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Update embedded/XMP metadata using JSON Patch document.
     *
     * @see
     * https://tools.ietf.org/html/rfc6902#page-6
     *
     * @note
     * Currently the service requires an etag, but spec states unguarded updates should be possible.
     * The argument `etag` will be updated to optional when the service supports it.
     * see: https://jira.corp.adobe.com/browse/SYSE-7940
     * and: https://jira.corp.adobe.com/browse/SYSE-5943
     *
     * @param {JSONPatchDocument | string}      data      Data to use as PATCH body
     * @param {string}                          etag      ETag of the embedded metadata resource to be updated
     * @param {Record<string, string>}          additionalHeaders     Additional headers to attach to HTTP Requests
     */
    patchEmbeddedMetadata(data: JSONPatchDocument | string, etag: string /** TODO: Revisit optional etag, see: https://jira.corp.adobe.com/browse/SYSE-5943 */, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'void'>>;
    /**
     * Update the asset's primary resource.
     *
     * @note Implicitly creates a new version of the asset.
     *
     * @param dataOrSliceCallback   Data to use for update, or callback to provide blocks for block upload.
     *                              If a callback is used, block upload will be used, regardless of size of the asset.
     * @param contentType           The content type of the resource to upload.
     * @param [size]                The size of the resource in bytes.
     * @param [etag]                The etag of the existing resource.
     * @param [md5]                 The MD5 hash of the resource, used for block upload finalize.
     * @param [additionalHeaders]   Additional headers to attach to HTTP Requests
     */
    updatePrimaryResource(dataOrSliceCallback: SliceableData | GetSliceCallback, contentType: string, size?: number, etag?: string, md5?: string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
        blockUpload?: AdobeBlockUpload;
    }>;
}
export declare abstract class AdobeFileBase extends Asset<AdobeFileData> implements AdobeFileBase {
    constructor(data: AdobeFileData, svc: AdobeHTTPService | ServiceConfig, links?: LinkSet);
    get width(): number | undefined;
    get length(): number | undefined;
    get renderable(): boolean | undefined;
}
export declare function getRendition<T extends AdobeResponseType = 'defaultbuffer'>(svc: AdobeHTTPService, asset: AdobeAsset, renditionOptions?: RenditionOptions, responseType?: T, linkProvider?: LinkProvider<RenditionOptions>, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<ResponseTypeMap[T], T>>;
/**
 * Perform block download of specified byte range.
 * @this
 * @param svc
 * @param assetOrPresignedUrl
 * @param startByte
 * @param endByte
 * @param resource
 * @param componentId
 * @param version
 * @param responseType
 * @returns
 */
export declare function doBlockDownload<T extends AdobeResponseType = 'defaultbuffer'>(this: OptionalContext<AdobeStreamableContext>, svc: AdobeHTTPService, assetOrPresignedUrl: AdobeAsset | string, startByte?: number, endByte?: number, resource?: LinkRelationKey, componentId?: string, version?: string, responseType?: T, additionalHeaders?: Record<string, string>): RepoDownloadStreamableReturn<T>;
/**
 * Update an asset's primary resource.
 *
 * @note Implicitly creates a new version of the asset.
 *
 * @param service               HTTPService to use.
 * @param asset                 Asset whose primary resource to update.
 * @param dataOrSliceCallback   Data to use for update, or callback to provide blocks for block upload.
 *                              If a callback is used, block upload will be used, regardless of size of the asset.
 * @param contentType           The content type of the resource to upload.
 * @param size                  The size of the resource in bytes.
 * @param etag                  The etag of the existing resource.
 * @param md5                   The MD5 hash of the resource, used for block upload finalize.
 * @param additionalHeaders     Additional headers to attach to HTTP requests
 */
export declare function updatePrimaryResource(svc: AdobeHTTPService | ServiceConfig, asset: AdobeAsset, dataOrSliceCallback: SliceableData | GetSliceCallback, contentType: string, size?: number, etag?: string, md5?: string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeRepoUploadResult, AdobeDCXError, {
    blockUpload?: AdobeBlockUpload;
}>;
//# sourceMappingURL=FileBase.d.ts.map