/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeAsset } from '@dcx/common-types';
export interface AdobeGenericCache<T> {
    /**
     * Return the key for the passed asset
     * @param asset     The asset to get the key for
     */
    getKey(asset: AdobeAsset): string | undefined;
    /**
     * Returns a value for the asset from the cache
     * @param asset     The asset key
     */
    getValueWithAsset(asset: AdobeAsset): Promise<T> | undefined;
    /**
     * Sets a promise for the key that can be resolved with the value
     * @param key       The key
     * @param repoId    The repoId
     */
    setPending(key: string, repoId?: string): void;
    /**
     * Returns value for key from cache
     * @param key       Cache key
     * @param repoId    The repositoryId
     */
    get(key: string, repoId?: string): Promise<T> | undefined;
    /**
     * Sets value for asset as key, if no repoId is set on the asset the asset will be stored in the session
     * @param value     The value to set
     * @param asset     The asset to use as the key
     */
    setValueWithAsset(value: T, asset: AdobeAsset): void;
    /**
     * Sets the key value pair
     * @param value     The value
     * @param key       The key
     * @param repoId    The repoId
     */
    set(value: T, key: string, repoId?: string): void;
    /**
     * Delete entry at key
     * @param {string} key
     */
    delete(key: string): void;
    /**
     * Delete entry with asset key
     * @param {AdobeAsset} asset
     */
    deleteWithAsset(asset: AdobeAsset): void;
}
export declare const DEFAULT_CACHE_MAX_ENTRIES = 100000;
/**
 * A cache that holds references to a limited number of values. Each time a value is accessed, it is moved to the head of a queue.
 * When a value is added to a full cache, the value at the end of that queue is evicted.
 */
export declare class GenericCache<T> implements AdobeGenericCache<T> {
    private values;
    private maxEntries;
    private defaultSessionKey;
    private promiseToResolveMap;
    constructor(maxEntries?: number, defaultSessionKey?: string);
    /**
     * Clear cache, resolve all pending promises to undefined
     */
    clear(): void;
    /**
     * Return the key for the passed asset
     * @param asset     The asset to get the key for
     */
    getKey(asset: AdobeAsset): string | undefined;
    /**
     * Returns a value for the asset from the cache
     * @param asset     The asset key
     */
    getValueWithAsset(asset: AdobeAsset): Promise<T> | undefined;
    /**
     * Sets a promise for the key that can be resolved with the value
     * @param key       The key
     * @param repoId    The repoId
     */
    setPending(key: string, repoId?: string): any;
    /**
     * Returns value for key from cache
     * @param key       Cache key
     * @param repoId    The repositoryId
     */
    get(key: string, repoId?: any): Promise<T> | undefined;
    /**
     * Sets value for asset as key, if no repoId is set on the asset the asset will be stored in the session
     * @param value     The value to set
     * @param asset     The asset to use as the key
     */
    setValueWithAsset(value: T | undefined, asset: AdobeAsset): void;
    /**
     * Sets the key value pair
     * @param value     The value
     * @param key       The key
     * @param repoId    The repoId
     */
    set(value: T, key: string, repoId?: any): void;
    delete(key: string, repoId?: any): void;
    deleteWithAsset(asset: AdobeAsset): void;
}
//# sourceMappingURL=GenericCache.d.ts.map