/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { ACPCopyResourceDesignator, ACPProblemResponse, ACPRepoMetadataResource, AdobeAsset, AdobeDCXComposite, AdobeDCXError, AdobeHTTPService, AdobeResponse, CopyResourceDesignator, JSONPatchDocument, LinkRelationKey, RequireAtLeastOne } from '@dcx/common-types';
import AdobePromise from '@dcx/promise';
import { AdobeOpsContext, OptionalContext } from './LeafContext';
import { ServiceConfig } from './Service';
import { RepoResponseResult } from './common';
export type AdobeOperation = 'move' | 'copy' | 'discard' | 'restore' | 'package' | 'delete' | 'copy_resources';
interface OperationSource {
    assetId?: string;
    path?: string;
    'repo:assetId'?: string;
    'repo:path'?: string;
    'if-none-match'?: string;
    'if-match'?: string;
    'repo:baseAssetId'?: string;
    baseAssetId?: string;
    version?: string;
    etag?: string;
    format?: string;
    'dc:format'?: string;
    'repo:version'?: string;
}
type AdobeOperationSource = OperationSource & RequireAtLeastOne<{
    repositoryId?: string;
    'repo:repositoryId'?: string;
}, 'repositoryId' | 'repo:repositoryId'>;
export type IDBasedOperationSource = RequireAtLeastOne<AdobeOperationSource, 'assetId' | 'repo:assetId'>;
/**
 * Path or ID based source for Operation API.
 */
export type PathOrIdAssetDesignator = RequireAtLeastOne<AdobeOperationSource, 'path' | 'repo:path' | 'assetId' | 'repo:assetId'> | AdobeDCXComposite;
export interface AdobeOperationDocument {
    op: AdobeOperation;
    target: AdobeOperationSource | AdobeOperationSource[];
    [key: string]: unknown;
}
export interface AdobeMoveCopyDocument extends AdobeOperationDocument {
    op: 'move' | 'copy';
    source: AdobeOperationSource | AdobeOperationSource[];
    intermediates?: boolean;
    asset?: AdobeAsset | ACPRepoMetadataResource;
}
export interface AdobeDiscardRestoreDocument extends AdobeOperationDocument {
    op: 'discard' | 'delete' | 'restore';
    recursive?: boolean;
    name?: string;
}
export interface AdobePackageDocument extends AdobeOperationDocument {
    op: 'package';
    source?: AdobeOperationSource | AdobeOperationSource[];
}
export interface AdobeCopyResourcesDocument extends Omit<AdobeOperationDocument, 'target'> {
    op: 'copy_resources';
    source: PathOrIdAssetDesignator;
    target: PathOrIdAssetDesignator;
    resources: CopyResourceDesignator[];
}
/**
 * @private Limit of resources that can be copied in a single COPY_RESOURCES operation.
 * {@see {@link https://git.corp.adobe.com/pages/caf/api-spec/chapters/responses/service_responses.html#limit-resource-count | Limit Resource Count Problem Type}}
 * {@see  {@link https://wiki.corp.adobe.com/pages/viewpage.action?spaceKey=dcxteam&title=Composite+Service+Resource+Copy+API+Spec#CompositeServiceResourceCopyAPISpec-HTTPResponseStatusCodes | CoS Resources Copy API Spec}}
 **/
export declare const COPY_RESOURCES_RESOURCE_LIMIT = 30;
/**
 * Used to narrow Any Operation Document to AdobeCopyResourcesDocument
 * @param doc Any Operation Document
 * @returns true if doc is AdobeCopyResourcesDocument
 */
export declare const isAdobeCopyResourcesDocument: (doc: AnyOperationDocument) => doc is AdobeCopyResourcesDocument;
export type AnyOperationDocument = AdobeMoveCopyDocument | AdobeCopyResourcesDocument | AdobeDiscardRestoreDocument | AdobePackageDocument;
interface OperationDocTypeMap {
    move: AdobeMoveCopyDocument;
    copy: AdobeMoveCopyDocument;
    copy_resources: AdobeCopyResourcesDocument;
    discard: AdobeDiscardRestoreDocument;
    delete: AdobeDiscardRestoreDocument;
    restore: AdobeDiscardRestoreDocument;
    package: AdobePackageDocument;
}
type OperationResultSource = Omit<OperationSource, 'assetId' | 'baseAssetId' | 'path' | 'etag' | 'format'>;
interface OperationResult {
    op: AdobeOperation;
    source?: OperationResultSource | OperationResultSource[];
    target?: OperationResultSource;
    asset?: ACPRepoMetadataResource;
    intermediates?: boolean;
    error?: ACPProblemResponse;
    resources?: ACPCopyResourceDesignator[];
}
export type AdobeOperationResult = Omit<OperationResult, 'error'> & {
    error?: AdobeDCXError<ACPProblemResponse>;
    _additionalData?: ACPProblemResponse;
};
export type CopyResourcesOperationResult = {
    source: OperationResultSource;
    target: OperationResultSource;
    resources: ACPCopyResourceDesignator[];
    asset?: AdobeAsset;
};
/**
 * Trigger server copy resources, and won't fall back to client copy on error 501
 * https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html
 *
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param svc               The http service
 * @param srcAsset          The source asset
 * @param destAsset         The destination asset
 * @param resources         List of resources designators
 * @param intermediates     Whether to create intermediate directories if missing.
 * @param manifestPatch     A JSON Patch Document to be applied to the target asset's manifest.
 * @param additionalHeaders Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>}
 */
export declare function copyResources(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, srcAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
    success: boolean;
}, 'json'>, AdobeDCXError>;
/**
 * Copies an asset for the src to the destination
 * User cannot copy component resources using this method
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param svc                   The http service
 * @param srcAsset              The source asset
 * @param destAsset             The destination asset
 * @param createIntermediates   Should intermediates be created?
 * @param overwriteExisting     Should an existing asset in dest be overwritten
 * @param additionalHeaders     Additional headers to attach to HTTP requests
 * @param manifestPatch         A JSON Patch Document to be applied to the target asset's manifest.
 */
export declare function copyAsset(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, srcAsset: PathOrIdAssetDesignator, destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting?: boolean, additionalHeaders?: Record<string, string>, manifestPatch?: JSONPatchDocument): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>;
/**
 * Normalized ResourceDesignator format.
 */
type NormalizedResourceDesignator = {
    reltype: LinkRelationKey;
    component_id: string;
    revision: string;
};
/**
 * Normalize a CopyResource Designator, the returned format will always include both source and target, in the normalizedResource designator
 */
export declare const normalizeCopyResourcesDesignator: (resourceDesignator: CopyResourceDesignator) => {
    source: NormalizedResourceDesignator;
    target: NormalizedResourceDesignator;
};
/**
 * Moves an asset from src to dest
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        The http service
 * @param {PathOrIdAssetDesignator} srcAsset       The source asset
 * @param {PathOrIdAssetDesignator} destAsset      The destination asset
 * @param {boolean} createIntermediates                 Should intermediates be created?
 * @param {boolean} overwriteExisting                   Should an existing asset in dest be overwritten
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
 */
export declare function moveAsset(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, srcAsset: PathOrIdAssetDesignator, destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>;
/**
 * Discards an asset
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        The http service
 * @param {PathOrIdAssetDesignator} asset          The asset to be discarded
 * @param {string} etag                                 The etag of the asset to be discarded
 * @param {boolean} recursive                           If asset has children, discard recursively?
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError>}
 */
export declare function discardAsset(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, asset: PathOrIdAssetDesignator, etag?: string, recursive?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
    success: boolean;
}, 'json'>, AdobeDCXError>;
/**
 * Delete an asset.
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        HTTPService to use.
 * @param {PathOrIdAssetDesignator} asset          Asset to delete.
 * @param {string} etag                                 Optional etag. If not provided will use "*", an unconditional delete.
 * @param {boolean} recursive                           Whether to delete the asset recursively. Required for directory assets.
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError>}
 */
export declare function deleteAsset(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, asset: PathOrIdAssetDesignator, etag?: string, recursive?: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
    success: boolean;
}, 'json'>, AdobeDCXError>;
/**
 * Restore a discarded asset
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc        HTTPService to use.
 * @param {IDBasedOperationSource} asset                The asset to restore
 * @param {Record<string, string>} additionalHeaders    Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>}
 */
export declare function restoreAsset(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, asset: IDBasedOperationSource, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeAsset, 'json'>, AdobeDCXError>;
/**
 * Package an asset (zip) and store at destination
 *
 * @this [OptionalLeafContext] Optional `this` scope that can provide an operations href.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc                                    HTTPService to use.
 * @param {PathOrIdAssetDesignator | PathOrIdAssetDesignator[]} sources   Source(s) to package
 * @param {PathOrIdAssetDesignator} destination                                Destination for zipped package
 * @param {boolean} createIntermediates                                             Should intermediates be created?
 * @param {boolean} overwriteExisting                                               Should an existing asset in dest be overwritten
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 *
 * @returns {AdobePromise<RepoResponseResult<{ success: boolean }, 'json'>, AdobeDCXError>}
 */
export declare function packageAssets(this: OptionalContext<AdobeOpsContext>, svc: AdobeHTTPService | ServiceConfig, sources: PathOrIdAssetDesignator | PathOrIdAssetDesignator[], destination: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting: boolean, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<{
    success: boolean;
}, 'json'>, AdobeDCXError>;
/**
 * Execute a POST on the ops href provided using the operation document as body.
 * Also split the response into an array of results, creating a DCX error for each failed operation.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc                                        HTTPService to use.
 * @param {string} opsEndpoint                                                          Operations endpoint
 * @param {AnyOperationDocument | AnyOperationDocument[] | string} operationDocument    Operations document
 * @param {Record<string, string>} additionalHeaders                                          Any additional headers to include in the request
 */
export declare function doBatchOperation(svc: AdobeHTTPService, opsEndpoint: string, operationDocument: AnyOperationDocument | AnyOperationDocument[] | string, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<AdobeOperationResult[], 'json'>, AdobeDCXError>;
/**
 * Execute a POST on the ops href provided using the operation document as body.
 *
 * @param {AdobeHTTPService | ServiceConfig} svc                                        HTTPService to use.
 * @param {string} opsEndpoint                                                          Operations endpoint
 * @param {AnyOperationDocument | AnyOperationDocument[] | string} operationDocument    Operations document
 * @param {Record<string, string>} additionalHeaders                                          Additional headers to include in the POST
 *
 * @returns {AdobePromise<AdobeResponse<'json'>, AdobeDCXError>}
 */
export declare function doOperation(svc: AdobeHTTPService, opsEndpoint: string, operationDocument: AnyOperationDocument | AnyOperationDocument[] | string, additionalHeaders?: Record<string, string>): AdobePromise<AdobeResponse<'json'>, AdobeDCXError>;
/**
 * OperationDocumentBuilder
 */
export declare class OperationDocumentBuilder {
    readonly opBatchLimit = 100;
    private _docs;
    private _batchSourceType;
    private _batchTargetType;
    getDocumentEntry<T extends AdobeOperation = AdobeOperation>(id: number): OperationDocTypeMap[T];
    getDocument(): AnyOperationDocument[];
    get entryCount(): number;
    copyResources(srcAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument, additionalHeaders?: Record<string, string>): OperationDocumentBuilder;
    copy(srcAsset: PathOrIdAssetDesignator, destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting?: boolean, addlFields?: Record<string, unknown>): OperationDocumentBuilder;
    move(srcAsset: PathOrIdAssetDesignator, destAsset: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting?: boolean, addlFields?: Record<string, unknown>): OperationDocumentBuilder;
    package(sources: PathOrIdAssetDesignator | PathOrIdAssetDesignator[], destination: PathOrIdAssetDesignator, createIntermediates: boolean, overwriteExisting?: boolean, addlFields?: Record<string, unknown>): OperationDocumentBuilder;
    discard(target: PathOrIdAssetDesignator, recursive: boolean, addlFields?: Record<string, unknown>): OperationDocumentBuilder;
    restore(target: IDBasedOperationSource, addlFields?: Record<string, unknown>): OperationDocumentBuilder;
    delete(target: PathOrIdAssetDesignator, recursive: boolean, addlFields?: Record<string, unknown>): OperationDocumentBuilder;
    private _assertUnderLimit;
    private _checkTargetType;
    private _checkSourceType;
    private _checkSourceOrTargetType;
    private _getSourceType;
}
export declare function newOperationDocBuilder(): OperationDocumentBuilder;
export declare function _copyResources(svc: AdobeHTTPService | ServiceConfig, sourceAsset: PathOrIdAssetDesignator, targetAsset: PathOrIdAssetDesignator, resources: CopyResourceDesignator[], intermediates?: boolean, manifestPatch?: JSONPatchDocument, additionalHeaders?: Record<string, string>): AdobePromise<RepoResponseResult<CopyResourcesOperationResult, 'json'>, AdobeDCXError>;
export {};
//# sourceMappingURL=operations.d.ts.map