/// <reference types="node" />

import { Duplex } from 'stream';
import type { IncomingMessage } from 'http';
import { Readable } from 'stream';
import { RequestOptions } from 'https';
import { Writable } from 'stream';

export declare const AccessPolicy = "http://ns.adobe.com/adobecloud/rel/ac/policy";

export declare type ACEInheritance = 'self' | 'deep';

export declare type ACEModifier = 'grant' | 'deny';

export declare type ACEPrivilege = 'ack' | 'read' | 'write' | 'modify' | 'delete' | 'full';

export declare interface ACPAccessControlEntry {
    'repo:principal': ACPPrincipal;
    'repo:modifier': ACEModifier;
    'repo:privileges': ACEPrivilege[];
    'repo:relations': LinkRelationKey[];
    'repo:inheritance': ACEInheritance;
}

export declare interface ACPAccessControlList {
    'repo:acl': ACPAccessControlEntry[];
}

export declare interface ACPCopyResourceDesignator {
    source: ACPResourceDesignator;
    target: Omit<ACPResourceDesignator, 'revision'> & {
        revision: string;
    };
}

export declare interface ACPIndexDocument {
    'repo:regions': string[];
    _links?: LinkSet;
    children?: any[];
}

declare type ACPPrincipal = CaseInsensitive<'all'> | CaseInsensitive<'authenticated'> | CaseInsensitive<'unauthenticated'> | {
    'xdm:provider': ACPXDMProvider;
    '@id': string;
    '@type': string;
};

export declare interface ACPProblemResponse {
    type: ACPProblemType;
    title: string;
    status: number;
}

export declare type ACPProblemType = 'http://ns.adobe.com/adobecloud/problem/responsetoolarge' | 'http://ns.adobe.com/adobecloud/problem/quotaexceeded' | 'http://ns.adobe.com/adobecloud/problem/dcxvalidation' | 'http://ns.adobe.com/adobecloud/problem/bulkrequestnotattempted';

export declare interface ACPRepoMetadataResource {
    'repo:repositoryId'?: string;
    'repo:etag'?: string;
    'repo:client-op-id'?: string;
    'repo:id'?: string;
    'repo:size'?: number;
    'repo:version'?: string;
    'repo:baseAssetId'?: string;
    'repo:assetId'?: string;
    'repo:assetClass'?: string;
    'repo:name'?: string;
    'repo:path'?: string;
    'dc:format'?: string;
    'repo:createdBy'?: string;
    'repo:createDate'?: string;
    'repo:modifyDate'?: string;
    'repo:modifiedBy'?: string;
    'repo:createdDate'?: string;
    'repo:lastModifiedDate'?: string;
    'repo:state'?: string;
    'storage:region'?: string;
    md5?: string;
    'repo:discardDate'?: string;
    'repo:discardedBy'?: string;
    'storage:deviceCreateDate'?: string;
    'storage:deviceModifyDate'?: string;
    'repo:defaultScheduledDeletionDuration'?: number;
    'repo:scheduledDeletionDate'?: string;
    'tiff:imageWidth'?: number;
    'tiff:imageLength'?: number;
    _embedded?: any;
    children?: any;
    _page?: any;
    _links?: any;
}

export declare interface ACPRepository {
    'repo:repositoryId': string;
    'repo:repositoryType': RepositoryType;
    'dc:title': any;
    'repo:createDate': string;
    'repo:owner': ACPRepositoryOwner;
    'repo:availableRegions': string[];
    _embedded: any;
}

export declare interface ACPRepositoryOwner {
    id: string;
    type: string;
}

export declare interface ACPResourceDesignator {
    reltype: LinkRelationKey;
    component_id?: string;
    revision?: string;
    etag?: string;
    content_md5?: string;
    content_length?: string;
}

/**
 * Transfer Documents are used to initialize, process, and complete block uploads.
 */
export declare interface ACPTransferDocument {
    'repo:size': number;
    'dc:format': string;
    'repo:reltype'?: LinkRelationKey;
    'repo:blocksize'?: number;
    'repo:md5'?: string;
    'repo:if-match'?: string;
    'repo:expires'?: string;
    'repo:resource'?: ACPResourceDesignator;
    component_id?: string;
    links?: {
        [LinkRelation.BLOCK_FINALIZE]: BasicLink;
        [LinkRelation.BLOCK_EXTEND]: BasicLink;
        [LinkRelation.BLOCK_TRANSFER]: BasicLink[];
    };
}

export declare interface ACPVersion {
    version?: string;
    created?: string;
    created_by?: string;
    _links?: LinkSet;
}

export declare interface ACPVersions {
    'repo:id'?: string;
    'repo:assetId'?: string;
    'repo:repositoryId'?: string;
    total_children?: number;
    children?: any;
    _page?: any;
    _links?: LinkSet;
}

export declare interface ACPXDMProvider {
    '@id': string;
}

export declare type AdditionalNodeOptions = RequestOptions & {
    authenticationAllowList?: string[];
    maxRedirects?: number;
};

export declare interface AdobeAsset {
    /**
     * The Repository ID of the Repository storing the Asset.
     */
    repositoryId?: string;
    /**
     * A unique identifier given to every addressable Asset in a given Repository.
     */
    assetId?: string;
    /**
     * ID of the root asset
     */
    baseAssetId?: string;
    /**
     * Identifier of the head version of the Asset. Not present for an Asset which does not yet have a version.
     */
    version?: string;
    /**
     * An Asset's location in the Directory hierarchy.
     */
    path?: string;
    /**
     * The name of the Asset.
     */
    name?: string;
    /**
     * The class of an Asset is inferred from its Media Type at the time the Asset is created.
     */
    assetClass?: string;
    /**
     * An ETag is an HTTP response header returned by an HTTP/1.1-compliant web server, used to determine change in content of a Resource at a given URL. This property is required if the Asset has a Primary Resource.
     */
    etag?: string;
    /**
     * The media type of the Resource.
     */
    format?: string;
    /**
     * The server date and time when the Resource was created in the Repository, such as when a File is first uploaded or a Directory is created by the server as the parent of a new Asset.
     */
    createDate?: string;
    /**
     * The server date and time when the Resource was last modified in the Repository, such as when a new version of an Asset is uploaded or a Directory's child Resource is added or removed.
     */
    modifyDate?: string;
    /**
     * A timestamp capturing the time at which this Asset was last modified on the client, and which may therefore precede the time at which the corresponding modification was uploaded to the Repository. (It can also be later, due to clock skew.) If not specified with any request that creates a new version of the Asset, it will be updated to repo:modifyDate.
     */
    deviceModifyDate?: string;
    /**
     * Time the asset was discarded directly or by inheritance. Does not exist for active assets.
     */
    discardDate?: string;
    /**
     * A timestamp capturing the time at which this Asset was created on the client, and which may therefore precede the time at which it was created in the Repository. (It can also be later, due to clock skew.) Can be set only at the time of Asset creation and, if not set, defaults to repo:createDate.
     */
    deviceCreateDate?: string;
    /**
     * The Asset's state, indicating whether the Asset is active, discarded, or deleted.
     */
    state?: string;
    /**
     * The ID of the user who initiated the action that caused the Resource to be created in the Repository
     */
    createdBy?: string;
    /**
     * The ID of the user who initiated the action that most recently caused the Resource to be modified in the Repository.
     */
    modifiedBy?: string;
    /**
     * Identifier of the user that discarded the asset directly or by inheritance. Does not exist for active assets.
     */
    discardedBy?: string;
    /**
     * The set of links associated with the Asset.
     */
    links?: LinkSet;
    /**
     * defaultScheduledDeletionDuration set at the time of directory creation
     */
    defaultScheduledDeletionDuration?: number;
    /**
     * scheduledDeletionDate set for non directory assets by the service
     */
    scheduledDeletionDate?: string;
    /**
     * The size of the Asset in bytes.
     */
    size?: number;
    width?: number;
    length?: number;
    renderable?: boolean;
    md5?: string;
    /**
     *  One of the ACP Content Regions, has an impact on the Links API. Can only be used with a service token for the LinksAPI.
     * @see {@link https://wiki.corp.adobe.com/pages/viewpage.action?pageId=2605664337#Proposal:LinksAPI(ResolveLite)-LinksAPI LinksAPI}
     */
    contentRegion?: string;
}

export declare type AdobeAssetEmbedded<T extends EmbeddableResource = never, U = {
    [K in EmbeddableResourceMap[T extends EmbeddableResourceString ? T : T extends EmbeddableResourceDesignator ? T['resource']['reltype'] : never]['returnKey']]: EmbeddableResourceUnmap[K];
}> = AdobeAsset & {
    embedded?: U;
};

/**
 * AdobeAsset requiring links
 */
export declare type AdobeAssetWithLinks = Partial<AdobeAsset> & {
    links: LinkSet;
};

export declare interface AdobeAuthProvider {
    /**
     * List of domains that are allowed to receive authentication headers.
     * Used for all outgoing requests and for redirects in Node.js.
     * For redirects in browser, the browser/XMLHTTPRequest handles the authentication header stripping.
     *
     * @default ["adobe.io"]
     */
    authenticationAllowList: string[];
    /**
     * Whether the AuthProvider is configured to skip authorization.
     * This is automatically set to true when no persistent auth callback
     * is set in the constructor.
     *
     * Can be manually set to the expected value after constructing.
     */
    isNoAuthMode: boolean;
    /**
     * Current API key.
     * To set, use setApiKey()
     */
    apiKey: string | undefined;
    /**
     * Current authorization token.
     * To set, use setAuthToken()
     */
    authToken: string | undefined;
    /**
     * Auth token scheme, used as the prefix for the Authorization header.
     * @default "Bearer"
     */
    authTokenScheme: string | undefined;
    /**
     * Attach listener to auth change event
     *
     * @param handler
     * @returns Function to remove handler
     */
    onChange(handler: AuthChangeHandler): DetachHandler;
    /**
     * Get authorization data as object
     * Example: { authToken: "abc", apiKey: "xyz"}
     */
    getAuthData(): AuthData;
    /**
     * Get authorization data wrapped in promise for chaining.
     * Does not do anything additional to getAuthData(), besides the promise.
     */
    getAuth(): Promise<AuthData>;
    /**
     * Set API key.
     *
     * @param apiKey The new API key
     */
    setApiKey(apiKey: string): void;
    /**
     * Set the current authentication token.
     * @note This triggers the "updated" event.
     *
     * @param authToken The new token
     */
    setAuthToken(authToken: string): void;
    /**
     * Explicitly set the state to "unauthenticated", triggering the
     * "unauthenticated" event, which in turn forces the persistent
     * auth refresh callback to be called (if any).
     *
     * Resolves to AuthData once the state becomes "updated".
     * If state is never changed, this promise will not resolve.
     */
    refreshAuth(): Promise<AuthData>;
    /**
     * Explicitly set the state to "updated", triggering the "updated" event.
     */
    resume(): void;
    /**
     * Remove current auth data, set state to "unauthenticated".
     */
    logout(): void;
    /**
     * Check if URL is allowed to receive authorization headers,
     * based on the authorizationAllowList.
     *
     * @param url URL to check
     */
    isAuthorizedURL(url: string): boolean;
    /**
     * Applies Authorization and X-Api-Key headers if the url is part of the allowed
     * and if the incoming headers do not contain a "null" value for their corresponding override.
     *
     * @note This should be used with normalized headers (ie. all lowercase keys) to
     * avoid the case where multiple values are set like { "X-Api-Key": null, "x-api-key": "abc" }
     *
     * @param headers Incoming headers
     * @returns Headers with auth data applied (lowercase keys)
     */
    applyAuthHeaders(url: string, headers: Record<string, string>): Record<string, string>;
}

export declare interface AdobeBlockDownload extends AdobeBlockTransfer {
    /**
     * data for the Block download.
     */
    buffer: Uint8Array;
    /**
     * Initialize the BlockUpload.
     * If the total size of the transfer is not known,
     * upload a single block and update sizes.
     *
     * @note
     * This method is called implicitly when calling `start` in a NOT_INITIALIZED state.
     */
    init(url?: string, totalSize?: number): IAdobePromise<AdobeBlockDownload>;
    /**
     * Start the download loop.
     */
    start(): IAdobePromise<AdobeBlockDownload>;
    /**
     * Pause download loop.
     */
    pause(): IAdobePromise<AdobeBlockDownload>;
}

export declare interface AdobeBlockTransfer {
    /**
     * The current state of the transfer.
     */
    state: BlockTransferState;
    /**
     * If defined, called on each block completion.
     */
    onProgress?: UploadProgressCallback;
    /**
     * Initialize the transfer.
     */
    init(): IAdobePromise<AdobeBlockTransfer>;
    /**
     * Start the transfer loop.
     */
    start(): IAdobePromise<AdobeBlockTransfer>;
    /**
     * Pause the transfer loop.
     */
    pause(): IAdobePromise<AdobeBlockTransfer>;
    /**
     * Resume the transfer loop from a paused state.
     */
    resume(): AdobeBlockTransfer;
    /**
     * Cancel the transfer.
     *
     * @note
     * A canceled transfer cannot be resumed.
     */
    cancel(): void;
}

export declare interface AdobeBlockUpload extends AdobeBlockTransfer {
    createdAsset?: AdobeAsset;
    uploadRecord?: AdobeComponentUploadRecord;
    finalizeResponse?: AdobeResponse;
    init(): IAdobePromise<AdobeBlockUpload, AdobeDCXError>;
    start(): IAdobePromise<AdobeBlockUpload, AdobeDCXError>;
    pause(): IAdobePromise<AdobeBlockUpload, AdobeDCXError>;
    uploadNextBlock(data: Buffer | Blob): IAdobePromise<AdobeResponse, AdobeDCXError>;
}

export declare interface AdobeComponentUploadRecord extends AdobeUploadRecord {
    location: string;
    revision: string;
    links: LinkSet;
}

export declare interface AdobeDCXBranch {
    originalManifestFormatVersion: number;
    compositeHref: string;
    /**
     * The id of the composite. Must be a unique among the nodes of the composite.
     *
     * <p>While not strictly read-only most clients do not ever have to modify this property.</p>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeId: string;
    /**
     * An AdobeDCXNode object that represents the root of the underlying manifest.
     * @type {AdobeDCXNode}
     * @memberof AdobeDCXBranch#
     * @readonly
     */
    rootNode: AdobeDCXNode;
    /**
     * The name of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    name?: string;
    /**
     * The type of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    type: string;
    /**
     * The asset id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeAssetId?: string;
    /**
     * The repository id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeRepositoryId?: string;
    /**
     * The etag of the manifest of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     * @readonly
     */
    manifestEtag?: string;
    /**
     * The editing state of the composite. Can be <em>'unmodified'</em>, <em>'modified'</em>,
     * <em>'pendingDelete'</em> or <em>'committedDelete'</em>.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeState: CompositeState;
    /**
     * The archival state of the composite. Can be <em>'active'</em> (default), <em>'pending'</em>,
     * or <em>'archived'</em>.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeArchivalState: CompositeArchivalState;
    /**
     * Whether the composite is bound to a composite on a server. If <em>false</em> for newly
     * created empty composites that have never been pulled from or pushed to the server.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Boolean}
     */
    isBound: boolean;
    /**
     * The version identifier of the banch.
     * Only valid if the branch has been pushed or pulled, undefined otherwise.
     *
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {string}
     */
    versionId?: string;
    /**
     * Whether the composite has has been modified in memory and
     * needs to be committed to local storage.
     *
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Boolean}
     */
    isDirty: boolean;
    data: AdobeDCXRootNodeData;
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    localData: string;
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    remoteData: string;
    /**
     * Lists any elements (type AdobeDCXElement) that have been created and have not yet
     * been used to update the corresponding child node or been abandoned.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Array<AdobeDCXElement>}
     */
    pendingElements: AdobeDCXElement[];
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    changeCount: number;
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    readOnly: boolean;
    /**
     * Restores the branch from its stringified representation (manifest). Throws if data is invalid.
     * @param   {String} data The stringified representation.
     * @returns {AdobeDCXBranch}.
     */
    parse(data: string): AdobeDCXBranch;
    /**
     * Returns the element with the given id or undefined if not found.
     * Throws if the node in question is not a valid element).
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @param   {String } id The id of the child node to look up.
     * @returns {AdobeDCXElement}
     */
    getElementWithId(id: string): AdobeDCXElement | undefined;
    /**
     * Returns the element with the given absolute path or undefined if not found.
     * Throws if the node in question is not a valid element).
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @memberof AdobeDCXBranch#
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXElement}
     */
    getElementWithAbsolutePath(path: string): AdobeDCXElement | undefined;
    /**
     * Creates a new element node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new element.
     * Throws if the path is invalid.
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @param   {String}                name            - The name of the new element.
     * @param   {String}                type            - The type of the new element.
     * @param   {String}                path            - The path of the new element.
     * @param   {String}                [nodeId]        - The id of the new child. If undefined the new child node will
     *                                                    get a random id.
     * @param   {Integer}               [index]         - If given and less than or equal to the current number of
     *                                                    children than the node gets inserted at the given index.
     *                                                    Otherwise it gets added to the end.
     * @param   {@link AdobeDCXNode}    [parentNode]    - The parent node to add the node to. Default parent is the
         *                                                    root node.
         * @returns {AdobeDCXElement}
         */
     addElement(name: string, type: string, path: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXElement;
     /**
      * Updates the data of the existing element in the branch.
      * Notice that the element object will no longer be tracked by the branch after updateElement
      * has been called. You will have to request the element again using getElementWithId or
      * getElementWithAbsolutePath if you want to keep working with it.
      * Throws if the element doesn't exist or if the update results in duplicate paths/ids.
      *
      * @param {AdobeDCXElement} element The modified element.
      * @returns {AdobeDCXNode} The updated node in the branch.
      */
     updateElement(element: AdobeDCXElement): AdobeDCXNode;
     /**
      * Cleans up an element that was previously created and has not been used to update the corresponding
      * node in the branch. Throws if the branch doesn't recognize the element (e.g. calling abandonElement
      * twice or after already having called updateElement).
      *
      * @param {AdobeDCXElement} element The element that is no longer needed.
      */
     abandonElement(element: AdobeDCXElement): void;
     /**
      * Returns the node with the given id or undefined if not found.
      * @param   {String}     id The id of the child node to look up.
      * @returns {AdobeDCXNode}
      */
     getChildWithId(id: string): AdobeDCXNode | undefined;
     /**
      * Returns the node with the given absolute path or undefined if not found.
      * @param   {String} path The absolute path.
      * @returns {AdobeDCXNode}
      */
     getChildWithAbsolutePath(path: string): AdobeDCXNode | undefined;
     /**
      * Generates and returns an array of the child nodes of the given parent node.
      * @example
      * var rootNodes = branch.getChildrenOf(branch.rootNode);
      * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
      * @returns {Array}
      */
     getChildrenOf(parentNode?: AdobeDCXNode): AdobeDCXNode[];
     /**
      * Creates a new node and inserts it into the children list of the given parent node or of the
      * root if no parent node is given.
      * Returns the new child node.
      * @param   {String}  [name]       The name of the new child. If undefined the child will not
      *                                 have a name.
      * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
      *                                 get a random id.
      * @param   {Integer} [index]      If given and less than or equal to the current number of
      *                                 children than the node gets inserted at the given index.
      *                                 Otherwise it gets added to the end.
      * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
      *                                 root node.
      * @returns {AdobeDCXNode}
      */
     addChild(name?: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;
     /**
      * Removes and returns the given child node from the branch.
      * @param   {AdobeDCXNode} node The child node to remove.
      * @returns {AdobeDCXNode}
      */
     removeChild(node: AdobeDCXNode): AdobeDCXNode;
     /**
      * Moves the existing child from its current parent/index to the given parent/index.
      * @param   {AdobeDCXNode} node         The child node to move
      * @param   {Integer}      [index]        If given and less than or equal to the current number of
      *                                      children than the node gets inserted at the given index.
      *                                      Otherwise it gets added to the end.
      * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
      *                                      the root.
      * @returns {AdobeDCXNode}
      */
     moveChild(node: AdobeDCXNode, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;
     /**
      * Copies the given child node as a new node into this branch. The node can be from the same or
      * from a different composite.
      *
      * <p>This function will try reuse the ids of any children and components of the copied node,
      * in order to minimize the amount of data that will later have to be uploaded, however, clients
      * must not rely on these ids being preserved in the copied objects.</p>
      *
      * <p>Fails if a node with the same id or same absolute path already exists.</p>
      *
      * <p>Notice: This method does not work without local storage (e.g. browser environment) if
      * used to copy between two composites stored at different endpoints.</p>
      *
      * @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
      *                                      newPath must be provided.
      * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
      *                                      then the new child node will be added to the root of the
      *                                      branch.
      * @param   {Integer}      [index]      If provided and less than or equal to the current number of
      *                                      children of the parentNode (or root) the child node gets
      *                                      inserted at the given index. Otherwise it gets added to
      *                                      the end.
      * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
      *                                      this a its path property and it will also receive a new
      *                                      random id (unless one is provided with the newId param).
      *                                      If left undefined then the copy of the node will keep
      *                                      the path of the original. In either case the function will
      *                                      fail if the resulting absolute path of the child or any
      *                                      of its children/components conflicts with an already
      *                                      existing absolute path.</p>
      *                                      <p>You must provide a newPath if you are copying the root
      *                                      node of a branch or element.</p>
      * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
      *                                      this a its id. If left undefined (and the newPath param
      *                                      is also undefined) then the copy will retain the id of
      *                                      the original. In either case the function will
      *                                      fail if the resulting id of the child or any
      *                                      of its children/components conflicts with an already
      *                                      existing id.
      * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
      *                                      when not using local storage.
      *                                      Gets called when the copy is done or has failed.
      * @returns {AdobeDCXNode}              Only returns the created child node if no callback is
      *                                      given.
      */
     copyChild(node: AdobeDCXNode, parentNode?: AdobeDCXNode, index?: number, newPath?: string, newId?: string, callback?: NodeCallback): AdobeDCXNode;
     /**
      * <p>Replaces the child node in this branch with a copy of the given node or branch with the same
      * id (or, if provided, with the given id). Fails if the child node does not exist in this branch.</p>
      *
      * <p>This function will try reuse the ids of any children and components of the copied node,
      * in order to minimize the amount of data that will later have to be uploaded, however, clients
      * must not rely on these ids being preserved in the copied objects.</p>
      *
      * <p>Notice: This method does not work without local storage (e.g. browser environment) if
      * used to copy between two different composites.</p>
      *
      * @param   {AdobeDCXNode} node       The child node to update from.
      * @param   {String}       [newPath]  If provided the copy of the component will be assigned
      *                                    this a its path property. Otherwise it will retain its original path.
      * @param   {String}       [newId]    If provided the copy of the child node will be assigned
      *                                    this as its id. Otherwise it will retain its original id.
      * @param   {NodeCallback} [callback] Optional when not copying between different composites or
      *                                    when not using local storage.
      *                                    Gets called when the copy is done or has failed.
      * @returns {AdobeDCXNode}            Only returns the created child node if no callback is
      *                                    given.
      */
     replaceChild(node: AdobeDCXNode, newPath?: string, newId?: string, callback?: NodeCallback): AdobeDCXNode;
     /**
      * Returns an array of all components in the branch.
      * @returns {Array}.
      */
     allComponents(): AdobeDCXComponent[];
     /**
      * Returns the component with the given id or undefined if not found.
      * @param   {String} id The id of the component to look up.
      * @returns {AdobeDCXComponent}
      */
     getComponentWithId(id: string): AdobeDCXComponent | undefined;
     /**
      * Returns the component with the given absolute path or undefined if not found.
      * @param   {String} path The absolute path of the desired component.
      * @returns {AdobeDCXComponent}
      */
     getComponentWithAbsolutePath(path: string): AdobeDCXComponent | undefined;
     /**
      * Returns an array containing the components of the given node.
      * @param   {AdobeDCXNode} parentNode The node whose components to return.
      * @returns {Array}
      */
     getComponentsOf(parentNode?: AdobeDCXNode): AdobeDCXComponent[];
     /**
      * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
      * @param error An IMCOMPLETE_COMPOSITE error
      * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
      */
     getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[];
     /**
      * <p>Creates and adds a component to the given parent node or to the root if no parent node is
      * given.</p>
      *
      * @param componentDescriptor The serialized component descriptor to use.
      * @param name          The name of the new component.
      * @param path          The path of the new component. Must satisfy uniquenes
      * @param relationship  The relationship of the new component.
      *                                            rules for components.
      * @param parentNode  The node to add the node to. Defaults to the root.
      * @returns {AdobeDCXComponent}               The new component.
      */
     addComponentWithComponentDescriptor(componentDescriptor: string, name: string, path: string, relationship?: LinkRelationKey, parentNode?: AdobeDCXNode): any;
     /**
      * <p>Creates and adds a component to the given parent node or to the root if no parent node is
      * given.</p>
      *
      * @param   {String}            name          The name of the new component.
      * @param   {String}            relationship  The relationship of the new component.
      * @param   {String}            path          The path of the new component. Must satisfy uniquenes
      *                                            rules for components.
      * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
      * @param   {Object}            uploadResults The upload results object returned by a previous call
      *                                            to AdobeDCXCompositeXfer.uploadAssetForNewComponent().
      * @returns {AdobeDCXComponent}               The new component.
      */
     addComponentWithUploadResults(name: string, relationship: string | undefined, path: string, parentNode: AdobeDCXNode | undefined, uploadResults: AdobeUploadResults): AdobeDCXComponent;
     /**
      * <strong>XHR-only</strong>
      *
      * <p>Updates the component record with the results of a recent upload of said component.</p>
      *
      * @param   {AdobeDCXComponent} component     The component.
      * @param   {Object}            uploadResults The upload results object returned by a previous
      *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
      * @returns {AdobeDCXComponent} The updated component.
      */
     updateComponentWithUploadResults(component: AdobeDCXComponent, uploadResults: AdobeUploadResults): AdobeDCXComponent;
     /**
      * Removes the component from the branch.
      * @param   {AdobeDCXComponent} component The component to remove.
      * @returns {AdobeDCXComponent} The removed component.
      */
     removeComponent(component: AdobeDCXComponent): AdobeDCXComponent;
     /**
      * Moves the component to the given node or the root if node is undefined
      * @param   {AdobeDCXComponent} component    The component to move.
      * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
      * @returns {AdobeDCXComponent} The moved component.
      */
     moveComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode): AdobeDCXComponent;
     /**
      * Copies the given component and adds it as a new component to this branch. Fails if the
      * component already exists.
      *
      * <p>Notice: This method does not work without local storage (browser environment) if
      * used to copy between two composites with different endpoints.</p>
      *
      * @param   {AdobeDCXComponent} component    The component to copy.
      * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
      *                                           provided then the component will be added to the
      *                                           root.
      * @param   {String}            [newPath]    If provided the copy of the component will be
      *                                           assigned this a its path property and it will also
      *                                           get assigned a random new id if none is provided via
      *                                           the <em>newId</em> param.
      * @param   {String}            [newId]      If provided the copy of the component will be assigned
      *                                           this a its id. If left undefined (and if newPath is
      *                                           undefined as well) then the copy of the component
      *                                           will retain the id of the original.
      *                                           This is useful when merging conflicting changes since
      *                                           it preserves the identity of components and avoids
      *                                           unnecessary network traffic.
      * @param   {ComponentCallback} [callback]   Optional when not copying between different
      *                                           composites or when copying without local storage.
      *                                           Gets called when the copy is done or has failed.
      * @returns {AdobeDCXComponent}              Only returns the new component when called without
      *                                           a callback.
      */
     copyComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode, newPath?: string, newId?: string): AdobeDCXComponent;
     copyComponent(component: AdobeDCXComponent, parentNode: AdobeDCXNode | undefined, newPath: string | undefined, newId: string | undefined, callback: ComponentCallback): void;
     /**
      * Replaces the matching component (same id) in this branch with a copy of the given component.
      * Fails if the component can't be found.
      *
      * <p>Notice: This method does not work without local storage (browser environment) if
      * used to copy between two different composites.</p>
      *
      * @param   {AdobeDCXComponent} component  The component to copy.
      * @param   {String}            [newPath]  If provided the copy of the component will be
      *                                         assigned this a its path property and it will also
      *                                         get assigned a new id if none is provided via the
      *                                         newId param.
      * @param   {String}            [newId]    If provided the copy of the component will be
      *                                         assigned this a its id property.
      * @param   {ComponentCallback} [callback] Optional when not copying between different
      *                                         composites. Gets called when the copy is done or has
      *                                         failed.
      * @returns {AdobeDCXComponent}            Only returns the new component when called without
      *                                         a callback.
      */
     replaceComponent(component: AdobeDCXComponent, newPath?: string, newId?: string): AdobeDCXComponent;
     replaceComponent(component: AdobeDCXComponent, newPath: string | undefined, newId: string | undefined, callback: ComponentCallback): void;
    }

    export declare type AdobeDCXBranchCore = any;

    export declare interface AdobeDCXComponent {
        /**
         * The id of the component. Must be a unique among the components of the composite.
         *
         * <p>Cannot be changed for a component that is part of a branch.</p>
         */
        id: string;
        /**
         * The asset id of the component in cloud storage.
         */
        assetId?: string;
        /**
         * The repository id of the component in cloud storage.
         */
        repositoryId?: string;
        /**
         * The name of the component.
         */
        name?: string;
        /**
         * The type of the component.
         */
        type: string;
        /**
         * The path of the component.
         */
        path: string;
        /**
         * The relationship of the component to its parent node and its sibling components.
         */
        relationship?: string;
        /**
         * The version of the component.
         */
        version?: string;
        /**
         * The edit state of the component.
         */
        state: string;
        /**
         * The etag of the component.
         */
        etag?: string;
        /**
         * The md5 of the component.
         */
        md5?: string;
        /**
         * The composite id of the owning composite.
         */
        readonly compositeId?: string;
        /**
         * The owner of the composite.
         */
        readonly owner?: AdobeDCXBranch | AdobeDCXElement;
        /**
         * The absolute path of the parent of the component.
         */
        readonly parentPath?: string;
        /**
         * The absolute path of the component.
         */
        readonly absolutePath?: string;
        /**
         * The content length of the component.
         */
        length?: number;
        /**
         * The width property of the component.
         */
        width?: number;
        /**
         * The height property of the component.
         */
        height?: number;
        /**
         * Returns an array of custom keys that are present at this component. Custom keys are manifest
         * properties that are not considered standard DCX properties.
         * @returns {Array}
         */
        getCustomKeys(): string[];
        /**
         * Returns the object or value for the given custom key.
         * @param   {String} key The custom key to look up.
         * @returns {*}
         */
        getValue<T = any>(key: string): T;
        /**
         * Sets the object or value for the given custom key.
         * @param {String} key   The custom key to set the value for.
         * @param {*}      value The value to set.
         */
        setValue<T = any>(key: string, value: T): void;
        /**
         * Removes the object or value for the given custom key.
         * @param {String} key The custom key to remove.
         */
        removeValue(key: string): void;
        /**
         * Returns the link with the given relationship as a JS object or undefined if the component
         * doesn't have such a link.
         * @param   {String} relationship The relationship of the link to the component.
         * @returns {Object} The link object with the given relationship as a JS object or undefined if
         *                   the component doesn't have such a link.
         */
        getLink(relationship: string): HalLink | undefined;
        /**
         * Removes the link with the given relationship.
         * @param {String} relationship The relationship of the link to the component.
         *
         */
        removeLink(relationship: string): void;
        /**
         * Sets the link with the given relationship to the given object.
         * @param {Object} link         A JS object representing the link.
         * @param {String} relationship The relationship of the link to the component.
         */
        setLink(link: HalLink, relationship: string): void;
        /**
         * Returns false if any of the properties of the given component is different from the properties
         * of this component. Recurses both data structures.
         *
         * @example
         * if (!component.isEqualTo(otherComponent, { unimportantProperty: true })) {
         *      // The components have different properties
         * }
         * @param   {AdobeDCXComponent} component            The component to compare with.
         * @param   {Object}            [propertiesToIgnore] An object having the properties that should
         *                                                 not be compared.
         * @returns {Boolean}
         */
        isEqualTo(component: AdobeDCXComponent, propertiesToIgnore?: Record<string, boolean>): boolean;
        /**
         * Get a serialized component descriptor.
         * @note Only works for RAPI. The Component must be attached to a branch.
         *
         * ```ts
         * const componentDescriptor = component.getComponentDescriptor();
         * ```
         * @returns {string}
         */
        getComponentDescriptor(): string;
    }

    export declare interface AdobeDCXComponentData extends AdobeDCXNodeData {
        /**
         * The etag of the component.
         *
         * <p>
         * Note: a (non-weak) entity tag value includes the leading and trailing double-quotes,
         * which therefore must be escaped in order to appear properly in the property value.
         * </p>
         */
        etag?: string;
        /**
         * The version of the last-synchronized version of this component.
         *
         * <p>
         * Note: In practice, in manifests version 5 and below the version property can be a string or a number.
         * </p>
         */
        version?: string;
        /**
         * The md5 hash of the last-synchronized version of this component.
         */
        md5?: string;
        /**
         * The length (size in bytes) of this component.
         */
        length?: number;
        /**
         * Used to translate between physical dimensions and pixel dimensions for an image.
         * Applications that convert this value to pixels per inch (by multiplying by 0.0254 meters per inch)
         * should round the resultant value to the nearest tenth of a pixel per inch. Conversely,
         * applications converting from pixels per inch should round to the nearest integer pixelsPerMeter.
         * This ensures a correct round trip between these two common units (e.g., 72.0 ppi -> "pixelsPerMeter":2835 -> 72.0 ppi).
         */
        pixelsPerMeter?: number;
    }

    export declare interface AdobeDCXComposite extends AdobeAsset {
        id?: string;
        href?: string;
        type?: string;
        assetId?: string;
        repositoryId?: string;
        collaborationType?: string;
        clientDataString?: string;
        readonly isBound: boolean;
        readonly current?: AdobeDCXBranch;
        /**
         * Loads the base manifest and instantiates a read-only branch if the manifest exists.
         *
         * <p>This method can only be called if the AdobeDCXComposite instance has been initialized with the
         * <em>xhrBaseBranchSupport</em> option.</p>
         *
         * @param {BranchCallback} [callback] Gets called on success or failure.
         * @returns                           Only returns a promise if no callback specified.
         */
        loadBaseBranch(): IAdobePromise<AdobeDCXBranch, AdobeDCXError>;
        loadBaseBranch(callback: BranchCallback): void;
        /**
         * Accepts the given merged branch as the new current branch, discards any temporary pull data
         * and also updates the base manifest. Is a no-op if no branch is provided and if there
         * is no pulled manifest in local storage (as may be the case if a pull didn't find any
         * modification on the server).
         *
         * @example
         * // NJS
         * pullComposite(composite, session, function (error, pulledBranch) {
         *      if (!error) {
         *          // The pull has succeeded. Now you need to incorporate the changes from the server
         *          // into your current branch.
         *          // Start the merge by making a mutable copy of pulledBranch
         *          var mergedBranch = pulledBranch.copy();
         *          // Now merge any changes you have made to current since the last successful push or
         *          // pull. You might want to also load the base branch to help you figure out what has
         *          // changed in either branch.
         *          ...
         *          // When done merging you need to make this call:
         *          composite.resolvePullWithBranch(mergedBranch, function (error, newCurrent) {
         *              if (!error) {
         *                  // Now you are done!
         *              }
         *          });
         *      }
         * });
         *
         * @example
         * // XHR
         * pullCompositeManifestOnly(composite, session, function (error, pulledBranch) {
         *      if (!error) {
         *          // The pull has succeeded. Now you need to incorporate the changes from the server
         *          // into your current branch. If you follow the recommendations for working in an
         *          // online environment you will not have modified current so you can just call:
         *          composite.resolvePullWithBranch(pulledBranch);
         *      }
         * });
         *
         *
         * <p><strong>The branch passed in must be derived from the pulled branch.</strong></p>
         *
         * @param {AdobeDCXBranch} branch     A branch which should become the new current branch. It
         *                                    should be the result of merging the previously pulled
         *                                    branch with any changes in current.
         * @param {BranchCallback} [callback] Optional if running in an <strong>XHR</strong>
         *                                    environment. Gets called on completion.
         * @return {AdobeDCXBranch}           Only returns the new current branch if called without
         *                                    a callback.
         */
        resolvePullWithBranch(branch: AdobeDCXBranch): void;
        resolvePullWithBranch(branch: AdobeDCXBranch, callback: BranchCallback): void;
        acceptPush(callback?: (err?: Error) => unknown): void | AdobeDCXBranch;
        resetBinding(): void;
        resetIdentity(): void;
    }

    export declare interface AdobeDCXElement {
        /**
         * An AdobeDCXNode object that represents the root of the underlying manifest.
         * @memberof AdobeDCXElement#
         * @type {AdobeDCXNode}
         * @readonly
         */
        rootNode: AdobeDCXNode;
        /**
         * The name of the composite.
         * @memberof AdobeDCXElement#
         * @type {String}
         */
        name?: string;
        /**
         * The type of the element.
         * @memberof AdobeDCXElement#
         * @type {String}
         */
        type: string;
        /**
         * The asset id of the composite that can be used to pull and push the composite.
         * @memberof AdobeDCXElement#
         * @type {String}
         * @readonly
         */
        compositeAssetId?: string;
        /**
         * The repository id of the composite that can be used to pull and push the composite.
         * @memberof AdobeDCXElement#
         * @type {String}
         * @readonly
         */
        compositeRepositoryId?: string;
        /**
         * Whether the composite has has been modified in memory and needs to be committed to local
         * storage.
         * @memberof AdobeDCXElement#
         * @readonly
         * @type {Boolean}
         */
        isDirty: boolean;
        /**
         * @memberof AdobeDCXElement#
         * @internal
         * @readonly
         */
        changeCount: number;
        /**
         * Returns the node with the given id or undefined if not found.
         * @param   {String} id The id of the child node to look up.
         * @returns {AdobeDCXNode}
         */
        getChildWithId(id: string): AdobeDCXNode | undefined;
        /**
         * Returns the node with the given absolute path or undefined if not found.
         * @param   {String} path The absolute path.
         * @returns {AdobeDCXNode}
         */
        getChildWithAbsolutePath(path: string): AdobeDCXNode | undefined;
        /**
         * Generates and returns an array of the child nodes of the given parent node.
         * @example
         * var rootNodes = element.getChildrenOf(element.rootNode);
         * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
         * @returns {Array}
         */
        getChildrenOf(parentNode: AdobeDCXNode): AdobeDCXNode[];
        /**
         * Creates a new node and inserts it into the children list of the given parent node or of the
         * root if no parent node is given.
         * Returns the new child node.
         * @param   {String}  [name]       The name of the new child. If undefined the child will not
         *                                 have a name.
         * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
         *                                 get a random id.
         * @param   {Integer} [index]      If given and less than or equal to the current number of
         *                                 children than the node gets inserted at the given index.
         *                                 Otherwise it gets added to the end.
         * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
         *                                 root node.
         * @returns {AdobeDCXNode}
         */
        addChild(name: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;
        /**
         * Removes and returns the given child node from the element.
         * @param   {AdobeDCXNode} node The child node to remove.
         * @returns {AdobeDCXNode}
         */
        removeChild(node: AdobeDCXNode): AdobeDCXNode;
        /**
         * Moves the existing child from its current parent/index to the given parent/index.
         * @param   {AdobeDCXNode} node         The child node to move
         * @param   {Integer}      index        If given and less than or equal to the current number of
         *                                      children than the node gets inserted at the given index.
         *                                      Otherwise it gets added to the end.
         * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
         *                                      the root.
         * @returns {AdobeDCXNode}
         */
        moveChild(node: AdobeDCXNode, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;
        /**
         * Copies the given child node as a new node into this element. The node can be from the same or
         * from a different composite.
         *
         * <p>This function will try reuse the ids of any children and components of the copied node,
         * in order to minimize the amount of data that will later have to be uploaded, however, clients
         * must not rely on these ids being preserved in the copied objects.</p>
         *
         * <p>Fails if a node with the same id or same absolute path already exists.</p>
         *
         * <p>Notice: This method does not work without local storage (e.g. browser environment) if
         * used to copy between two composites stored at different endpoints.</p>
         *
         * @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
         *                                      newPath must be provided.
         * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
         *                                      then the new child node will be added to the root of the
         *                                      branch.
         * @param   {Integer}      [index]      If provided and less than or equal to the current number of
         *                                      children of the parentNode (or root) the child node gets
         *                                      inserted at the given index. Otherwise it gets added to
         *                                      the end.
         * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
         *                                      this a its path property and it will also receive a new
         *                                      random id (unless one is provided with the newId param).
         *                                      If left undefined then the copy of the node will keep
         *                                      the path of the original. In either case the function will
         *                                      fail if the resulting absolute path of the child or any
         *                                      of its children/components conflicts with an already
         *                                      existing absolute path.</p>
         *                                      <p>You must provide a newPath if you are copying the root
         *                                      node of a branch or element.</p>
         * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
         *                                      this a its id. If left undefined (and the newPath param
         *                                      is also undefined) then the copy will retain the id of
         *                                      the original. In either case the function will
         *                                      fail if the resulting id of the child or any
         *                                      of its children/components conflicts with an already
         *                                      existing id.
         * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
         *                                      when not using local storage.
         *                                      Gets called when the copy is done or has failed.
         * @returns {AdobeDCXNode}              Only returns the created child node if no callback is
         *                                      given.
         */
        copyChild(node: AdobeDCXNode, parentNode?: AdobeDCXNode, index?: number, newPath?: string, newId?: string): AdobeDCXNode;
        copyChild(node: AdobeDCXNode, parentNode: AdobeDCXNode | undefined, index: number | undefined, newPath: string | undefined, newId: string | undefined, callback: NodeCallback): void;
        /**
         * Returns an array of all components in the element.
         * @returns {Array}.
         */
        allComponents(): AdobeDCXComponent[];
        /**
         * Returns the component with the given id or undefined if not found.
         * @param   {String} id The id of the component to look up.
         * @returns {AdobeDCXComponent}
         */
        getComponentWithId(id: string): AdobeDCXComponent | undefined;
        /**
         * Returns the component with the given absolute path or undefined if not found.
         * @param   {String} path The absolute path of the desired component.
         * @returns {AdobeDCXComponent}
         */
        getComponentWithAbsolutePath(path: string): AdobeDCXComponent | undefined;
        /**
         * Returns an array containing the components of the given node.
         * @param   {AdobeDCXNode} parentNode The node whose components to return.
         * @returns {Array<AdobeDCXComponent>}
         */
        getComponentsOf(parentNode: AdobeDCXNode): AdobeDCXComponent[];
        /**
         * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
         * @param error An IMCOMPLETE_COMPOSITE error
         * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
         */
        getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[];
        /**
         * <p>Creates and adds a component to the given parent node or to the root if no parent node is
         * given.</p>
         *
         * @param componentDescriptor The serialized component descriptor to use.
         * @param name          The name of the new component.
         * @param path          The path of the new component. Must satisfy uniquenes
         * @param relationship  The relationship of the new component.
         *                                            rules for components.
         * @param parentNode  The node to add the node to. Defaults to the root.
         * @returns {AdobeDCXComponent}               The new component.
         */
        addComponentWithComponentDescriptor(componentDescriptor: string, name: string, path: string, relationship?: LinkRelationKey, parentNode?: AdobeDCXNode): any;
        /**
         * <p>Creates and adds a component to the given parent node or to the root if no parent node is
         * given.</p>
         *
         * @param   {String}            name          The name of the new component.
         * @param   {String}            relationship  The relationship of the new component.
         * @param   {String}            path          The path of the new component. Must satisfy uniquenes
         *                                            rules for components.
         * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
         * @param   {Object}            uploadResults The upload results object returned by a previous call
         *                                            to AdobeDCXCompositeXfer.uploadAssetForComponent().
         * @returns {AdobeDCXComponent}               The new component.
         */
        addComponentWithUploadResults(name: string, relationship: string, path: string, parentNode?: AdobeDCXNode, uploadResults?: AdobeUploadResults): AdobeDCXComponent;
        /**
         * <p>Updates the component record with the results of a recent upload of said component.</p>
         *
         * @param   {AdobeDCXComponent} component     The component.
         * @param   {Object}            uploadResults The upload results object returned by a previous
         *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
         * @returns {AdobeDCXComponent} The updated component.
         */
        updateComponentWithUploadResults(component: AdobeDCXComponent, uploadResults: any): AdobeDCXComponent;
        /**
         * Removes the component from the element.
         * @param   {AdobeDCXComponent} component The component to remove.
         * @returns {AdobeDCXComponent} The removed component.
         */
        removeComponent(component: AdobeDCXComponent): AdobeDCXComponent;
        /**
         * Moves the component to the given node or the root if node is undefined
         * @param   {AdobeDCXComponent} component    The component to move.
         * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
         * @returns {AdobeDCXComponent} The moved component.
         */
        moveComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode): AdobeDCXComponent;
        /**
         * Copies the given component and adds it as a new component to this branch. Fails if the
         * component already exists.
         *
         * <p>Notice: This method does not work without local storage (browser environment) if
         * used to copy between two composites with different endpoints.</p>
         *
         * @param   {AdobeDCXComponent} component    The component to copy.
         * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
         *                                           provided then the component will be added to the
         *                                           root.
         * @param   {String}            [newPath]    If provided the copy of the component will be
         *                                           assigned this a its path property and it will also
         *                                           get assigned a random new id if none is provided via
         *                                           the <em>newId</em> param.
         * @param   {String}            [newId]      If provided the copy of the component will be assigned
         *                                           this a its id. If left undefined (and if newPath is
         *                                           undefined as well) then the copy of the component
         *                                           will retain the id of the original.
         *                                           This is useful when merging conflicting changes since
         *                                           it preserves the identity of components and avoids
         *                                           unnecessary network traffic.
         * @param   {ComponentCallback} [callback]   Optional when not copying between different
         *                                           composites or when copying without local storage.
         *                                           Gets called when the copy is done or has failed.
         * @returns {AdobeDCXComponent}              Only returns the new component when called without
         *                                           a callback.
         */
        copyComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode, newPath?: string, newId?: string): AdobeDCXComponent;
        copyComponent(component: AdobeDCXComponent, parentNode: AdobeDCXNode | undefined, newPath: string | undefined, newId: string | undefined, callback: ComponentCallback): void;
    }

    export declare interface AdobeDCXError<T = unknown> extends Error {
        readonly name: 'AdobeDCXError';
        message: string;
        code: string;
        stack?: string;
        underlyingError?: AdobeDCXError | Error;
        additionalData: T;
        failedComponents?: FailedComponent[];
        problemType?: string;
        response?: AdobeResponse;
    }

    export declare interface AdobeDCXNode {
        /**
         * The id of the node. Must be a unique among the nodes of the composite.
         * <p>Cannot be changed for a node that is part of a branch or element.</p>
         * @memberof AdobeDCXNode#
         * @type {String}
         */
        id: string;
        /**
         * The name of the node.
         *
         * @memberof AdobeDCXNode#
         * @type {String}
         */
        name?: string;
        /**
         * The type of the node.
         *
         * @memberof AdobeDCXNode#
         * @type {String}
         */
        type: string;
        /**
         * The path property of the node.
         *
         * @memberof AdobeDCXNode#
         * @type {String}
         */
        path?: string;
        /**
         * The relationship of the node to its parent node.
         *
         * @memberof AdobeDCXNode#
         * @type {String}
         */
        relationship?: string;
        readonly owner?: AdobeDCXBranch | AdobeDCXElement;
        /**
         * The absolute path of the parent of the node.
         *
         * @memberof AdobeDCXNode#
         * @readonly
         * @type {String}
         */
        readonly parentPath?: string;
        /**
         * The absolute path of the node.
         *
         * @memberof AdobeDCXNode#
         * @readonly
         * @type {String}
         */
        readonly absolutePath?: string;
        /**
         * Whether this node is the root of its branch or element.
         *
         * @memberof AdobeDCXNode#
         * @readonly
         * @type {Boolean}
         */
        readonly isRoot: boolean;
        /**
         * Returns an array of non-standard keys that are present at this node.
         * @returns {Array} An array of all non-standard property keys.
         */
        getCustomKeys(): string[];
        /**
         * Returns the object or value for the given key.
         * @param   {String} key The custom key to look up.
         * @returns {*} The value or object for the key.
         */
        getValue<T = any>(key: string): T;
        /**
         * Returns the object or value for the given key.
         * @param {String}   key   The custom key to set the value for.
         * @param {*} value The value or object.
         */
        setValue<T = any>(key: string, value: T): void;
        /**
         * Removes the object or value for the given key.
         * @param {String} key The custom key to remove.
         */
        removeValue(key: string): void;
        /**
         * Creates and returns a deep copy of the node sans any of its components or children.
         * @returns {AdobeDCXNode} The copy.
         */
        /**
         * Removes the link with the given relationship.
         * @param {String} relationship The relationship of the link to the node.
         *
         */
        removeLink(relationship: string): void;
        /**
         * Sets the link with the given relationship to the given object.
         * @param {Object} link         A JS object representing the link.
         * @param {String} relationship The relationship of the link to the node.
         */
        setLink(link: HalLink, relationship: string): void;
        /**
         * Returns the link with the given relationship as a JS object or undefined if the node
         * doesn't have such a link.
         * @param   {String} relationship The relationship of the link to the node.
         * @returns {Object} The link with the given relationship as a JS object or undefined if
         *                   the node doesn't have such a link.
         */
        getLink(relationship: string): HalLink | undefined;
        /**
         * Returns false if any of the properties of the given node is different from the properties
         * of this node. Recurses both data structures.
         * @param   {AdobeDCXNode} node                          The node to compare with.
         * @param   {Array}        [nodePropertiesToIgnore]      Optional. An object having the properties
         *                                                       that should not be compared for node.
         * @param   {Array}        [componentPropertiesToIgnore] Optional. An object having the properties
         *                                                       that should not be compared for components.
         * @returns {Boolean}
         */
        isEqualTo(node: AdobeDCXNode, nodePropertiesToIgnore?: Record<string, boolean>, componentPropertiesToIgnore?: Record<string, boolean>): any;
    }

    export declare interface AdobeDCXNodeData {
        /**
         * The id of a node should be set to a UUID when the node is created, and should never be changed.
         * Node IDs, being stable, are suitable for use in conflict resolution;
         * they allow reliable detection of when a node has been deleted or moved.
         */
        id: string;
        /**
         * The name of a node is intended to be used for display purposes only.
         * Names do not play any role in paths or references.
         * The name is required for a node that is an element; see Composition and Embedding.
         */
        name?: string;
        /**
         * The type property records the media type of the node.
         * When the value of this property ends in "+dcx", the node is an element; see Composition and Embedding.
         */
        type: string;
        /**
         * Since a composite format (type) can evolve over time, there is an optional numeric property to represent
         * the version of that type (if not specified, the value is understood to be zero).
         */
        typeVersion?: number;
        /**
         * Paths are used when resolving references.
         */
        path?: string;
        /**
         * The rel property records the relationship of this node to its parent node.
         * Relationships should be selected from the set of Common Link Relation Types defined for CC.
         */
        rel?: string;
        /**
         * The width of the component, as appropriate for that type of component (e.g., integer pixels for a PNG).
         *
         * <p>
         * Note: Elements with an intrinsic 2D size should record their width and height in the element's root node.
         * This is primarily used with renditions.
         * Along with the type property, this allows clients to find the best-suited rendition
         * (format and size) directly from the manifest itself.
         * </p>
         */
        width?: number;
        /**
         * The height of the component, as appropriate for that type of component (e.g., integer pixels for a PNG).
         *
         * <p>
         * Note: Elements with an intrinsic 2D size should record their width and height in the element's root node.
         * This is primarily used with renditions.
         * Along with the type property, this allows clients to find the best-suited rendition
         * (format and size) directly from the manifest itself.
         * </p>
         */
        height?: number;
        /**
         * Whether or not this composite has been modified since last synchronized.
         *
         * @enum {string}
         */
        state: AdobeDCXState;
        /**
         * Most composites have some internal structure, which is encoded as a tree.
         * This attribute optionally contains the list of children of the root node.
         * @see https://git.corp.adobe.com/caf/xcm/blob/master/docs/dcx.md#structure
         */
        children?: AdobeDCXNodeData[];
        /**
         * Composites are composed of components, which are listed in this array.
         * @see https://git.corp.adobe.com/caf/xcm/blob/master/docs/dcx.md#components
         */
        components?: AdobeDCXComponentData[];
        /**
         * A JSON HAL object that records related links, such as an original source location.
         *
         * @see http://tools.ietf.org/html/draft-kelly-json-hal-06
         */
        _links?: {
            [rel: string]: HalLink;
        };
    }

    export declare interface AdobeDCXRootNodeData extends AdobeDCXNodeData {
        /**
         * The creation date of the entire composite (not just the manifest).
         * It's an RFC 3339 timestamp.
         * @example
         * ```
         * 1996-12-19T16:39:57-08:00
         * ```
         */
        created?: string;
        /**
         * The modification date of the entire composite (not just the manifest).
         * It's an RFC 3339 timestamp.
         * @example
         * ```
         * 1996-12-19T16:39:57-08:00
         * ```
         */
        modified?: string;
        /**
         * An array of strings, intended to be assigned by the user, and used for organization or search purposes.
         */
        tags?: string[];
        /**
         * The current version of the manifest syntax, as recorded by the framework implementation upon writing
         * the manifest. Clients need never consult this value, but framework implementations may take it into
         * account when reading manifests.
         */
        'manifest-format-version': number;
        /**
         * The units for the coordinate system (if any) of this composite.
         * Can be used in a DCX Element to override the default units of the enclosing composite (e.g., DCX2D).
         * Valid values are: (pixel, point, inch, mm, cm)
         *
         * @enum {string}
         */
        units?: AdobeDCXUnit;
    }

    export declare type AdobeDCXState = 'unmodified' | 'modified' | 'pendingDelete' | 'committedDelete';

    export declare type AdobeDCXUnit = 'pixel' | 'point' | 'inch' | 'mm' | 'cm';

    /**
     * Interface for the refactored HTTPService.
     * For now the interface is defined in src/HTTPService.ts
     */
    export declare interface AdobeHTTPService {
        /**
         * An inactive service will error out with a AdobeDCXError.SERVICE_IS_INACTIVE error
         * whenever a request is being made. Clients can set isActive to false when the user
         * has explicitly logged out.
         * @default true
         */
        isActive: boolean;
        /**
         * Whether the platform handles redirects natively.
         * Returns true for browser and false for Node,
         * although Node builds may also be configured to handle redirects.
         */
        handlesRedirects: boolean;
        /**
         * If true then XHR instances will have the withCredentials property set which means that user
         * credentials (stored in a cookie) will be used in cross origin requests.
         */
        crossOriginCredentials: boolean;
        /**
         * The maximum number of outstanding requests over this HTTP Service instance.
         * The default value is 5, to stay below browser connection limits, but the limit
         * can be increased if you have special circumstances that warrant it.
         * Remember that there is no use in exceeding the bandwith capacity of your client
         * or of the host, and that in error cases it's better to have half of your
         * uploads complete than all of your uploads half complete.
         */
        maxOutstanding: number;
        server?: string;
        /**
         * Feature flags are intended to be temporary in nature.
         * These values are expected to be able to be enabled/disabled for testing/release purposes.
         */
        featureFlags: any;
        /**
         * Used for redirects & outgoing requests.
         *
         * Sets the authenticationAllowList on AdobeAuthProvider.
         */
        authenticationAllowList: string[];
        /**
         * The AdobeAuthProvider instance.
         * If an instance is not provided at construction, one will be created.
         */
        authProvider: AdobeAuthProvider;
        /**
         * Sets additional headers.
         * @param {Object} additionalHeaders An object containing key-value pairs for each additional header to supply.
         */
        setAdditionalHeaders(hdrs: Record<string, string>): void;
        /**
         * Sets additional options.
         * @param {Object} additionalOptions An object containing additional options to supply with each http request (in Node only).
         */
        setAdditionalNodeOptions(options: AdditionalNodeOptions): void;
        /**
         * Sets the timeout value for requests.
         * This is the per-request timeout, not related to backoff timeouts.
         * To set the backoff timeout, provide a value in the `retryOptions`
         * property of the invoke options.
         * @param {Integer} timeout timeout value in ms or undefined if no timeout
         */
        setTimeout(timeout?: number): void;
        /**
         * Sets the authentication token and resumes the service.
         * @param {String} token The authentication token, if not provided, logs out
         */
        setAuthToken(token?: string): void;
        /**
         * Sets the API key.
         * @param {String} apiKey The apikey to supply with each request.
         */
        setApiKey(key?: string): void;
        /**
         * Sets a function that denotes whether a Promise should resolve or reject given a status
         * By default, this is function that always returns true meaning any status code will resolve.
         * @param {(status?: number) => boolean} fn
         */
        setValidateStatus(fn?: (status?: number) => boolean): void;
        /**
         * Set options for retrying failed requests
         * @param {BackoffOptions<any>} retryOptions
         */
        setRetryOptions(retryOptions: BackoffOptions<never>): void;
        /**
         * Sets callbacks before and after invocation hook.
         * (For testing)
         * @private
         * @param before a hook to call before each http request is issued. Signature: before(req).
         * @param after  a hook to be called after each http request is initiated: Signature: after(req, xhr).
         */
        setRequestHooks(before?: PreRequestHook, after?: PostRequestHook): void;
        /**
         * Call this when you do not use an auth token for authentication and you need to let the
         * service know that you have renewed the authentication cookie so that the service can
         * resume to make requests.
         */
        resume(): void;
        invoke<T extends AdobeResponseType = any>(method: HTTPMethod, href: string, headers?: Record<string, string>, body?: BodyType, options?: InvokeOptions<T>): IAdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>;
        /**
         * Issue a request.
         * By default, the request is a GET.
         * The body param specifies the data, if any.
         * The generic T denotes the responseType for type completion/checking.
         *
         * Note: use EITHER callback or promise, using both may lead to undefined resolution.
         *
         * @example
         * ```js
         * // with callback
         * service.invoke<SomeInterface>('GET', 'https://server.url/path', {}, null,
         *  { responseType: 'json' },
         *  (err, xhr, data) => {
         *    // do stuff
         *    // data is typeof === 'object' if response was valid JSON
         *    // data is Type `SomeInterface`
         *  });
         * ```
         *
         * * @example
         * ```js
         * // with promise, async/await
         * try {
         *   const [xhr, data] = await service.invoke<SomeInterface>(
         *      'GET',
         *      'https://server.url/path',
         *      {},
         *      null,
         *      { responseType: 'json' });
         *   // data is typeof === 'object'
         *   // data is Type `SomeInterface`
         * } catch(e) {
         *   // handle network error
         * }
         * ```
         * @param  {String}            method      - HTTP method
         * @param  {String}            href        - The URL; may be relative or absolute.
         *                                          If relative, the `server` property will be prepended (if exists)
         *                                          If absolute (contains endpoint & protocol), will be sent as-is
         * @param  {Record<string, string | string[]>} headers
         *                                         - The request headers
         * @param  {BodyType}          [body]      - The request body (Buffer/ArrayBufferView/Blob: data)
         * @param  {InvokeOptions}     [options]   - Object capturing additional request options.
         * @param  {ResponseCallback}  callback    - Called when the response arrives. Signature: callback(error, xhr)
         * @return {AdobePromise<[AdobeResponse<T>, T], AdobeDCXError, RequestDescriptor>}  -
         *                                          An AdobePromise with extended props of a RequestDescriptor.
         *                                          Resolves to a tuple of [AdobeResponse, Data] where data is the type of
         *                                          the `responseType` option. For type completion, use the generic `T` with
         *                                          the expected responseType.
         */
        invoke<T extends AdobeResponseType = any>(method: HTTPMethod, href: string, headers?: Record<string, string>, body?: BodyType, options?: InvokeOptions<T>, callback?: ResponseCallback<T>): IAdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>;
    }

    /**
     * Options for initializing a new HTTPService
     */
    export declare interface AdobeHTTPServiceOptions {
        maxOutstanding?: number;
        useAuthProvider?: boolean;
        timeout?: number;
        crossOriginCredentials?: boolean;
        server?: string;
        preferFetch?: boolean;
        requestIdPrefix?: string;
    }

    /**
     * Minimal data required to use an asset with API calls.
     * Needs to have one of:
     *  1. Links
     *  2. Asset ID and Repo ID
     *  3. Path and Repo ID
     */
    export declare type AdobeMinimalAsset = AssetWithPathOrId | AdobeAssetWithLinks;

    declare type AdobeRepoAPISession = any;

    export declare interface AdobeRepoMetadata {
        repositoryId: string;
        assetId: string;
        assetClass: string;
        name: string;
        path: string;
        format: string;
        created: Date;
        createdBy?: string;
        modified: Date;
        modifiedBy?: string;
        etag?: string;
        size?: number;
        state: string;
        links: LinkSet;
        defaultScheduledDeletionDuration?: number;
        scheduledDeletionDate?: string;
        width?: number;
        length?: number;
    }

    export declare interface AdobeRepoOwner {
        id: string;
        type: string;
    }

    export declare interface AdobeRepository {
        repositoryId: string;
        repositoryType: string;
        availableRegions: string[];
        createDate?: string;
        title?: any;
        owner?: AdobeRepoOwner;
        links?: LinkSet;
    }

    export declare interface AdobeRepoUploadResult {
        /**
         * Whether block upload was used.
         * Defines what the other conditional properties hold.
         */
        isBlockUpload: boolean;
        /**
         * Response from the upload, or the response from the
         * block upload finalize call if block upload was used.
         */
        response: AdobeResponse<'json'>;
        /**
         * The uploaded data asset or component data.
         */
        result: Partial<AdobeComponentUploadRecord>;
        /**
         * The block upload instance, if block upload was used.
         */
        blockUpload?: AdobeBlockUpload;
        /**
         * The asset which was used for the upload.
         * May contain both repositoryId and assetId, OR links, OR all three,
         * depending on the asset used as an argument.
         */
        asset: {
            repositoryId?: string;
            assetId?: string;
            links?: LinkSet;
        };
    }

    export declare interface AdobeRequest<T extends AdobeResponseType> {
        id: string;
        descriptor: RequestDescriptor;
        getPromise(): Promise<AdobeResponse<T>>;
        cancel: (err?: AdobeDCXError) => void;
    }

    export declare interface AdobeResponse<T extends AdobeResponseType = any> {
        statusCode: number;
        headers: Record<string, string>;
        responseType?: T;
        response: ResponseTypeMap[T];
        name?: string;
        message: string;
        code?: string;
        requestDesc?: RequestDescriptor;
        xhr?: AdobeXhr<T>;
    }

    export declare type AdobeResponseType = keyof ResponseTypeMap;

    /*************************************************************************
     *
     * ADOBE CONFIDENTIAL
     * ___________________
     *
     * @license
     * Copyright 2020 Adobe Inc.
     * All Rights Reserved.
     *
     * NOTICE:  All information contained herein is, and remains
     * the property of Adobe Inc. and its suppliers,
     * if any.  The intellectual and technical concepts contained
     * herein are proprietary to Adobe Inc. and its
     * suppliers and are protected by trade secret or copyright law.
     * Dissemination of this information or reproduction of this material
     * is strictly forbidden unless prior written permission is obtained
     * from Adobe Inc.
     **************************************************************************/
    /**
     * The base session interface which other sesssions extend
     */
    export declare interface AdobeSessionBase {
        authenticationAllowList: string[];
    }

    /**
     * Holds the results of an individual successful component upload.
     * Describes a single component, which in context of an UploadResults instance,
     * can be used to fully describe the component.
     */
    export declare interface AdobeUploadRecord {
        id: string;
        etag: string;
        version: string;
        md5: string;
        length: number;
        type: string;
    }

    export declare interface AdobeUploadResults {
        /**
         * Key-value pairs, mapping componentId => UploadRecord
         */
        records: Record<string, AdobeUploadRecord>;
        /**
         * Composite ID (UUID)
         */
        compositeId?: string;
        /**
         * Composite cloud asset ID (URN)
         */
        compositeAssetId: string;
        /**
         * Repository ID
         *
         * Must be defined for RAPI DCXComposites
         */
        repositoryId?: string;
        /**
         * Add an UploadRecord for componentId to the results.
         *
         * @param componentId
         * @param record
         */
        addUploadRecord(componentId: string, record: AdobeUploadRecord): void;
        /**
         * Get a serialized component descriptor for a specific component in UploadResults.
         *
         * @note Only works for RAPI. The UploadResults object must contain a repositoryId.
         *
         * @param {string}                  componentId     - The component ID to serialize
         * @returns {string}
         */
        getComponentDescriptor(componentId: string): string;
        /**
         * Get a URL that can be used to download the component.
         * If size is over threshold allowed for single download, return presigned direct-download URL.
         *
         * @note Only works for RAPI. The UploadResults object must contain a repositoryId.
         *
         * @param {AdobeRepoAPISession}     session         - Repo session to use
         * @param {string}                  componentId     - Component ID to get the URL for
         */
        getComponentURL(session: AdobeRepoAPISession, componentId: string): IAdobePromise<{
            isPresignedUrl: boolean;
            url: string;
            response: AdobeResponse<'json'> | undefined;
        }, AdobeDCXError>;
        /**
         * Get the component data, using either single download or parallelized block download.
         *
         * @note Only works for RAPI. The UploadResults object must contain a repositoryId.
         *
         * @param {AdobeRepoAPISession}     session         - Repo session to use
         * @param {string}                  componentId     - Component ID to get data for
         * @param {AdobeResponseType}       [responseType]  - Response type for the data, defaults to Buffer (Node) or ArrayViewBuffer (Browser).
         */
        getComponent<T extends AdobeResponseType>(session: AdobeRepoAPISession, componentId: string, responseType?: T): RepoDownloadStreamableReturn<T>;
    }

    export declare interface AdobeXhr<T extends AdobeResponseType = AdobeResponseType> {
        href: string;
        method: HTTPMethod;
        body?: BodyType;
        headers: Record<string, string | string[]>;
        xhr: XMLHttpRequest | AdobeXhrNode;
        getStatus(): number | undefined;
        isSent(): boolean;
        abort(): void;
        isAborted(): boolean;
        isError(): boolean;
        onProgress(handler: ProgressListener): () => void;
        getResponseData(): ResponseTypeMap[T] | undefined;
        getResponseHeader(header: string): string | undefined;
        getAllResponseHeaders(): Record<string, string | string[]>;
        getResponse(): AdobeResponse<T> | undefined;
        getResponseDataAsJSON(): Promise<any>;
        getErrorCode(): number | string;
        send(url: string, method: HTTPMethod, body?: BodyType, headers?: Record<string, string | string[]>, responseType?: AdobeResponseType, options?: XhrOptions<T>): Promise<AdobeXhr<T>>;
    }

    export declare type AdobeXhrNode = XMLHttpRequest & {
        setNodeOptions: (opts: AdditionalNodeOptions) => void;
    };

    export declare type AnyObject = {
        [s: string]: any;
    };

    /**
     * Asset Designator
     * See Resource Designator without component link relation
     */
    declare type AssetResourceDesignator = {
        reltype: Omit<LinkRelationKey, LinkRelation.COMPONENT>;
    };

    /**
     * AdobeAsset requiring either `path` or `assetId`
     */
    export declare type AssetWithPathOrId = RequireAtLeastOne<AdobeAsset, 'assetId' | 'path'>;

    /**
     * AdobeAsset requiring either ()`repositoryId` & `path`) or `assetId`
     */
    export declare type AssetWithRepoAndPathOrId = AdobeAsset & (Required<Pick<AdobeAsset, 'path' | 'repositoryId'>> | Required<Pick<AdobeAsset, 'assetId'>>);

    export declare type AuthChangeHandler = (event: AuthEvent, provider?: AdobeAuthProvider) => void | Promise<void>;

    /*************************************************************************
     *
     * ADOBE CONFIDENTIAL
     * ___________________
     *
     * @license
     * Copyright 2020 Adobe Inc.
     * All Rights Reserved.
     *
     * NOTICE:  All information contained herein is, and remains
     * the property of Adobe Inc. and its suppliers,
     * if any.  The intellectual and technical concepts contained
     * herein are proprietary to Adobe Inc. and its
     * suppliers and are protected by trade secret or copyright law.
     * Dissemination of this information or reproduction of this material
     * is strictly forbidden unless prior written permission is obtained
     * from Adobe Inc.
     **************************************************************************/
    /**
     * AuthProvider related types
     */
    export declare interface AuthData {
        authToken?: string;
        apiKey?: string;
    }

    /**
     * Gets passed into createHTTPService() and called back whenever the service needs a new
     * authentication token.
     * @callback AuthenticationCallback@callback AuthenticationCallback
     *    @param {AdobeHTTPService | AdobeAuthProvider} serviceOrProvider Service callback use is deprecated,
     *                                                  use {@link AdobeHTTPServiceOptions} `useAuthProvider` set to true.
     *                                                  The service-based callback will be removed in the future.
     */
    export declare type AuthenticationCallback = (serviceOrProvider?: AdobeHTTPService | AdobeAuthProvider) => void | Promise<void>;

    export declare type AuthEvent = 'unauthenticated' | 'updated';

    export declare interface Backoff<T extends AdobeResponseType> {
        getPromise: () => Promise<AdobeXhr<T>>;
        cancel: () => void;
        onProgress: (h: (p1: number, p2: number) => void) => void;
        getSnapshot(): BackoffSnapshot<T>;
    }

    export declare type BackoffIncreaser = (count?: number, last?: number, initial?: number) => number;

    export declare interface BackoffOptions<T extends AdobeResponseType> extends XhrOptions<T> {
        /**
         * Response type of the request
         */
        responseType?: AdobeResponseType;
        /**
         * Authentication callback, if defined used for fetching new token.
         *
         * @note if the URL is not on the authentication allow list, the promise
         * will reject to DCXError#UNAUTHORIZED
         *
         */
        authCallback?: (url: string, headers: Record<string, string>) => Promise<Record<string, string>>;
        /**
         * Array of progress listeners
         */
        progressListeners?: ((sor: number, t: number) => void)[];
        /**
         * An array of numbers or regular expressions to check the status code against
         * to determine whether to retry. If the list contains the status code or one
         * of the regular expressions matches, the request will be retried.
         *
         * @default [/^(?!^501$|^507$)^(5\d{2})$|429$/]
         */
        retryCodes?: (number | RegExp)[];
        /**
         * A total amount of time to wait in milliseconds for all combined request waits.
         * Does not factor in the duration of the request, only the duration of the waits between retries.
         *
         * @default 72000
         */
        timeoutAfter?: number;
        /**
         * A function that returns the next duration in milliseconds to wait before retrying a request.
         *
         * @default ExponentialBackoff
         */
        increase?: BackoffIncreaser;
        /**
         * Whether to attempt to use the failure response's `Retry-After` header.
         * Handles both Date or seconds directive, depending on header.
         * If no retry-after header is found, will default to using the increaser function.
         *
         * @default true
         * @see https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Retry-After#directives
         */
        preferRetryAfterHeader?: boolean;
        /**
         * Number of milliseconds to wait before attempting the first retry.
         *
         * @default 2000
         */
        initialWait?: number;
        /**
         * Number of milliseconds at which to stop increasing the wait duration.
         * If the increaser function is customized, this will not be honored and
         * it is up to clients to cap their increaser function.
         *
         * @default 32000
         */
        maxWait?: number;
        /**
         * Disable all retries entirely
         *
         * @default false
         */
        disableRetry?: boolean;
        /**
         * Retry upon network error
         *
         * @default true
         */
        retryNetworkError?: boolean;
        /**
         * @internal
         * @private
         */
        forceXhr?: any;
        /**
         * Called when the request is about to be scheduled for some duration in the future.
         */
        preScheduleCallback?: ScheduleCallback<T>;
        /**
         * Called immediately after the request has been scheduled for retry.
         */
        postScheduleCallback?: ScheduleCallback<T>;
        /**
         * Called immediately before the retry request is sent.
         */
        preCallback?: PrePostBackoffCallback<T>;
        /**
         * Called immediately on response of the retry request.
         */
        postCallback?: PrePostBackoffCallback<T>;
        /**
         * Response header to check for a URL to poll, when a specific response code is encountered.
         * Used only if polling is enabled for the response code encountered.
         */
        pollHeader?: string;
        /**
         * HTTP verb to use for polling, if it is encountered.
         *
         * @default "GET"
         */
        pollMethod?: HTTPMethod;
        /**
         * HTTP response codes to consider poll-able
         * If the response from the initial HTTPService#invoke call matches one of these,
         * and if `pollHeader` is exists, and the response headers contains a value for that key,
         * perform polling on that header value.
         *
         * @default []
         * @see BackoffOptions#retryCodes
         */
        pollCodes?: (number | RegExp)[];
        /** @internal */
        descriptor?: RequestDescriptor;
    }

    export declare interface BackoffSnapshot<T extends AdobeResponseType> {
        count: number;
        canceled: boolean;
        timedOut: boolean;
        requests: AdobeXhr<T>[];
        totalWaited: number;
        duration: number;
        requestPending: boolean;
        waitingFor: number;
    }

    export declare type BasicLink = HalLink & {
        mode?: LinkMode;
    };

    declare interface BindingProperties {
        etag?: string;
        version?: string;
        md5?: string;
        [k: string]: unknown;
    }

    /**
     * Response body format from api/:download
     */
    export declare interface BlockDownloadBody {
        /**
         * Presigned URL to download asset.
         * Allows use of Range requests
         */
        href: string;
        /**
         * Total size of the asset, in bytes.
         */
        size: number;
        /**
         * Media type of the asset.
         */
        type: string;
    }

    export declare type BlockTransferDocument = ACPTransferDocument;

    export declare type BlockTransferState = 'NOT_INITIALIZED' | 'INITIALIZING' | 'INITIALIZED' | 'WAITING' | 'STARTED' | 'PAUSING' | 'PAUSED' | 'CANCELED' | 'ERROR' | 'FINALIZING' | 'COMPLETE';

    export declare type BodyType = ArrayBufferView | ArrayBuffer | Buffer | Uint8Array | Blob | string;

    export declare type BranchCallback = (error?: AdobeDCXError, branch?: AdobeDCXBranch) => unknown;

    declare type CaseInsensitive<T extends string> = Uppercase<T> | Lowercase<T> | Capitalize<T> | Uncapitalize<T>;

    declare type Children = (Element_2 | Node_2)[];

    declare interface CommonProperties {
        id: Id;
        name?: Name;
        _links?: HALResource;
        [k: string]: unknown;
    }

    declare type Component = CommonProperties & ComponentProperties & BindingProperties & MetadataProperties & CustomProperties;

    export declare type ComponentCallback = (error?: Error, component?: AdobeDCXComponent) => unknown;

    declare interface ComponentProperties {
        type?: NonDcxType;
        path?: Path;
        state?: State;
        length?: Length;
        rel?: Rel;
        [k: string]: unknown;
    }

    /**
     * Resource designator for a component.
     * In some cases revision is required, others optional,
     * those should be validated at runtime.
     */
    export declare interface ComponentResourceDesignator {
        reltype: 'http://ns.adobe.com/adobecloud/rel/component';
        component_id: string;
        revision?: string;
    }

    declare type Components = Component[];

    /**
     * this is used for getCompositeComponentsUrlsForUpload
     */
    export declare type ComponentUploadInfo = {
        /**
         * the content type of the component
         */
        contentType: string;
        /**
         * the size of the component
         */
        size: number;
        /**
         * the id of the component
         */
        componentId: string;
    };

    /**
     * The parameters for a component upload request.
     * The HTTP request parameters without body
     */
    declare type ComponentUploadRequestParameters = Omit<HTTPRequestParameters, 'body'>;

    /**
     * The archival state of the composite. Can be <em>'active'</em> (default), <em>'pending'</em>,
     * or <em>'archived'</em>.
     * @type {String}
     */
    export declare type CompositeArchivalState = 'pending' | 'active' | 'archived';

    /**
     * The editing state of the composite. Can be <em>'unmodified'</em>, <em>'modified'</em>,
     * <em>'pendingDelete'</em> or <em>'committedDelete'</em>.
     * @type {String}
     */
    export declare type CompositeState = 'unmodified' | 'modified' | 'pendingDelete' | 'committedDelete';

    declare type CopyRelTypeForSource = `${CopyRelTypeForTarget | LinkRelation.EFFECTIVE_PRIVILAGES | LinkRelation.REPO_METADATA | LinkRelation.RENDITION | LinkRelation.VERSION_HISTORY}`;

    /**
     * see {@link | https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html#eligible-resources}
     */
    declare type CopyRelTypeForTarget = `${LinkRelation.PRIMARY | LinkRelation.ACL_POLICY | LinkRelation.COMPONENT | LinkRelation.MANIFEST | LinkRelation.APP_METADATA | LinkRelation.EMBEDDED_METADATA}`;

    /**
     * COPY_RESOURCES API Resource Designator
     * See {@link | https://git.corp.adobe.com/pages/caf/api-spec/chapters/operations/copying_resources.html#specifying-resources}
     */
    export declare type CopyResourceDesignator = {
        source: {
            reltype: CopyRelTypeForSource;
        } | Required<ComponentResourceDesignator>;
        target: {
            reltype: CopyRelTypeForTarget;
        } | Omit<ComponentResourceDesignator, 'revision'>;
    } | ResourceDesignator;

    /**
     * The date and time the resource was created. For a digital file, this need not match a file-system creation time. For a freshly created resource, it should be close to that time, modulo the time taken to write the file. Later file transfer, copying, and so on, can make the file-system time arbitrarily different.
     */
    declare type CreateDate = string;

    declare interface CustomProperties {
        /**
         * This interface was referenced by `CustomProperties`'s JSON-Schema definition
         * via the `patternProperty` "^(?=.{3,1024}$)[\x01-\x7F]+#{1}[\x01-\x7F]+$".
         */
        [k: string]: string | {
            [k: string]: unknown;
        } | unknown[] | number | boolean | null;
    }

    declare type DcxType = Format & {
        [k: string]: unknown;
    };

    /**
     * The parsed object representation of a ComponentDescriptor serialized object.
     */
    export declare interface DeserializedComponentDescriptor {
        /**
         * Serialization version
         */
        versionId: string;
        /**
         * Component MIME type
         */
        type: string;
        /**
         * Composite asset ID (URN)
         */
        cloudAssetId: string;
        /**
         * Composite ID (UUID)
         */
        compositeId?: string;
        /**
         * Component ID (UUID)
         */
        componentId: string;
        /**
         * Component revision
         */
        componentRevisionId: string;
        /**
         * Composite repository ID
         */
        repositoryId?: string;
        /**
         * Value of component data hash, in format hashType
         */
        hashValue?: string;
        /**
         * Hash format for hashValue
         */
        hashType?: string;
        /**
         * Component etag
         */
        etag?: string;
        /**
         * Expiration, used by other DCX libs
         */
        cloudExpiration?: string;
        /**
         * Component data size
         */
        size?: number;
    }

    export declare type DetachHandler = () => void;

    export declare type EffectivePrivileges = PartialRecord<LinkRelationKey | '*', Privilege[]>;

    declare type Element_2 = NodeElementProperties & {
        type: DcxType;
        children?: Children;
        components?: Components;
        [k: string]: unknown;
    };

    export declare type EmbeddableResource = EmbeddableResourceDesignator | EmbeddableResourceString;

    declare type EmbeddableResourceDesignator = {
        resource: {
            reltype: 'http://ns.adobe.com/adobecloud/rel/metadata/application' | 'http://ns.adobe.com/adobecloud/rel/ac/effective';
        };
        selectors?: string[];
    };

    export declare interface EmbeddableResourceMap {
        'http://ns.adobe.com/adobecloud/rel/repository': {
            returnKey: 'RepositoryResource';
            returnValue: AdobeRepository;
        };
        'http://ns.adobe.com/adobecloud/rel/ac/effective': {
            returnKey: 'EffectivePrivileges';
            returnValue: EffectivePrivileges;
        };
        'http://ns.adobe.com/adobecloud/rel/metadata/application': {
            returnKey: 'AppMetadata';
            returnValue: Record<string, unknown> | undefined;
        };
    }

    declare type EmbeddableResourceString = 'http://ns.adobe.com/adobecloud/rel/repository' | 'http://ns.adobe.com/adobecloud/rel/ac/effective' | 'http://ns.adobe.com/adobecloud/rel/metadata/application';

    declare interface EmbeddableResourceUnmap {
        RepositoryResource: Partial<AdobeRepository>;
        EffectivePrivileges: EffectivePrivileges;
        AppMetadata: string | Record<string, unknown> | undefined;
    }

    export declare type EnumMap<C, T> = {
        [P in keyof C]: T;
    };

    export declare interface FailedComponent {
        error: AdobeDCXError;
        component: AdobeDCXComponent;
    }

    /**
     * The parameters for a finalize block upload request. body is required.
     */
    declare type FinalizeBlockUploadRequestParameters = HTTPRequestParameters & {
        readonly body: string;
    };

    /**
     * The file format, physical medium, or dimensions of the resource. Recommended best practice is to use a controlled vocabulary such as the list of Internet Media Types [MIME].
     */
    declare type Format = string;

    /**
     * Get slice callback type
     * @param startByte         Where the buffer slice should start.
     * @param endByte           Where the buffer slice should end (not inclusive)
     */
    export declare type GetSliceCallback<T extends SliceableData = SliceableData> = (startByte: number, endByte: number) => Promise<T>;

    declare interface Hal {
        /**
         * It is an object whose property names are link relation types (as defined by [RFC5988](https://tools.ietf.org/html/rfc5988) and values are either a Link Object or an array of Link Objects.  The subject resource of these links is the Resource Object of which the containing `_links` object is a property.
         */
        _links?: Record<string, any>;
        /**
         * It is an object whose property names are link relation types (as defined by [RFC5988](https://tools.ietf.org/html/rfc5988)) and values are either a Resource Object or an array of Resource Objects.\n\nEmbedded Resources MAY be a full, partial, or inconsistent version of the representation served from the target URI.
         */
        _embedded?: Record<string, any>;
        [k: string]: unknown;
    }

    /**
     * HAL-Forms property
     *
     * This type describes a single field in HAL-Forms
     */
    export declare interface HalFormsProperty {
        /**
         * Name of the form field
         */
        name: string;
        /**
         * Human-readable label for the field
         */
        prompt?: string;
        /**
         * Is the field read-only?
         */
        readOnly?: boolean;
        /**
         * Regular expression for validation the input
         */
        regex?: string;
        /**
         * Is this a required field?
         *
         * Defaults to false.
         */
        required?: boolean;
        /**
         * Whether the value element contains a URI Template.
         *
         * Defaults to false.
         */
        templated?: boolean;
        /**
         * Default or pre-existing value for the field
         */
        value?: string;
    }

    /**
     * HAL-Forms Template.
     *
     * @see http://rwcbook.github.io/hal-forms/
     */
    export declare interface HalFormsTemplate {
        /**
         * Mime-type for the form.
         *
         * If omitted, application/json should be assumed
         */
        contentType?: string;
        /**
         * HTTP Method
         */
        method: string;
        /**
         * List of properties in this HAL Form.
         *
         * If this is not set, the implication is that there are no properties.
         */
        properties?: HalFormsProperty[];
        /**
         * Human-readable title for the form
         */
        title?: string;
    }

    /**
     * ref: https://github.com/evert/hal-types
     */
    /**
     * A single HAL link
     */
    export declare interface HalLink {
        /**
         * Target of the link
         */
        href: string;
        /**
         * Content-type of the target resource
         */
        type?: string;
        /**
         * If set to true, the href should be interpeted as a URI-Template (RFC6570).
         */
        templated?: boolean;
        /**
         * Human readable title for the link
         */
        title?: string;
        /**
         * Language of the target resource, such as 'en-CA'.
         */
        hreflang?: string;
        /**
         * Link hints, as defined in draft-nottingham-link-hint
         */
        hints?: LinkHints;
    }

    declare type HALResource = Hal;

    /**
     * A HAL document
     */
    export declare interface HalResource {
        /**
         * List of links, indexed by their relationship.
         *
         * Each value is either a Link, or an array of Links
         */
        _links: {
            self: HalLink;
            [rel: string]: HalLink | HalLink[];
        };
        [property: string]: any;
        /**
         * Embedded resources
         */
        _embedded?: {
            [rel: string]: HalResource | HalResource[];
        };
        /**
         * HAL-FORMS
         */
        _templates?: {
            /**
             * Default form. The only valid value is 'default'
             */
            default: HalFormsTemplate;
        };
    }

    declare type Height = number;

    export declare type HTTPMethod = 'GET' | 'POST' | 'PUT' | 'HEAD' | 'OPTIONS' | 'PATCH' | 'DELETE' | 'delete' | 'get' | 'post' | 'put' | 'head' | 'options';

    /**
     * The parameters for an HTTP request.
     */
    declare type HTTPRequestParameters = {
        /**
         * The URL to request.
         */
        readonly href: string;
        /**
         * The HTTP method to use.
         */
        readonly method: HTTPMethod;
        /**
         * The HTTP headers to use.
         */
        readonly headers?: Record<string, string>;
        /**
         * The HTTP body to use.
         */
        readonly body?: BodyType;
    };

    export declare type IAdobePromise<T = any, E = any, X = unknown> = _IAdobePromise<T, E, X> & X;

    declare interface _IAdobePromise<T = any, E = any, X = unknown> {
        canceled: boolean;
        props: X;
        [Symbol.toStringTag]: string;
        /**
         * Get internal promise
         */
        getPromise<T2 = T>(): Promise<T2>;
        /**
         * Register then handler
         * @param onResolve
         * @param onReject
         */
        then<T2 = T, E2 = E, X2 = X>(onResolve: onResolveFn<T, T2> | null | undefined, onReject?: onRejectFn<E2> | null | undefined): IAdobePromise<T2, E2, X & X2>;
        /**
         * Return a new promise chain using global Promise instance.
         * @param onResolve
         * @param onReject
         */
        parallel<T2 = T, E2 = E>(onResolve: onResolveFn<T, T2>, onReject?: onRejectFn<E2, any>): Promise<T2>;
        /**
         * Register error handler
         * @param onReject
         */
        catch<E2 = E, R = void>(onReject: onRejectFn<E2, R>): IAdobePromise<T, E2, X>;
        /**
         * Register handler called after entire promise chain has completed.
         *
         * @note
         * Depends on global Promise's finally() method.
         *
         * @param fn
         */
        finally(fn: () => void): IAdobePromise<T, E, X>;
        /**
         * Call cancel handlers, mark promise as canceled.
         * @param {unknown} reason
         */
        cancel(reason?: unknown): void;
        /**
         * Alias to AdobePromise#cancel
         * @param {unknown} reason
         */
        abort(reason?: unknown): void;
        /**
         * Register a cancel handler
         * @param handler
         */
        onCancel(handler: (reason?: unknown) => void): void;
    }

    declare type Id = string;

    export declare interface IndexDocument<T extends EmbeddableResource = 'http://ns.adobe.com/adobecloud/rel/repository'> {
        regions: string[];
        links?: LinkSet;
        assignedDirectories?: AdobeAssetEmbedded<'http://ns.adobe.com/adobecloud/rel/repository' | T>[];
    }

    export declare type InternalClass<T extends new (...args: any) => any> = new (...args: ConstructorParameters<T>) => InstanceType<T>;

    export declare type InternalFunction<T extends (...args: any) => any> = (...args: Parameters<T>) => ReturnType<T>;

    declare interface InvokeOptions<T extends AdobeResponseType = any> {
        responseType?: T;
        noSoonerThen?: number;
        reuseRequestDesc?: Partial<RequestDescriptor> & {
            id: string;
        };
        retryOptions?: Pick<BackoffOptions<T>, 'authCallback' | 'retryCodes' | 'timeoutAfter' | 'increase' | 'preferRetryAfterHeader' | 'initialWait' | 'maxWait' | 'preScheduleCallback' | 'postScheduleCallback' | 'disableRetry' | 'pollCodes' | 'pollHeader' | 'pollMethod'>;
        autoParseJson?: boolean;
        timeout?: number;
        isStatusValid?: (status?: number, response?: AdobeResponse) => boolean | AdobeDCXError;
        additionalNodeOptions?: AdditionalNodeOptions;
        isExternalRequest?: boolean;
    }
    export { InvokeOptions as AdobeRequestOptions }
    export { InvokeOptions }

    export declare type JSONPatchDocument = JSONPatchOperation[];

    /*************************************************************************
     *
     * ADOBE CONFIDENTIAL
     * ___________________
     *
     * @license
     * Copyright 2021 Adobe Inc.
     * All Rights Reserved.
     *
     * NOTICE:  All information contained herein is, and remains
     * the property of Adobe Inc. and its suppliers,
     * if any.  The intellectual and technical concepts contained
     * herein are proprietary to Adobe Inc. and its
     * suppliers and are protected by trade secret or copyright law.
     * Dissemination of this information or reproduction of this material
     * is strictly forbidden unless prior written permission is obtained
     * from Adobe Inc.
     **************************************************************************/
    /**
     * JSON Patch types
     */
    export declare interface JSONPatchOperation {
        op: 'add' | 'remove' | 'replace' | 'move' | 'copy';
        path?: string;
        value?: unknown;
        from?: string;
    }

    export declare type KeysOfType<O, T> = {
        [K in keyof O]: O[K] extends T ? K : never;
    }[keyof O];

    export declare type KeysOfUnion<T> = T extends any ? keyof T : never;

    declare type Length = number;

    export declare type Link = BasicLink | BasicLink[];

    export declare interface LinkedResource {
        links?: LinkSet;
        /**
         * Sets the assets link
         * @param links   Links to set
         */
        setLinks(links: LinkSet): void;
        /**
         * Returns the link for the provided relationship
         * @param relationship      The link relationship to return the link for
         * @return                  The request link if exists
         */
        getLink(relationship: string): Link | undefined;
        /**
         * Sets the link for a relationship
         * @param relationship      The link relationship to set the link for
         * @param link              The link to set
         */
        setLink(relationship: string, link: Link): void;
        /**
         * Removes a relationship link
         * @param relationship      The link relationship to remove
         */
        removeLink(relationship: string): void;
        /**
         * Returns the href property of a link relationship
         * @param relationship      The relationship key
         * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
         * @return                  The href if the relationship and href exists on the provided links. Else undefined
         */
        getLinkHref(relationship: string, linkMode?: LinkMode): string | undefined;
        /**
         * Returns a specific property value from a link relationship
         * @param relationship      The relationship key
         * @param property          The property to pull from the relationship
         * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
         * @return                  The property if the relationship and property exist on the provided links. Else undefined
         */
        getLinkProperty(relationship: string, property: string, linkMode?: LinkMode): string | undefined;
        /**
         * Returns the result href from a templated link with values
         * @param relationship      The relationship key
         * @param values            The key/values to bind to the template
         * @param linkMode          If the relationship contains an array of links, allows specifying the exact mode to return the property for.
         * @return                  The expanded href template if the relationship exists and the link is marked as templated=true. Else undefined
         */
        getLinkHrefTemplated(relationship: string, values: Record<string, (string | number) | (string | number)[] | Record<string, string | number>>, linkMode?: LinkMode): string | undefined;
    }

    declare interface LinkHintAuthScheme {
        /**
         * A registered authenticaiton scheme such as 'Basic', 'Digest' or 'Bearer'.
         */
        scheme: string;
        /**
         * Realm information from the authentation scheme, similar to the "realm"
         * attribute on the WWW-Authentication header.
         */
        realm?: string;
    }

    /**
     * A supported format, indexed by media type
     */
    declare interface LinkHintFormats {
        [mediaType: string]: {
            /**
             * Relevant links to the format.
             *
             * Mainly this could contain a 'profile' or 'describedby'
             * link. These both describe a schema for the format
             */
            links?: {
                [rel: string]: HalLink;
            };
            /**
             * If set to true, it indicates that this format is
             * deprecated
             */
            deprecated?: boolean;
        };
    }

    /**
     * Link hints.
     *
     * See: https://tools.ietf.org/html/draft-nottingham-link-hint
     */
    export declare interface LinkHints {
        /**
         * List of HTTP methods that can be used on the target
         * resource
         */
        allow?: string[];
        /**
         * List of supported formats (used in for example GET responses
         * or PUT requests
         */
        formats?: LinkHintFormats;
        /**
         * List of links that exists at the context of the hint.
         *
         * This allows to specify links that exists at the target
         * resource. For example, you might add an 'edit-form' link
         * on a link, allowing clients to render an edit button
         * for a target.
         */
        links?: {
            [rel: string]: HalLink;
        };
        /**
         * A list of media-types the target resource supports for
         * POST request bodies
         */
        'accept-post'?: LinkHintFormats;
        /**
         * Array of media-types that the target resource supports for
         * PATCH request bodies.
         *
         * Unlike 'accept-post', this does not support objects with
         * schema links or the deprecated flag.
         */
        'accept-patch'?: string[];
        /**
         * Array of range-specifiers the resource supports.
         *
         * The values are equivalent to the contents of the Accept-Ranges
         * HTTP header
         */
        'accept-ranges'?: string[];
        /**
         * List of values for the Prefer header that the resource
         * supports.
         */
        'accept-prefer'?: string[];
        /**
         * A hint that the target resource requires the use of precondition
         * headers such as If-Modified-Since or If-Match for state-changing
         * requests such as PUT, POST, DELETE.
         */
        'precondition-req'?: ('etag' | 'last-modified')[];
        /**
         * A list of authentication schemes that are required at the target
         * resource
         */
        'auth-schemes'?: LinkHintAuthScheme[];
        /**
         * A status for the resource.
         *
         * Indicates that the resource is deprecated, or 'gone' (would result in a
         * 410 status code).
         */
        status?: 'deprecated' | 'gone';
    }

    export declare type LinkMode = 'id' | 'path';

    /**
     * LinkRelation enumeration
     * Defines the possible relation types for HAL links
     */
    export declare const enum LinkRelation {
        ACCESS_CHECK = "http://ns.adobe.com/adobecloud/rel/ac/check",
        ACL_POLICY = "http://ns.adobe.com/adobecloud/rel/ac/policy",
        ANNOTATIONS = "http://ns.adobe.com/adobecloud/rel/annotations",
        APP_METADATA = "http://ns.adobe.com/adobecloud/rel/metadata/application",
        BASE_DIRECTORY = "http://ns.adobe.com/adobecloud/rel/directory/base",
        BLOCK_DOWNLOAD = "http://ns.adobe.com/adobecloud/rel/download",
        BLOCK_EXTEND = "http://ns.adobe.com/adobecloud/rel/block/extend",
        BLOCK_FINALIZE = "http://ns.adobe.com/adobecloud/rel/block/finalize",
        BLOCK_TRANSFER = "http://ns.adobe.com/adobecloud/rel/block/transfer",
        BLOCK_UPLOAD_INIT = "http://ns.adobe.com/adobecloud/rel/block/init",
        BULK_REQUEST = "http://ns.adobe.com/adobecloud/rel/bulk",
        COMPONENT = "http://ns.adobe.com/adobecloud/rel/component",
        CREATE = "http://ns.adobe.com/adobecloud/rel/create",
        DESCRIBED_BY = "describedBy",
        DIRECTORY = "http://ns.adobe.com/adobecloud/rel/directory",
        DISCARD = "http://ns.adobe.com/adobecloud/rel/discard",
        EFFECTIVE_PRIVILAGES = "http://ns.adobe.com/adobecloud/rel/ac/effective",
        EMBEDDED_METADATA = "http://ns.adobe.com/adobecloud/rel/metadata/embedded",
        ID = "http://ns.adobe.com/adobecloud/rel/id",
        MANIFEST = "http://ns.adobe.com/adobecloud/rel/manifest",
        PAGE = "http://ns.adobe.com/adobecloud/rel/page",
        PATH = "http://ns.adobe.com/adobecloud/rel/path",
        PRIMARY = "http://ns.adobe.com/adobecloud/rel/primary",
        RENDITION = "http://ns.adobe.com/adobecloud/rel/rendition",
        REPO_METADATA = "http://ns.adobe.com/adobecloud/rel/metadata/repository",
        REPO_OPS = "http://ns.adobe.com/adobecloud/rel/ops",
        REPOSITORY = "http://ns.adobe.com/adobecloud/rel/repository",
        RESOLVE_BY_ID = "http://ns.adobe.com/adobecloud/rel/resolve/id",
        RESOLVE_BY_PATH = "http://ns.adobe.com/adobecloud/rel/resolve/path",
        RESTORE = "http://ns.adobe.com/adobecloud/rel/restore",
        VERSION_HISTORY = "version-history"
    }

    /**
     * Repository API HalLink keys
     */
    export declare type LinkRelationKey = `${LinkRelation}`;

    export declare type LinkSet = PartialRecord<LinkRelationKey, Link> & {
        [LinkRelation.RENDITION]?: RenditionLink | RenditionLink[];
    };

    /**
     * This file was automatically generated by json-schema-to-typescript.
     * DO NOT MODIFY IT BY HAND. Instead, modify the source JSONSchema file,
     * and run json-schema-to-typescript to regenerate this file.
     */
    /**
     * All DCX composite manifests conform to this schema.
     */
    export declare type ManifestData = Element_2 & RootProperties;

    declare type ManifestFormatVersion = number;

    declare interface MetadataProperties {
        width?: Width;
        height?: Height;
        pixelsPerMeter?: PixelsPerMeter;
        [k: string]: unknown;
    }

    /**
     * The date and time the resource was last modified. NOTE: The value of this property is not necessarily the same as the file’s system modification date because it is typically set before the file is saved.
     */
    declare type ModifyDate = string;

    /**
     * Name is equivalent to the text(localized) property of the default item in the dc:title array.
     */
    declare type Name = string;

    declare type Node_2 = NodeElementProperties & {
        type?: NonDcxType;
        children?: Children;
        components?: Components;
        [k: string]: unknown;
    };

    /**
     * Gets passed into copyChild(), replaceChild and called back whenever the operation has finished.
     * @callback NodeCallback@callback NodeCallback
     *    @param {Error}         error
     *    @param {AdobeDCXNode}  node
     */
    export declare type NodeCallback = (error: Error, node?: AdobeDCXNode) => unknown;

    /**
     * Properties that are common to nodes and elements
     */
    declare type NodeElementProperties = CommonProperties & NodeProperties & MetadataProperties & CustomProperties;

    declare interface NodeProperties {
        rel?: Rel;
        typeVersion?: TypeVersion;
        path?: Path;
        created?: CreateDate;
        modified?: ModifyDate;
        tags?: Subject;
        units?: Unit;
        [k: string]: unknown;
    }

    declare type NonDcxType = Format & {
        [k: string]: unknown;
    };

    /**
     * Rejected state handler function
     */
    declare type onRejectFn<P = unknown, R = unknown> = (reason: P) => R | PromiseLike<R>;

    /*************************************************************************
     *
     * ADOBE CONFIDENTIAL
     * ___________________
     *
     * @license
     * Copyright 2021 Adobe Inc.
     * All Rights Reserved.
     *
     * NOTICE:  All information contained herein is, and remains
     * the property of Adobe Inc. and its suppliers,
     * if any.  The intellectual and technical concepts contained
     * herein are proprietary to Adobe Inc. and its
     * suppliers and are protected by trade secret or copyright law.
     * Dissemination of this information or reproduction of this material
     * is strictly forbidden unless prior written permission is obtained
     * from Adobe Inc.
     **************************************************************************/
    /**
     * This type should be identical to that defined in AdobePromise.
     *
     * It should only be used as needed by APIs defined in common-types
     * to avoid dependencies on the promise module.
     */
    /**
     * Resolved state handler function
     */
    declare type onResolveFn<P = unknown, R = unknown> = (result: P) => R | PromiseLike<R>;

    export declare type Optional<T> = T | unknown | undefined | void;

    export declare type Overwrite<T1, T2> = {
        [P in Exclude<keyof T1, keyof T2>]: T1[P];
    } & T2;

    export declare type PartialRecord<K extends keyof any, T> = {
        [P in K]?: T;
    };

    declare type PatchableRepoMetadataProperties = 'storage:deviceCreateDate' | 'storage:deviceModifyDate' | 'storage:deviceId' | 'repo:defaultScheduledDeletionDuration';

    declare type Path = string & {
        [k: string]: unknown;
    };

    export declare type PersistentRefreshCallback = (provider?: AdobeAuthProvider) => void | Promise<void>;

    declare type PixelsPerMeter = number;

    /**
     * Union of platform-specific Duplex stream types
     */
    export declare type PlatformDuplexSteam = (ReadableStream<Uint8Array> & WritableStream) | Duplex;

    /**
     * Union of platform-specific Readable stream types
     */
    export declare type PlatformReadableStream = ReadableStream<Uint8Array> | Readable;

    /**
     * Union of platform-specific Writable stream types
     */
    export declare type PlatformWritableStream = WritableStream | Writable;

    export declare type PostRequestHook = (xhr?: AdobeXhr<any>, backoffSnapshot?: BackoffSnapshot<any>) => void;

    export declare type PrePostBackoffCallback<T extends AdobeResponseType> = (xhr?: AdobeXhr<T>, snapshot?: BackoffSnapshot<T>) => void | Promise<void>;

    export declare type PrePostCallback<T extends AdobeResponseType> = (xhr?: AdobeXhr<T>) => void;

    export declare type PreRequestHook = (xhr?: AdobeXhr<any>, backoffSnapshot?: BackoffSnapshot<any>) => void;

    export declare type Privilege = 'none' | 'ack' | 'read' | 'write' | 'delete';

    /**
     * Callback for monitoring progress of the letious transfer functions.
     * @callback ProgressCallback@callback ProgressCallback
     *    @param {Integer} bytesCompleted   - Bytes sent or received so far
     *    @param {Integer} totalBytes       - Total bytes to send or receive
     *    @param {boolean} [indeterminate]  - Specifies that the total bytes could not be determined
     */
    export declare type ProgressCallback = (bytesCompleted: number, totalBytes: number, indeterminate?: boolean | undefined) => void;

    export declare type ProgressListener = (sentOrReceivedBytes: number, totalBytes: number, indeterminateTotalBytes?: boolean | undefined) => void;

    declare type Rel = string;

    export declare type RenditionLink = BasicLink & {
        type?: RenditionType;
        width?: number;
        height?: number;
    };

    /*************************************************************************
     *
     * ADOBE CONFIDENTIAL
     * ___________________
     *
     * @license
     * Copyright 2020 Adobe Inc.
     * All Rights Reserved.
     *
     * NOTICE:  All information contained herein is, and remains
     * the property of Adobe Inc. and its suppliers,
     * if any.  The intellectual and technical concepts contained
     * herein are proprietary to Adobe Inc. and its
     * suppliers and are protected by trade secret or copyright law.
     * Dissemination of this information or reproduction of this material
     * is strictly forbidden unless prior written permission is obtained
     * from Adobe Inc.
     **************************************************************************/
    /**
     * Types related to Renditions
     */
    export declare interface RenditionOptions {
        size?: number;
        type?: RenditionType;
        page?: number;
        fragment?: string;
        /**
         * deprecated in v8 - use `resource` instead
         * To be removed in v9
         * @deprecated
         */
        revision?: string;
        /**
         * deprecated in v8 - use `resource` instead
         * To be removed in v9
         * @deprecated
         */
        component_id?: string;
        /**
         * A Stringified resource designator.
         * This format is not compatible with the older top-level `component_id` and `revision` parameters.
         * @see {@link https://wiki.corp.adobe.com/display/CA/Proposal%3A+Add+resource+Parameter+to+api%3Arendition }
         */
        resource?: string;
    }

    export declare type RenditionType = 'image/jpeg' | 'image/jpg' | 'image/png' | 'image/gif' | 'application/vnd.adobe.ccv.videometadata';

    /**
     * Defines the return type for a function that can return a readable stream.
     * If returning a stream, the promise resolves with the stream.
     *
     * When not requesting a stream, the promise may be extended with a blockDownload property, depending on the size of the transfer.
     */
    export declare type RepoDownloadStreamableReturn<T extends AdobeResponseType = AdobeResponseType> = IAdobePromise<AdobeResponse<T>, AdobeDCXError, T extends 'stream' ? {
        blockDownload: never;
    } : {
        blockDownload?: AdobeBlockDownload;
    }>;

    export declare type RepoMetaPatch = {
        [k in PatchableRepoMetadataProperties]?: string | number;
    };

    export declare type RepositoryType = 'storage' | 'index';

    /**
     * Gets return from methods that initiate a network request. With the exception of the `progress`
     * property clients must treat all properties as read-only.
     * @typedef {Object} RequestDescriptor@typedef {Object} RequestDescriptor
     *   @property {String}             method      The HTTP method.
     *   @property {String}             href        The href of the request.
     *   @property {Object}             headers     The HTTP request headers.
     *   @property {ProgressCallback}   [progress]  A function that gets called peridically with progress updates.
     */
    export declare interface RequestDescriptor {
        id: string;
        method: HTTPMethod;
        href: string;
        headers: Record<string, string | number>;
        progress?: ProgressCallback;
        noSoonerThen?: number;
        autoParseJson?: boolean;
        token?: any;
        queueStartTime?: number;
        startTime?: number;
        options?: InvokeOptions<any>;
        body?: any;
        watch?<T = any>(prop: string, handler: (next?: T, previous?: T) => void): void;
    }

    export declare type RequestMethod = 'get' | 'put' | 'post' | 'options' | 'head';

    export declare type RequireAtLeastOne<T, Keys extends keyof T = keyof T> = Pick<T, Exclude<keyof T, Keys>> & {
        [K in Keys]-?: Required<Pick<T, K>> & Partial<Record<Exclude<Keys, K>, T[K] | undefined>>;
    }[Keys];

    export declare type RequireSome<T, R extends keyof T = keyof T> = Required<Pick<T, R>> & T;

    /**
     * The set of links that can be fetched using the resolveBy API.
     * Currently only Repo Metadata is a valid param, but that will change over time.
     */
    export declare type ResolvableResourceRelations = 'http://ns.adobe.com/adobecloud/rel/metadata/repository';

    /**
     * Resource Designator
     * See {@link | https://git.corp.adobe.com/pages/caf/api-spec/chapters/syntax.html#resource-designators}
     */
    export declare type ResourceDesignator = ResourceDesignatorNonTemplated | ComponentResourceDesignator | AssetResourceDesignator;

    /**
     * Some resources can be specified using only the link href.
     */
    declare type ResourceDesignatorNonTemplated = 'http://ns.adobe.com/adobecloud/rel/metadata/repository';

    /**
     * Invoke response callback
     */
    export declare interface ResponseCallback<T extends AdobeResponseType = any> {
        (error?: undefined, xhr?: AdobeResponse<T>, responseData?: T): void;
        (error?: AdobeDCXError, xhr?: AdobeResponse<T>, responseData?: T): void;
    }

    export declare interface ResponseTypeMap {
        text: string;
        blob: Blob;
        json: any;
        defaultbuffer: Uint8Array | Buffer | ArrayBuffer;
        buffer: Buffer;
        arraybuffer: ArrayBuffer;
        stream: ReadableStream<Uint8Array> | IncomingMessage;
        any: any;
        void: null;
    }

    declare interface RootProperties {
        created?: CreateDate;
        modified?: ModifyDate;
        typeVersion?: TypeVersion;
        'manifest-format-version': ManifestFormatVersion;
        [k: string]: unknown;
    }

    declare type ScheduleCallback<T extends AdobeResponseType> = (snapshot?: BackoffSnapshot<T>) => void | Promise<void>;

    /**
     * Data that can be sliced
     */
    export declare type SliceableData = ArrayBuffer | Blob | string | TypedArray;

    declare type State = 'modified' | 'unmodified' | 'pendingDelete' | 'committedDelete';

    /*************************************************************************
     *
     * ADOBE CONFIDENTIAL
     * ___________________
     *
     * @license
     * Copyright 2020 Adobe Inc.
     * All Rights Reserved.
     *
     * NOTICE:  All information contained herein is, and remains
     * the property of Adobe Inc. and its suppliers,
     * if any.  The intellectual and technical concepts contained
     * herein are proprietary to Adobe Inc. and its
     * suppliers and are protected by trade secret or copyright law.
     * Dissemination of this information or reproduction of this material
     * is strictly forbidden unless prior written permission is obtained
     * from Adobe Inc.
     **************************************************************************/
    /**
     * Common types
     */
    export declare interface StringLike {
        toString(): string;
    }

    /**
     * Set of descriptive phrases or keywords that describe the content of the resource.
     */
    declare type Subject = {
        '@set': string[];
        [k: string]: unknown;
    } | string[];

    declare type TypedArray = Int8Array | Uint8Array | Uint8ClampedArray | Int16Array | Uint16Array | Int32Array | Uint32Array | Float32Array | Float64Array | BigInt64Array | BigUint64Array;

    declare type TypeVersion = number;

    /**
     * Unit used for artboard coordinates (`x`, `y`, `width` and `height`). For example: `inch`, `mm`, `pixel`, `pica`, `point` (default is `pixel`).
     */
    declare type Unit = string;

    /**
     * Callback for monitoring progress of a upload transfer.
     * @callback UploadProgressCallback@callback UploadProgressCallback
     *    @param {Integer} bytesCompleted
     *    @param {Integer} totalBytes
     *    @param {boolean} indeterminate
     */
    export declare type UploadProgressCallback = ProgressCallback & {
        (bytesCompleted: number, totalBytes: number, indeterminate: boolean): void;
    };

    /**
     * `UploadRecipe`  includes the parameters required to issue requests to upload a component of a provided size.
     */
    export declare type UploadRecipe = {
        /**
         * The size of the block to use for uploading.
         */
        blockSize: Readonly<number>;
        /**
         * The upload request parameters.
         */
        uploadRequestParameters: ComponentUploadRequestParameters[];
        /**
         * The finalize request parameters.
         */
        finalizeRequestParameters?: FinalizeBlockUploadRequestParameters;
    };

    declare type Width = number;

    export declare interface XhrOptions<T extends AdobeResponseType> {
        withCredentials?: boolean;
        /**
         * @internal
         * @private
         */
        forceXhr?: any;
        fetch?: typeof fetch;
        preCallback?: PrePostCallback<T>;
        postCallback?: PrePostCallback<T>;
        timeout?: number;
        autoParseJson?: boolean;
        additionalNodeOptions?: AdditionalNodeOptions;
        preferFetch?: boolean;
    }

    export declare type XhrResponse<T extends AdobeResponseType = AdobeResponseType> = Pick<AdobeResponse<T>, 'statusCode' | 'headers' | 'responseType' | 'response' | 'message' | 'xhr'> & {
        originalResponseData?: string;
    };

    export { }
