/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { RequireAtLeastOne } from './common';
import { LinkSet } from './link';
import { AdobeRepository, EffectivePrivileges } from './resources';

export interface AdobeAsset {
    /**
     * The Repository ID of the Repository storing the Asset.
     */
    repositoryId?: string;
    /**
     * A unique identifier given to every addressable Asset in a given Repository.
     */
    assetId?: string;
    /**
     * ID of the root asset
     */
    baseAssetId?: string;
    /**
     * Identifier of the head version of the Asset. Not present for an Asset which does not yet have a version.
     */
    version?: string;
    /**
     * An Asset's location in the Directory hierarchy.
     */
    path?: string;
    /**
     * The name of the Asset.
     */
    name?: string;
    /**
     * The class of an Asset is inferred from its Media Type at the time the Asset is created.
     */
    assetClass?: string;
    /**
     * An ETag is an HTTP response header returned by an HTTP/1.1-compliant web server, used to determine change in content of a Resource at a given URL. This property is required if the Asset has a Primary Resource.
     */
    etag?: string;
    /**
     * The media type of the Resource.
     */
    format?: string;
    /**
     * The server date and time when the Resource was created in the Repository, such as when a File is first uploaded or a Directory is created by the server as the parent of a new Asset.
     */
    createDate?: string;
    /**
     * The server date and time when the Resource was last modified in the Repository, such as when a new version of an Asset is uploaded or a Directory's child Resource is added or removed.
     */
    modifyDate?: string;
    /**
     * A timestamp capturing the time at which this Asset was last modified on the client, and which may therefore precede the time at which the corresponding modification was uploaded to the Repository. (It can also be later, due to clock skew.) If not specified with any request that creates a new version of the Asset, it will be updated to repo:modifyDate.
     */
    deviceModifyDate?: string;
    /**
     * Time the asset was discarded directly or by inheritance. Does not exist for active assets.
     */
    discardDate?: string;
    /**
     * A timestamp capturing the time at which this Asset was created on the client, and which may therefore precede the time at which it was created in the Repository. (It can also be later, due to clock skew.) Can be set only at the time of Asset creation and, if not set, defaults to repo:createDate.
     */
    deviceCreateDate?: string;
    /**
     * The Asset's state, indicating whether the Asset is active, discarded, or deleted.
     */
    state?: string;
    /**
     * The ID of the user who initiated the action that caused the Resource to be created in the Repository
     */
    createdBy?: string;
    /**
     * The ID of the user who initiated the action that most recently caused the Resource to be modified in the Repository.
     */
    modifiedBy?: string;
    /**
     * Identifier of the user that discarded the asset directly or by inheritance. Does not exist for active assets.
     */
    discardedBy?: string;
    /**
     * The set of links associated with the Asset.
     */
    links?: LinkSet;
    /**
     * defaultScheduledDeletionDuration set at the time of directory creation
     */
    defaultScheduledDeletionDuration?: number;
    /**
     * scheduledDeletionDate set for non directory assets by the service
     */
    scheduledDeletionDate?: string;
    /**
     * The size of the Asset in bytes.
     */
    size?: number; //Likely just on file
    width?: number;
    length?: number;
    renderable?: boolean;

    md5?: string;
    /**
     *  One of the ACP Content Regions, has an impact on the Links API. Can only be used with a service token for the LinksAPI.
     * @see {@link https://wiki.corp.adobe.com/pages/viewpage.action?pageId=2605664337#Proposal:LinksAPI(ResolveLite)-LinksAPI LinksAPI}
     */
    contentRegion?: string;

    // [key: string]: string | boolean | number | LinkSet;
}

/**
 * AdobeAsset requiring either `path` or `assetId`
 */
export type AssetWithPathOrId = RequireAtLeastOne<AdobeAsset, 'assetId' | 'path'>;

/**
 * AdobeAsset requiring either ()`repositoryId` & `path`) or `assetId`
 */
export type AssetWithRepoAndPathOrId = AdobeAsset &
    (Required<Pick<AdobeAsset, 'path' | 'repositoryId'>> | Required<Pick<AdobeAsset, 'assetId'>>);

/**
 * AdobeAsset requiring links
 */
export type AdobeAssetWithLinks = Partial<AdobeAsset> & { links: LinkSet };

/**
 * Minimal data required to use an asset with API calls.
 * Needs to have one of:
 *  1. Links
 *  2. Asset ID and Repo ID
 *  3. Path and Repo ID
 */
export type AdobeMinimalAsset = AssetWithPathOrId | AdobeAssetWithLinks;

type PatchableRepoMetadataProperties =
    | 'storage:deviceCreateDate'
    | 'storage:deviceModifyDate'
    | 'storage:deviceId'
    | 'repo:defaultScheduledDeletionDuration';
export type RepoMetaPatch = { [k in PatchableRepoMetadataProperties]?: string | number };
export interface EmbeddableResourceMap {
    'http://ns.adobe.com/adobecloud/rel/repository': {
        returnKey: 'RepositoryResource';
        returnValue: AdobeRepository;
    };
    'http://ns.adobe.com/adobecloud/rel/ac/effective': {
        returnKey: 'EffectivePrivileges';
        returnValue: EffectivePrivileges;
    };
    'http://ns.adobe.com/adobecloud/rel/metadata/application': {
        returnKey: 'AppMetadata';
        returnValue: Record<string, unknown> | undefined;
    };
}

type EmbeddableResourceString =
    | 'http://ns.adobe.com/adobecloud/rel/repository'
    | 'http://ns.adobe.com/adobecloud/rel/ac/effective'
    | 'http://ns.adobe.com/adobecloud/rel/metadata/application';

type EmbeddableResourceDesignator = {
    resource: {
        reltype:
            | 'http://ns.adobe.com/adobecloud/rel/metadata/application'
            | 'http://ns.adobe.com/adobecloud/rel/ac/effective';
    };
    selectors?: string[];
};

export type EmbeddableResource = EmbeddableResourceDesignator | EmbeddableResourceString;

interface EmbeddableResourceUnmap {
    RepositoryResource: Partial<AdobeRepository>;
    EffectivePrivileges: EffectivePrivileges;
    AppMetadata: string | Record<string, unknown> | undefined;
}

export type AdobeAssetEmbedded<
    T extends EmbeddableResource = never,
    U = {
        [K in EmbeddableResourceMap[T extends EmbeddableResourceString
            ? T // If it's a string use the string directly
            : T extends EmbeddableResourceDesignator // If it is not a string, check if is it an EmbeddableResourceDesignator
            ? T['resource']['reltype'] // If it is an EmbeddableResourceDesignator, use its reltype property
            : never]['returnKey']]: EmbeddableResourceUnmap[K]; // No EmbeddableResource
    },
> = AdobeAsset & {
    embedded?: U;
};
