/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { HalLink } from '../hal';
import { AdobeDCXBranch } from './Branch';
import { AdobeDCXElement } from './Element';
import { AdobeDCXNodeData } from './Node';

export type ComponentCallback = (error?: Error, component?: AdobeDCXComponent) => unknown;

export interface AdobeDCXComponentData extends AdobeDCXNodeData {
    /**
     * The etag of the component.
     *
     * <p>
     * Note: a (non-weak) entity tag value includes the leading and trailing double-quotes,
     * which therefore must be escaped in order to appear properly in the property value.
     * </p>
     */
    etag?: string;

    /**
     * The version of the last-synchronized version of this component.
     *
     * <p>
     * Note: In practice, in manifests version 5 and below the version property can be a string or a number.
     * </p>
     */
    version?: string;

    /**
     * The md5 hash of the last-synchronized version of this component.
     */
    md5?: string;

    /**
     * The length (size in bytes) of this component.
     */
    length?: number;

    /**
     * Used to translate between physical dimensions and pixel dimensions for an image.
     * Applications that convert this value to pixels per inch (by multiplying by 0.0254 meters per inch)
     * should round the resultant value to the nearest tenth of a pixel per inch. Conversely,
     * applications converting from pixels per inch should round to the nearest integer pixelsPerMeter.
     * This ensures a correct round trip between these two common units (e.g., 72.0 ppi -> "pixelsPerMeter":2835 -> 72.0 ppi).
     */
    pixelsPerMeter?: number;
}

export interface AdobeDCXComponent {
    /**
     * The id of the component. Must be a unique among the components of the composite.
     *
     * <p>Cannot be changed for a component that is part of a branch.</p>
     */
    id: string;

    /**
     * The asset id of the component in cloud storage.
     */
    assetId?: string;

    /**
     * The repository id of the component in cloud storage.
     */
    repositoryId?: string;

    /**
     * The name of the component.
     */
    name?: string;

    /**
     * The type of the component.
     */
    type: string;

    /**
     * The path of the component.
     */
    path: string;

    /**
     * The relationship of the component to its parent node and its sibling components.
     */
    relationship?: string;

    /**
     * The version of the component.
     */
    version?: string;

    /**
     * The edit state of the component.
     */
    state: string;

    /**
     * The etag of the component.
     */
    etag?: string;

    /**
     * The md5 of the component.
     */
    md5?: string;

    /**
     * The composite id of the owning composite.
     */
    readonly compositeId?: string;

    /**
     * The owner of the composite.
     */
    readonly owner?: AdobeDCXBranch | AdobeDCXElement;

    /**
     * The absolute path of the parent of the component.
     */
    readonly parentPath?: string;

    /**
     * The absolute path of the component.
     */
    readonly absolutePath?: string;

    /**
     * The content length of the component.
     */
    length?: number;

    /**
     * The width property of the component.
     */
    width?: number;

    /**
     * The height property of the component.
     */
    height?: number;

    /**
     * Returns an array of custom keys that are present at this component. Custom keys are manifest
     * properties that are not considered standard DCX properties.
     * @returns {Array}
     */
    getCustomKeys(): string[];

    /**
     * Returns the object or value for the given custom key.
     * @param   {String} key The custom key to look up.
     * @returns {*}
     */
    getValue<T = any>(key: string): T;

    /**
     * Sets the object or value for the given custom key.
     * @param {String} key   The custom key to set the value for.
     * @param {*}      value The value to set.
     */
    setValue<T = any>(key: string, value: T): void;

    /**
     * Removes the object or value for the given custom key.
     * @param {String} key The custom key to remove.
     */
    removeValue(key: string): void;

    /**
     * Returns the link with the given relationship as a JS object or undefined if the component
     * doesn't have such a link.
     * @param   {String} relationship The relationship of the link to the component.
     * @returns {Object} The link object with the given relationship as a JS object or undefined if
     *                   the component doesn't have such a link.
     */
    getLink(relationship: string): HalLink | undefined;

    /**
     * Removes the link with the given relationship.
     * @param {String} relationship The relationship of the link to the component.
     *
     */
    removeLink(relationship: string): void;

    /**
     * Sets the link with the given relationship to the given object.
     * @param {Object} link         A JS object representing the link.
     * @param {String} relationship The relationship of the link to the component.
     */
    setLink(link: HalLink, relationship: string): void;

    /**
     * Returns false if any of the properties of the given component is different from the properties
     * of this component. Recurses both data structures.
     *
     * @example
     * if (!component.isEqualTo(otherComponent, { unimportantProperty: true })) {
     *      // The components have different properties
     * }
     * @param   {AdobeDCXComponent} component            The component to compare with.
     * @param   {Object}            [propertiesToIgnore] An object having the properties that should
     *                                                 not be compared.
     * @returns {Boolean}
     */
    isEqualTo(component: AdobeDCXComponent, propertiesToIgnore?: Record<string, boolean>): boolean;
    /**
     * Get a serialized component descriptor.
     * @note Only works for RAPI. The Component must be attached to a branch.
     *
     * ```ts
     * const componentDescriptor = component.getComponentDescriptor();
     * ```
     * @returns {string}
     */
    getComponentDescriptor(): string;
}
