/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeDCXError } from '../error';
import { LinkRelationKey } from '../link';
import { AdobeDCXComponent, ComponentCallback } from './Component';
import { AdobeDCXNode } from './Node';
import { AdobeUploadResults } from './UploadResults';

/**
 * Gets passed into copyChild(), replaceChild and called back whenever the operation has finished.
 * @callback NodeCallback
 *    @param {Error}         error
 *    @param {AdobeDCXNode}  node
 */
export type NodeCallback = (error: Error, node?: AdobeDCXNode) => unknown;

export interface AdobeDCXElement {
    /**
     * An AdobeDCXNode object that represents the root of the underlying manifest.
     * @memberof AdobeDCXElement#
     * @type {AdobeDCXNode}
     * @readonly
     */
    rootNode: AdobeDCXNode;

    /**
     * The name of the composite.
     * @memberof AdobeDCXElement#
     * @type {String}
     */
    name?: string;

    /**
     * The type of the element.
     * @memberof AdobeDCXElement#
     * @type {String}
     */
    type: string;

    /**
     * The asset id of the composite that can be used to pull and push the composite.
     * @memberof AdobeDCXElement#
     * @type {String}
     * @readonly
     */
    compositeAssetId?: string;

    /**
     * The repository id of the composite that can be used to pull and push the composite.
     * @memberof AdobeDCXElement#
     * @type {String}
     * @readonly
     */
    compositeRepositoryId?: string;

    /**
     * Whether the composite has has been modified in memory and needs to be committed to local
     * storage.
     * @memberof AdobeDCXElement#
     * @readonly
     * @type {Boolean}
     */
    isDirty: boolean;

    /**
     * @memberof AdobeDCXElement#
     * @internal
     * @readonly
     */
    changeCount: number;

    /**
     * Returns the node with the given id or undefined if not found.
     * @param   {String} id The id of the child node to look up.
     * @returns {AdobeDCXNode}
     */
    getChildWithId(id: string): AdobeDCXNode | undefined;

    /**
     * Returns the node with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXNode}
     */
    getChildWithAbsolutePath(path: string): AdobeDCXNode | undefined;

    /**
     * Generates and returns an array of the child nodes of the given parent node.
     * @example
     * var rootNodes = element.getChildrenOf(element.rootNode);
     * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
     * @returns {Array}
     */
    getChildrenOf(parentNode: AdobeDCXNode): AdobeDCXNode[];

    /**
     * Creates a new node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new child node.
     * @param   {String}  [name]       The name of the new child. If undefined the child will not
     *                                 have a name.
     * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
     *                                 get a random id.
     * @param   {Integer} [index]      If given and less than or equal to the current number of
     *                                 children than the node gets inserted at the given index.
     *                                 Otherwise it gets added to the end.
     * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
     *                                 root node.
     * @returns {AdobeDCXNode}
     */
    addChild(name: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;

    /**
     * Removes and returns the given child node from the element.
     * @param   {AdobeDCXNode} node The child node to remove.
     * @returns {AdobeDCXNode}
     */
    removeChild(node: AdobeDCXNode): AdobeDCXNode;

    /**
     * Moves the existing child from its current parent/index to the given parent/index.
     * @param   {AdobeDCXNode} node         The child node to move
     * @param   {Integer}      index        If given and less than or equal to the current number of
     *                                      children than the node gets inserted at the given index.
     *                                      Otherwise it gets added to the end.
     * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
     *                                      the root.
     * @returns {AdobeDCXNode}
     */
    moveChild(node: AdobeDCXNode, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;

    /**
     * Copies the given child node as a new node into this element. The node can be from the same or
     * from a different composite.
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Fails if a node with the same id or same absolute path already exists.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two composites stored at different endpoints.</p>
     *
     * @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
     *                                      newPath must be provided.
     * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
     *                                      then the new child node will be added to the root of the
     *                                      branch.
     * @param   {Integer}      [index]      If provided and less than or equal to the current number of
     *                                      children of the parentNode (or root) the child node gets
     *                                      inserted at the given index. Otherwise it gets added to
     *                                      the end.
     * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
     *                                      this a its path property and it will also receive a new
     *                                      random id (unless one is provided with the newId param).
     *                                      If left undefined then the copy of the node will keep
     *                                      the path of the original. In either case the function will
     *                                      fail if the resulting absolute path of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing absolute path.</p>
     *                                      <p>You must provide a newPath if you are copying the root
     *                                      node of a branch or element.</p>
     * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
     *                                      this a its id. If left undefined (and the newPath param
     *                                      is also undefined) then the copy will retain the id of
     *                                      the original. In either case the function will
     *                                      fail if the resulting id of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                      when not using local storage.
     *                                      Gets called when the copy is done or has failed.
     * @returns {AdobeDCXNode}              Only returns the created child node if no callback is
     *                                      given.
     */
    copyChild(
        node: AdobeDCXNode,
        parentNode?: AdobeDCXNode,
        index?: number,
        newPath?: string,
        newId?: string,
    ): AdobeDCXNode;
    copyChild(
        node: AdobeDCXNode,
        parentNode: AdobeDCXNode | undefined,
        index: number | undefined,
        newPath: string | undefined,
        newId: string | undefined,
        callback: NodeCallback,
    ): void;

    /**
     * Returns an array of all components in the element.
     * @returns {Array}.
     */
    allComponents(): AdobeDCXComponent[];

    /**
     * Returns the component with the given id or undefined if not found.
     * @param   {String} id The id of the component to look up.
     * @returns {AdobeDCXComponent}
     */
    getComponentWithId(id: string): AdobeDCXComponent | undefined;

    /**
     * Returns the component with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path of the desired component.
     * @returns {AdobeDCXComponent}
     */
    getComponentWithAbsolutePath(path: string): AdobeDCXComponent | undefined;

    /**
     * Returns an array containing the components of the given node.
     * @param   {AdobeDCXNode} parentNode The node whose components to return.
     * @returns {Array<AdobeDCXComponent>}
     */
    getComponentsOf(parentNode: AdobeDCXNode): AdobeDCXComponent[];

    /**
     * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
     * @param error An IMCOMPLETE_COMPOSITE error
     * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
     */
    getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[];

    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param componentDescriptor The serialized component descriptor to use.
     * @param name          The name of the new component.
     * @param path          The path of the new component. Must satisfy uniquenes
     * @param relationship  The relationship of the new component.
     *                                            rules for components.
     * @param parentNode  The node to add the node to. Defaults to the root.
     * @returns {AdobeDCXComponent}               The new component.
     */
    addComponentWithComponentDescriptor(
        componentDescriptor: string,
        name: string,
        path: string,
        relationship?: LinkRelationKey,
        parentNode?: AdobeDCXNode,
    );

    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param   {String}            name          The name of the new component.
     * @param   {String}            relationship  The relationship of the new component.
     * @param   {String}            path          The path of the new component. Must satisfy uniquenes
     *                                            rules for components.
     * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
     * @param   {Object}            uploadResults The upload results object returned by a previous call
     *                                            to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent}               The new component.
     */
    addComponentWithUploadResults(
        name: string,
        relationship: string,
        path: string,
        parentNode?: AdobeDCXNode,
        uploadResults?: AdobeUploadResults,
    ): AdobeDCXComponent;

    /**
     * <p>Updates the component record with the results of a recent upload of said component.</p>
     *
     * @param   {AdobeDCXComponent} component     The component.
     * @param   {Object}            uploadResults The upload results object returned by a previous
     *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent} The updated component.
     */
    updateComponentWithUploadResults(component: AdobeDCXComponent, uploadResults: any): AdobeDCXComponent;

    /**
     * Removes the component from the element.
     * @param   {AdobeDCXComponent} component The component to remove.
     * @returns {AdobeDCXComponent} The removed component.
     */
    removeComponent(component: AdobeDCXComponent): AdobeDCXComponent;

    /**
     * Moves the component to the given node or the root if node is undefined
     * @param   {AdobeDCXComponent} component    The component to move.
     * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
     * @returns {AdobeDCXComponent} The moved component.
     */
    moveComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode): AdobeDCXComponent;

    /**
     * Copies the given component and adds it as a new component to this branch. Fails if the
     * component already exists.
     *
     * <p>Notice: This method does not work without local storage (browser environment) if
     * used to copy between two composites with different endpoints.</p>
     *
     * @param   {AdobeDCXComponent} component    The component to copy.
     * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
     *                                           provided then the component will be added to the
     *                                           root.
     * @param   {String}            [newPath]    If provided the copy of the component will be
     *                                           assigned this a its path property and it will also
     *                                           get assigned a random new id if none is provided via
     *                                           the <em>newId</em> param.
     * @param   {String}            [newId]      If provided the copy of the component will be assigned
     *                                           this a its id. If left undefined (and if newPath is
     *                                           undefined as well) then the copy of the component
     *                                           will retain the id of the original.
     *                                           This is useful when merging conflicting changes since
     *                                           it preserves the identity of components and avoids
     *                                           unnecessary network traffic.
     * @param   {ComponentCallback} [callback]   Optional when not copying between different
     *                                           composites or when copying without local storage.
     *                                           Gets called when the copy is done or has failed.
     * @returns {AdobeDCXComponent}              Only returns the new component when called without
     *                                           a callback.
     */
    copyComponent(
        component: AdobeDCXComponent,
        parentNode?: AdobeDCXNode,
        newPath?: string,
        newId?: string,
    ): AdobeDCXComponent;
    copyComponent(
        component: AdobeDCXComponent,
        parentNode: AdobeDCXNode | undefined,
        newPath: string | undefined,
        newId: string | undefined,
        callback: ComponentCallback,
    ): void;
}
