/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { HalLink } from '../hal';
import { AdobeDCXBranch } from './Branch';
import { AdobeDCXComponentData } from './Component';
import { AdobeDCXElement } from './Element';

export type AdobeDCXState = 'unmodified' | 'modified' | 'pendingDelete' | 'committedDelete';
export type AdobeDCXUnit = 'pixel' | 'point' | 'inch' | 'mm' | 'cm';

export interface AdobeDCXNodeData {
    /**
     * The id of a node should be set to a UUID when the node is created, and should never be changed.
     * Node IDs, being stable, are suitable for use in conflict resolution;
     * they allow reliable detection of when a node has been deleted or moved.
     */
    id: string;

    /**
     * The name of a node is intended to be used for display purposes only.
     * Names do not play any role in paths or references.
     * The name is required for a node that is an element; see Composition and Embedding.
     */
    name?: string;

    /**
     * The type property records the media type of the node.
     * When the value of this property ends in "+dcx", the node is an element; see Composition and Embedding.
     */
    type: string;

    /**
     * Since a composite format (type) can evolve over time, there is an optional numeric property to represent
     * the version of that type (if not specified, the value is understood to be zero).
     */
    typeVersion?: number;

    /**
     * Paths are used when resolving references.
     */
    path?: string;

    /**
     * The rel property records the relationship of this node to its parent node.
     * Relationships should be selected from the set of Common Link Relation Types defined for CC.
     */
    rel?: string;

    /**
     * The width of the component, as appropriate for that type of component (e.g., integer pixels for a PNG).
     *
     * <p>
     * Note: Elements with an intrinsic 2D size should record their width and height in the element's root node.
     * This is primarily used with renditions.
     * Along with the type property, this allows clients to find the best-suited rendition
     * (format and size) directly from the manifest itself.
     * </p>
     */
    width?: number;

    /**
     * The height of the component, as appropriate for that type of component (e.g., integer pixels for a PNG).
     *
     * <p>
     * Note: Elements with an intrinsic 2D size should record their width and height in the element's root node.
     * This is primarily used with renditions.
     * Along with the type property, this allows clients to find the best-suited rendition
     * (format and size) directly from the manifest itself.
     * </p>
     */
    height?: number;

    /**
     * Whether or not this composite has been modified since last synchronized.
     *
     * @enum {string}
     */
    state: AdobeDCXState;

    /**
     * Most composites have some internal structure, which is encoded as a tree.
     * This attribute optionally contains the list of children of the root node.
     * @see https://git.corp.adobe.com/caf/xcm/blob/master/docs/dcx.md#structure
     */
    children?: AdobeDCXNodeData[];

    /**
     * Composites are composed of components, which are listed in this array.
     * @see https://git.corp.adobe.com/caf/xcm/blob/master/docs/dcx.md#components
     */
    components?: AdobeDCXComponentData[];

    /**
     * A JSON HAL object that records related links, such as an original source location.
     *
     * @see http://tools.ietf.org/html/draft-kelly-json-hal-06
     */
    _links?: {
        [rel: string]: HalLink;
    };
}

export interface AdobeDCXRootNodeData extends AdobeDCXNodeData {
    /**
     * The creation date of the entire composite (not just the manifest).
     * It's an RFC 3339 timestamp.
     * @example
     * ```
     * 1996-12-19T16:39:57-08:00
     * ```
     */
    created?: string;

    /**
     * The modification date of the entire composite (not just the manifest).
     * It's an RFC 3339 timestamp.
     * @example
     * ```
     * 1996-12-19T16:39:57-08:00
     * ```
     */
    modified?: string;

    /**
     * An array of strings, intended to be assigned by the user, and used for organization or search purposes.
     */
    tags?: string[];

    /**
     * The current version of the manifest syntax, as recorded by the framework implementation upon writing
     * the manifest. Clients need never consult this value, but framework implementations may take it into
     * account when reading manifests.
     */
    'manifest-format-version': number;

    /**
     * The units for the coordinate system (if any) of this composite.
     * Can be used in a DCX Element to override the default units of the enclosing composite (e.g., DCX2D).
     * Valid values are: (pixel, point, inch, mm, cm)
     *
     * @enum {string}
     */
    units?: AdobeDCXUnit;
}

export interface AdobeDCXNode {
    /**
     * The id of the node. Must be a unique among the nodes of the composite.
     * <p>Cannot be changed for a node that is part of a branch or element.</p>
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    id: string;
    /**
     * The name of the node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    name?: string;
    /**
     * The type of the node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    type: string;
    /**
     * The path property of the node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    path?: string;

    /**
     * The relationship of the node to its parent node.
     *
     * @memberof AdobeDCXNode#
     * @type {String}
     */
    relationship?: string;

    readonly owner?: AdobeDCXBranch | AdobeDCXElement;

    /**
     * The absolute path of the parent of the node.
     *
     * @memberof AdobeDCXNode#
     * @readonly
     * @type {String}
     */
    readonly parentPath?: string;

    /**
     * The absolute path of the node.
     *
     * @memberof AdobeDCXNode#
     * @readonly
     * @type {String}
     */
    readonly absolutePath?: string;

    /**
     * Whether this node is the root of its branch or element.
     *
     * @memberof AdobeDCXNode#
     * @readonly
     * @type {Boolean}
     */
    readonly isRoot: boolean;

    /**
     * Returns an array of non-standard keys that are present at this node.
     * @returns {Array} An array of all non-standard property keys.
     */
    getCustomKeys(): string[];

    /**
     * Returns the object or value for the given key.
     * @param   {String} key The custom key to look up.
     * @returns {*} The value or object for the key.
     */
    getValue<T = any>(key: string): T;

    /**
     * Returns the object or value for the given key.
     * @param {String}   key   The custom key to set the value for.
     * @param {*} value The value or object.
     */
    setValue<T = any>(key: string, value: T): void;

    /**
     * Removes the object or value for the given key.
     * @param {String} key The custom key to remove.
     */
    removeValue(key: string): void;

    /**
     * Creates and returns a deep copy of the node sans any of its components or children.
     * @returns {AdobeDCXNode} The copy.
     */

    /**
     * Removes the link with the given relationship.
     * @param {String} relationship The relationship of the link to the node.
     *
     */
    removeLink(relationship: string): void;

    /**
     * Sets the link with the given relationship to the given object.
     * @param {Object} link         A JS object representing the link.
     * @param {String} relationship The relationship of the link to the node.
     */
    setLink(link: HalLink, relationship: string): void;

    /**
     * Returns the link with the given relationship as a JS object or undefined if the node
     * doesn't have such a link.
     * @param   {String} relationship The relationship of the link to the node.
     * @returns {Object} The link with the given relationship as a JS object or undefined if
     *                   the node doesn't have such a link.
     */
    getLink(relationship: string): HalLink | undefined;

    /**
     * Returns false if any of the properties of the given node is different from the properties
     * of this node. Recurses both data structures.
     * @param   {AdobeDCXNode} node                          The node to compare with.
     * @param   {Array}        [nodePropertiesToIgnore]      Optional. An object having the properties
     *                                                       that should not be compared for node.
     * @param   {Array}        [componentPropertiesToIgnore] Optional. An object having the properties
     *                                                       that should not be compared for components.
     * @returns {Boolean}
     */
    isEqualTo(
        node: AdobeDCXNode,
        nodePropertiesToIgnore?: Record<string, boolean>,
        componentPropertiesToIgnore?: Record<string, boolean>,
    );
}
