/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { RepoDownloadStreamableReturn } from '../AdobeStream';
import { AdobeDCXError } from '../error';
import { AdobeResponse, AdobeResponseType, ProgressCallback } from '../http';
import { LinkSet } from '../link';
import { IAdobePromise } from '../promise/AdobePromise';

// TODO: Move AdobeRepoAPISession interface to common-types
type AdobeRepoAPISession = any;

/**
 * Holds the results of an individual successful component upload.
 * Describes a single component, which in context of an UploadResults instance,
 * can be used to fully describe the component.
 */
export interface AdobeUploadRecord {
    id: string;
    etag: string;
    version: string;
    md5: string;
    length: number;
    type: string;
}

export interface AdobeComponentUploadRecord extends AdobeUploadRecord {
    location: string;
    revision: string;
    links: LinkSet;
}

/**
 * Callback for monitoring progress of a upload transfer.
 * @callback UploadProgressCallback
 *    @param {Integer} bytesCompleted
 *    @param {Integer} totalBytes
 *    @param {boolean} indeterminate
 */
export type UploadProgressCallback = ProgressCallback & {
    (bytesCompleted: number, totalBytes: number, indeterminate: boolean): void;
};

/**
 * The parsed object representation of a ComponentDescriptor serialized object.
 */
export interface DeserializedComponentDescriptor {
    /**
     * Serialization version
     */
    versionId: string;

    /**
     * Component MIME type
     */
    type: string;

    /**
     * Composite asset ID (URN)
     */
    cloudAssetId: string;

    /**
     * Composite ID (UUID)
     */
    compositeId?: string;

    /**
     * Component ID (UUID)
     */
    componentId: string;

    /**
     * Component revision
     */
    componentRevisionId: string;

    /**
     * Composite repository ID
     */
    repositoryId?: string;

    /**
     * Value of component data hash, in format hashType
     */
    hashValue?: string;

    /**
     * Hash format for hashValue
     */
    hashType?: string;

    /**
     * Component etag
     */
    etag?: string;

    /**
     * Expiration, used by other DCX libs
     */
    cloudExpiration?: string;

    /**
     * Component data size
     */
    size?: number;
}

export interface AdobeUploadResults {
    /**
     * Key-value pairs, mapping componentId => UploadRecord
     */
    records: Record<string, AdobeUploadRecord>;

    /**
     * Composite ID (UUID)
     */
    compositeId?: string;

    /**
     * Composite cloud asset ID (URN)
     */
    compositeAssetId: string;

    /**
     * Repository ID
     *
     * Must be defined for RAPI DCXComposites
     */
    repositoryId?: string;

    /**
     * Add an UploadRecord for componentId to the results.
     *
     * @param componentId
     * @param record
     */
    addUploadRecord(componentId: string, record: AdobeUploadRecord): void;

    /**
     * Get a serialized component descriptor for a specific component in UploadResults.
     *
     * @note Only works for RAPI. The UploadResults object must contain a repositoryId.
     *
     * @param {string}                  componentId     - The component ID to serialize
     * @returns {string}
     */
    getComponentDescriptor(componentId: string): string;

    /**
     * Get a URL that can be used to download the component.
     * If size is over threshold allowed for single download, return presigned direct-download URL.
     *
     * @note Only works for RAPI. The UploadResults object must contain a repositoryId.
     *
     * @param {AdobeRepoAPISession}     session         - Repo session to use
     * @param {string}                  componentId     - Component ID to get the URL for
     */
    getComponentURL(
        session: AdobeRepoAPISession,
        componentId: string,
    ): IAdobePromise<
        { isPresignedUrl: boolean; url: string; response: AdobeResponse<'json'> | undefined },
        AdobeDCXError
    >;

    /**
     * Get the component data, using either single download or parallelized block download.
     *
     * @note Only works for RAPI. The UploadResults object must contain a repositoryId.
     *
     * @param {AdobeRepoAPISession}     session         - Repo session to use
     * @param {string}                  componentId     - Component ID to get data for
     * @param {AdobeResponseType}       [responseType]  - Response type for the data, defaults to Buffer (Node) or ArrayViewBuffer (Browser).
     */
    getComponent<T extends AdobeResponseType>(
        session: AdobeRepoAPISession,
        componentId: string,
        responseType?: T,
    ): RepoDownloadStreamableReturn<T>;
}
