/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeResponseType } from '.';
import { HTTPMethod } from './AdobeHttpService';
import { RequestDescriptor } from './AdobeRequest';
import { AdobeXhr, XhrOptions } from './AdobeXhr';

export type BackoffIncreaser = (count?: number, last?: number, initial?: number) => number;

export interface BackoffSnapshot<T extends AdobeResponseType> {
    count: number;
    canceled: boolean;
    timedOut: boolean;
    requests: AdobeXhr<T>[];
    totalWaited: number;
    duration: number;
    requestPending: boolean;
    waitingFor: number;
}

export interface Backoff<T extends AdobeResponseType> {
    getPromise: () => Promise<AdobeXhr<T>>;
    cancel: () => void;
    onProgress: (h: (p1: number, p2: number) => void) => void;
    getSnapshot(): BackoffSnapshot<T>;
}

type ScheduleCallback<T extends AdobeResponseType> = (snapshot?: BackoffSnapshot<T>) => void | Promise<void>;

export type PrePostBackoffCallback<T extends AdobeResponseType> = (
    xhr?: AdobeXhr<T>,
    snapshot?: BackoffSnapshot<T>,
) => void | Promise<void>;

export interface BackoffOptions<T extends AdobeResponseType> extends XhrOptions<T> {
    /**
     * Response type of the request
     */
    responseType?: AdobeResponseType;

    /**
     * Authentication callback, if defined used for fetching new token.
     *
     * @note if the URL is not on the authentication allow list, the promise
     * will reject to DCXError#UNAUTHORIZED
     *
     */
    authCallback?: (url: string, headers: Record<string, string>) => Promise<Record<string, string>>;

    /**
     * Array of progress listeners
     */
    progressListeners?: ((sor: number, t: number) => void)[];

    /**
     * An array of numbers or regular expressions to check the status code against
     * to determine whether to retry. If the list contains the status code or one
     * of the regular expressions matches, the request will be retried.
     *
     * @default [/^(?!^501$|^507$)^(5\d{2})$|429$/]
     */
    retryCodes?: (number | RegExp)[];

    /**
     * A total amount of time to wait in milliseconds for all combined request waits.
     * Does not factor in the duration of the request, only the duration of the waits between retries.
     *
     * @default 72000
     */
    timeoutAfter?: number;

    /**
     * A function that returns the next duration in milliseconds to wait before retrying a request.
     *
     * @default ExponentialBackoff
     */
    increase?: BackoffIncreaser;

    /**
     * Whether to attempt to use the failure response's `Retry-After` header.
     * Handles both Date or seconds directive, depending on header.
     * If no retry-after header is found, will default to using the increaser function.
     *
     * @default true
     * @see https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Retry-After#directives
     */
    preferRetryAfterHeader?: boolean;

    /**
     * Number of milliseconds to wait before attempting the first retry.
     *
     * @default 2000
     */
    initialWait?: number;

    /**
     * Number of milliseconds at which to stop increasing the wait duration.
     * If the increaser function is customized, this will not be honored and
     * it is up to clients to cap their increaser function.
     *
     * @default 32000
     */
    maxWait?: number;

    /**
     * Disable all retries entirely
     *
     * @default false
     */
    disableRetry?: boolean;

    /**
     * Retry upon network error
     *
     * @default true
     */
    retryNetworkError?: boolean;

    /**
     * @internal
     * @private
     */
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    forceXhr?: any;

    /**
     * Called when the request is about to be scheduled for some duration in the future.
     */
    preScheduleCallback?: ScheduleCallback<T>;

    /**
     * Called immediately after the request has been scheduled for retry.
     */
    postScheduleCallback?: ScheduleCallback<T>;

    /**
     * Called immediately before the retry request is sent.
     */
    preCallback?: PrePostBackoffCallback<T>;

    /**
     * Called immediately on response of the retry request.
     */
    postCallback?: PrePostBackoffCallback<T>;

    /**
     * Response header to check for a URL to poll, when a specific response code is encountered.
     * Used only if polling is enabled for the response code encountered.
     */
    pollHeader?: string;

    /**
     * HTTP verb to use for polling, if it is encountered.
     *
     * @default "GET"
     */
    pollMethod?: HTTPMethod;

    /**
     * HTTP response codes to consider poll-able
     * If the response from the initial HTTPService#invoke call matches one of these,
     * and if `pollHeader` is exists, and the response headers contains a value for that key,
     * perform polling on that header value.
     *
     * @default []
     * @see BackoffOptions#retryCodes
     */
    pollCodes?: (number | RegExp)[];

    /** @internal */
    descriptor?: RequestDescriptor;
}
