/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { AdobeResponseType } from '.';
import { AdobeDCXError } from '../error';
import { IAdobePromise } from '../promise/AdobePromise';
import { AdobeAuthProvider } from './AdobeAuthProvider';
import { BackoffOptions, BackoffSnapshot } from './AdobeBackoff';
import { BodyType, RequestDescriptor } from './AdobeRequest';
import { AdobeResponse } from './AdobeResponse';
import { AdobeXhr } from './AdobeXhr';
import { AdditionalNodeOptions } from './AdobeXhrNode';

/**
 * Invoke response callback
 */
export interface ResponseCallback<T extends AdobeResponseType = any> {
    (error?: undefined, xhr?: AdobeResponse<T>, responseData?: T): void;
    (error?: AdobeDCXError, xhr?: AdobeResponse<T>, responseData?: T): void;
}

/**
 * Options for initializing a new HTTPService
 */
export interface AdobeHTTPServiceOptions {
    maxOutstanding?: number;
    useAuthProvider?: boolean;
    timeout?: number;
    crossOriginCredentials?: boolean;
    server?: string;
    preferFetch?: boolean;
    requestIdPrefix?: string;
}

/*
 * Options for a request.
 * @typedef {Object} InvokeOptions
 *   @property {String}             [responseType]      Possible values: `'blob'`, `'text'`, `'buffer'`, `'arraybuffer'`, `stream`
 *   @property {Time}               [noSoonerThen]      JavaScript Time value (e.g. `Date.now()`)
 *   @property {RequestDescriptor}  [reuseRequestDesc]  An inactive request descriptor previously obtained by a
 *                                                      call to this `invoke()`.
 *   @property {Boolean}            [isExternalRequest] If `true` secure headers will not be sent.
 */
export interface InvokeOptions<T extends AdobeResponseType = any> {
    responseType?: T;
    noSoonerThen?: number;
    reuseRequestDesc?: Partial<RequestDescriptor> & { id: string };
    retryOptions?: Pick<
        BackoffOptions<T>,
        | 'authCallback'
        | 'retryCodes'
        | 'timeoutAfter'
        | 'increase'
        | 'preferRetryAfterHeader'
        | 'initialWait'
        | 'maxWait'
        | 'preScheduleCallback'
        | 'postScheduleCallback'
        | 'disableRetry'
        | 'pollCodes'
        | 'pollHeader'
        | 'pollMethod'
    >;
    autoParseJson?: boolean;
    timeout?: number;
    isStatusValid?: (status?: number, response?: AdobeResponse) => boolean | AdobeDCXError;
    additionalNodeOptions?: AdditionalNodeOptions;
    isExternalRequest?: boolean;
}

/**
 * Gets passed into createHTTPService() and called back whenever the service needs a new
 * authentication token.
 * @callback AuthenticationCallback
 *    @param {AdobeHTTPService | AdobeAuthProvider} serviceOrProvider Service callback use is deprecated,
 *                                                  use {@link AdobeHTTPServiceOptions} `useAuthProvider` set to true.
 *                                                  The service-based callback will be removed in the future.
 */
export type AuthenticationCallback = (serviceOrProvider?: AdobeHTTPService | AdobeAuthProvider) => void | Promise<void>;

export type RequestMethod = 'get' | 'put' | 'post' | 'options' | 'head';

export type HTTPMethod =
    | 'GET'
    | 'POST'
    | 'PUT'
    | 'HEAD'
    | 'OPTIONS'
    | 'PATCH'
    | 'DELETE'
    | 'delete'
    | 'get'
    | 'post'
    | 'put'
    | 'head'
    | 'options';

export type PreRequestHook = (xhr?: AdobeXhr<any>, backoffSnapshot?: BackoffSnapshot<any>) => void;
export type PostRequestHook = (xhr?: AdobeXhr<any>, backoffSnapshot?: BackoffSnapshot<any>) => void;

/**
 * Interface for the refactored HTTPService.
 * For now the interface is defined in src/HTTPService.ts
 */
export interface AdobeHTTPService {
    /**
     * An inactive service will error out with a AdobeDCXError.SERVICE_IS_INACTIVE error
     * whenever a request is being made. Clients can set isActive to false when the user
     * has explicitly logged out.
     * @default true
     */
    isActive: boolean;

    /**
     * Whether the platform handles redirects natively.
     * Returns true for browser and false for Node,
     * although Node builds may also be configured to handle redirects.
     */
    handlesRedirects: boolean;

    /**
     * If true then XHR instances will have the withCredentials property set which means that user
     * credentials (stored in a cookie) will be used in cross origin requests.
     */
    crossOriginCredentials: boolean;

    /**
     * The maximum number of outstanding requests over this HTTP Service instance.
     * The default value is 5, to stay below browser connection limits, but the limit
     * can be increased if you have special circumstances that warrant it.
     * Remember that there is no use in exceeding the bandwith capacity of your client
     * or of the host, and that in error cases it's better to have half of your
     * uploads complete than all of your uploads half complete.
     */
    maxOutstanding: number;
    server?: string;
    /**
     * Feature flags are intended to be temporary in nature.
     * These values are expected to be able to be enabled/disabled for testing/release purposes.
     */
    featureFlags: any;

    /**
     * Used for redirects & outgoing requests.
     *
     * Sets the authenticationAllowList on AdobeAuthProvider.
     */
    authenticationAllowList: string[];

    /**
     * The AdobeAuthProvider instance.
     * If an instance is not provided at construction, one will be created.
     */
    authProvider: AdobeAuthProvider;

    /**
     * Sets additional headers.
     * @param {Object} additionalHeaders An object containing key-value pairs for each additional header to supply.
     */
    setAdditionalHeaders(hdrs: Record<string, string>): void;

    /**
     * Sets additional options.
     * @param {Object} additionalOptions An object containing additional options to supply with each http request (in Node only).
     */
    setAdditionalNodeOptions(options: AdditionalNodeOptions): void;

    /**
     * Sets the timeout value for requests.
     * This is the per-request timeout, not related to backoff timeouts.
     * To set the backoff timeout, provide a value in the `retryOptions`
     * property of the invoke options.
     * @param {Integer} timeout timeout value in ms or undefined if no timeout
     */
    setTimeout(timeout?: number): void;

    /**
     * Sets the authentication token and resumes the service.
     * @param {String} token The authentication token, if not provided, logs out
     */
    setAuthToken(token?: string): void;

    /**
     * Sets the API key.
     * @param {String} apiKey The apikey to supply with each request.
     */
    setApiKey(key?: string): void;

    /**
     * Sets a function that denotes whether a Promise should resolve or reject given a status
     * By default, this is function that always returns true meaning any status code will resolve.
     * @param {(status?: number) => boolean} fn
     */
    setValidateStatus(fn?: (status?: number) => boolean): void;

    /**
     * Set options for retrying failed requests
     * @param {BackoffOptions<any>} retryOptions
     */
    setRetryOptions(retryOptions: BackoffOptions<never>): void;

    /**
     * Sets callbacks before and after invocation hook.
     * (For testing)
     * @private
     * @param before a hook to call before each http request is issued. Signature: before(req).
     * @param after  a hook to be called after each http request is initiated: Signature: after(req, xhr).
     */
    setRequestHooks(before?: PreRequestHook, after?: PostRequestHook): void;

    /**
     * Call this when you do not use an auth token for authentication and you need to let the
     * service know that you have renewed the authentication cookie so that the service can
     * resume to make requests.
     */
    resume(): void;

    invoke<T extends AdobeResponseType = any>(
        method: HTTPMethod,
        href: string,
        headers?: Record<string, string>,
        body?: BodyType,
        options?: InvokeOptions<T>,
    ): IAdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>;
    /**
     * Issue a request.
     * By default, the request is a GET.
     * The body param specifies the data, if any.
     * The generic T denotes the responseType for type completion/checking.
     *
     * Note: use EITHER callback or promise, using both may lead to undefined resolution.
     *
     * @example
     * ```js
     * // with callback
     * service.invoke<SomeInterface>('GET', 'https://server.url/path', {}, null,
     *  { responseType: 'json' },
     *  (err, xhr, data) => {
     *    // do stuff
     *    // data is typeof === 'object' if response was valid JSON
     *    // data is Type `SomeInterface`
     *  });
     * ```
     *
     * * @example
     * ```js
     * // with promise, async/await
     * try {
     *   const [xhr, data] = await service.invoke<SomeInterface>(
     *      'GET',
     *      'https://server.url/path',
     *      {},
     *      null,
     *      { responseType: 'json' });
     *   // data is typeof === 'object'
     *   // data is Type `SomeInterface`
     * } catch(e) {
     *   // handle network error
     * }
     * ```
     * @param  {String}            method      - HTTP method
     * @param  {String}            href        - The URL; may be relative or absolute.
     *                                          If relative, the `server` property will be prepended (if exists)
     *                                          If absolute (contains endpoint & protocol), will be sent as-is
     * @param  {Record<string, string | string[]>} headers
     *                                         - The request headers
     * @param  {BodyType}          [body]      - The request body (Buffer/ArrayBufferView/Blob: data)
     * @param  {InvokeOptions}     [options]   - Object capturing additional request options.
     * @param  {ResponseCallback}  callback    - Called when the response arrives. Signature: callback(error, xhr)
     * @return {AdobePromise<[AdobeResponse<T>, T], AdobeDCXError, RequestDescriptor>}  -
     *                                          An AdobePromise with extended props of a RequestDescriptor.
     *                                          Resolves to a tuple of [AdobeResponse, Data] where data is the type of
     *                                          the `responseType` option. For type completion, use the generic `T` with
     *                                          the expected responseType.
     */
    invoke<T extends AdobeResponseType = any>(
        method: HTTPMethod,
        href: string,
        headers?: Record<string, string>,
        body?: BodyType,
        options?: InvokeOptions<T>,
        callback?: ResponseCallback<T>,
    ): IAdobePromise<AdobeResponse<T>, AdobeDCXError, RequestDescriptor>;
}
