/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeDCXError } from '../error';
import { LinkRelationKey } from '../link';
import { AdobeDCXComponent, ComponentCallback } from './Component';
import { AdobeDCXElement, NodeCallback } from './Element';
import { AdobeDCXNode, AdobeDCXRootNodeData } from './Node';
import { AdobeUploadResults } from './UploadResults';
/**
 * The archival state of the composite. Can be <em>'active'</em> (default), <em>'pending'</em>,
 * or <em>'archived'</em>.
 * @type {String}
 */
export type CompositeArchivalState = 'pending' | 'active' | 'archived';
/**
 * The editing state of the composite. Can be <em>'unmodified'</em>, <em>'modified'</em>,
 * <em>'pendingDelete'</em> or <em>'committedDelete'</em>.
 * @type {String}
 */
export type CompositeState = 'unmodified' | 'modified' | 'pendingDelete' | 'committedDelete';
export type AdobeDCXBranchCore = any;
export interface AdobeDCXBranch {
    originalManifestFormatVersion: number;
    compositeHref: string;
    /**
     * The id of the composite. Must be a unique among the nodes of the composite.
     *
     * <p>While not strictly read-only most clients do not ever have to modify this property.</p>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeId: string;
    /**
     * An AdobeDCXNode object that represents the root of the underlying manifest.
     * @type {AdobeDCXNode}
     * @memberof AdobeDCXBranch#
     * @readonly
     */
    rootNode: AdobeDCXNode;
    /**
     * The name of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    name?: string;
    /**
     * The type of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    type: string;
    /**
     * The asset id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeAssetId?: string;
    /**
     * The repository id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeRepositoryId?: string;
    /**
     * The etag of the manifest of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     * @readonly
     */
    manifestEtag?: string;
    /**
     * The editing state of the composite. Can be <em>'unmodified'</em>, <em>'modified'</em>,
     * <em>'pendingDelete'</em> or <em>'committedDelete'</em>.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeState: CompositeState;
    /**
     * The archival state of the composite. Can be <em>'active'</em> (default), <em>'pending'</em>,
     * or <em>'archived'</em>.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    compositeArchivalState: CompositeArchivalState;
    /**
     * Whether the composite is bound to a composite on a server. If <em>false</em> for newly
     * created empty composites that have never been pulled from or pushed to the server.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Boolean}
     */
    isBound: boolean;
    /**
     * The version identifier of the banch.
     * Only valid if the branch has been pushed or pulled, undefined otherwise.
     *
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {string}
     */
    versionId?: string;
    /**
     * Whether the composite has has been modified in memory and
     * needs to be committed to local storage.
     *
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Boolean}
     */
    isDirty: boolean;
    data: AdobeDCXRootNodeData;
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    localData: string;
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    remoteData: string;
    /**
     * Lists any elements (type AdobeDCXElement) that have been created and have not yet
     * been used to update the corresponding child node or been abandoned.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Array<AdobeDCXElement>}
     */
    pendingElements: AdobeDCXElement[];
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    changeCount: number;
    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    readOnly: boolean;
    /**
     * Restores the branch from its stringified representation (manifest). Throws if data is invalid.
     * @param   {String} data The stringified representation.
     * @returns {AdobeDCXBranch}.
     */
    parse(data: string): AdobeDCXBranch;
    /**
     * Returns the element with the given id or undefined if not found.
     * Throws if the node in question is not a valid element).
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @param   {String } id The id of the child node to look up.
     * @returns {AdobeDCXElement}
     */
    getElementWithId(id: string): AdobeDCXElement | undefined;
    /**
     * Returns the element with the given absolute path or undefined if not found.
     * Throws if the node in question is not a valid element).
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @memberof AdobeDCXBranch#
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXElement}
     */
    getElementWithAbsolutePath(path: string): AdobeDCXElement | undefined;
    /**
     * Creates a new element node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new element.
     * Throws if the path is invalid.
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @param   {String}                name            - The name of the new element.
     * @param   {String}                type            - The type of the new element.
     * @param   {String}                path            - The path of the new element.
     * @param   {String}                [nodeId]        - The id of the new child. If undefined the new child node will
     *                                                    get a random id.
     * @param   {Integer}               [index]         - If given and less than or equal to the current number of
     *                                                    children than the node gets inserted at the given index.
     *                                                    Otherwise it gets added to the end.
     * @param   {@link AdobeDCXNode}    [parentNode]    - The parent node to add the node to. Default parent is the
     *                                                    root node.
     * @returns {AdobeDCXElement}
     */
    addElement(name: string, type: string, path: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXElement;
    /**
     * Updates the data of the existing element in the branch.
     * Notice that the element object will no longer be tracked by the branch after updateElement
     * has been called. You will have to request the element again using getElementWithId or
     * getElementWithAbsolutePath if you want to keep working with it.
     * Throws if the element doesn't exist or if the update results in duplicate paths/ids.
     *
     * @param {AdobeDCXElement} element The modified element.
     * @returns {AdobeDCXNode} The updated node in the branch.
     */
    updateElement(element: AdobeDCXElement): AdobeDCXNode;
    /**
     * Cleans up an element that was previously created and has not been used to update the corresponding
     * node in the branch. Throws if the branch doesn't recognize the element (e.g. calling abandonElement
     * twice or after already having called updateElement).
     *
     * @param {AdobeDCXElement} element The element that is no longer needed.
     */
    abandonElement(element: AdobeDCXElement): void;
    /**
     * Returns the node with the given id or undefined if not found.
     * @param   {String}     id The id of the child node to look up.
     * @returns {AdobeDCXNode}
     */
    getChildWithId(id: string): AdobeDCXNode | undefined;
    /**
     * Returns the node with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXNode}
     */
    getChildWithAbsolutePath(path: string): AdobeDCXNode | undefined;
    /**
     * Generates and returns an array of the child nodes of the given parent node.
     * @example
     * var rootNodes = branch.getChildrenOf(branch.rootNode);
     * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
     * @returns {Array}
     */
    getChildrenOf(parentNode?: AdobeDCXNode): AdobeDCXNode[];
    /**
     * Creates a new node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new child node.
     * @param   {String}  [name]       The name of the new child. If undefined the child will not
     *                                 have a name.
     * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
     *                                 get a random id.
     * @param   {Integer} [index]      If given and less than or equal to the current number of
     *                                 children than the node gets inserted at the given index.
     *                                 Otherwise it gets added to the end.
     * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
     *                                 root node.
     * @returns {AdobeDCXNode}
     */
    addChild(name?: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;
    /**
     * Removes and returns the given child node from the branch.
     * @param   {AdobeDCXNode} node The child node to remove.
     * @returns {AdobeDCXNode}
     */
    removeChild(node: AdobeDCXNode): AdobeDCXNode;
    /**
     * Moves the existing child from its current parent/index to the given parent/index.
     * @param   {AdobeDCXNode} node         The child node to move
     * @param   {Integer}      [index]        If given and less than or equal to the current number of
     *                                      children than the node gets inserted at the given index.
     *                                      Otherwise it gets added to the end.
     * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
     *                                      the root.
     * @returns {AdobeDCXNode}
     */
    moveChild(node: AdobeDCXNode, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode;
    /**
     * Copies the given child node as a new node into this branch. The node can be from the same or
     * from a different composite.
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Fails if a node with the same id or same absolute path already exists.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two composites stored at different endpoints.</p>
     *
     * @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
     *                                      newPath must be provided.
     * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
     *                                      then the new child node will be added to the root of the
     *                                      branch.
     * @param   {Integer}      [index]      If provided and less than or equal to the current number of
     *                                      children of the parentNode (or root) the child node gets
     *                                      inserted at the given index. Otherwise it gets added to
     *                                      the end.
     * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
     *                                      this a its path property and it will also receive a new
     *                                      random id (unless one is provided with the newId param).
     *                                      If left undefined then the copy of the node will keep
     *                                      the path of the original. In either case the function will
     *                                      fail if the resulting absolute path of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing absolute path.</p>
     *                                      <p>You must provide a newPath if you are copying the root
     *                                      node of a branch or element.</p>
     * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
     *                                      this a its id. If left undefined (and the newPath param
     *                                      is also undefined) then the copy will retain the id of
     *                                      the original. In either case the function will
     *                                      fail if the resulting id of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                      when not using local storage.
     *                                      Gets called when the copy is done or has failed.
     * @returns {AdobeDCXNode}              Only returns the created child node if no callback is
     *                                      given.
     */
    copyChild(node: AdobeDCXNode, parentNode?: AdobeDCXNode, index?: number, newPath?: string, newId?: string, callback?: NodeCallback): AdobeDCXNode;
    /**
     * <p>Replaces the child node in this branch with a copy of the given node or branch with the same
     * id (or, if provided, with the given id). Fails if the child node does not exist in this branch.</p>
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two different composites.</p>
     *
     * @param   {AdobeDCXNode} node       The child node to update from.
     * @param   {String}       [newPath]  If provided the copy of the component will be assigned
     *                                    this a its path property. Otherwise it will retain its original path.
     * @param   {String}       [newId]    If provided the copy of the child node will be assigned
     *                                    this as its id. Otherwise it will retain its original id.
     * @param   {NodeCallback} [callback] Optional when not copying between different composites or
     *                                    when not using local storage.
     *                                    Gets called when the copy is done or has failed.
     * @returns {AdobeDCXNode}            Only returns the created child node if no callback is
     *                                    given.
     */
    replaceChild(node: AdobeDCXNode, newPath?: string, newId?: string, callback?: NodeCallback): AdobeDCXNode;
    /**
     * Returns an array of all components in the branch.
     * @returns {Array}.
     */
    allComponents(): AdobeDCXComponent[];
    /**
     * Returns the component with the given id or undefined if not found.
     * @param   {String} id The id of the component to look up.
     * @returns {AdobeDCXComponent}
     */
    getComponentWithId(id: string): AdobeDCXComponent | undefined;
    /**
     * Returns the component with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path of the desired component.
     * @returns {AdobeDCXComponent}
     */
    getComponentWithAbsolutePath(path: string): AdobeDCXComponent | undefined;
    /**
     * Returns an array containing the components of the given node.
     * @param   {AdobeDCXNode} parentNode The node whose components to return.
     * @returns {Array}
     */
    getComponentsOf(parentNode?: AdobeDCXNode): AdobeDCXComponent[];
    /**
     * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
     * @param error An IMCOMPLETE_COMPOSITE error
     * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
     */
    getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[];
    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param componentDescriptor The serialized component descriptor to use.
     * @param name          The name of the new component.
     * @param path          The path of the new component. Must satisfy uniquenes
     * @param relationship  The relationship of the new component.
     *                                            rules for components.
     * @param parentNode  The node to add the node to. Defaults to the root.
     * @returns {AdobeDCXComponent}               The new component.
     */
    addComponentWithComponentDescriptor(componentDescriptor: string, name: string, path: string, relationship?: LinkRelationKey, parentNode?: AdobeDCXNode): any;
    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param   {String}            name          The name of the new component.
     * @param   {String}            relationship  The relationship of the new component.
     * @param   {String}            path          The path of the new component. Must satisfy uniquenes
     *                                            rules for components.
     * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
     * @param   {Object}            uploadResults The upload results object returned by a previous call
     *                                            to AdobeDCXCompositeXfer.uploadAssetForNewComponent().
     * @returns {AdobeDCXComponent}               The new component.
     */
    addComponentWithUploadResults(name: string, relationship: string | undefined, path: string, parentNode: AdobeDCXNode | undefined, uploadResults: AdobeUploadResults): AdobeDCXComponent;
    /**
     * <strong>XHR-only</strong>
     *
     * <p>Updates the component record with the results of a recent upload of said component.</p>
     *
     * @param   {AdobeDCXComponent} component     The component.
     * @param   {Object}            uploadResults The upload results object returned by a previous
     *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent} The updated component.
     */
    updateComponentWithUploadResults(component: AdobeDCXComponent, uploadResults: AdobeUploadResults): AdobeDCXComponent;
    /**
     * Removes the component from the branch.
     * @param   {AdobeDCXComponent} component The component to remove.
     * @returns {AdobeDCXComponent} The removed component.
     */
    removeComponent(component: AdobeDCXComponent): AdobeDCXComponent;
    /**
     * Moves the component to the given node or the root if node is undefined
     * @param   {AdobeDCXComponent} component    The component to move.
     * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
     * @returns {AdobeDCXComponent} The moved component.
     */
    moveComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode): AdobeDCXComponent;
    /**
     * Copies the given component and adds it as a new component to this branch. Fails if the
     * component already exists.
     *
     * <p>Notice: This method does not work without local storage (browser environment) if
     * used to copy between two composites with different endpoints.</p>
     *
     * @param   {AdobeDCXComponent} component    The component to copy.
     * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
     *                                           provided then the component will be added to the
     *                                           root.
     * @param   {String}            [newPath]    If provided the copy of the component will be
     *                                           assigned this a its path property and it will also
     *                                           get assigned a random new id if none is provided via
     *                                           the <em>newId</em> param.
     * @param   {String}            [newId]      If provided the copy of the component will be assigned
     *                                           this a its id. If left undefined (and if newPath is
     *                                           undefined as well) then the copy of the component
     *                                           will retain the id of the original.
     *                                           This is useful when merging conflicting changes since
     *                                           it preserves the identity of components and avoids
     *                                           unnecessary network traffic.
     * @param   {ComponentCallback} [callback]   Optional when not copying between different
     *                                           composites or when copying without local storage.
     *                                           Gets called when the copy is done or has failed.
     * @returns {AdobeDCXComponent}              Only returns the new component when called without
     *                                           a callback.
     */
    copyComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode, newPath?: string, newId?: string): AdobeDCXComponent;
    copyComponent(component: AdobeDCXComponent, parentNode: AdobeDCXNode | undefined, newPath: string | undefined, newId: string | undefined, callback: ComponentCallback): void;
    /**
     * Replaces the matching component (same id) in this branch with a copy of the given component.
     * Fails if the component can't be found.
     *
     * <p>Notice: This method does not work without local storage (browser environment) if
     * used to copy between two different composites.</p>
     *
     * @param   {AdobeDCXComponent} component  The component to copy.
     * @param   {String}            [newPath]  If provided the copy of the component will be
     *                                         assigned this a its path property and it will also
     *                                         get assigned a new id if none is provided via the
     *                                         newId param.
     * @param   {String}            [newId]    If provided the copy of the component will be
     *                                         assigned this a its id property.
     * @param   {ComponentCallback} [callback] Optional when not copying between different
     *                                         composites. Gets called when the copy is done or has
     *                                         failed.
     * @returns {AdobeDCXComponent}            Only returns the new component when called without
     *                                         a callback.
     */
    replaceComponent(component: AdobeDCXComponent, newPath?: string, newId?: string): AdobeDCXComponent;
    replaceComponent(component: AdobeDCXComponent, newPath: string | undefined, newId: string | undefined, callback: ComponentCallback): void;
}
//# sourceMappingURL=Branch.d.ts.map