/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
import { AdobeAsset } from '../AdobeAsset';
import { AdobeDCXError } from '../error';
import { IAdobePromise } from '../promise/AdobePromise';
import { AdobeDCXBranch } from './Branch';
export type BranchCallback = (error?: AdobeDCXError, branch?: AdobeDCXBranch) => unknown;
export interface AdobeDCXComposite extends AdobeAsset {
    id?: string;
    href?: string;
    type?: string;
    assetId?: string;
    repositoryId?: string;
    collaborationType?: string;
    clientDataString?: string;
    readonly isBound: boolean;
    readonly current?: AdobeDCXBranch;
    /**
     * Loads the base manifest and instantiates a read-only branch if the manifest exists.
     *
     * <p>This method can only be called if the AdobeDCXComposite instance has been initialized with the
     * <em>xhrBaseBranchSupport</em> option.</p>
     *
     * @param {BranchCallback} [callback] Gets called on success or failure.
     * @returns                           Only returns a promise if no callback specified.
     */
    loadBaseBranch(): IAdobePromise<AdobeDCXBranch, AdobeDCXError>;
    loadBaseBranch(callback: BranchCallback): void;
    /**
     * Accepts the given merged branch as the new current branch, discards any temporary pull data
     * and also updates the base manifest. Is a no-op if no branch is provided and if there
     * is no pulled manifest in local storage (as may be the case if a pull didn't find any
     * modification on the server).
     *
     * @example
     * // NJS
     * pullComposite(composite, session, function (error, pulledBranch) {
     *      if (!error) {
     *          // The pull has succeeded. Now you need to incorporate the changes from the server
     *          // into your current branch.
     *          // Start the merge by making a mutable copy of pulledBranch
     *          var mergedBranch = pulledBranch.copy();
     *          // Now merge any changes you have made to current since the last successful push or
     *          // pull. You might want to also load the base branch to help you figure out what has
     *          // changed in either branch.
     *          ...
     *          // When done merging you need to make this call:
     *          composite.resolvePullWithBranch(mergedBranch, function (error, newCurrent) {
     *              if (!error) {
     *                  // Now you are done!
     *              }
     *          });
     *      }
     * });
     *
     * @example
     * // XHR
     * pullCompositeManifestOnly(composite, session, function (error, pulledBranch) {
     *      if (!error) {
     *          // The pull has succeeded. Now you need to incorporate the changes from the server
     *          // into your current branch. If you follow the recommendations for working in an
     *          // online environment you will not have modified current so you can just call:
     *          composite.resolvePullWithBranch(pulledBranch);
     *      }
     * });
     *
     *
     * <p><strong>The branch passed in must be derived from the pulled branch.</strong></p>
     *
     * @param {AdobeDCXBranch} branch     A branch which should become the new current branch. It
     *                                    should be the result of merging the previously pulled
     *                                    branch with any changes in current.
     * @param {BranchCallback} [callback] Optional if running in an <strong>XHR</strong>
     *                                    environment. Gets called on completion.
     * @return {AdobeDCXBranch}           Only returns the new current branch if called without
     *                                    a callback.
     */
    resolvePullWithBranch(branch: AdobeDCXBranch): void;
    resolvePullWithBranch(branch: AdobeDCXBranch, callback: BranchCallback): void;
    acceptPush(callback?: (err?: Error) => unknown): void | AdobeDCXBranch;
    resetBinding(): void;
    resetIdentity(): void;
}
//# sourceMappingURL=Composite.d.ts.map