/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/
/**
 * AuthProvider related types
 */
export interface AuthData {
    authToken?: string;
    apiKey?: string;
}
export type AuthEvent = 'unauthenticated' | 'updated';
export type AuthChangeHandler = (event: AuthEvent, provider?: AdobeAuthProvider) => void | Promise<void>;
export type DetachHandler = () => void;
export type PersistentRefreshCallback = (provider?: AdobeAuthProvider) => void | Promise<void>;
export interface AdobeAuthProvider {
    /**
     * List of domains that are allowed to receive authentication headers.
     * Used for all outgoing requests and for redirects in Node.js.
     * For redirects in browser, the browser/XMLHTTPRequest handles the authentication header stripping.
     *
     * @default ["adobe.io"]
     */
    authenticationAllowList: string[];
    /**
     * Whether the AuthProvider is configured to skip authorization.
     * This is automatically set to true when no persistent auth callback
     * is set in the constructor.
     *
     * Can be manually set to the expected value after constructing.
     */
    isNoAuthMode: boolean;
    /**
     * Current API key.
     * To set, use setApiKey()
     */
    apiKey: string | undefined;
    /**
     * Current authorization token.
     * To set, use setAuthToken()
     */
    authToken: string | undefined;
    /**
     * Auth token scheme, used as the prefix for the Authorization header.
     * @default "Bearer"
     */
    authTokenScheme: string | undefined;
    /**
     * Attach listener to auth change event
     *
     * @param handler
     * @returns Function to remove handler
     */
    onChange(handler: AuthChangeHandler): DetachHandler;
    /**
     * Get authorization data as object
     * Example: { authToken: "abc", apiKey: "xyz"}
     */
    getAuthData(): AuthData;
    /**
     * Get authorization data wrapped in promise for chaining.
     * Does not do anything additional to getAuthData(), besides the promise.
     */
    getAuth(): Promise<AuthData>;
    /**
     * Set API key.
     *
     * @param apiKey The new API key
     */
    setApiKey(apiKey: string): void;
    /**
     * Set the current authentication token.
     * @note This triggers the "updated" event.
     *
     * @param authToken The new token
     */
    setAuthToken(authToken: string): void;
    /**
     * Explicitly set the state to "unauthenticated", triggering the
     * "unauthenticated" event, which in turn forces the persistent
     * auth refresh callback to be called (if any).
     *
     * Resolves to AuthData once the state becomes "updated".
     * If state is never changed, this promise will not resolve.
     */
    refreshAuth(): Promise<AuthData>;
    /**
     * Explicitly set the state to "updated", triggering the "updated" event.
     */
    resume(): void;
    /**
     * Remove current auth data, set state to "unauthenticated".
     */
    logout(): void;
    /**
     * Check if URL is allowed to receive authorization headers,
     * based on the authorizationAllowList.
     *
     * @param url URL to check
     */
    isAuthorizedURL(url: string): boolean;
    /**
     * Applies Authorization and X-Api-Key headers if the url is part of the allowed
     * and if the incoming headers do not contain a "null" value for their corresponding override.
     *
     * @note This should be used with normalized headers (ie. all lowercase keys) to
     * avoid the case where multiple values are set like { "X-Api-Key": null, "x-api-key": "abc" }
     *
     * @param headers Incoming headers
     * @returns Headers with auth data applied (lowercase keys)
     */
    applyAuthHeaders(url: string, headers: Record<string, string>): Record<string, string>;
}
//# sourceMappingURL=AdobeAuthProvider.d.ts.map