# @dcx/dcx-js

The [dcx-js module](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html) consists of classes and factory methods that support the creation, reading and writing of DCX-based composites.

Digital Composite Technology, or DCX, is a framework for developing cloud-aware formats at Adobe. It enables the synthesis of complex formats out of aggregations of smaller components, treated as a unit. It was created to address the monolithic, complex file formats of the past that are poorly suited for the requirements of mobile devices and cloud computing. 

For a deep dive on DCX Concepts visit the [DCX Wiki](https://wiki.corp.adobe.com/pages/viewpage.action?pageId=850006456)

## Prerequisite Understanding

Before going into how to use the dcx-js module, it's important to understand the history of the library and how the current incarnation differs from previous versions.

The dcx-js library was originally built to work with the the Creative Cloud Storage service, more commonly know as CC Storage. CC Storage is an API and service that provides access to User Repositories. This CC Storage service is actively maintained but is not being actively upgraded and clients should favor using it's successor the Content Storage Service (CSS) instead. 

The Content Storage Service (CSS) is the [Repository API](https://git.corp.adobe.com/pages/caf/api-spec/) implementation via which clients access both User Repositories and Organizational Repositories. It is the future of the Adobe Cloud Platform and is both actively maintained and upgraded. 

The [AdobeRepoAPISession](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html) class is used to interface with the ACP (CSS/CoS) via a promise based API.

> New clients of dcx-js should be using the Repository API via [AdobeRepoAPISession](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html) to interface with cloud storage and not the CC Storage Service.

The rest of this README assumes you are using the *Repository API Session*, if you are looking for documentation on working with the CC Storage Session [see here](./README-ccStorage.md).

## Getting Started

### Creating an HTTP Service

The [AdobeRepoAPISession](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_repo_api_session.adoberepoapisession.html) requires an instance of the [AdobeHTTPService](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_http.httpservice.html) class in order to make requests. Refer to [Authentication](/packages/http/README.md#authentication) for more details on creating an HTTP Service instance.

```ts
const { createHTTPService } = require('@dcx/http');

const service = createHTTPService(function (service) {
    // get a new token asynchronously
    myGetToken(function (token) {
        service.setAuthToken(token);
    });
});
service.setApiKey('MyApiKey');
service.setAdditionalHeaders( {'foo': 'bar'} );
```

### Creating a Repository API Session

Session creation is done via a factory method that can be imported from the [dcx-js](/packages/dcx-js) module.

```ts
const { createRepoAPISession } = require('@dcx/dcx-js');
const session = createRepoAPISession(service, 'https://platform-cs-stage.adobe.io');
```

### Importing the Transfer Interface 

The [dcx-js](/packages/dcx-js) module exports a [Transfer Interface](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html) that can be used to import all the APIs and classes that are part of the dcx-js module.

```ts
const { pullCompositeManifestOnly, pushComposite } = require('@dcx/dcx-js');
```

## Working with Composites

### Instantiating a Composite

`AdobeDCXComposite` is the class that represents a composite. You need to create an instance of it to work with a composite.

#### Creating a composite

The [newDCXComposite](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html#newdcxcomposite) factory method can be used to create a new unbound composite as well as bound composite to an existing asset in the cloud.

```ts
export function newDCXComposite(
    assetId?: string,
    repositoryId?: string,
    name?: string,
    id?: string,
    type?: string,
    links?: Record<string, Link>,
    options: AdobeDCXCompositeOptions = {}
): AdobeDCXComposite
```

The `assetId` argument is the asset ID of the composite if it already exists in the cloud.

The `repositoryId` argument is the repository ID of the composite if it already exists in the cloud.

The `name` argument is the name of the new composite. It does *not* have to be unique and should reflect whatever name the user has chosen for the new composite.

The `id` argument is a user generated uuid assigned to the composite id property in root node of the manifest.

The `type` argument must identify the content type of the composite. See https://wiki.corp.adobe.com/display/dma/Manifest+Syntax for more details.

The `links` argument can be provided if the client already has cached links for this composite, if this argument is provided then you don't need to provide the assetId or repository id for bound composites.

The `options` argument provide further composite configuration options. See [DCX Options](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/interfaces/_dcx_dcx_js.dcxoptions-1.html) for more info.

##### Creating a new unbound composite

To create an unbound composite, you don't have to provide either the `repositoryId` or `assetId` but you must provide `name` and `type`.

```ts
const { newDCXComposite } = require('@dcx/dcx-js');
const { generateUuid } = require('@dcx/util');
const unboundComposite = newDCXComposite(undefined, 
                                         undefined, 
                                         name, 
                                         generateUuid(), 
                                         'demo/test+dcx'
);
```

This composite now just exists locally, in order to persist this composite to the cloud you will need to use the [createComposite](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html#createcomposite-1) method.

```ts
const { createComposite } = require('@dcx/dcx-js');
// Create an object that represents the parent directory that will contain the new composite
const parentDirectory = {
    assetId: 'urn:aaid:sc:US:fe58269e-de3d-493b-ab7d-0d2b79d50e85',
    repositoryId: '09EC14AC5E84F4630A494030@AdobeOrg'
};

// Create a new composite cloud asset at the specified path.
const composite = await createComposite(session, 
                                        unboundComposite,
                                        parentDirectory,
                                        'example.file'  //Path
);
```

#### Creating a bound composite instance

To create a bound composite, you must provide `repositoryId` and `assetId` and/or `links`.

```ts
const { newDCXComposite } = require('@dcx/dcx-js');
const composite = newDCXComposite(assetId, 
                                  repoId, 
                                  name, 
                                  generateUuid(), 
                                  'demo/test+dcx');
```

Notice that at this point the composite has not been pulled from the server. So the composite object is little more than a container for the assetId and repoId of the composite on the server. You will need to explicitly `pull` the composite in order to start working with it.

```ts
const { pullCompositeManifestOnly } = require('@dcx/dcx-js');
const branch = await pullCompositeManifestOnly(session, composite);
composite.resolvePullWithBranch(branch);
```

#### Creating a composite from a snapshot

Consumers looking for the most direct path to creating a composite with the fewest number of requests might opt to perform the create "all-at-once" (local and remote, with data). One alternative form for creating a composite is to provide a snapshot (which can be obtained via `repoAPISession.getPrimaryResource` with an existing composite) (See [Creating Composites in RAPI](https://git.corp.adobe.com/pages/caf/api-spec/chapters/composites.html#creating-composites)).

```ts
import { createComposite, newDCXComposite } from '@dcx/dcx-js';

const composite = await createComposite(
    session,
    newDCXComposite(
        undefined, // assetId
        undefined, // repository id
        name, // composite name
        undefined, // composite id
        format, // composite format / media-type
    ),
    rootDir, // root of the create action
    relativePath, //
    undefined,
    undefined,
    snapshot,
    snapshot!.byteLength,
);

```

##### Downloading a snapshot
```ts
const { createRepoAPISession } = require('@dcx/dcx-js');
const session = createRepoAPISession(service, 'https://platform-cs-stage.adobe.io');

async function downloadSnapshot(asset: AdobeAsset | AdobeDCXComposite) {
    const {response: snapshot} = await session.getPrimaryResource(composite)
    return snapshot;
}
```

#### Navigating a composite branch
As you begin to work with composites you may find yourself in need of accessing components at various paths in composites. Much of this navigation can happen by utilizing the methods available on a given `branch`. 
```ts
composite.resolvePullWithBranch(branch);
// Components at a specific known path
const deeplyNestedComponents = composite.current.getComponentsOf(
    composite.current.getChildWithAbsolutePath('/some/deeply/nested/path/of/resources')
);
// Components with a specific ID
const specificallyTargetedComponent = composite.current.getComponentWithId(storedComponentId);

// all components in the asset
const allComponents = composite.current.allComponents();
```

#### Creating a composite by duplicating a composite, branch or element

This API allows clients to create a new composite based on an existing composite, branch or element. The result is a new local, in-memory composite which you can push to the server using the regular `pushComposite()` API, which will use server-to-server copy requests to duplicate the component files.

```ts
const { newCompositeAsCopyOf } = require('@dcx/dcx-js');
var compositeOrBranchOrElement = ...;
var copiedComposite = newCompositeAsCopyOf(compositeOrBranchOrElement, 'Name of Copy');
```

The following restrictions apply:

- Both the existing and the new composite must have the same endpoint. If not `pushComposite()` will error out.
- You must not pass in a composite that has not been pushed or pulled yet.
- This first implementation does not yet copy over local files. So at this point in time you can not copy composites with locally added/modified components. Also you have to call `downloadComponents()` after a successful `pushComposite()`.

## Exploring a Composite
 
Once you have a `AdobeDCXComposite` object you can explore it. Composites expose their DOM in branches ([AdobeDCXBranch](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/classes/_dcx_dcx_js.adobedcxbranch-1.html)). When working with a composite you usually use its `current` branch:

```ts
let current = composite.current;
console.log('Name of the composite: ' + current.name);
console.log('Value of custom property: ' + current.rootNode.getValue('my#customProperty'));
let rootComponents = current.getComponentsOf(current.rootNode);
console.log('Number of components at the root level: ' + rootComponents.length);
let pagesNode = current.getChildWithAbsolutePath('/pages');
let pages = current.getChildrenOf(pagesNode);
console.log('Number of pages in composite: ' + pages.length);
let page1 = pages[0];
console.log('Name of the first page: ' + page1.name);
```

## Editing a Composite

The `current` branch is the only branch that is editable. You can modify properties and the structural makeup of the composite:

```ts
let current = composite.current;
current.name = 'My document';
current.rootNode.setValue('my#customProperty', 'foo');
let pagesNode = current.addChild('pages', undefined, 0, current.rootNode);
pagesNode.path = 'pages';
let page = current.addChild('page 1', undefined, 0, pagesNode);
page.setValue('my#backgroundColor', '#FFFFFF');
```

In a browser (XHR) environment working offline is not supported and thus conflicts are unlikely to  occur. We recommend the following pattern:

1. Pull a composite and display it
2. When the user has made an edit, pull the composite again
3. Try to make the edit to the newly pulled composite. If that fails (e.g. the node that the user
wants to edit is gone) bail out (and notify the user)
4. Push immediately
5. If conflict either bail out or go back to #2

The main idea is to avoid having to merge conflicting changes. Let the user make modifications in small units and keep the time that passes between a pull and a push to a minimum. If a conflict does occur it is usually straight forward to pull again and make the modification again or bail out if the modification is no longer possible.

### Components

#### Uploading a composite component

The components of a composite are always associated with a file-based asset, and adding a new component typically happens in response to the user picking a local file. Since file access in a browser environment is very restricted you will need to upload the component asset immediately. In order to minimize the chance of a conflict we recommend uploading any new or changed component files before pulling and updating the DOM. 

The `uploadNewComponent` API accepts a data parameter that can be either a string, a buffer or a [GetSliceCallback](../common-types/src/block_transfer.ts). If the component data is less then `maxSingleTransferSize` then providing the data as a buffer is the recommended approach. If the size of the data is greater then `maxSingleTransferSize` it is recommended for clients to provide a [GetSliceCallback](../common-types/src/block_transfer.ts) function instead that will slice the data into smaller chunks that can then be uploaded by dcx-js via block transfer. If the data is greater then `maxSingleTransferSize` but a buffer is still provided, dcx-js will take care of slicing the data and transfer the object via block transfer.

See the [Component Upload Example](../assets/README.md#blockUpload) to see how to define and provide a [GetSliceCallback](../common-types/src/block_transfer.ts) function.

```ts
const { uploadNewComponent, pullCompositeManifestOnly, pushComposite} = require('@dcx/dcx-js');
const { generateUuid } = require('@dcx/util');

// Upload the component
const uploadResults = await uploadNewComponent(session, 
                                               composite, 
                                               data, 
                                               'image/png', 
                                               generateUuid()
);

// Pull the latest DOM and update local composite to avoid conflicts
const pulledBranch = await pullCompositeManifestOnly(session, composite);
composite.resolvePullWithBranch(pulledBranch);

// Update the DOM with the upload results
composite.current.addComponentWithUploadResults(name, 
                                                relationship, 
                                                path, 
                                                parentNode, 
                                                uploadResults
);

// Push the updated composite
await pushComposite(session, composite, false);
```

#### Copy existing cloud asset for component

> Support for copyAssetForNewComponent will be added with the completion of [DCX-3653](https://jira.corp.adobe.com/browse/DCX-3653)

If the asset of the new component is already in the cloud (must be same service endpoint!) you can instruct the transfer code to do a server-to-server copy rather than an upload:

```ts
TODO: Copy Asset for New Component Example
```

#### Updating a component asset

Updating a component asset with a new file asset follows the same pattern as creating a new component:

```ts
const { uploadComponent, pullCompositeManifestOnly, pushComposite } = require('@dcx/dcx-js');

// Resolves to the component at the path
const component = composite.current.getComponentWithAbsolutePath(
    '/rendition.png'
);

// True if the component exists
if (component) {
    // Upload the new version of the component
    const uploadResults = await uploadComponent(
        session,
        composite,
        component,
        data
    );

    // Pull the latest DOM and update local composite to avoid conflicts
    composite.resolvePullWithBranch(await pullCompositeManifestOnly(session, composite));

    // Update the DOM with the upload results of the updated component
    composite.current.updateComponentWithUploadResults(component, uploadResults);

    // Push the composite with the updated component
    await pushComposite(session, composite, false);
} else {
    // Existing component does not exist
}
```

#### Adding a component to a new composite.

If you are uploading or copying components for a new local composite constructed by calling `newDCXComposite` `newCompositeAsCopyOf` then you must first call `createComposite` in order to create a composite resource collection on the server at the desired href and resolve this to an asset id.

```ts
const { newDCXComposite, createComposite } = require('@dcx/dcx-js');
const compositeName = 'sample-composite';
const repositoryId = '09EC14AC5E84F4630A494030@AdobeOrg';
const composite = newDCXComposite(undefined, repositoryId, compositeName, undefined, 'sample/test+dcx');
await createComposite(session, 
                      composite, 
                      { path: '/users/{user-id}/cloud-content', repositoryId: repositoryId}, 
                      compositeName
);
```

### Elements

An element is a special kind of child node which represents an embedded composite. It must have a name, a path and a type.

You can create an element by either copying the root node of a composite into another composite or you can use the `addElement()` API on the `current` branch of a composite to create a new element:

```ts
let node = current.copyChild(otherComposite.current.rootNode,
                             parentNode, 
                             4,
                             'background', 
                             undefined,
                             callback
);

let element = current.addElement('untitled',
                                 'application/vnd.my.type+dcx',
                                 'background', 
                                 undefined, 
                                 4,
                                 parentNode
);
```

Notice that `copyChild()` returns a regular `AdobeDCXNode` whereas `addElement()` returns an instance of `AdobeDCXElement`. This difference is important since these classes have different APIs and require different handling.

The `AdobeDCXElement` class provides APIs that are similar to those of a branch. You can use them to add/update/remove both child nodes and components like you would use a branch of a separate composite. An element operates on its own in-memory copy of the child node, changes you make do not affect the composite until you call `updateElement()`.

The composite will keep track of any element that you get with `addElement()` or with one of the `getElementWithXxx()` APIs. You must call either `updateElement()` or `abandonElement()` on the branch once you are done working with an element so that the composite can do any necessary cleanup.

```ts
const node = current.updateElement(element);

current.abandonElement(element);
```

Notice that `updateElement()` returns the updated child node.

If you want to modify an existing element you can ask the branch to instantiate the element for you:

```ts
const element = current.getElementWithAbsolutePath('/pages/page1/background');
```
Again, you must call either `updateElement()` or `abandonElement()` when you are done.

### Pushing a New or Updated Composite to the Server

When it is time to submit your new or updated composite to the server you call `pushComposite()`. If you are pushing a new composite for the first time then you must have already called the `createComposite` method to resolve the destination href of the composite to an asset id that is reflected in the `composite.assetId` property.

```ts
const { pushComposite} = require('@dcx/dcx-js');
const promise = pushComposite(session, composite, false);
...
if (changeMyMind) {
    // You can cancel a push if it hasn't completed yet
    promise.cancel();
}
```

#### Composite Integrity Error Handling
When pushing a composite, it is possible for an `INCOMPLETE_COMPOSITE` error to be thrown. This error indicates that the composite manifest includes references to components that do not exist in cloud storage. One potential cause for this is if a manifest is updated after a component is uploaded, but the manifest is not pushed for an extended period of time. One solution is to re-upload the component and replace the . Another approach is to remove the missing component and re-push the manifest which might look like the following example:
```ts
async function pushAndRemoveMissingComponents(session, composite) {
    try {
        return await pushComposite(session, composite, true, 1);
    } catch (error: AdobeDCXError) {
        if (error.code === DCXError.INCOMPLETE_COMPOSITE) {
            composite.current.getMissingComponentsFromError(error)
                .forEach(component => composite.current.removeComponent(component));
            return await pushAndRemoveMissingComponents(session, composite);
        }
        throw error;            
    }
}
```

### Pulling a Composite from a Server

This presumes that you have a composite instance that already exists on the server. Further, this section assumes that you have not made any local changes to the composite that haven't been pushed yet. If that is not the case or if your push has failed with a conflict you will have to consult the Conflicts section below.

In order to pull a composites manifest from the server you must provide both the assetId as well as the repositoryId the asset is contained in.

```ts
const { pullCompositeManifestOnly, newDCXComposite } = require('@dcx/dcx-js');
const composite = newDCXComposite(
    'urn:aaid:sc:US:b7257935-9ea6-4ae2-a6a3-09d1de18a63b',
    '09EC14AC5E84F4630A494030@AdobeOrg'
);
const pullPromise = pullCompositeManifestOnly(session, composite);
...
if (changeMyMind) {
    // You can cancel a pull if it hasn't completed yet
    pullPromise.cancel();
}
```

### (De)serializing UploadResults
Component uploads through the `uploadComponent()` and `uploadNewComponent()` APIs return an object which can be used to get a serialized version of a component to transfer to other clients, and then converted back to an UploadResults instance.

```ts
const { uploadNewComponent, uploadResultsFromComponentDescriptor } = require('@dcx/dcx-js');

const uploadResults = await uploadNewComponent(
        session, // RepoAPISession instance
        composite, // DCXComposite instance
        data, // or getSliceCallback
        type, // mime type
        componentId, // optional
        size, // optional
        md5 // optional
);

// Get a serialized component descriptor
const descriptor = uploadResults.getComponentDescriptor(componentId);

// Get UploadResults from component descriptor
const deserializedUploadResults = uploadResultsFromComponentDescriptor(descriptor);

// Get component URL suitable for downloading (may be presigned URL)
const { isPresignedUrl, url } = await deserializedUploadResults.getComponentURL(session, componentId);

// Get component data
const { result, response } = await deserializedUploadResults.getComponent(session, componentId, responseType);
```

### Working with XMP Embedded Metadata

DCX uses the Extensible Metadata Platform (XMP) to manage metadata about the content and provenance of a composite. 

While there is no official javascript library provided by Adobe to work with XMP as of v5.13 we are adding official support for XMP in dcx-js. This support will come for free when using the repository session and no updates are required by client who just want basic metadata persisted about their composites.

The dcx-js implementation differs from other DCX libraries as it uses JSON Patch rather than updating the metadata as XML, which means fewer network calls and less data over the wire. 

Behind the scenes the library uses a bulk request with sub-version updates, which cuts down on network requests and avoids duplicate version entries. 

XMP support is also extensible for clients, allowing them to integrate with the full XMP library and perform their own updates if they desire.

There are **4 modes** for working with XMP metadata, the mode used is determined by the [XMPConfig](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html#xmpconfig) argument in pushComposite.

#### Managed Mode
All fields are managed by dcx-js. creatorTool is set to `dcx-js`.

```ts
await pushComposite(repoSession, 
                                      composite, 
                                      true, 
                                      1,
                                      {
                                          mode: XMPModes.MANAGED
                                      });
```

The 5th parameter of the pushComposite API is an [XMPConfig](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html#xmpconfig) object. This is where clients can define the XMP strategy they want to use. For `Managed` mode this object is optional. If this is the first time the composite is being pushed, `Managed` mode will take care of creating the XMP component and adding it to the root components on the composite. An example of the default XMP properties added are below.

```json
{
    "xmp:ModifyDate": "2021-06-10T16:47:08.828Z",
    "xmpMM:History": {
        "@list": [
            {
                "stEvt:softwareAgent": "dcx-js",
                "stEvt:action": "created",
                "stEvt:when": "2021-06-10T16:47:08.828Z",
                "stEvt:instanceID": "xmp.iid:144fc002-1f68-4038-89e0-e09ce05a52d3"
            }
        ]
    },
    "xmp:MetadataDate": "2021-06-10T16:47:08.828Z",
    "xmp:CreatorTool": "dcx-js",
    "xmp:CreateDate": "2021-06-10T16:47:08.828Z",
    "@id": "",
    "xmpMM:InstanceID": "xmp.iid:144fc002-1f68-4038-89e0-e09ce05a52d3",
    "@context": {
        "xmpMM": "http://ns.adobe.com/xap/1.0/mm/",
        "stEvt": "http://ns.adobe.com/xap/1.0/sType/ResourceEvent#",
        "xmp": "http://ns.adobe.com/xap/1.0/"
    },
    "xmpMM:DocumentID": "xmp.did:144fc002-1f68-4038-89e0-e09ce05a52d3",
    "xmpMM:OriginalDocumentID": "xmp.did:144fc002-1f68-4038-89e0-e09ce05a52d3"
}
```

If this is a subsequent push, `Managed` mode will update the `xmp:ModifyDate` and add a new `saved` event to the history list.

#### Partially Managed Mode
In the `Partially Managed` mode, specific fields (eg. creatorTool & modifyDate) can be replaced by the client in the [XMPConfig](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_dcx_js.html#xmpconfig) object.

```ts
await pushComposite(repoSession, 
                                      composite, 
                                      true, 
                                      1,
                                      {
                                         mode: XMPModes.PARTIALLY_MANAGED,
                                         creatorTool: 'cc-web',
                                         modifyDate: '2021-06-10T17:56:03.812Z'
                                      });
```

#### Client Managed Mode
In this mode clients takeover complete management of the XMP metadata persisted. On initial push dcx-js will use the provided XML string to create the XMP component. 

The [initializeXMPXML](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_xmp_util.html#initializexmpxml) utility method inside the [@dcx/xmp-util](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_xmp_util.html) module is provided to make it easy to construct an `initialXMPXML` string required to create the XMP component. The method provides a few arguments to customize initial values like creatorTool, docId, createDate, history events and derivedWithData. The `initialXMPXML` argument just requires a RDF-XML compliant string so clients can also construct this themselves and are not required to use the [initializeXMPXML](https://dcx.ci.corp.adobe.com/job/dcx-js-stats/dcx-js_20Docs/modules/_dcx_xmp_util.html#initializexmpxml) utility method.

```ts
import { initializeXMPXML } from '@dcx/xmp-util';

const initialXMP = initializeXMPXML('cc-web', '123');
await pushComposite(repoSession, 
                                      composite, 
                                      false, 
                                      undefined, 
                                      {
                                          mode: XMPModes.CLIENT_MANAGED,
                                          initialXMPXML: initialXMP
                                      });
```


On subsequent pushes dcx-js will use the provided JSON Patch document in the `xmpPatch` argument to patch embedded metadata alongside the manifest push. This results in a single version, but leaves all validation and error handling to the client application.

```ts
await pushComposite(repoSession, 
                                      composite, 
                                      false, 
                                      1, 
                                      {
                                          mode: XMPModes.CLIENT_MANAGED,
                                          xmpPatch: [
                                            {
                                                op: 'add',
                                                path: '/@context/tiff',
                                                value: 'http://ns.adobe.com/tiff/1.0/'
                                            },
                                            {
                                                op: 'add',
                                                path: '/tiff:Orientation',
                                                value: '1'
                                            }
                                          ]
                                      });
```

The request above shows patching the XMP document with a new property. This requires two patch operations, one to add the new namespace to the context object, and the second to add the new property. Providing a xmpPatch argument does mean that the onus is on clients to also provide a history event patch operation if they wish to have new entry added to the history list.


> See [@dcx/xmp-util](https://git.corp.adobe.com/DMA/dcx-js/tree/dev/packages/xmp-util) for convenience APIs for XMP Patching.

#### Unmanaged Mode
In Unmanaged mode dcx-js skips the embedded metadata update entirely. It is the caller's responsibility to handle the creation and updates to XMP metadata however they choose. If using this mode, no XMP document will be created during dcx-js initial push. Note that later updates by client applications will lead to an additional version being created.
 
 ```ts
await pushComposite(repoSession, 
                                      composite, 
                                      true, 
                                      1,
                                      {
                                          mode: XMPModes.UNMANAGED
                                      });
```

 > When using mode MANAGED or PARTIALLY_MANAGED, no mode needs to be specified. Properties that can be customized by clients can be provided in the options object. If another mode is explicitly defined, those fields may be ignored.
 
 > If using mode CLIENT_MANAGED, initialXMPXML is only used for initial pushes or when the manifest does not yet contain a metadata component. If not provided, dcx-js uses the default XMP content. Similarly, xmpPatch is only used for bound composites that contain a metadata component. If not provided, no update will be performed on update pushes.

### Conflicts

**TODO: Write this section**

### Advanced Editing

**TODO: Write this section**
