/* istanbul ignore file */

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { getPubsAsset, RepoResponseResult } from '@dcx/assets';
import { AdobeAsset, AdobeDCXComposite, AdobeDCXError, RequestDescriptor } from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { AdobeRepoAPISession } from '@dcx/repo-api-session';
import { noOp } from '@dcx/util';
import AdobeCommunityPublicationRecord from './AdobeCommunityPublicationRecord';
import AdobeCommunitySession from './AdobeCommunitySession';
import AdobeRemixData from './AdobeRemixData';

/**
 * @class AdobeCommunityPlatform
 * @classdesc AdobeCommunityPlatform is a static class that implements methods for publishing/remixing DCX composites to/from CP.
 * <p>Clients do not need to require() the class since a reference to it is available via their instance
 * of {@link AdobeDCX}.
 * @hideconstructor
 */

/**
 * Callback for monitoring progress of the various transfer functions.
 * @callback ProgressCallback
 *    @param {Integer} bytesCompleted
 *    @param {Integer} totalBytes
 */

////////////////////////////////////////////////////////////////////////////////////////////////
// XferContext
////////////////////////////////////////////////////////////////////////////////////////////////

/**
 * @class
 * @hideconstructor
 * @classdesc An XferContext object is returned when a transfer is initiated.
 * Clients can use it to abort the transfer or register a progress handler with the context.
 * @property {Integer}          [bytesTransfered]   The current number of bytes transfered.
 * @property {Integer}          [bytesTotal]        The total number of bytes to transfer.
 * @property {ProgressCallback} [onProgress]        Set this to a progressCallback function if you
 *                                                  want to receives progress callbacks.
 * @property {Array}            [failedComponents]  Upon failure this it is an array of
 *                                                  { component: component, error: error } for
 *                                                  each component that failed to down or upload.
 */
class XferContext {
    private _session: any;
    private _callback: any;
    private _cleanup: any;
    private _componentsPending: any;
    private _bytesTransfered: number;
    private _bytesTotal: number;
    private _onProgress: any;
    private _reportProgress: any;
    private _aborted = false;

    constructor(session: AdobeCommunitySession, callback: any, cleanup?: any) {
        this._session = session;
        this._callback = callback;
        this._cleanup = cleanup;

        this._componentsPending = 0;
        this._bytesTransfered = 0;
        this._bytesTotal = 0;
        this._onProgress = noOp;

        this._reportProgress = (incr: number) => {
            if (incr > 0) {
                this._bytesTransfered += incr;
                const progress = this._onProgress;
                if (progress) {
                    progress(this._bytesTransfered, this._bytesTotal);
                }
            }
        };
    }

    /**
     * @internal
     */
    get xferComplete() {
        return this._xferComplete;
    }

    /**
     * Is true if the xfer has been aborted.
     * @memberof XferContext#
     * @type {Boolean}
     */
    public get aborted() {
        return this._aborted;
    }

    /**
     * The number of bytes already transfered
     * @memberof XferContext#
     * @type {Integer}
     */
    public get bytesTransfered() {
        return this._bytesTransfered;
    }

    /**
     * The total number of bytes to transfer
     * @memberof XferContext#
     * @type {Integer}
     */
    public get bytesTotal() {
        return this._bytesTotal;
    }

    /**
     * The progress callback that should get called on progress.
     * @memberof XferContext#
     * @type {ProgressCallback}
     */
    public set onProgress(progressCallback) {
        this._onProgress = progressCallback;
    }
    public get onProgress() {
        return this._onProgress;
    }

    /**
     * Call the client callback. Repeated calls will get ignored.
     * @private
     */
    private _callCallback(callbackError: any, result?: any) {
        const callback = this._callback;
        if (callback) {
            this._callback = undefined;
            return callback(callbackError, result);
        }
    }

    /**
     * Called when the xfer has either succeeded or failed.
     * @private
     */
    private _xferComplete(completionError: any, result?: any) {
        const cleanup = this._cleanup;
        if (cleanup) {
            this._cleanup = undefined;
            cleanup(completionError, result); // cleanup must call callback
        } else {
            // No cleanup: call client callback directly
            return this._callCallback(completionError, result);
        }
    }

    /**
     * Call this to abort the transfer.
     * @param {Error} [error] If not undefined then the callback for the xfer gets
     *                        invoked and passed this as the first (error) argument.
     */
    public abort(abortError: AdobeDCXError | Error) {
        if (!this._aborted) {
            this._aborted = true;
            this._session._service.abortAllWithToken(this);
        }
        if (!abortError) {
            // No error provided: undefine client callback so that it doesn't get called and
            // create a generic error so that our logic knows that the operation has failed.
            this._callback = undefined;
            abortError = new Error('Aborted'); // Create a generic error
        }
        return this._xferComplete(abortError, null);
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////
// Public Transfer APIs
////////////////////////////////////////////////////////////////////////////////////////////////

export class AdobeCommunityPlatform {
    private static pubsAsset: AdobeAsset;
    /**
     * @private
     */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    private constructor() {}

    /**
     Create composite at pubs location.
        @param session      The repo session to use
        @param relPath      Path at which new composite will be created in the cloud, relative to /pubs
        @param contentType  Content type of composite
     */
    public static async createCompositeAtPubs(
        session: AdobeRepoAPISession,
        relPath: string,
        contentType: string,
    ): Promise<RepoResponseResult<AdobeAsset>> {
        if (!AdobeCommunityPlatform.pubsAsset) {
            AdobeCommunityPlatform.pubsAsset = await getPubsAsset(session.serviceConfig);
        }

        return await session.createComposite(
            {
                assetId: AdobeCommunityPlatform.pubsAsset.assetId,
                repositoryId: AdobeCommunityPlatform.pubsAsset.repositoryId,
            },
            relPath,
            contentType,
        );
    }

    /**
     Publish composite to Community Platform.
        @param composite Composite to be published
        @param compositeHref Path of the composite directory in ccStorage
        @param pubRecord
        @param [remixData]
        @param [creatorTool]
        @param callback signature: callback(error, pubRecord)
     */
    public static publishCompositeAlreadyPushedToPubs(
        composite: AdobeDCXComposite,
        pubRecord: AdobeCommunityPublicationRecord,
        remixData: AdobeRemixData | undefined,
        creatorTool: string | undefined,
        communitySession: AdobeCommunitySession,
        callback: any,
    ) {
        if (!pubRecord || pubRecord.resourcePath == null || pubRecord.communityId == null) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'ResourcePath & communityId must be set on publication record for the composite to be published.',
            );
        }

        const publishContext = new XferContext(communitySession, callback);

        const updateAsset = pubRecord.assetId !== null;
        let req: RequestDescriptor;

        if (composite.current) {
            communitySession.updatePublicationRecordData(pubRecord, composite.current);
        }

        if (!updateAsset) {
            req = communitySession.publishCompositeAtResource(
                composite.type as string,
                pubRecord,
                function (publishError: AdobeDCXError, cpHref: string) {
                    return publishContext.xferComplete(publishError, publishError ? undefined : pubRecord);
                },
            );
            if (req) {
                req.token = publishContext;
            }

            return publishContext;
        }
        req = communitySession.updateCpMetadataAtResource(
            composite.type as string,
            pubRecord,
            function (publishError?: AdobeDCXError, cpHref?: string) {
                return publishContext.xferComplete(publishError, publishError ? undefined : pubRecord);
            },
        );
        if (req) {
            req.token = publishContext;
        }

        return publishContext;
    }

    public static getCpMetadata(
        communitySession: AdobeCommunitySession,
        pubRecord: AdobeCommunityPublicationRecord,
        callback: any,
    ) {
        if (!pubRecord || pubRecord.assetId == null || pubRecord.communityId == null) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'Asset Id & communityId must be set on publication record to fetch its CP metadata.',
            );
        }
        return communitySession.getCpMetadata(pubRecord, callback);
    }
}

export default AdobeCommunityPlatform;
