/* eslint-disable no-useless-escape */
/* istanbul ignore file */

/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import { HeaderKeys, HTTPMethods, JSONMediaType } from '@dcx/assets';
import {
    AdobeDCXBranch,
    AdobeDCXComponent,
    AdobeDCXComposite,
    AdobeHTTPService,
    AdobeResponse,
    AdobeResponseType,
    RequestDescriptor,
} from '@dcx/common-types';
import { AdobeDCXError, networkError, unexpectedResponse } from '@dcx/error';
import { appendPathElements, ensureRelativeHrefStartsWithSlash, parseURI } from '@dcx/util';
import AdobeCommunityPublicationRecord, {
    AdobeCommunityPublicationRecordMetadata,
} from './AdobeCommunityPublicationRecord';
import DCXComponent from './AdobeDCXComponent';
import { SessionBase } from './AdobeSessionBase';

// override for storage/community session
// all statuses resolve, none reject.
const isStatusValid = () => true;
const retryOptions = { disableRetry: true };
const autoParseJson = false;

const CONTENT_API_PREFIX = '/content/2/';
const CONTENT_API_ASSET_BASE = /^\/?content\/2\/[^\/]+\/[^\/]+/;
const ASSETS_API_ASSET_BASE = /^\/?api\/v2\/[^\/]+\/assets\/[^\/]+/;
const CONTENT_API_FORMAT_MAPPING: { 'image/jpeg': 'jpg'; 'image/png': 'png'; [key: string]: string } = {
    'image/jpeg': 'jpg',
    'image/png': 'png',
};

//******************************************************************************
// Public API
//******************************************************************************

/**
 * @class
 * @classdesc  Session interface for the CP service.
 * <p>The constructor for AdobeCommunitySession is private. See {@link AdobeDCX}
 * to learn how to create an AdobeCommunitySession and what to use it for.</p>
 * @augments AdobeSessionBase
 * @hideconstructor
 * @param httpService the HTTP service
 * @param server      the url for the storage server
 */
export class AdobeCommunitySession extends SessionBase {
    readonly name = 'AdobeCommunitySession';

    public constructor(httpService: AdobeHTTPService, server: string) {
        super(httpService, server);

        if (!this._endPoint) {
            throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'Could not determine endpoint from: ' + server);
        }
    }

    /** @private */
    publishCompositeAtResource(type: string, pubRecord: AdobeCommunityPublicationRecord, callback: any) {
        // See https://cp-docs.corp.adobe.com/sdks/cp/API-v2/assets/publishasset
        let href = '/api/v2/' + pubRecord.communityId + '/assets';
        href = this._resolveUrl(href);
        const data = {
            resource_path: this.assetIdFromSSResourceHref(pubRecord.resourcePath as string),
            resource_type: type,
            metadata: this.metadataFromPubRecord(pubRecord),
        };
        const headers = { [HeaderKeys.CONTENT_TYPE]: JSONMediaType };

        return this._service.invoke(
            HTTPMethods.POST,
            href,
            headers,
            JSON.stringify(data),
            { responseType: 'text', autoParseJson, isStatusValid, retryOptions },
            function (invokeError, response) {
                function assetIdFromCPAssetHref(locationHref: string) {
                    const assetIdPattern = '^.*?\\/api\\/v2\\/([^/]*)/assets\\/([^\\/]*).*$';
                    const matcher = locationHref.match(assetIdPattern);
                    if (!matcher) {
                        return null;
                    }
                    return matcher[2];
                }

                if (invokeError) {
                    return callback(networkError('Error publishing composite', invokeError, response));
                }
                const statusCode = response.statusCode;
                if (statusCode === 201) {
                    const loc = response.headers['location'];
                    if (!loc) {
                        invokeError = new AdobeDCXError(
                            AdobeDCXError.INVALID_DATA,
                            'Missing location header returned on response by server',
                            undefined,
                            response,
                        );
                        return callback(
                            unexpectedResponse('Unexpected response publishing composite', invokeError, response),
                        );
                    }

                    const assetId = assetIdFromCPAssetHref(loc);
                    if (assetId) {
                        pubRecord.assetId = assetId;
                        return callback(undefined, loc);
                    }
                }
                return callback(unexpectedResponse('Unexpected response publishing composite', invokeError, response));
            },
        );
    }

    /** @internal */
    public updateCpMetadataAtResource(
        type: string,
        pubRecord: AdobeCommunityPublicationRecord,
        callback: (error?: AdobeDCXError, result?: any) => void,
    ) {
        // See https://cp-docs.corp.adobe.com/sdks/cp/API-v2/assets/updateassetmetadata
        let href = '/api/v2/' + pubRecord.communityId + '/assets/' + pubRecord.assetId;
        href = this._resolveUrl(href);
        const data = this.metadataFromPubRecord(pubRecord);
        const headers = { [HeaderKeys.CONTENT_TYPE]: JSONMediaType };

        return this._service.invoke(
            HTTPMethods.PUT,
            href,
            headers,
            JSON.stringify(data),
            { responseType: 'text', autoParseJson, isStatusValid, retryOptions },
            (invokeError, response) => {
                if (invokeError) {
                    return callback(networkError('Error publishing composite', invokeError, response));
                }
                const statusCode = response.statusCode;
                if (statusCode === 200) {
                    return callback(undefined, href);
                }
                return callback(unexpectedResponse('Unexpected response publishing composite', invokeError, response));
            },
        );
    }

    /** @internal */
    getCpMetadata(pubRecord: AdobeCommunityPublicationRecord, callback: any) {
        // See https://cp-docs.corp.adobe.com/sdks/cp/API-v2/assets/getassetmetadata
        let href = '/api/v2/' + pubRecord.communityId + '/assets/' + pubRecord.assetId;
        href = this._resolveUrl(href);

        this._service.invoke(
            HTTPMethods.GET,
            href,
            {},
            undefined,
            { responseType: 'text', autoParseJson, isStatusValid, retryOptions },
            function (invokeError, response) {
                if (invokeError) {
                    return callback(networkError('Error getting metadata for CP composite', invokeError, response));
                }
                const statusCode = response.statusCode;
                if (statusCode === 200) {
                    let json;
                    const buf = response.response || (response as unknown as string);
                    try {
                        json = JSON.parse(buf);

                        pubRecord.resourcePath = json['resource_path'];

                        // These are optional
                        pubRecord.description = json['description'] || null;
                        pubRecord.title = json['title'] || null;
                        pubRecord.alias = json['alias'] || null;
                        pubRecord.undiscoverable = json['undiscoverable'] || false;
                        pubRecord.isPrivate = json['private'] || false;

                        let ja = json['tags'];
                        if (ja && ja !== null && ja.length > 0) {
                            pubRecord.tags = ja;
                        }

                        let jo = json['_embedded'];
                        if (jo && jo !== null) {
                            ja = json['creators'];
                            if (ja && ja !== null && ja.length > 0) {
                                const ids: string[] = [];
                                for (let i = 0; i < ja.length; ++i) {
                                    const jo2 = ja[i];
                                    if (jo2 && jo2 !== null) {
                                        const id = jo2['id'];
                                        if (id && id !== null) {
                                            ids.push(id);
                                        }
                                    }
                                }
                                if (ids.length !== 0) {
                                    pubRecord.creatorIds = ids;
                                }
                            }
                        }

                        jo = json['category_id'];
                        if (jo && jo !== null) {
                            pubRecord.categoryId = jo['id'] || null;
                        }

                        ja = json['sub_categories'];
                        if (ja && ja !== null && ja.length > 0) {
                            pubRecord.subCategoryIds = [];
                            for (let i = 0; i < ja.length; ++i) {
                                jo = ja[i];
                                pubRecord.subCategoryIds.push(jo['id']);
                            }
                        }

                        jo = json['custom'];
                        if (jo && jo !== null) {
                            pubRecord.custom = jo;
                        }
                    } catch (x) {
                        const responseError = new AdobeDCXError(
                            AdobeDCXError.INVALID_DATA,
                            'Invalid JSON returned by server',
                            x as Error,
                            response,
                        );
                        return callback(responseError);
                    }
                    return callback(undefined, pubRecord);
                }
                return callback(
                    unexpectedResponse(
                        'Unexpected response while getting metadata for CP composite',
                        invokeError,
                        response,
                    ),
                );
            },
        );
    }

    /** @private */
    updatePublicationRecordData(pubRecord: AdobeCommunityPublicationRecord, pushedBranch: AdobeDCXBranch) {
        const version = pushedBranch.versionId;
        pubRecord.mainResource = 'manifest';
        if (version) {
            pubRecord.mainResourceVersion = version;
        }
        if (pubRecord.artworkComponentId !== null) {
            const artworkComponent = pushedBranch.getComponentWithId(pubRecord.artworkComponentId);
            if (artworkComponent == null) {
                throw new AdobeDCXError(
                    AdobeDCXError.COMPONENT_DOWNLOAD_ERROR,
                    'Bad component ID = ' + pubRecord.artworkComponentId,
                );
            }
            // TODO: Revisit this, seems incorrect
            let href = pubRecord.resourcePath;
            href = this.assetIdFromSSResourceHref(href as string) as string;
            href += artworkComponent.absolutePath;
            pubRecord.artworkResource = href;
            pubRecord.artworkResourceVersion = artworkComponent.version as string;
        }
    }

    /**
     * Gets the href for the manifest of the given composite.
     * @private
     * @param composite the composite.
     */
    public getCompositeManifestHref(composite: AdobeDCXComposite, versionId: string, getHrefCallback: any): string {
        if (versionId) {
            throw new AdobeDCXError(
                AdobeDCXError.INVALID_PARAMS,
                'AdobeCommunitySession does not support version manifests.',
            );
        }

        if (!composite.assetId) {
            throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'Composite must be bound.');
        } else if (!this.isProperAssetId(composite.assetId)) {
            throw new AdobeDCXError(AdobeDCXError.INVALID_STATE, 'Composite asset id has invalid format.');
        }

        const href = this._resolveUrl(composite.assetId);
        if (!href) {
            throw new AdobeDCXError(AdobeDCXError.WRONG_ENDPOINT, 'Wrong endpoint: ' + composite.assetId);
        }
        return getHrefCallback(undefined, appendPathElements(href, 'manifest'));
    }

    /**
     * Gets a component from the server.
     * NOTE that this does not specify an If-Match header. We want to get the component
     * whether the etags match or not, for instance when we don't have the local file present.
     * @private
     * @param   {AdobeDCXComponent} component    The component to download.
     * @param   {String}            responseType The expected data type of the response.
     * @param   {Function}          callback     Callback
     * @returns {Object}            the requestDescriptor for the request.
     */
    public getComponent(
        component: AdobeDCXComponent,
        responseType: AdobeResponseType,
        callback: any,
    ): RequestDescriptor {
        const defaultRequestDesc: RequestDescriptor = {} as RequestDescriptor;
        let requestDesc: RequestDescriptor = defaultRequestDesc;

        this.getComponentHref(component, component.version as string, (error, href) => {
            if (error) {
                return callback(error);
            }
            requestDesc = requestDesc || defaultRequestDesc;
            return this._getAsset(href as string, undefined, responseType, requestDesc, callback);
        });
        return requestDesc || defaultRequestDesc;
    }

    /**
     * Gets a rendition of a component from the server.
     * NOTE that this does not specify an If-Match header. We want to get the component
     * whether the etags match or not, for instance when we don't have the local file present.
     * @private
     * @param   {AdobeDCXComponent} component    The component to download.
     * @param   {String}            type         The media type of the rendition 'image/jpeg' or 'image/png'.
     * @param   {String}            dimension    'width' or 'height'
     * @param   {Integer}           size         The requested number of pixels in the dimension.
     * @param   {String}            responseType The expected data type of the response.
     * @param   {Function}          callback     Callback
     * @returns {Object}            the requestDescriptor for the request.
     */
    public getRenditionOfComponent(
        component: AdobeDCXComponent,
        type: string,
        dimension: string,
        size: number,
        responseType: AdobeResponseType,
        callback: any,
    ) {
        const defaultRequestDesc: RequestDescriptor = {} as RequestDescriptor;
        let requestDesc: RequestDescriptor = defaultRequestDesc;

        this.getComponentRenditionHref(component, component.version as string, type, dimension, size, (error, href) => {
            if (error) {
                return callback(error);
            }
            requestDesc = requestDesc || defaultRequestDesc;
            return this._getAsset(href as string, { accept: type }, responseType, requestDesc, callback);
        });
        return requestDesc || defaultRequestDesc;
    }

    /** @private */
    public getCompositeManifest(composite: AdobeDCXComposite, versionId: string, etag: string, callback: any) {
        let href = composite.assetId;
        if (href) {
            href = this._resolveUrl(href);
        }
        if (href) {
            href = appendPathElements(href, 'manifest');
        } else {
            return callback(new AdobeDCXError(AdobeDCXError.WRONG_ENDPOINT, 'Wrong endpoint: ' + composite.assetId));
        }
        return this.getAssetAsType(href, 'text', etag, callback);
    }

    /**
     * Issue a simple HEAD request.
     * @private
     * @return         the requestDesc
     * @param href     the URL
     * @param callback signature: callback(error, response)
     */
    public headRequest(href: string, callback: (err: Error, response?: AdobeResponse) => void): RequestDescriptor {
        return this._service.invoke(
            HTTPMethods.HEAD,
            href,
            {},
            undefined,
            { responseType: 'text', autoParseJson, isStatusValid, retryOptions },
            function (error, response) {
                if (error) {
                    return callback(error); // return raw error from http
                }
                callback(error, response);
            },
        ) as RequestDescriptor;
    }

    /** @internal */
    public getComponentHref(
        component: AdobeDCXComponent,
        version: string,
        getHrefCallback: (err?: AdobeDCXError, href?: string) => void | RequestDescriptor,
    ): void {
        const owner = (component as DCXComponent)._owner;
        if (!owner || !owner.compositeAssetId) {
            throw new AdobeDCXError(
                AdobeDCXError.INVALID_STATE,
                'Component must be part of a branch of a bound composite.',
            );
        } else if (!this.isProperAssetId(owner.compositeAssetId)) {
            throw new AdobeDCXError(AdobeDCXError.INVALID_STATE, 'Composite asset id has invalid format.');
        }

        if (owner._core) {
            const sourceAssetInfo = owner._core._getSourceAssetInfoOfComponent(component as DCXComponent);
            if (sourceAssetInfo) {
                getHrefCallback(
                    undefined,
                    this._constructComponentHref(
                        sourceAssetInfo.compositeAssetId,
                        sourceAssetInfo.componentPath,
                        sourceAssetInfo.componentVersion,
                    ),
                );
                return;
            }
        }

        getHrefCallback(
            undefined,
            this._constructComponentHref(owner.compositeAssetId, component.absolutePath as string, version),
        );
    }

    /** @internal */
    public getComponentRenditionHref(
        component: AdobeDCXComponent,
        version: string,
        type: string,
        dimension: string,
        size: number,
        getHrefCallback: (err?: AdobeDCXError, href?: string) => void,
    ): void {
        const owner = (component as DCXComponent)._owner;
        if (!owner || !owner.compositeAssetId) {
            throw new AdobeDCXError(
                AdobeDCXError.INVALID_STATE,
                'Component must be part of a branch of a bound composite.',
            );
        } else if (!this.isProperAssetId(owner.compositeAssetId)) {
            throw new AdobeDCXError(AdobeDCXError.INVALID_STATE, 'Composite asset id has invalid format.');
        }

        if (owner._core) {
            const sourceAssetInfo = owner._core._getSourceAssetInfoOfComponent(component as DCXComponent);
            if (sourceAssetInfo) {
                throw new AdobeDCXError(
                    AdobeDCXError.INVALID_STATE,
                    'Getting a rendition of a source href not implemented.',
                );
            }
        }
        return getHrefCallback(
            undefined,
            this._constructComponentRenditionHref(
                owner.compositeAssetId,
                component.absolutePath as string,
                version,
                type,
                dimension,
                size,
            ),
        );
    }

    /**
     * Checks string prefix to see if it conforms to a proper asset id (href) for the CCStorage service
     * @private
     * @param   {String} assetId The asset id in question
     * @returns {Boolean}
     */
    private isProperAssetId(assetId: string) {
        return ensureRelativeHrefStartsWithSlash(assetId);
    }

    //******************************************************************************
    // Internal Methods
    //******************************************************************************

    /** @private */
    private _constructComponentHref(compositeHref: string, componentAbsolutePath: string, componentVersion: string) {
        const relPath = componentAbsolutePath.slice(1);
        let href = appendPathElements(this._resolveUrl(compositeHref), relPath);
        if (href && typeof componentVersion !== 'undefined') {
            if (this._hrefUsesContentApis(compositeHref)) {
                href += '/version/' + componentVersion;
            } else {
                href += '?version=' + componentVersion;
            }
        }
        return href;
    }

    /** @private */
    private _constructComponentRenditionHref(
        compositeHref: string,
        componentAbsolutePath: string,
        componentVersion: string,
        type: string,
        dimension: string,
        size: number,
    ) {
        let matches, href;
        // get the server-side path for the component
        const componentRelPath = componentAbsolutePath.slice(1);

        const path = parseURI(compositeHref).path;
        if (path.slice(0, CONTENT_API_PREFIX.length) === CONTENT_API_PREFIX) {
            // An href using the content API has this makeup:
            //      /content/2/{libraries|dcx}/{asset_id}/content
            // we need to return this:
            //      /content/2/{libraries|dcx}/{asset_id}/rendition/{path}/version/{version}/format/{format}/dimension/{dimension}/size/{size}
            const format = CONTENT_API_FORMAT_MAPPING[type];
            if (!format) {
                throw new AdobeDCXError(AdobeDCXError.INVALID_PARAMS, 'Unsupported rendition media type: ' + type);
            }
            matches = CONTENT_API_ASSET_BASE.exec(path);
            if (matches && matches.length > 0) {
                href = this._resolveUrl(
                    appendPathElements(
                        matches[0],
                        'rendition',
                        componentRelPath,
                        'version',
                        componentVersion,
                        'format',
                        format,
                        'dimension',
                        dimension,
                        'size',
                        String(size),
                    ),
                );
            } else {
                throw new AdobeDCXError(
                    AdobeDCXError.INVALID_PARAMS,
                    'Failed to parse composite href: ' + compositeHref,
                );
            }
        } else {
            // Otherwise we assume that the href follows this recipe:
            //      /api/v2/{community_id}/assets/{asset_id}/original
            // and we need to return this:
            //      /api/v2/{community_id}/assets/{asset_id}/rendition/{dimension}/{size}/{path}?version={version}
            matches = ASSETS_API_ASSET_BASE.exec(path);
            if (matches && matches.length > 0) {
                href = this._resolveUrl(
                    appendPathElements(matches[0], 'rendition', dimension, String(size), componentRelPath) +
                        '?version=' +
                        componentVersion,
                );
            } else {
                throw new AdobeDCXError(
                    AdobeDCXError.INVALID_PARAMS,
                    'Failed to parse composite href: ' + compositeHref,
                );
            }
        }
        return href;
    }

    /** @private */
    private _hrefUsesContentApis(href: string) {
        const path = parseURI(href).path;
        return path.slice(0, CONTENT_API_PREFIX.length) === CONTENT_API_PREFIX;
    }

    /** @private */
    private assetIdFromSSResourceHref(href: string): string | null {
        const SSResourcePattern = '^\\/pubs\\/([^\\/]*).*$';
        const matcher = href.match(SSResourcePattern);
        if (!matcher) {
            return null;
        }
        return matcher[1];
    }

    /** @private */
    private metadataFromPubRecord(pubRecord: AdobeCommunityPublicationRecord): AdobeCommunityPublicationRecordMetadata {
        const builder: AdobeCommunityPublicationRecordMetadata = {};

        // skip assetId, resourcePath
        if (pubRecord.title !== null) {
            builder['title'] = pubRecord.title;
        }
        if (pubRecord.description !== null) {
            builder['description'] = pubRecord.description;
        }
        if (pubRecord.alias !== null) {
            builder['alias'] = pubRecord.alias;
        }
        builder['undiscoverable'] = pubRecord.undiscoverable;
        if (pubRecord.isPrivate) {
            builder['private'] = pubRecord.isPrivate;
        }
        if (Array.isArray(pubRecord.tags)) {
            builder['tags'] = [...pubRecord.tags];
        }
        if (Array.isArray(pubRecord.creatorIds)) {
            builder['creator_ids'] = [...pubRecord.creatorIds];
        }
        if (pubRecord.categoryId !== null) {
            builder['category_id'] = pubRecord.categoryId;
        }
        if (Array.isArray(pubRecord.subCategoryIds)) {
            builder['sub_category_ids'] = [...pubRecord.subCategoryIds];
        }
        if (pubRecord.custom !== null) {
            builder['custom'] = pubRecord.custom;
        }
        if (pubRecord.mainResource !== null && pubRecord.mainResourceVersion !== null) {
            const resBuilder: Record<string, string> = {};
            resBuilder['resource_path'] = pubRecord.mainResource;
            resBuilder['resource_version'] = pubRecord.mainResourceVersion;
            builder['main_resource'] = resBuilder;
        }
        if (pubRecord.artworkResource !== null && pubRecord.artworkResourceVersion !== null) {
            const artworkBuilder: Record<string, string> = {};
            artworkBuilder['resource_path'] = pubRecord.artworkResource;
            artworkBuilder['resource_version'] = pubRecord.artworkResourceVersion;
            builder['artwork'] = artworkBuilder;
        }
        return builder;
    }
}

export default AdobeCommunitySession;
