/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * @license
 * Copyright 2020 Adobe Inc.
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Inc. and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Inc. and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Inc.
 **************************************************************************/

import {
    AdobeDCXComponent,
    AdobeDCXComponentData,
    AdobeDCXElement,
    AdobeDCXError,
    AdobeDCXNode,
    AdobeDCXNodeData,
    AdobeDCXRootNodeData,
    AdobeUploadResults,
    ComponentCallback,
    CompositeArchivalState,
    CompositeState,
    AdobeDCXBranch as IAdobeDCXBranch,
    LinkRelationKey,
    LinkSet,
    NodeCallback,
} from '@dcx/common-types';
import { DCXError } from '@dcx/error';
import { newDebug } from '@dcx/logger';
import { generateUuid, isObject, validateParams } from '@dcx/util';
import { AdobeDCXBranchCore } from './AdobeDCXBranchCore';
import { AdobeDCXComponent as DCXComponent } from './AdobeDCXComponent';
import DCXElement from './AdobeDCXElement';
import { AdobeDCXLocalNodeData, AdobeDCXNode as DCXNode } from './AdobeDCXNode';
import { COMPOSITE_ARCHIVAL_STATES, COMPOSITE_STATES } from './enum';
import { DerivationType } from './util/xmp';

const dbg = newDebug('dcx:dcxjs:dcxbranch');

const MANIFEST_FORMAT_VERSION = 6;

/**
 * @class
 * @classdesc A branch of a composite represents a specific version of the composite.
 * The main branch is the `current` branch, which also allows editing a composite.
 * Other branches are usually read-only and only available when running under Node.js.
 * These branches are `base`, `pushed` and `pulled`.
 *
 * The constructor for `AdobeDCXBranch` is private. Branch instances are getting
 * instantiated on an as needed-basis and can be accessed via an instance of
 * {@link AdobeDCXComposite}.</p>
 * @hideconstructor
 * @param {Object}  [data]      private
 * @param {Boolean} [readOnly]  private
 * @param {String} [assetId]  private
 */
export class AdobeDCXBranch implements IAdobeDCXBranch {
    public originalManifestFormatVersion!: number;
    public compositeHref!: string;

    /** @internal */
    _core!: AdobeDCXBranchCore;

    /** @internal */
    _data!: AdobeDCXRootNodeData & AdobeDCXLocalNodeData;

    /** @internal */
    protected _pendingElements: DCXElement[];

    /** @internal */
    _versionId?: string;

    /** @internal */
    _derivationType: DerivationType = DerivationType.NONE;

    /** @internal */
    _derivationDatetime?: string;

    constructor(data?: AdobeDCXRootNodeData, readOnly?: boolean, assetId?: string, repoId?: string) {
        if (data) {
            this._setData(data, readOnly, assetId, repoId); // creates _core
        } else {
            const id = generateUuid();
            data = { 'manifest-format-version': MANIFEST_FORMAT_VERSION, id: id } as AdobeDCXRootNodeData;
            this.originalManifestFormatVersion = MANIFEST_FORMAT_VERSION;
            this._core = new AdobeDCXBranchCore(data, this, readOnly);
            this._core._compositeAssetId = assetId;
            this._core._compositeRepositoryId = repoId;

            this._data = data;
        }
        this._pendingElements = [];
    }

    public get data() {
        return this._data;
    }
    public set data(data: AdobeDCXRootNodeData) {
        this._data = data;
    }

    /**
     * The id of the composite. Must be a unique among the nodes of the composite.
     *
     * <p>While not strictly read-only most clients do not ever have to modify this property.</p>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get compositeId() {
        return this._core.rootNode.id;
    }
    set compositeId(id) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof id !== 'string' || id === '') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a non-empty string.');
        } else if (core.allNodes[id]) {
            throw new DCXError(DCXError.DUPLICATE_VALUE, 'There is already a node with the same id.');
        } else {
            delete core.allNodes[this._data.id];
            this._data.id = id;
            core.allNodes[id] = this._core.rootNode;
            core._setDirty();
        }
    }
    /**
     * An AdobeDCXNode object that represents the root of the underlying manifest.
     * @type {AdobeDCXNode}
     * @memberof AdobeDCXBranch#
     * @readonly
     */
    get rootNode(): DCXNode {
        return this._core.rootNode;
    }
    set rootNode(node: DCXNode) {
        this._core.rootNode = node;
    }

    /**
     * The name of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get name(): string | undefined {
        return this._core.rootNode.name;
    }
    set name(name: string | undefined) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof name !== 'string' && typeof name !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            core.rootNode.name = name;
        }
    }

    /**
     * The type of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get type() {
        return this._core.rootNode.type;
    }
    set type(type) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof type !== 'string' || type === '') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a non-empty string.');
        } else {
            core.rootNode.type = type;
        }
    }

    /**
     * The asset id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get compositeAssetId(): string | undefined {
        return this._core._compositeAssetId;
    }
    set compositeAssetId(assetId: string | undefined) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof assetId !== 'string' && typeof assetId !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            // Don't mark the branch as dirty
            core._compositeAssetId = assetId;
        }
    }

    /**
     * The repository id of the composite that can be used to pull and push the composite.
     * <strong>Do not modify this for a bound composite.</strong>
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get compositeRepositoryId() {
        return this._core._compositeRepositoryId;
    }
    set compositeRepositoryId(repoId: string | undefined) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        validateParams(['repoId', repoId, 'string', true]);
        // Don't mark the branch as dirty
        core._compositeRepositoryId = repoId;
    }

    /**
     * The links of the composite.
     * @memberof AdobeDCXBranch#
     * @type {LinkSet}
     */
    get compositeLinks() {
        return this._core._compositeLinks;
    }
    set compositeLinks(links: LinkSet | undefined) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        validateParams(['links', links, 'object', true]);
        // Don't mark the branch as dirty
        core._compositeLinks = links;
    }

    /**
     * @internal
     * Check if the branch is bound with a repositoryId.
     * If so, treat the composite as intended for R-API (no pendingArchive/pendingDelete states).
     */
    get _isRepoComposite(): boolean {
        return !!this._core._compositeRepositoryId;
    }

    /**
     * @private
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get _collaborationType(): string | undefined {
        return this._data.local ? this._data.local.collaborationType : undefined;
    }
    set _collaborationType(value: string | undefined) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else {
            if (value) {
                core._local().collaborationType = value;
            } else {
                delete core._local().collaborationType;
            }
            core._setDirty(true);
        }
    }

    /**
     * Allow clients to set an arbitrary string containing local data to store on the branch. This is private because you
     * should always go through the DCXComposite API.
     * @private
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get _clientDataString() {
        return this._data.local ? this._data.local.clientDataString : undefined;
    }
    set _clientDataString(value) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else {
            if (value) {
                core._local().clientDataString = value;
            } else {
                delete core._local().clientDataString;
            }
            core._setDirty(true);
        }
    }

    /**
     * The etag of the manifest of the composite.
     * @memberof AdobeDCXBranch#
     * @type {String}
     * @readonly
     */
    get manifestEtag(): string | undefined {
        return this._data.local ? this._data.local.manifestEtag : undefined;
    }
    set manifestEtag(etag: string | undefined) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof etag !== 'string' && typeof etag !== 'undefined') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string or undefined.');
        } else {
            core._local().manifestEtag = etag;
            core._setDirty(true); // preserve compositeState!
        }
    }

    /**
     * The editing state of the composite. Can be <em>'unmodified'</em>, <em>'modified'</em>,
     * <em>'pendingDelete'</em> or <em>'committedDelete'</em>.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get compositeState(): CompositeState {
        return this._data.state || COMPOSITE_STATES.unmodified;
    }
    set compositeState(state: CompositeState) {
        const core = this._core;

        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof state !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else if (!Object.values(COMPOSITE_STATES).includes(state)) {
            throw new DCXError(
                DCXError.INVALID_PARAMS,
                'State must be "modified", "unmodified", "pendingDelete", or "committedDelete".',
            );
        } else if (
            this._isRepoComposite &&
            (state === COMPOSITE_STATES.pendingDelete || state === COMPOSITE_STATES.committedDelete)
        ) {
            // cannot use 2 step delete for R-API composites
            throw new DCXError(
                DCXError.INVALID_STATE,
                'R-API composites must be deleted in one step using RepoAPISession.',
            );
        } else if (
            (!this._isRepoComposite &&
                (state === COMPOSITE_STATES.pendingDelete || state === COMPOSITE_STATES.committedDelete) &&
                this.compositeArchivalState === COMPOSITE_ARCHIVAL_STATES.pending) ||
            this.compositeArchivalState === COMPOSITE_ARCHIVAL_STATES.archived
        ) {
            throw new DCXError(DCXError.INVALID_STATE, 'Cannot delete an archived composite.');
        } else {
            this._data.state = state;
            core._setDirty(true); // preserve compositeState!
        }
    }

    /**
     * The archival state of the composite. Can be <em>'active'</em> (default), <em>'pending'</em>,
     * or <em>'archived'</em>.
     * @memberof AdobeDCXBranch#
     * @type {String}
     */
    get compositeArchivalState(): CompositeArchivalState {
        return this._core._local().archivalState || COMPOSITE_ARCHIVAL_STATES.active;
    }
    set compositeArchivalState(state: CompositeArchivalState) {
        const core = this._core;
        if (core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        } else if (typeof state !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Expecting a string.');
        } else if (!Object.values(COMPOSITE_ARCHIVAL_STATES).includes(state)) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'State must be "pending", "archived", or "active".');
        } else if (this._isRepoComposite && state === COMPOSITE_ARCHIVAL_STATES.pending) {
            throw new DCXError(
                DCXError.INVALID_STATE,
                'R-API composites must be discarded in one step using RepoAPISession#discardAsset.',
            );
        } else if (this.compositeArchivalState === COMPOSITE_ARCHIVAL_STATES.archived) {
            throw new DCXError(DCXError.INVALID_STATE, 'Composite has already been archived.');
        } else if (
            state !== COMPOSITE_ARCHIVAL_STATES.active &&
            (this.compositeState === COMPOSITE_STATES.pendingDelete ||
                this.compositeState === COMPOSITE_STATES.committedDelete)
        ) {
            throw new DCXError(DCXError.INVALID_STATE, 'Cannot archive a deleted composite.');
        } else {
            if (state === COMPOSITE_ARCHIVAL_STATES.active) {
                delete this._core._local().archivalState;
            } else {
                this._core._local().archivalState = state as 'archived' | 'pending';
            }
            core._setDirty(true); // preserve compositeState!
        }
    }

    /**
     * Whether the composite is bound to a composite on a server. If <em>false</em> for newly
     * created empty composites that have never been pulled from or pushed to the server.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Boolean}
     */
    get isBound() {
        return !!(this.manifestEtag && this.compositeAssetId);
    }
    set isBound(_: boolean) {
        throw new DCXError(DCXError.READ_ONLY, 'Cannot set the property isBound. Use resetIdentity instead.');
    }

    /**
     * The version identifier of the banch.
     * Only valid if the branch has been pulled or pushed, undefined otherwise.
     *
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {String}
     */
    get versionId(): string | undefined {
        return this._versionId;
    }
    set versionId(versionId: string | undefined) {
        if (this._core._readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        this._versionId = versionId;
    }

    /**
     * @private
     * @memberof AdobeDCXBranch#
     * @type {Boolean}
     */
    get _isDirty() {
        return this._core._isDirty;
    }
    set _isDirty(value) {
        this._core._isDirty = value;
    }

    /**
     * Whether the composite has has been modified in memory and needs to be committed to local
     * storage.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Boolean}
     */
    get isDirty() {
        return this._core.isDirty;
    }
    set isDirty(value) {
        this._core.isDirty = value;
    }

    /**
     * @memberof AdobeDCXBranch#
     * @internal
     */
    get localData() {
        return this._core._stringify(false, true);
    }
    set localData(_) {
        throw new DCXError(DCXError.READ_ONLY, 'Cannot set the property localData.');
    }

    /**
     * @memberof AdobeDCXBranch#
     * @internal
     */
    get remoteData() {
        return this._core._stringify(true);
    }
    set remoteData(_) {
        throw new DCXError(DCXError.READ_ONLY, 'Cannot set the property remoteData.');
    }

    /**
     * Lists any elements (type AdobeDCXElement) that have been created and have not yet
     * been used to update the corresponding child node or been abandoned.
     * @memberof AdobeDCXBranch#
     * @readonly
     * @type {Array}
     */
    get pendingElements(): AdobeDCXElement[] {
        return this._pendingElements;
    }
    set pendingElements(_) {
        throw new DCXError(DCXError.READ_ONLY, 'Cannot set the property pendingElements.');
    }

    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    get changeCount() {
        return this._core.changeCount;
    }
    set changeCount(value) {
        this._core.changeCount = value;
    }

    /**
     * @memberof AdobeDCXBranch#
     * @private
     */
    get readOnly() {
        return this._core._readOnly;
    }
    set readOnly(value) {
        this._core._readOnly = value;
    }

    /**
     * @internal
     */
    static _newBranchAsCopyOfCore(coreToCopy: AdobeDCXBranchCore): AdobeDCXBranch {
        dbg('_newBranchAsCopyOfCore()');

        const data = JSON.parse(JSON.stringify(coreToCopy._data));
        const branch = new AdobeDCXBranch(data);
        const core = branch._core;
        core._sourceCompositeInfo = {
            links: coreToCopy._compositeLinks as LinkSet,
            repositoryId: coreToCopy._compositeRepositoryId as string,
            assetId: coreToCopy._compositeAssetId as string,
        };

        branch._resetIdentity(function (resetComponent) {
            const origComponent = coreToCopy.getComponentWithId(resetComponent.id) as DCXComponent;
            if (!origComponent) {
                throw new DCXError(DCXError.INVALID_STATE, 'Could not find original component.');
            }
            if (origComponent.state === COMPOSITE_STATES.modified || !origComponent.etag) {
                throw new DCXError(
                    DCXError.INVALID_STATE,
                    'Cannot create a new composite from a branch that contains modified or unbound components.',
                );
            }
            const origAssetId = origComponent._owner!.compositeAssetId;
            if (!origAssetId) {
                throw new DCXError(
                    DCXError.INVALID_STATE,
                    'Cannot make a copy of component of a composite that has ' +
                        'not been bound with an asset id. Component id =' +
                        origComponent.id,
                );
            }

            const origRepositoryId = origComponent._owner!.compositeRepositoryId;

            dbg('_nBACOC() origAssetId', origAssetId);
            dbg('_nBACOC() origRepositoryId', origRepositoryId);
            dbg('_nBACOC() resetComponent', resetComponent);

            // For each component in the reset branch we set its source asset info
            // to reflect the original component
            core._setSourceAssetInfoOfComponent(
                {
                    compositeAssetId: origAssetId,
                    componentId: origComponent.id,
                    componentVersion: origComponent.version as string,
                    componentPath: origComponent.absolutePath as string,
                    repositoryId: origRepositoryId,
                },
                resetComponent,
            );
        });

        return branch;
    }

    //******************************************************************************
    // Persistence
    //******************************************************************************

    /**
     * Restores the branch from its stringified representation (manifest). Throws if data is invalid.
     * @internal
     * @param   {String} data The stringified representation.
     * @returns {AdobeDCXBranch}.
     */
    public parse(data: string): AdobeDCXBranch {
        // verify params
        if (typeof data !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Param "data" must be a string');
        }
        let newData: AdobeDCXRootNodeData & AdobeDCXLocalNodeData;

        // parse the string
        try {
            newData = JSON.parse(data);
        } catch (e) {
            throw new DCXError(
                DCXError.INVALID_JSON,
                'Manifest could not be parsed. Underlying error: ' + (e as Error).message,
                e as Error,
            );
        }

        // set the new data
        return this._setData(
            newData,
            this._core._readOnly,
            this._core._compositeAssetId,
            this._core._compositeRepositoryId,
        );
    }

    //******************************************************************************
    // Elements
    //******************************************************************************

    /**
     * Returns the element with the given id or undefined if not found.
     * Throws if the node in question is not a valid element).
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @param   {String } id The id of the child node to look up.
     * @returns {AdobeDCXElement}
     */
    public getElementWithId(id: string): AdobeDCXElement | undefined {
        return this._createElement(this._core.getChildWithId(id) as DCXNode);
    }

    /**
     * Returns the element with the given absolute path or undefined if not found.
     * Throws if the node in question is not a valid element).
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @memberof AdobeDCXBranch#
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXElement}
     */
    public getElementWithAbsolutePath(path: string): AdobeDCXElement | undefined {
        return this._createElement(this._core.getChildWithAbsolutePath(path) as DCXNode);
    }

    /**
     * Creates a new element node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new element.
     * Throws if the path is invalid.
     * Notice that the branch keeps track of pending element objects. You must either call
     * updateElement or abandonElement when you are done with it so that the branch can update
     * its data and perform any necessary clean up tasks.
     * @param   {String}                name            - The name of the new element.
     * @param   {String}                type            - The type of the new element.
     * @param   {String}                path            - The path of the new element.
     * @param   {String}                [nodeId]        - The id of the new child. If undefined the new child node will
     *                                                    get a random id.
     * @param   {Integer}               [index]         - If given and less than or equal to the current number of
     *                                                    children than the node gets inserted at the given index.
     *                                                    Otherwise it gets added to the end.
     * @param   {@link AdobeDCXNode}    [parentNode]    - The parent node to add the node to. Default parent is the
     *                                                    root node.
     * @returns {AdobeDCXElement}
     */
    public addElement(
        name: string,
        type: string,
        path: string,
        nodeId?: string,
        index?: number,
        parentNode?: AdobeDCXNode,
    ): AdobeDCXElement {
        if (typeof name !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Element must have a name of type string');
        }
        if (typeof type !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Element must have a type of type string');
        }
        if (typeof path !== 'string') {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Element must have a path of type string');
        }
        const node = this._core.addChild(name, nodeId, index, parentNode);
        try {
            // Using try-catch because setting the path of the node can fail.
            node.path = path;
        } catch (x) {
            // Need to remove the node from the branch.
            this._core.removeChild(node);
            // Now we can safely re-throw the exception.
            throw x;
        }
        node._data.type = type;
        return this._createElement(node);
    }

    /**
     * Updates the data of the existing element in the branch.
     * Notice that the element object will no longer be tracked by the branch after updateElement
     * has been called. You will have to request the element again using getElementWithId or
     * getElementWithAbsolutePath if you want to keep working with it.
     * Throws if the element doesn't exist or if the update results in duplicate paths/ids.
     *
     * @param {AdobeDCXElement} element The modified element.
     * @returns {AdobeDCXNode} The updated node in the branch.
     */
    public updateElement(element: AdobeDCXElement): AdobeDCXNode {
        const elementNode = this.getChildWithId(element.rootNode.id);
        if (!elementNode) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Element does not exist in branch.');
        }
        const node = this._core.replaceChild(element.rootNode, elementNode.path);
        this._deleteElement(element);
        return node;
    }

    /**
     * Cleans up an element that was previously created and has not been used to update the corresponding
     * node in the branch. Throws if the branch doesn't recognize the element (e.g. calling abandonElement
     * twice or after already having called updateElement).
     *
     * @param {AdobeDCXElement} element The element that is no longer needed.
     */
    public abandonElement(element: AdobeDCXElement) {
        this._deleteElement(element);
    }

    //******************************************************************************
    // Children (Nodes)
    //******************************************************************************

    /**
     * Returns the node with the given id or undefined if not found.
     * @param   {String } id The id of the child node to look up.
     * @returns {AdobeDCXNode}
     */
    public getChildWithId(id: string): AdobeDCXNode | undefined {
        return this._core.getChildWithId(id);
    }

    /**
     * Returns the node with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path.
     * @returns {AdobeDCXNode}
     */
    public getChildWithAbsolutePath(path: string): AdobeDCXNode | undefined {
        return this._core.getChildWithAbsolutePath(path);
    }

    /**
     * Generates and returns an array of the child nodes of the given parent node.
     * @example
     * var rootNodes = branch.getChildrenOf(branch.rootNode);
     * @param   {AdobeDCXNode} parentNode The parent node to return the children for.
     * @returns {Array}
     */
    public getChildrenOf(parentNode?: AdobeDCXNode): AdobeDCXNode[] {
        return this._core.getChildrenOf(parentNode);
    }

    /**
     * Creates a new node and inserts it into the children list of the given parent node or of the
     * root if no parent node is given.
     * Returns the new child node.
     * @param   {String}  [name]       The name of the new child. If undefined the child will not
     *                                 have a name.
     * @param   {String}  [nodeId]     The id of the new child. If undefined the new child node will
     *                                 get a random id.
     * @param   {Integer} [index]      If given and less than or equal to the current number of
     *                                 children than the node gets inserted at the given index.
     *                                 Otherwise it gets added to the end.
     * @param   {String}  [parentNode] The parent node to add the node to. Default parent is the
     *                                 root node.
     * @returns {AdobeDCXNode}
     */
    public addChild(name?: string, nodeId?: string, index?: number, parentNode?: AdobeDCXNode): AdobeDCXNode {
        return this._core.addChild(name, nodeId, index, parentNode);
    }

    /**
     * Removes and returns the given child node from the branch.
     * @param   {AdobeDCXNode} node The child node to remove.
     * @returns {AdobeDCXNode}
     */
    public removeChild(node: AdobeDCXNode): AdobeDCXNode {
        return this._core.removeChild(node);
    }

    /**
     * Moves the existing child from its current parent/index to the given parent/index.
     * @param   {AdobeDCXNode} node         The child node to move
     * @param   {Integer}      [index]      If given and less than or equal to the current number of
     *                                      children than the node gets inserted at the given index.
     *                                      Otherwise it gets added to the end.
     * @param   {AdobeDCXNode} [parentNode] The parent node to move the node to. Default parent is
     *                                      the root.
     * @returns {AdobeDCXNode}
     */
    public moveChild(
        node: AdobeDCXNode,
        index: number | null = null,
        parentNode: AdobeDCXNode | null = null,
    ): AdobeDCXNode {
        return this._core.moveChild(node, index, parentNode);
    }

    /**
     * Copies the given child node as a new node into this branch. The node can be from the same or
     * from a different composite.
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Fails if a node with the same id or same absolute path already exists.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two composites stored at different endpoints.</p>
     *
     * @param   {AdobeDCXNode} node         The child node to copy. If it is the root node then
     *                                      newPath must be provided.
     * @param   {AdobeDCXNode} [parentNode] The parent node to copy the child node to. If undefined
     *                                      then the new child node will be added to the root of the
     *                                      branch.
     * @param   {Integer}      [index]      If provided and less than or equal to the current number of
     *                                      children of the parentNode (or root) the child node gets
     *                                      inserted at the given index. Otherwise it gets added to
     *                                      the end.
     * @param   {String}       [newPath]    <p>If provided, the copy of the child node will be assigned
     *                                      this a its path property and it will also receive a new
     *                                      random id (unless one is provided with the newId param).
     *                                      If left undefined then the copy of the node will keep
     *                                      the path of the original. In either case the function will
     *                                      fail if the resulting absolute path of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing absolute path.</p>
     *                                      <p>You must provide a newPath if you are copying the root
     *                                      node of a branch or element.</p>
     * @param   {String}       [newId]      If provided, the copy of the child node will be assigned
     *                                      this a its id. If left undefined (and the newPath param
     *                                      is also undefined) then the copy will retain the id of
     *                                      the original. In either case the function will
     *                                      fail if the resulting id of the child or any
     *                                      of its children/components conflicts with an already
     *                                      existing id.
     * @param   {NodeCallback} [callback]   Optional when not copying between different composites or
     *                                      when not using local storage.
     *                                      Gets called when the copy is done or has failed.
     * @returns {AdobeDCXNode}              Only returns the created child node if no callback is
     *                                      given.
     */
    public copyChild(
        node: AdobeDCXNode,
        parentNode?: AdobeDCXNode,
        index?: number,
        newPath?: string,
        newId?: string,
        callback?: undefined,
    ): AdobeDCXNode;
    public copyChild(
        node: AdobeDCXNode,
        parentNode: AdobeDCXNode | undefined,
        index: number | undefined,
        newPath: string | undefined,
        newId: string | undefined,
        callback: NodeCallback,
    ): void;
    public copyChild(
        node: AdobeDCXNode,
        parentNode?: AdobeDCXNode,
        index?: number,
        newPath?: string,
        newId?: string,
        callback?: NodeCallback,
    ): AdobeDCXNode | void {
        return this._core.copyChild(node, parentNode, index, newPath, newId, callback as NodeCallback);
    }

    /**
     * <p>Replaces the child node in this branch with a copy of the given node or branch with the same
     * id (or, if provided, with the given id). Fails if the child node does not exist in this branch.</p>
     *
     * <p>This function will try reuse the ids of any children and components of the copied node,
     * in order to minimize the amount of data that will later have to be uploaded, however, clients
     * must not rely on these ids being preserved in the copied objects.</p>
     *
     * <p>Notice: This method does not work without local storage (e.g. browser environment) if
     * used to copy between two different composites.</p>
     *
     * @param   {AdobeDCXNode} node       The child node to update from.
     * @param   {String}       [newPath]  If provided the copy of the component will be assigned
     *                                    this a its path property. Otherwise it will retain its original path.
     * @param   {String}       [newId]    If provided the copy of the child node will be assigned
     *                                    this as its id. Otherwise it will retain its original id.
     * @param   {NodeCallback} [callback] Optional when not copying between different composites or
     *                                    when not using local storage.
     *                                    Gets called when the copy is done or has failed.
     * @returns {AdobeDCXNode}            Only returns the created child node if no callback is
     *                                    given.
     */
    public replaceChild(node: AdobeDCXNode, newPath?: string, newId?: string): AdobeDCXNode;
    public replaceChild(
        node: AdobeDCXNode,
        newPath: string | undefined,
        newId: string | undefined,
        callback: NodeCallback,
    ): void;
    public replaceChild(
        node: AdobeDCXNode,
        newPath?: string,
        newId?: string,
        callback?: NodeCallback,
    ): AdobeDCXNode | void {
        return this._core.replaceChild(node, newPath, newId, callback as NodeCallback);
    }

    //******************************************************************************
    // Components
    //******************************************************************************

    /**
     * Returns an array of all components in the branch.
     * @returns {Array}.
     */
    public allComponents(): AdobeDCXComponent[] {
        return this._core.allComponents();
    }

    /**
     * Returns the component with the given id or undefined if not found.
     * @param   {String} id The id of the component to look up.
     * @returns {AdobeDCXComponent}
     */
    public getComponentWithId(id: string): AdobeDCXComponent | undefined {
        return this._core.getComponentWithId(id);
    }

    /**
     * Returns the component with the given absolute path or undefined if not found.
     * @param   {String} path The absolute path of the desired component.
     * @returns {AdobeDCXComponent}
     */
    public getComponentWithAbsolutePath(path: string): AdobeDCXComponent | undefined {
        return this._core.getComponentWithAbsolutePath(path);
    }

    /**
     * Returns an array containing the components of the given node.
     * @param   {AdobeDCXNode} parentNode The node whose components to return.
     * @returns {Array}
     */
    public getComponentsOf(parentNode?: AdobeDCXNode): AdobeDCXComponent[] {
        return this._core.getComponentsOf(parentNode);
    }

    /**
     * Given an INCOMPLETE_COMPOSITE DCXError, this will attempt to return the invalid components so they can be removed or updated
     * @param error An IMCOMPLETE_COMPOSITE error
     * @returns Array of components from this branch that are declared missing in the INCOMPLETE_COMPOSITE error report
     */
    public getMissingComponentsFromError(error: AdobeDCXError): AdobeDCXComponent[] {
        return this._core.getMissingComponentsFromError(error);
    }

    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param   {String}            name          The name of the new component.
     * @param   {String}            relationship  The relationship of the new component.
     * @param   {String}            path          The path of the new component. Must satisfy uniquenes
     *                                            rules for components.
     * @param   {AdobeDCXNode}      [parentNode]  The node to add the node to. Defaults to the root.
     * @param   {Object}            uploadResults The upload results object returned by a previous call
     *                                            to AdobeDCXCompositeXfer.uploadAssetForNewComponent().
     * @returns {AdobeDCXComponent}               The new component.
     */
    public addComponentWithUploadResults(
        name: string,
        relationship: string,
        path: string,
        parentNode: AdobeDCXNode | undefined = undefined,
        uploadResults: AdobeUploadResults,
    ) {
        return this._core.addComponentWithUploadResults(name, relationship, path, parentNode, uploadResults);
    }

    /**
     * <p>Creates and adds a component to the given parent node or to the root if no parent node is
     * given.</p>
     *
     * @param componentDescriptor The serialized component descriptor to use.
     * @param name          The name of the new component.
     * @param path          The path of the new component. Must satisfy uniquenes rules for components.
     * @param relationship  The relationship of the new component.
     * @param parentNode  The node to add the node to. Defaults to the root.
     * @returns {AdobeDCXComponent}               The new component.
     */
    public addComponentWithComponentDescriptor(
        componentDescriptor: string,
        name: string,
        path: string,
        relationship?: LinkRelationKey,
        parentNode?: AdobeDCXNode,
    ) {
        return this._core.addComponentWithComponentDescriptor(
            componentDescriptor,
            name,
            path,
            relationship,
            parentNode,
        );
    }

    /**
     * <strong>XHR-only</strong>
     *
     * <p>Updates the component record with the results of a recent upload of said component.</p>
     *
     * @param   {AdobeDCXComponent} component     The component.
     * @param   {Object}            uploadResults The upload results object returned by a previous
     *                                            call to AdobeDCXCompositeXfer.uploadAssetForComponent().
     * @returns {AdobeDCXComponent} The updated component.
     */
    public updateComponentWithUploadResults(component: AdobeDCXComponent, uploadResults: AdobeUploadResults) {
        return this._core.updateComponentWithUploadResults(component as DCXComponent, uploadResults);
    }

    /**
     * Removes the component from the branch.
     * @param   {AdobeDCXComponent} component The component to remove.
     * @returns {AdobeDCXComponent} The removed component.
     */
    public removeComponent(component: AdobeDCXComponent): AdobeDCXComponent {
        return this._core.removeComponent(component as DCXComponent);
    }

    /**
     * Moves the component to the given node or the root if node is undefined
     * @param   {AdobeDCXComponent} component    The component to move.
     * @param   {AdobeDCXNode}      [parentNode] The node to move the component to.
     * @returns {AdobeDCXComponent} The moved component.
     */
    public moveComponent(component: AdobeDCXComponent, parentNode?: AdobeDCXNode): AdobeDCXComponent {
        return this._core.moveComponent(component as DCXComponent, parentNode as DCXNode);
    }

    /**
     * Copies the given component and adds it as a new component to this branch. Fails if the
     * component already exists.
     *
     * <p>Notice: This method does not work without local storage (browser environment) if
     * used to copy between two composites with different endpoints.</p>
     *
     * @param   {AdobeDCXComponent} component    The component to copy.
     * @param   {AdobeDCXNode}      [parentNode] The node to copy the component to. If none is
     *                                           provided then the component will be added to the
     *                                           root.
     * @param   {String}            [newPath]    If provided the copy of the component will be
     *                                           assigned this a its path property and it will also
     *                                           get assigned a random new id if none is provided via
     *                                           the <em>newId</em> param.
     * @param   {String}            [newId]      If provided the copy of the component will be assigned
     *                                           this a its id. If left undefined (and if newPath is
     *                                           undefined as well) then the copy of the component
     *                                           will retain the id of the original.
     *                                           This is useful when merging conflicting changes since
     *                                           it preserves the identity of components and avoids
     *                                           unnecessary network traffic.
     * @param   {ComponentCallback} [callback]   Optional when not copying between different
     *                                           composites or when copying without local storage.
     *                                           Gets called when the copy is done or has failed.
     * @returns {AdobeDCXComponent}              Only returns the new component when called without
     *                                           a callback.
     */
    public copyComponent(
        component: AdobeDCXComponent,
        parentNode?: AdobeDCXNode,
        newPath?: string,
        newId?: string,
    ): AdobeDCXComponent;
    public copyComponent(
        component: AdobeDCXComponent,
        parentNode: AdobeDCXNode | undefined,
        newPath: string | undefined,
        newId: string | undefined,
        callback: ComponentCallback,
    ): void;
    public copyComponent(
        component: AdobeDCXComponent,
        parentNode?: AdobeDCXNode,
        newPath?: string,
        newId?: string,
        callback?: ComponentCallback,
    ): AdobeDCXComponent | void {
        return this._core.copyComponent(component as DCXComponent, parentNode as DCXNode, newPath, newId, callback);
    }

    /**
     * Replaces the matching component (same id) in this branch with a copy of the given component.
     * Fails if the component can't be found.
     *
     * <p>Notice: This method does not work without local storage (browser environment) if
     * used to copy between two different composites.</p>
     *
     * @param   {AdobeDCXComponent} component  The component to copy.
     * @param   {String}            [newPath]  If provided the copy of the component will be
     *                                         assigned this a its path property and it will also
     *                                         get assigned a new id if none is provided via the
     *                                         newId param.
     * @param   {String}            [newId]    If provided the copy of the component will be
     *                                         assigned this a its id property.
     * @param   {ComponentCallback} [callback] Optional when not copying between different
     *                                         composites. Gets called when the copy is done or has
     *                                         failed.
     * @returns {AdobeDCXComponent}            Only returns the new component when called without
     *                                         a callback.
     */
    public replaceComponent(component: AdobeDCXComponent, newPath?: string, newId?: string): AdobeDCXComponent;
    public replaceComponent(
        component: AdobeDCXComponent,
        newPath: string | undefined,
        newId: string | undefined,
        callback: ComponentCallback,
    ): void;
    public replaceComponent(
        component: AdobeDCXComponent,
        newPath?: string,
        newId?: string,
        callback?: ComponentCallback,
    ): AdobeDCXComponent | void {
        return this._core.replaceComponent(component as DCXComponent, newPath, newId, callback);
    }

    //******************************************************************************
    // Miscellaneous
    //******************************************************************************

    /**
     * Creates and returns a read-write deep copy of the branch. The copy uses the same
     * local storage as the original. Use this to create a mutable copy of an otherwise
     * read-only branch like e.g. pulled.
     * @internal
     * @returns {AdobeDCXBranch}
     */
    public copy(): AdobeDCXBranch {
        const copiedBranch = new AdobeDCXBranch(
            JSON.parse(JSON.stringify(this._data)),
            false,
            this._core._compositeAssetId,
            this._core._compositeRepositoryId,
        );
        copiedBranch._derivationType = this._derivationType;
        copiedBranch._derivationDatetime = this._derivationDatetime;
        if (this._core._sourceAssetInfoLookup) {
            const sourceLookupAsString = JSON.stringify(this._core._sourceAssetInfoLookup);
            copiedBranch._core._sourceAssetInfoLookup = JSON.parse(sourceLookupAsString);
        }
        return copiedBranch;
    }

    //******************************************************************************
    // Private
    //******************************************************************************

    /**
     * Only checks that the type is object.
     * Should NOT be used for verifying contents of the object, unless modified.
     *
     * @param o
     */
    private _isNodeLike(o: unknown): o is DCXNode {
        return isObject(o);
    }

    // elements
    /**
     * @private
     * @param {Object} node The node
     */
    private _createElement(node: undefined): void;
    private _createElement(node: DCXNode | DCXComponent): DCXElement;
    private _createElement(node?: DCXNode | DCXComponent): DCXElement | void {
        if (!node) {
            return;
        }

        // extract the data
        const data = JSON.parse(JSON.stringify(node._data));
        data['manifest-format-version'] = this._data['manifest-format-version'];
        delete data.path; // remove the path property

        // instantiate a new element object
        const newElement = new DCXElement(data, this, this._core._readOnly);
        this._pendingElements.push(newElement);

        return newElement;
    }

    /**
     * @private
     * @param {Object} element The element
     */
    private _deleteElement(element: AdobeDCXElement) {
        const index = this._pendingElements.indexOf(element as DCXElement);
        if (index < 0) {
            throw new DCXError(DCXError.INVALID_PARAMS, 'Unknown element.');
        }
        this._pendingElements.splice(index, 1);

        // TODO: Delete element manifest
    }

    /**
     * Returns an error if the passed in data is not a valid manifest. Returns null if everything is OK.
     * @private
     * @param   {Object} data The value object to verify.
     * @returns {Error}
     */
    private _verify(data: AdobeDCXRootNodeData) {
        if (typeof data.id !== 'string') {
            return new DCXError(DCXError.INVALID_DATA, 'Manifest is missing an id of type string');
        }
        if (typeof data.name !== 'string') {
            return new DCXError(DCXError.INVALID_DATA, 'Manifest is missing a name of type string');
        }
        if (typeof data.type !== 'string') {
            return new DCXError(DCXError.INVALID_DATA, 'Manifest is missing a type of type string');
        }
        if (typeof data['manifest-format-version'] !== 'number') {
            return new DCXError(DCXError.INVALID_DATA, 'Manifest is missing a manifest-format-version of type number');
        }

        return null;
    }

    /**
     * Verifies and sets the new data and creates the caches. Throws if invalid data.
     * @private
     * @param   {Object}         data Data
     * @param {Boolean} readOnly Whether the branch is read-only.
     * @param {String}  assetId.
     * @returns {AdobeDCXBranch} The branch.
     */
    private _setData(
        data: AdobeDCXRootNodeData & AdobeDCXLocalNodeData,
        readOnly?: boolean,
        assetId?: string,
        repositoryId?: string,
    ) {
        this.originalManifestFormatVersion = data['manifest-format-version'];
        convertDataIfNecessary(data);

        // Upgrade local data version if necessary
        if (data.local) {
            // We went back and forth regarding whether the href saved in the local section is
            // relative or absolute. Since it now can be either we do no longer convert the local
            // section but we should be setting the correct version here so that we can support
            // conversion in the future if it becomes necessary
            data.local.version = 2;
        }

        const err = this._verify(data);
        if (err === null) {
            // Ensure that we always write out the version that we understand:
            data['manifest-format-version'] = MANIFEST_FORMAT_VERSION;
            this._core = new AdobeDCXBranchCore(data, this, readOnly);
            this._core._compositeAssetId = assetId;
            this._core._compositeRepositoryId = repositoryId;

            this._data = data;

            return this;
        }
        throw err;
    }

    /**
     * Resets the binding and (optionally) the identity of the branch
     * @private
     * @param {Boolean} retainId
     */
    private _reset(retainId: boolean, componentCallback?: (resetComponent: AdobeDCXComponentData) => unknown) {
        if (this.readOnly) {
            throw new DCXError(DCXError.READ_ONLY, 'This object is read-only.');
        }
        if (this._data.local) {
            delete this._data.local.manifestEtag;
            delete this._core._compositeAssetId;
            delete this._core._compositeRepositoryId;
        }
        if (!retainId) {
            this._data.id = generateUuid();
        }

        this._data.state = COMPOSITE_STATES.modified;

        this._core._recursiveReset(this._data, componentCallback);

        this._core._setDirty();
    }

    /**
     * Resets the binding of the branch while retaining its id.
     * @internal
     */
    _resetBinding(componentCallback?: (resetComponent: AdobeDCXComponentData) => void) {
        this._reset(true, componentCallback);
    }

    /**
     * Resets the binding and ids of the branch.
     * @internal
     */
    _resetIdentity(componentCallback?: (resetComponent: AdobeDCXComponentData) => void) {
        this._reset(false, componentCallback);
    }

    /** @internal */
    _local() {
        return this._core._local();
    }

    /**
     * Sets the dirty flag and (optionally) the compositeState
     * @internal
     * @param {Boolean} preserveCompositeState
     */
    _setDirty(preserveCompositeState?: boolean) {
        this._core._setDirty(preserveCompositeState);
    }

    /**
     * Updates the path of a node. Caller must dirty the branch.
     * @internal
     * @param {Object} node
     * @param {String} newPath
     */
    _setPathOfNode(node: AdobeDCXNode, newPath?: string) {
        this._core._setPathOfNode(node as DCXNode, newPath);
    }

    /**
     * Verifies the integrity of the in-memory structures of the branch. Looks for incorrect caches/lookup tables, incorrect
     * object references and cycles/duplicate objects. Also, optionally, verifies that all asset files exist locally.
     * @internal
     * @param   {Boolean}  shouldBeComplete Whether to check for the existence of all component assets. @Warning: This check is using synchronous file system calls.
     * @param   {Function} logger           Optional. A function that gets called for every error found. Signature: function (string)
     * @param   {Object}   fs               Optional. The file system object to use in the shouldBeComplete check
     * @returns {Array}    Array of errors or null if everything is OK.
     */
    _verifyIntegrity(shouldBeComplete?: boolean, logger?: any, fs?: any) {
        return this._core._verifyIntegrity(shouldBeComplete, logger, fs);
    }
}

// conversion

function convertDataFrom5To6(data: AdobeDCXNodeData, isRoot = false) {
    // Version 6 requires that all component version properties are strings
    // We also ensure that we do not have a path property at the root
    // Notice that this function is recursive

    if (isRoot) {
        if (data.path) {
            delete data.path;
        }
    }

    const components = data.components;
    if (typeof components === 'object') {
        for (let i = 0; i < components.length; i++) {
            const component = components[i];
            if (typeof component.version === 'number') {
                component.version = (component.version as number).toString();
            }
        }
    }
    const children = data.children;
    if (typeof children === 'object') {
        for (let i = 0; i < children.length; i++) {
            const child = children[i];
            convertDataFrom5To6(child);
        }
    }
}

/**
 * @private
 * @param {Object} data Data
 */
function convertDataIfNecessary(data: AdobeDCXRootNodeData) {
    const version = data['manifest-format-version'];

    if (version < MANIFEST_FORMAT_VERSION) {
        if (version < 4) {
            throw new DCXError(
                DCXError.INVALID_DATA,
                'Encountered manifest format version ' +
                    data['manifest-format-version'] +
                    '. Format version conversion not yet implemented for that version.',
            );
        }
        if (version < 6) {
            convertDataFrom5To6(data, true);
        }
    }
}

export default AdobeDCXBranch;
